﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <nn/fssystem/save/fs_HierarchicalDuplexStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

// テストクラス
class FsHierarchicalDuplexStorageTest : public ::testing::Test
{
public:
    // メタ情報を作成します。
    static nn::Result CreateMetaInformation(
            nn::fssystem::save::HierarchicalDuplexMetaInformation* pMeta,
            const nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam& inputParam,
            int64_t sizeData
        ) NN_NOEXCEPT
    {
        return nn::fssystem::save::HierarchicalDuplexStorageControlArea::CreateMetaInformation(
            pMeta,
            inputParam,
            sizeData
        );
    }
};

// メタデータの正当性テスト
TEST_F(FsHierarchicalDuplexStorageTest, MetaData)
{
    static const int64_t Giga = 1024 * 1024 * 1024;
    static const int64_t SizeOriginalBody = 4 * Giga;

    for( uint32_t blockL1 = 128; blockL1 <= MaxBlockSize; blockL1 *= 2 )
    {
        for( uint32_t blockL2 = 128; blockL2 <= MaxBlockSize; blockL2 *= 2 )
        {
            nn::fssystem::save::HierarchicalDuplexMetaInformation meta;

            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
            inputParam.sizeBlockLevel[0] = blockL1;
            inputParam.sizeBlockLevel[1] = blockL2;

            NNT_ASSERT_RESULT_SUCCESS(
                CreateMetaInformation(&meta, inputParam, SizeOriginalBody));

            int64_t sizeBody = meta.infoDuplex.info[0].size * 2 +
                               meta.infoDuplex.info[1].size * 2 +
                               meta.infoDuplex.info[2].size * 2;

            // 大まかな値を計算
            const int64_t sizeL3 = SizeOriginalBody * 2;
            const int64_t sizeL2 = (((((sizeL3 / 2) + blockL2 - 1) / blockL2) + 7) / 8) * 2;
            const int64_t sizeL1 = (((((sizeL2 / 2) + blockL1 - 1) / blockL1) + 7) / 8) * 2;

            const int64_t paddingL1 = nn::fssystem::dbm::DuplexBitmap::QuerySize(blockL1 * 8);
            const int64_t paddingL2 = nn::fssystem::dbm::DuplexBitmap::QuerySize(blockL2 * 8);
            ASSERT_TRUE(
                (sizeBody >= sizeL3 + sizeL2 + sizeL1) &&
                (sizeBody <= sizeL3 + sizeL2 + sizeL1 + paddingL1 * 2 + paddingL2 * 2));

            // だいたい意図通りかチェック
            ASSERT_EQ(static_cast<int64_t>(0), meta.infoDuplex.info[0].offset);
            ASSERT_EQ(blockL1, 1 << meta.infoDuplex.info[1].orderBlock);
            ASSERT_EQ(blockL2, 1 << meta.infoDuplex.info[2].orderBlock);

            ASSERT_TRUE(
                meta.infoDuplex.info[0].size >= sizeL1 / 2 &&
                meta.infoDuplex.info[0].size < sizeL1 / 2 + paddingL1);
        }
    }
}

// 2重化されていることをテスト
TEST_F(FsHierarchicalDuplexStorageTest, MountAndUnmount)
{
    static const int64_t SizeOriginalBody = 64 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 128 * 1024;

    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
    inputParam.sizeBlockLevel[0] = 32;
    inputParam.sizeBlockLevel[1] = 64;

    // メタ作り
    nn::fssystem::save::HierarchicalDuplexMetaInformation meta;
    NNT_ASSERT_RESULT_SUCCESS(
        CreateMetaInformation(
            &meta,
            inputParam,
            SizeOriginalBody
        )
    );

    const int64_t sizeControl = sizeof(meta);
    const int64_t sizeBody = meta.infoDuplex.info[0].size * 2 +
                             meta.infoDuplex.info[1].size * 2 +
                             meta.infoDuplex.info[2].size * 2;

    // 管理領域に使うメモリファイルの作成
    nnt::fs::util::SafeMemoryStorage baseStorageControl;
    baseStorageControl.Initialize(sizeControl);

    // 本体に使うメモリファイルの作成
    nnt::fs::util::SafeMemoryStorage baseStorageBody;
    baseStorageBody.Initialize(sizeBody);

    std::memset(baseStorageBody.GetBuffer(), 0, static_cast<size_t>(sizeBody));

    // 管理領域はフォーマット
    nn::fssystem::save::HierarchicalDuplexStorageControlArea controlArea;
    NNT_ASSERT_RESULT_SUCCESS(
        controlArea.Format(
            nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
            inputParam,
            SizeOriginalBody
        )
    );

    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);

    // 中間バッファ作り
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(&bufCache[0]),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    // 排他制御オブジェクト
    nn::os::Mutex locker(true);

    // ストレージ
    int64_t offset = 0;
    nn::fs::SubStorage storageBitmapMasterA(
        &baseStorageBody, offset, meta.infoDuplex.info[0].size);
    offset += meta.infoDuplex.info[0].size;
    nn::fs::SubStorage storageBitmapMasterB(
        &baseStorageBody, offset, meta.infoDuplex.info[0].size);
    offset += meta.infoDuplex.info[0].size;
    nn::fs::SubStorage storageBitmapL1A(
        &baseStorageBody, offset, meta.infoDuplex.info[1].size);
    offset += meta.infoDuplex.info[1].size;
    nn::fs::SubStorage storageBitmapL1B(
        &baseStorageBody, offset, meta.infoDuplex.info[1].size);
    offset += meta.infoDuplex.info[1].size;
    nn::fs::SubStorage storageDataA(
        &baseStorageBody, offset, meta.infoDuplex.info[2].size);
    offset += meta.infoDuplex.info[2].size;
    nn::fs::SubStorage storageDataB(
        &baseStorageBody, offset, meta.infoDuplex.info[2].size);
    offset += meta.infoDuplex.info[2].size;

    nnt::fs::util::Vector<char> tmp1(SizeOriginalBody);
    nnt::fs::util::Vector<char> tmp2(SizeOriginalBody);
    char* ptr1 = &tmp1[0];
    char* ptr2 = &tmp2[0];

    std::memset(ptr1, 0xAA, SizeOriginalBody);
    std::memset(ptr2, 0x00, SizeOriginalBody);

    // 本体はマウント
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                false,
                &cacheBuffer,
                &locker
            )
        );

        NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, 32));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, 32));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, 32));
    }

    // 同じ側をマウント
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                false,
                &cacheBuffer,
                &locker
            )
        );

        // 更新されていない
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, 32));
        std::memset(ptr1, 0x00, 32);
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, 32));

        // 書き込む
        std::memset(ptr1, 0xAA, SizeOriginalBody);
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, 8192));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, 8192));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, 8192));
    }

    // 次に違う側をマウント
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage;
        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                true,
                &cacheBuffer,
                &locker
            )
        );

        // 更新されている
        size_t sizeTest = 8192;
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, sizeTest));
        std::memset(ptr1, 0xAA, sizeTest);
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));

        // 書き込みチェック
        sizeTest = SizeOriginalBody / 2;
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, sizeTest));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, sizeTest));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));

        sizeTest = 64;
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(SizeOriginalBody - sizeTest, ptr1, sizeTest));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(SizeOriginalBody - sizeTest, ptr2, sizeTest));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));

        sizeTest = 150;
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(SizeOriginalBody - sizeTest, ptr1, sizeTest));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(SizeOriginalBody - sizeTest, ptr2, sizeTest));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));

        sizeTest = SizeOriginalBody;
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, sizeTest));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, sizeTest));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));

        std::memset(ptr1, 0x99, sizeTest);
        NNT_ASSERT_RESULT_SUCCESS(storage.Write(0, ptr1, sizeTest));
        NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, ptr2, sizeTest));
        ASSERT_EQ(0, std::memcmp(ptr1, ptr2, sizeTest));
    }
} // NOLINT(impl/function_size)

// 2重化ビットマップ部分のフォーマットテスト
TEST_F(FsHierarchicalDuplexStorageTest, FormatBitmap)
{
    static const int CacheCountMax = 2 * 1024;
    static const size_t CacheBlockSize = 4 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // フォーマットのテスト
    for( uint32_t blockL1 = 32; blockL1 <= 512; blockL1 *= 2 )
    {
        for( uint32_t blockL2 = 32; blockL2 <= 512; blockL2 *= 2 )
        {
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
            inputParam.sizeBlockLevel[0] = blockL1;
            inputParam.sizeBlockLevel[1] = blockL2;

            size_t sizeOriginalBody =
                std::uniform_int_distribution<size_t>(1, 32)(mt) * blockL2;
            nn::fssystem::save::HierarchicalDuplexMetaInformation meta;

            NNT_ASSERT_RESULT_SUCCESS(
                CreateMetaInformation(
                    &meta,
                    inputParam,
                    sizeOriginalBody
                )
            );

            const int64_t sizeControl = sizeof(meta);
            const int64_t sizeBody = meta.infoDuplex.info[0].size * 2 +
                                     meta.infoDuplex.info[1].size * 2 +
                                     meta.infoDuplex.info[2].size * 2;

            // 管理領域に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageControl;
            baseStorageControl.Initialize(sizeControl);

            std::memset(baseStorageControl.GetBuffer(), 0xFF, static_cast<size_t>(sizeControl));

            // 本体に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageBody;
            baseStorageBody.Initialize(sizeBody);

            std::memset(baseStorageBody.GetBuffer(), 0xFF, static_cast<size_t>(sizeBody));

            // 管理領域をフォーマット
            nn::fs::SubStorage storageControlArea(&baseStorageControl, 0, sizeControl);
            nn::fssystem::save::HierarchicalDuplexStorageControlArea controlAreaHieDuplex;
            NNT_ASSERT_RESULT_SUCCESS(
                controlAreaHieDuplex.Format(
                    storageControlArea,
                    inputParam,
                    sizeOriginalBody
                )
            );

            // キャッシュバッファマネージャを初期化します。
            const size_t cacheCount = std::uniform_int_distribution<>(8, 12)(mt);
            const size_t sizeBlock = GenerateRandomBlockSize(8 * 1024, MaxBlockSize);
            const size_t sizeCacheBuffer = sizeBlock * cacheCount;
            nnt::fs::util::Vector<char> bufCache(sizeCacheBuffer);

            nn::fssystem::FileSystemBufferManager cacheBuffer;
            NNT_ASSERT_RESULT_SUCCESS(
                cacheBuffer.Initialize(
                    CacheCountMax,
                    reinterpret_cast<uintptr_t>(&bufCache[0]),
                    sizeCacheBuffer,
                    CacheBlockSize
                )
            );

            // 排他制御オブジェクト
            nn::os::Mutex locker(true);

            // ストレージ
            int64_t offset = 0;
            nn::fs::SubStorage storageBitmapMasterA(
                &baseStorageBody, offset, meta.infoDuplex.info[0].size);
            offset += meta.infoDuplex.info[0].size;
            nn::fs::SubStorage storageBitmapMasterB(
                &baseStorageBody, offset, meta.infoDuplex.info[0].size);
            offset += meta.infoDuplex.info[0].size;
            nn::fs::SubStorage storageBitmapL1A(
                &baseStorageBody, offset, meta.infoDuplex.info[1].size);
            offset += meta.infoDuplex.info[1].size;
            nn::fs::SubStorage storageBitmapL1B(
                &baseStorageBody, offset, meta.infoDuplex.info[1].size);
            offset += meta.infoDuplex.info[1].size;

            const int64_t sizeBitmap = offset;

            nn::fs::SubStorage storageDataA(
                &baseStorageBody, offset, meta.infoDuplex.info[2].size);
            offset += meta.infoDuplex.info[2].size;
            nn::fs::SubStorage storageDataB(
                &baseStorageBody, offset, meta.infoDuplex.info[2].size);
            offset += meta.infoDuplex.info[2].size;

            // 二重化ビットマップをフォーマットします。
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalDuplexStorage::Format(
                    meta.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    &cacheBuffer
                )
            );

            // 二重化ビットマップ部分の値だけがフォーマットによってクリアされていることを確認
            char* ptr = reinterpret_cast<char*>(baseStorageBody.GetBuffer());
            for( int64_t i = 0; i < sizeBody; ++i )
            {
                if( i < sizeBitmap )
                {
                    ASSERT_EQ(0x00, static_cast<uint8_t>(ptr[i]));
                }
                else
                {
                    ASSERT_EQ(0xFF, static_cast<uint8_t>(ptr[i]));
                }
            }
        }
    }
}

// ストレージの読み書きテスト
TEST_F(FsHierarchicalDuplexStorageTest, ReadWrite)
{
    static const size_t TestSize = 8 * 1024 * 1024;
    static const size_t TestLoop = 10;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 4 * 1024 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);
    nnt::fs::util::Vector<char> readBuf(TestSize);
    nnt::fs::util::Vector<char> writeBuf(TestSize);
    nnt::fs::util::Vector<char> writeBuf2(TestSize);

    char* pReadBuf = &readBuf[0];
    char* pReadBuf2 = &readBuf[0];
    char* pWriteBuf = &writeBuf[0];
    char* pWriteBuf2 = &writeBuf2[0];

    for( uint32_t blockL1 = 128; blockL1 <= MaxBlockSize; blockL1 *= 2 )
    {
        NN_SDK_LOG(".");
        for( uint32_t blockL2 = 128; blockL2 <= MaxBlockSize; blockL2 *= 2 )
        {
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
            inputParam.sizeBlockLevel[0] = blockL1;
            inputParam.sizeBlockLevel[1] = blockL2;

            size_t sizeOriginalBody = std::uniform_int_distribution<>(1, 64)(mt) * blockL2;
            if( TestSize < sizeOriginalBody )
            {
                sizeOriginalBody = TestSize;
            }

            // メタ作り
            nn::fssystem::save::HierarchicalDuplexMetaInformation meta;
            NNT_ASSERT_RESULT_SUCCESS(
                CreateMetaInformation(
                    &meta,
                    inputParam,
                    sizeOriginalBody
                )
            );

            const int64_t sizeControl = sizeof(meta);
            const int64_t sizeBody = meta.infoDuplex.info[0].size * 2 +
                                     meta.infoDuplex.info[1].size * 2 +
                                     meta.infoDuplex.info[2].size * 2;

            // 管理領域に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageControl;
            baseStorageControl.Initialize(sizeControl);

            std::memset(baseStorageControl.GetBuffer(), 0xFF, static_cast<size_t>(sizeControl));

            // 本体に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageBody;
            baseStorageBody.Initialize(sizeBody);

            std::memset(baseStorageBody.GetBuffer(), 0xFF, static_cast<size_t>(sizeBody));

            // 管理領域をフォーマット
            nn::fssystem::save::HierarchicalDuplexStorageControlArea controlArea;
            NNT_ASSERT_RESULT_SUCCESS(
                controlArea.Format(
                    nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
                    inputParam,
                    sizeOriginalBody
                )
            );

            // キャッシュバッファマネージャを初期化
            nn::fssystem::FileSystemBufferManager cacheBuffer;
            NNT_ASSERT_RESULT_SUCCESS(
                cacheBuffer.Initialize(
                    CacheCountMax,
                    reinterpret_cast<uintptr_t>(&bufCache[0]),
                    CacheBufferSize,
                    CacheBlockSize
                )
            );

            // 排他制御オブジェクト
            nn::os::Mutex locker(true);

            // ストレージ
            int64_t storageOffset = 0;
            nn::fs::SubStorage storageBitmapMasterA(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[0].size);
            storageOffset += meta.infoDuplex.info[0].size;
            nn::fs::SubStorage storageBitmapMasterB(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[0].size);
            storageOffset += meta.infoDuplex.info[0].size;
            nn::fs::SubStorage storageBitmapL1A(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[1].size);
            storageOffset += meta.infoDuplex.info[1].size;
            nn::fs::SubStorage storageBitmapL1B(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[1].size);
            storageOffset += meta.infoDuplex.info[1].size;
            nn::fs::SubStorage storageDataA(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[2].size);
            storageOffset += meta.infoDuplex.info[2].size;
            nn::fs::SubStorage storageDataB(
                &baseStorageBody, storageOffset, meta.infoDuplex.info[2].size);
            storageOffset += meta.infoDuplex.info[2].size;

            // 二重化ビットマップをフォーマットします。
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalDuplexStorage::Format(
                    meta.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    &cacheBuffer
                )
            );

            bool isSelectBit = false;

            for( int32_t m = 0; m < TestLoop; m++ )
            {
                // 二重化ファイルをマウントします。
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage1;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage1.Initialize(
                            meta.infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    if( m == 0 )
                    {
                        std::memset(pWriteBuf2, 0xFF, sizeOriginalBody);
                        NNT_ASSERT_RESULT_SUCCESS(
                            storage1.Write(0, pWriteBuf2, sizeOriginalBody));
                    }

                    // ランダムなオフセット、サイズで読み書きを行います。
                    for( int32_t i = 0; i < TestLoop; ++i )
                    {
                        const int64_t offset =
                            std::uniform_int_distribution<int64_t>(0, sizeOriginalBody - 1)(mt);
                        const size_t size =
                            std::uniform_int_distribution<size_t>(
                                1, sizeOriginalBody - static_cast<size_t>(offset))(mt);

                        std::memset(pWriteBuf, (i + 1) & 0xFF, size);

                        // データを書き込みます。
                        ASSERT_TRUE(offset + size < TestSize);
                        std::memcpy(pWriteBuf2 + offset, pWriteBuf, size);
                        NNT_ASSERT_RESULT_SUCCESS(storage1.Write(offset, pWriteBuf, size));

                        if( std::uniform_int_distribution<>(0, 7)(mt) == 0 )
                        {
                            // データを読み込んでベリファイします。
                            NNT_ASSERT_RESULT_SUCCESS(storage1.Read(offset, pReadBuf, size));
                            if( 0 != std::memcmp(pReadBuf, pWriteBuf, size) )
                            {
                                uint32_t count = 0;

                                NN_SDK_LOG(
                                    "%u, size=%u, L1L2=%d, OrgBody=%u loop=%d\n",
                                    static_cast<uint32_t>(offset),
                                    static_cast<uint32_t>(size),
                                    blockL1,
                                    static_cast<uint32_t>(sizeOriginalBody),
                                    i
                                );

                                for( uint32_t j = 0; (j < size) && (count < 30); ++j )
                                {
                                    if( pReadBuf[j] != pWriteBuf[j] )
                                    {
                                        NN_SDK_LOG(
                                            "[%u]: %d -> %d\n",
                                            static_cast<uint32_t>(offset + j),
                                            static_cast<int32_t>(pWriteBuf[j]),
                                            static_cast<int32_t>(pReadBuf[j])
                                        );

                                        count++;
                                    }
                                }
                                ASSERT_TRUE(false);
                            }
                        }
                    }

                    //  元のデータが書き換わらないことを確認するために保存しておきます。
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage1.Read(0, pReadBuf2, sizeOriginalBody)
                    );
                }

                // 書き込み先を示すビット変えて再度二重化ファイルをマウントします。
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage2;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage2.Initialize(
                            meta.infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            !isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    // ランダムなオフセット、サイズで読み書きを行います。
                    for( int32_t i = 0; i < TestLoop; i++ )
                    {
                        const int64_t offset =
                            std::uniform_int_distribution<int64_t>(0, sizeOriginalBody - 1)(mt);
                        const size_t size =
                            std::uniform_int_distribution<size_t>(
                                1, sizeOriginalBody - static_cast<size_t>(offset))(mt);

                        std::memset(pWriteBuf, (i + 1) & 0xFF, size);

                        // データを書き込みます。
                        ASSERT_TRUE(offset + size < TestSize);
                        std::memcpy(pWriteBuf2 + offset, pWriteBuf, size);
                        NNT_ASSERT_RESULT_SUCCESS(storage2.Write(offset, pWriteBuf, size));

                        if( std::uniform_int_distribution<>(0, 7)(mt) == 0 )
                        {
                            // データを読み込んでベリファイします。
                            NNT_ASSERT_RESULT_SUCCESS(storage2.Read(offset, pReadBuf, size));

                            if( 0 != std::memcmp(pReadBuf, pWriteBuf, size) )
                            {
                                uint32_t count = 0;
                                NN_SDK_LOG(
                                    "%u, size=%u, L1L2=%d, OrgBody=%u loop=%d\n",
                                    static_cast<uint32_t>(offset),
                                    static_cast<uint32_t>(size),
                                    blockL1,
                                    static_cast<uint32_t>(sizeOriginalBody),
                                    i
                                );

                                for( uint32_t j = 0; (j < size) && (count < 20); j++ )
                                {
                                    if( pReadBuf[j] != pWriteBuf[j] )
                                    {
                                        NN_SDK_LOG(
                                            "[%u]: %d -> %d\n",
                                            static_cast<uint32_t>(offset + j),
                                            static_cast<int32_t>(pWriteBuf[j]),
                                            static_cast<int32_t>(pReadBuf[j])
                                        );

                                        count++;
                                    }
                                }
                                ASSERT_TRUE(false);
                            }
                        }
                    }
                }

                {
                    // 書き込み先を示すビットを戻して、以前のデータが
                    // 書き換わっていないことを確認します。
                    nn::fssystem::save::HierarchicalDuplexStorage storage3;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage3.Initialize(
                            meta.infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    NNT_ASSERT_RESULT_SUCCESS(
                        storage3.Read(0, pReadBuf, sizeOriginalBody)
                    );
                    ASSERT_EQ(
                        0,
                        std::memcmp(pReadBuf2, pReadBuf, sizeOriginalBody)
                    );
                }
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// 4 GB を超えるオフセットの読み書きテスト
TEST_F(FsHierarchicalDuplexStorageTest, ReadWriteLarge)
{
    static const int64_t TestSize = static_cast<int64_t>(64) * 1024 * 1024 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 4 * 1024 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);

    const uint32_t blockL1 = 16 * 1024;
    const uint32_t blockL2 = 16 * 1024;
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
    inputParam.sizeBlockLevel[0] = blockL1;
    inputParam.sizeBlockLevel[1] = blockL2;

    int64_t sizeOriginalBody = TestSize;

    // メタ作り
    nn::fssystem::save::HierarchicalDuplexMetaInformation meta;
    NNT_ASSERT_RESULT_SUCCESS(
        CreateMetaInformation(
            &meta,
            inputParam,
            sizeOriginalBody
        )
    );

    const int64_t sizeControl = sizeof(meta);
    const int64_t sizeBody = meta.infoDuplex.info[0].size * 2 +
                             meta.infoDuplex.info[1].size * 2 +
                             meta.infoDuplex.info[2].size * 2;

    // 管理領域に使うメモリファイルの作成
    nnt::fs::util::SafeMemoryStorage baseStorageControl;
    baseStorageControl.Initialize(sizeControl);

    std::memset(baseStorageControl.GetBuffer(), 0xFF, static_cast<size_t>(sizeControl));

    // 本体に使うメモリファイルの作成
    nnt::fs::util::VirtualMemoryStorage baseStorageBody;
    baseStorageBody.Initialize(sizeBody);

    // 管理領域をフォーマット
    nn::fssystem::save::HierarchicalDuplexStorageControlArea controlArea;
    NNT_ASSERT_RESULT_SUCCESS(
        controlArea.Format(
            nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
            inputParam,
            sizeOriginalBody
        )
    );

    // キャッシュバッファマネージャを初期化
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(
            CacheCountMax,
            reinterpret_cast<uintptr_t>(&bufCache[0]),
            CacheBufferSize,
            CacheBlockSize
        )
    );

    // 排他制御オブジェクト
    nn::os::Mutex locker(true);

    // ストレージ
    int64_t storageOffset = 0;
    nn::fs::SubStorage storageBitmapMasterA(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[0].size);
    storageOffset += meta.infoDuplex.info[0].size;
    nn::fs::SubStorage storageBitmapMasterB(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[0].size);
    storageOffset += meta.infoDuplex.info[0].size;
    nn::fs::SubStorage storageBitmapL1A(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[1].size);
    storageOffset += meta.infoDuplex.info[1].size;
    nn::fs::SubStorage storageBitmapL1B(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[1].size);
    storageOffset += meta.infoDuplex.info[1].size;
    nn::fs::SubStorage storageDataA(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[2].size);
    storageOffset += meta.infoDuplex.info[2].size;
    nn::fs::SubStorage storageDataB(
        &baseStorageBody, storageOffset, meta.infoDuplex.info[2].size);
    storageOffset += meta.infoDuplex.info[2].size;

    // 二重化ビットマップをフォーマットします。
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::HierarchicalDuplexStorage::Format(
            meta.infoDuplex,
            storageBitmapMasterA,
            storageBitmapMasterB,
            storageBitmapL1A,
            storageBitmapL1B,
            &cacheBuffer
        )
    );

    bool isSelectBit = false;

    static const int64_t OffsetList[] = {
        0,
        static_cast<int64_t>(4) * 1024 * 1024 * 1024,
        static_cast<int64_t>(8) * 1024 * 1024 * 1024,
        static_cast<int64_t>(16) * 1024 * 1024 * 1024,
        static_cast<int64_t>(32) * 1024 * 1024 * 1024,
    };
    static const size_t OffsetListLength = sizeof(OffsetList) / sizeof(int64_t);

    // バッファ初期化
    static const size_t AccessSize = 1024;
    std::unique_ptr<char> readBuffer(new char[AccessSize]);
    std::unique_ptr<char> writeBufferList1[OffsetListLength];
    for( auto& writeBuffer : writeBufferList1 )
    {
        writeBuffer.reset(new char[AccessSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), AccessSize);
    }
    std::unique_ptr<char> writeBufferList2[OffsetListLength];
    for( auto& writeBuffer : writeBufferList2 )
    {
        writeBuffer.reset(new char[AccessSize]);
        nnt::fs::util::FillBufferWithRandomValue(writeBuffer.get(), AccessSize);
    }

    // 二重化ファイルをマウントします。
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage1;
        NNT_ASSERT_RESULT_SUCCESS(
            storage1.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                isSelectBit,
                &cacheBuffer,
                &locker
            )
        );

        // 書き込み
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage1.Write(OffsetList[i], writeBufferList1[i].get(), AccessSize));
        }

        // 書き込んだデータが読めるかテスト
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage1.Read(OffsetList[i], readBuffer.get(), AccessSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList1[i].get(), readBuffer.get(), AccessSize);
        }
    }

    // ビットを反転させてマウントします。
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage2;
        NNT_ASSERT_RESULT_SUCCESS(
            storage2.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                !isSelectBit,
                &cacheBuffer,
                &locker
            )
        );

        // 前回書き込んだデータが読めるかテスト
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage2.Read(OffsetList[i], readBuffer.get(), AccessSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList1[i].get(), readBuffer.get(), AccessSize);
        }

        // 別のデータを書き込み
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage2.Write(OffsetList[i], writeBufferList2[i].get(), AccessSize));
        }

        // 書き込んだデータが読めるかテスト
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage2.Read(OffsetList[i], readBuffer.get(), AccessSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList2[i].get(), readBuffer.get(), AccessSize);
        }
    }

    // ビットを反転させずにマウントします。
    {
        nn::fssystem::save::HierarchicalDuplexStorage storage3;
        NNT_ASSERT_RESULT_SUCCESS(
            storage3.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                !isSelectBit,
                &cacheBuffer,
                &locker
            )
        );

        // 前回書き込んだデータではなくその前のデータが読めるかテスト
        for( size_t i = 0; i < OffsetListLength; ++i )
        {
            NNT_ASSERT_RESULT_SUCCESS(storage3.Read(OffsetList[i], readBuffer.get(), AccessSize));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBufferList1[i].get(), readBuffer.get(), AccessSize);
        }
    }
} // NOLINT(impl/function_size)

// フラッシュテスト
TEST_F(FsHierarchicalDuplexStorageTest, Flush)
{
    static const size_t TestSize = 1024 * 1024;
    static const uint32_t TestLoop = 10;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const int32_t CacheBufferSize = 4 * 1024 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];

    nn::os::Mutex locker(true);
    bool isSelectBit = false;

    for( int i = 0; i < TestLoop; ++i )
    {
        // キャッシュバッファマネージャを初期化
        nn::fssystem::FileSystemBufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&bufCache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );

        int64_t offsetBase = std::uniform_int_distribution<>(0, 127)(mt);
        size_t blockL1 = GenerateRandomBlockSize(4, MaxBlockSize);
        size_t blockL2 = GenerateRandomBlockSize(32, MaxBlockSize);

        // ストレージサイズをブロックサイズ単位に合わせる
        size_t sizeStorage = std::uniform_int_distribution<size_t>(1, TestSize)(mt);
        sizeStorage = nn::util::align_up(sizeStorage, blockL2);

        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
        inputParam.sizeBlockLevel[0] = blockL1;
        inputParam.sizeBlockLevel[1] = blockL2;

        // メタ作り
        nn::fssystem::save::HierarchicalDuplexSizeSet sizes = {};
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::QuerySize(
                &sizes,
                inputParam,
                sizeStorage
            )
        );

        int64_t sizeDuplexControl = sizeof(nn::fssystem::save::HierarchicalDuplexMetaInformation);
        int64_t sizeDuplexBody = sizes.layeredBitmapSizes[0] * 2 +
                                 sizes.layeredBitmapSizes[1] * 2 +
                                 sizes.bodySize * 2;

        // 管理領域に使うメモリファイルの作成
        nnt::fs::util::AccessCountedMemoryStorage storageDuplexControl;
        storageDuplexControl.Initialize(sizeDuplexControl + offsetBase * 2);

        // 本体に使うメモリファイルの作成
        nnt::fs::util::AccessCountedMemoryStorage storageDuplexBody;
        storageDuplexBody.Initialize(sizeDuplexBody + offsetBase * 2);

        // 管理領域をフォーマット
        nn::fs::SubStorage
            storageControlArea(&storageDuplexControl, offsetBase, sizeDuplexControl);

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::Format(
                storageControlArea,
                inputParam,
                sizeStorage
            )
        );

        // 管理領域をマウントしてメタデータを取得
        nn::fssystem::save::HierarchicalDuplexStorageControlArea controlArea;
        NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

        const nn::fssystem::save::HierarchicalDuplexMetaInformation& meta = controlArea.GetMeta();

        // ストレージ
        int64_t offsetStorage = 0;
        nn::fs::SubStorage storageBitmapMasterA(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapMasterB(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapL1A(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageBitmapL1B(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageDataA(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;
        nn::fs::SubStorage storageDataB(
            &storageDuplexBody, offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;

        // 二重化ビットマップをフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorage::Format(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                &cacheBuffer
            )
        );

        controlArea.Finalize();

        NNT_ASSERT_RESULT_SUCCESS(storageDuplexControl.GetSize(&sizeDuplexControl));

        // ファイルをマウントします。
        {
            // 管理領域をマウントしてメタデータを取得
            NNT_ASSERT_RESULT_SUCCESS(controlArea.Initialize(storageControlArea));

            isSelectBit = !isSelectBit;

            // 二重化ファイルをマウント
            nn::fssystem::save::HierarchicalDuplexStorage storageDuplex;
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Initialize(
                    meta.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    storageDataA,
                    storageDataB,
                    isSelectBit,
                    &cacheBuffer,
                    &locker
                )
            );

            controlArea.Finalize();

            storageDuplexControl.ResetAccessCounter();
            storageDuplexBody.ResetAccessCounter();

            // 適当な位置に書き込み
            for( int j = 0; j <= 3; ++j )
            {
                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(0, sizeStorage - 1)(mt);
                const size_t size =
                    std::uniform_int_distribution<size_t>(
                        1, sizeStorage - static_cast<size_t>(offset))(mt);
                std::memset(ptr, j + 1, size);
                NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Write(offset, ptr, size));
            }

            // flush == true の書き込み処理でも管理領域は更新されない
            ASSERT_EQ(0, storageDuplexControl.GetFlushTimes());
            ASSERT_EQ(0, storageDuplexBody.GetFlushTimes());

            // 0 バイト書き込み
            for( int j = 0; j <= 3; ++j )
            {
                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(0, sizeStorage - 1)(mt);

                storageDuplexControl.ResetAccessCounter();
                storageDuplexBody.ResetAccessCounter();

                // サイズ 0 書き込み
                NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Write(offset, ptr, 0));

                // 書き込みは発生しない
                ASSERT_EQ(0, storageDuplexControl.GetFlushTimes());
                ASSERT_EQ(0, storageDuplexBody.GetFlushTimes());

                // オフセットに 0、サイズ 0 書き込み
                NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Write(0, ptr, 0));

                storageDuplexControl.ResetAccessCounter();
                storageDuplexBody.ResetAccessCounter();

                // 管理領域は更新されない
                ASSERT_EQ(0, storageDuplexControl.GetFlushTimes());
                ASSERT_EQ(0, storageDuplexBody.GetFlushTimes());
            }

            storageDuplexControl.ResetAccessCounter();
            storageDuplexBody.ResetAccessCounter();

            // 範囲外に書き込み
            for( int j = 0; j <= 3; ++j )
            {
                const int64_t offset =
                    std::uniform_int_distribution<int64_t>(sizeStorage + 1, sizeStorage * 2)(mt);
                const size_t size =
                    std::uniform_int_distribution<size_t>(1, sizeStorage)(mt);
                NNT_ASSERT_RESULT_FAILURE(
                    nn::fs::ResultInvalidOffset,
                    storageDuplex.Write(offset, ptr, size)
                );
            }

            // 範囲外エラー発生時に書き込みは行われない
            ASSERT_EQ(0, storageDuplexControl.GetWriteTimes());
            ASSERT_EQ(0, storageDuplexBody.GetWriteTimes());
        }
    }
} // NOLINT(impl/function_size)

#if 0
TEST_F(FsHierarchicalDuplexStorageTest, BurstAccess)
{
    static const size_t TEST_SIZE = 256 * 1024;

    nn::fssystem::save::Random rnd(0);

    nnt::fs::util::Vector<uint8_t> abuf(TEST_SIZE);
    uint8_t *buf = &abuf[0];

    nn::os::Mutex criticalSection((nn::WithInitialize()));

    static const int32_t CACHE_SIZE = 128 * 1024;
    nnt::fs::util::Vector<uint8_t> bufCache(CACHE_SIZE);
    nn::fssystem::save::BlockCacheBufferManager cacheBuffer;
    bool selectBit = false;

    int64_t sizeStorage = TEST_SIZE;
    int64_t offset = 0;
    size_t sizeBlock = GenerateRandomBlockSize(8, MaxBlockSize);

    NN_UNUSED_VAR(sizeBlock);

    // キャッシュバッファマネージャを初期化します。
    nn::fnd::MemoryRange range(
        reinterpret_cast<uptr>(&bufCache[0]),
        reinterpret_cast<uptr>(&bufCache[0] + CACHE_SIZE)
    );
    cacheBuffer.Initialize(range);

    size_t blockL1 = 128;
    size_t blockL2 = 512;
    // ストレージサイズをブロックサイズ単位に合わせる
    sizeStorage = nn::util::align_up(sizeStorage, blockL2);

    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParamDuplex;
    inputParamDuplex.sizeBlockLevel[0] = blockL1;
    inputParamDuplex.sizeBlockLevel[1] = blockL2;

    int64_t sizeDuplexControl;
    int64_t sizeDuplexBody;
    int32_t sizeResult;

    // メタ作り
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::QuerySize(
            &sizeDuplexControl,
            &sizeDuplexBody,
            inputParamDuplex,
            sizeStorage
        )
    );

    // 管理領域に使うメモリファイルの生成
    AccessCountedMemoryStorage baseStorageDuplexControl;
    NNT_ASSERT_RESULT_SUCCESS(baseStorageDuplexControl.Create(sizeDuplexControl));

    // 本体に使うメモリファイルの生成
    AccessCountedMemoryStorage baseStorageDuplexBody;
    NNT_ASSERT_RESULT_SUCCESS(baseStorageDuplexBody.Create(sizeDuplexBody));

    // 管理領域をフォーマット
    nn::fssystem::save::SubStorageSet storageControlArea(&baseStorageDuplexControl, offset, sizeDuplexControl);
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::Format(
            storageControlArea,
            inputParamDuplex,
            sizeStorage
        )
    );

    // 管理領域をマウントしてメタデータを取得
    nn::fssystem::save::HierarchicalDuplexStorageControlArea controlAreaHieDuplex;
    NNT_ASSERT_RESULT_SUCCESS(controlAreaHieDuplex.Initialize(storageControlArea));

    const nn::fssystem::save::HierarchicalDuplexMetaInformation& meta = controlAreaHieDuplex.GetMeta();

    // 二重化ビットマップをフォーマットします。
    nn::fssystem::save::SubStorageSet storageBitmap(&baseStorageDuplexBody, offset, sizeDuplexBody);
    NNT_ASSERT_RESULT_SUCCESS(
        nn::fssystem::save::HierarchicalDuplexStorage::Format(
            meta.infoDuplex,
            storageBitmap,
            &cacheBuffer,
            &criticalSection
        )
    );

    controlAreaHieDuplex.Finalize();

    // ファイルをマウントします。
    NNT_ASSERT_RESULT_SUCCESS(storageDuplexControl.GetSize(&sizeDuplexControl));

    // 管理領域をマウントしてメタデータを取得
    NNT_ASSERT_RESULT_SUCCESS(controlAreaHieDuplex.Initialize(storageControlArea));

    // 二重化ファイルをマウントします。
    nn::fssystem::save::HierarchicalDuplexStorage storageDuplex;
    NNT_ASSERT_RESULT_SUCCESS(
        storageDuplex.Initialize(
            meta.infoDuplex,
            storageBitmap,
            selectBit,
            &cacheBuffer,
            &criticalSection
        )
    );

    controlAreaHieDuplex.Finalize();

    // アクセス回数リセット
    baseStorageDuplexControl.ResetAccessCounter();
    baseStorageDuplexBody.ResetAccessCounter();

    // 一度全体を読み込みます。
    baseStorageDuplexBody.ResetAccessCounter();
    NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Read(&sizeResult, 0, buf, sizeStorage));
    ASSERT_EQ(static_cast<size_t>(sizeResult), sizeStorage);

    ASSERT_EQ(2, baseStorageDuplexBody.GetReadTimes());
    ASSERT_EQ(0, baseStorageDuplexBody.GetFlushTimes());
    ASSERT_EQ(0, baseStorageDuplexBody.GetWriteTimes());

    // 再度全体を読み込みます。
    storageDuplexBody.ResetAccessCounter();
    NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Read(&sizeResult, 0, buf, sizeStorage));
    ASSERT_EQ(static_cast<size_t>(sizeResult), sizeStorage);

    // ビットマップに乗っている部分はキャッシュされています
    ASSERT_EQ(1, baseStorageDuplexBody.GetReadTimes());
    ASSERT_EQ(0, baseStorageDuplexBody.GetFlushTimes());
    ASSERT_EQ(0, baseStorageDuplexBody.GetWriteTimes());

    // 途中まで書き込みます
    baseStorageDuplexBody.ResetAccessCounter();
    NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Write(&sizeResult, 0, buf, sizeStorage / 2, nn::fs::WriteOption(false)));
    ASSERT_EQ(static_cast<size_t>(sizeResult), sizeStorage / 2);

    // ビットマップに乗っている部分はキャッシュされています
    ASSERT_EQ(0, baseStorageDuplexBody.GetReadTimes());
    ASSERT_EQ(0, baseStorageDuplexBody.GetFlushTimes());
    ASSERT_EQ(1, baseStorageDuplexBody.GetWriteTimes());

    // Flushしつつ書き込みます
    baseStorageDuplexBody.ResetAccessCounter();
    NNT_ASSERT_RESULT_SUCCESS(storageDuplex.Write(&sizeResult, 0, buf, 0, nn::fs::WriteOption(true)));
    ASSERT_EQ(0. static_cast<size_t>(sizeResult));

    // ビットマップと、本体部の二回書き込みが発生します
    ASSERT_EQ(0, baseStorageDuplexBody.GetReadTimes());
    ASSERT_EQ(2, baseStorageDuplexBody.GetFlushTimes());
    ASSERT_EQ(2, baseStorageDuplexBody.GetWriteTimes());

    cacheBuffer.Finalize();
    criticalSection.Finalize();
} // NOLINT(impl/function_size)
#endif

// 拡張テストで使用する、ランダムなオフセット、サイズで読み書きを行う関数
// 読み書き用バッファは呼び出し元から渡し、毎回確保しないようにする
void RandomWriteVerify(
         nn::fssystem::save::HierarchicalDuplexStorage* pStorage,
         size_t sizeOriginalBody,
         std::mt19937* pMt,
         char* pReadBuf,
         char* pWriteBuf
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pStorage);
    NN_SDK_REQUIRES_NOT_NULL(pMt);

    static const size_t TestLoop = 10;

    for( int32_t i = 0; i < TestLoop; ++i )
    {
        const int64_t offset = std::uniform_int_distribution<int64_t>(
                                   0,
                                   sizeOriginalBody - 1
                               )(*pMt);
        const size_t size = std::uniform_int_distribution<size_t>(
                                1,
                                static_cast<size_t>(sizeOriginalBody - offset)
                            )(*pMt);

        std::memset(pWriteBuf, (i + 1) & 0xFF, size);

        // データを書き込み
        NNT_ASSERT_RESULT_SUCCESS(pStorage->Write(offset, pWriteBuf, size));

        if( std::uniform_int_distribution<int>(0, 7)(*pMt) == 0 )
        {
            // データを読み込んでベリファイ
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Read(offset, pReadBuf, size));
            NNT_FS_UTIL_ASSERT_MEMCMPEQ(pWriteBuf, pReadBuf, size);
        }
    }
}

// 拡張テストで使用する、SubStorage の内容コピーを行う関数
void CopySubStorage(
         nn::fs::SubStorage* pDstStorage,
         nn::fs::SubStorage* pSrcStorage
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDstStorage);
    NN_SDK_REQUIRES_NOT_NULL(pSrcStorage);

    int64_t sizeDst;
    NNT_ASSERT_RESULT_SUCCESS(pDstStorage->GetSize(&sizeDst));

    int64_t sizeSrc;
    NNT_ASSERT_RESULT_SUCCESS(pSrcStorage->GetSize(&sizeSrc));
    ASSERT_GE(sizeDst, sizeSrc);

    // size_t 単位の読み書きを繰り返してストレージの内容をコピー
    int64_t offset = 0;
    for( size_t sizeBuf = static_cast<size_t>(sizeSrc); sizeSrc > 0; sizeSrc -= sizeBuf )
    {
        nnt::fs::util::Vector<char> buf(sizeBuf);
        NNT_ASSERT_RESULT_SUCCESS(pSrcStorage->Read(offset, &buf[0], sizeBuf));
        NNT_ASSERT_RESULT_SUCCESS(pDstStorage->Write(offset, &buf[0], sizeBuf));
        offset += sizeBuf;
    }
}

// ストレージの拡張テスト
TEST_F(FsHierarchicalDuplexStorageTest, ExpandHeavy)
{
    static const size_t TestSizeMax = 8 * 1024 * 1024;
    static const size_t TestSizeMaxExpand = TestSizeMax * 4;
    static const size_t TestLoop = 10;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 8 * 1024 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> bufCache(CacheBufferSize);
    nnt::fs::util::Vector<char> readBuf(TestSizeMaxExpand);
    nnt::fs::util::Vector<char> readBuf2(TestSizeMaxExpand);
    nnt::fs::util::Vector<char> writeBuf(TestSizeMaxExpand);

    char* pReadBuf = &readBuf[0];
    char* pReadBuf2 = &readBuf2[0];
    char* pWriteBuf = &writeBuf[0];

    for( int blockL1Order = 7; blockL1Order <= 15; ++blockL1Order )
    {
        const int blockL1 = 2 << blockL1Order;
        NN_SDK_LOG(".");
        for( int blockL2Order = 7; blockL2Order <= 15; ++blockL2Order )
        {
            const int blockL2 = 2 << blockL2Order;

            nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
            inputParam.sizeBlockLevel[0] = blockL1;
            inputParam.sizeBlockLevel[1] = blockL2;

            size_t sizeOriginalBody = std::uniform_int_distribution<size_t>(1, 32)(mt) * blockL2;
            if( TestSizeMax < sizeOriginalBody )
            {
                sizeOriginalBody = TestSizeMax;
            }

            // ストレージサイズの取得
            nn::fssystem::save::HierarchicalDuplexSizeSet sizeSet;
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalDuplexStorageControlArea::QuerySize(
                    &sizeSet, inputParam, sizeOriginalBody
                )
            );

            const size_t sizeControl =
                sizeof(nn::fssystem::save::HierarchicalDuplexMetaInformation);
            const int64_t sizeBody = sizeSet.layeredBitmapSizes[0] * 2 +
                                     sizeSet.layeredBitmapSizes[1] * 2 +
                                     sizeSet.bodySize * 2;

            // 管理領域に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageControl(sizeControl);
            baseStorageControl.FillBuffer(0, sizeControl, 0xFF);

            // 本体に使うメモリファイルの作成
            nnt::fs::util::SafeMemoryStorage baseStorageBody(sizeBody);
            baseStorageBody.FillBuffer(0, static_cast<size_t>(sizeBody), 0xFF);

            // 管理領域をフォーマット
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalDuplexStorageControlArea::Format(
                    nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
                    inputParam,
                    sizeOriginalBody
                )
            );

            // キャッシュバッファマネージャを初期化
            nn::fssystem::FileSystemBufferManager cacheBuffer;
            NNT_ASSERT_RESULT_SUCCESS(
                cacheBuffer.Initialize(
                    CacheCountMax,
                    reinterpret_cast<uintptr_t>(&bufCache[0]),
                    CacheBufferSize,
                    CacheBlockSize
                )
            );

            // 排他制御オブジェクト
            nn::os::Mutex locker(true);

            nn::fssystem::save::HierarchicalDuplexInformation infoDuplex;
            int64_t offset = 0;

            // マスタービットマップの情報
            infoDuplex.info[0].offset = offset;
            infoDuplex.info[0].size = sizeSet.layeredBitmapSizes[0];
            infoDuplex.info[0].orderBlock = 0;

            // マスタービットマップのストレージ
            nn::fs::SubStorage storageBitmapMasterA(
                &baseStorageBody, offset, sizeSet.layeredBitmapSizes[0]
            );
            offset += sizeSet.layeredBitmapSizes[0];
            nn::fs::SubStorage storageBitmapMasterB(
                &baseStorageBody, offset, sizeSet.layeredBitmapSizes[0]
            );
            offset += sizeSet.layeredBitmapSizes[0];

            // L1 の情報
            infoDuplex.info[1].offset = offset;
            infoDuplex.info[1].size = sizeSet.layeredBitmapSizes[1];
            infoDuplex.info[1].orderBlock = blockL1Order;

            // L1 のストレージ
            nn::fs::SubStorage storageBitmapL1A(
                &baseStorageBody, offset, sizeSet.layeredBitmapSizes[1]
            );
            offset += sizeSet.layeredBitmapSizes[1];
            nn::fs::SubStorage storageBitmapL1B(
                &baseStorageBody, offset, sizeSet.layeredBitmapSizes[1]
            );
            offset += sizeSet.layeredBitmapSizes[1];

            // データ本体の情報
            infoDuplex.info[2].offset = offset;
            infoDuplex.info[2].size = sizeSet.bodySize;
            infoDuplex.info[2].orderBlock = blockL2Order;

            // データ本体のストレージ
            nn::fs::SubStorage storageDataA(
                &baseStorageBody, offset, sizeSet.bodySize
            );
            offset += sizeSet.bodySize;
            nn::fs::SubStorage storageDataB(
                &baseStorageBody, offset, sizeSet.bodySize
            );
            offset += sizeSet.bodySize;

            // 二重化ビットマップをフォーマット
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::HierarchicalDuplexStorage::Format(
                    infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    &cacheBuffer
                )
            );

            bool isSelectBit = false;

            for( int i = 0; i < TestLoop; ++i )
            {
                // 二重化ファイルをマウント
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage1;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage1.Initialize(
                            infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    // ランダムなオフセット、サイズで読み書きを行う
                    RandomWriteVerify(&storage1, sizeOriginalBody, &mt, pReadBuf, pWriteBuf);

                    //  元のデータが書き換わらないことを確認するために保存
                    NNT_ASSERT_RESULT_SUCCESS(storage1.Read(0, pReadBuf2, sizeOriginalBody));
                }

                // 書き込み先を示すビット変えて再度二重化ファイルをマウント
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage2;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage2.Initialize(
                            infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            !isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    // ランダムなオフセット、サイズで読み書きを行う
                    RandomWriteVerify(&storage2, sizeOriginalBody, &mt, pReadBuf, pWriteBuf);
                }

                {
                    // 書き込み先を示すビットを戻して、以前のデータが
                    // 書き換わっていないことを確認
                    nn::fssystem::save::HierarchicalDuplexStorage storage3;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage3.Initialize(
                            infoDuplex,
                            storageBitmapMasterA,
                            storageBitmapMasterB,
                            storageBitmapL1A,
                            storageBitmapL1B,
                            storageDataA,
                            storageDataB,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    NNT_ASSERT_RESULT_SUCCESS(storage3.SwapDuplexBitmap());

                    NNT_ASSERT_RESULT_SUCCESS(storage3.Read(0, pReadBuf, sizeOriginalBody));
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(pReadBuf2, pReadBuf, sizeOriginalBody);
                }

                // 拡張
                size_t sizeOriginalBodyExpand =
                    sizeOriginalBody +
                    std::uniform_int_distribution<size_t>(0, 128)(mt) * blockL2;
                if( TestSizeMaxExpand < sizeOriginalBodyExpand )
                {
                    sizeOriginalBodyExpand = TestSizeMaxExpand;
                }

                // 拡張に必要なストレージサイズの取得
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::HierarchicalDuplexStorageControlArea::QuerySize(
                        &sizeSet, inputParam, sizeOriginalBodyExpand
                    )
                );

                const int64_t sizeBodyExpand = sizeSet.layeredBitmapSizes[0] * 2 +
                                               sizeSet.layeredBitmapSizes[1] * 2 +
                                               sizeSet.bodySize * 2;

                // 本体に使うメモリファイルの作成
                nnt::fs::util::SafeMemoryStorage baseStorageBodyExpand(sizeBodyExpand);
                baseStorageBodyExpand.FillBuffer(0, static_cast<size_t>(sizeBodyExpand), 0xFF);

                // 管理領域を更新
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::HierarchicalDuplexStorageControlArea::Expand(
                        nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
                        sizeOriginalBodyExpand
                    )
                );

                nn::fssystem::save::HierarchicalDuplexInformation infoDuplexExpand;
                int64_t offsetInfo = 0;

                // マスタービットマップの情報
                infoDuplexExpand.info[0].offset = offsetInfo;
                infoDuplexExpand.info[0].size = sizeSet.layeredBitmapSizes[0];
                infoDuplexExpand.info[0].orderBlock = 0;

                // マスタービットマップのストレージ
                nn::fs::SubStorage storageBitmapMasterAExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.layeredBitmapSizes[0]
                );
                offsetInfo += sizeSet.layeredBitmapSizes[0];
                nn::fs::SubStorage storageBitmapMasterBExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.layeredBitmapSizes[0]
                );
                offsetInfo += sizeSet.layeredBitmapSizes[0];

                // 拡張前のストレージから内容をコピー
                CopySubStorage(&storageBitmapMasterAExpand, &storageBitmapMasterA);
                CopySubStorage(&storageBitmapMasterBExpand, &storageBitmapMasterB);

                // L1 の情報
                infoDuplexExpand.info[1].offset = offsetInfo;
                infoDuplexExpand.info[1].size = sizeSet.layeredBitmapSizes[1];
                infoDuplexExpand.info[1].orderBlock = blockL1Order;

                // L1 のストレージ
                nn::fs::SubStorage storageBitmapL1AExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.layeredBitmapSizes[1]
                );
                offsetInfo += sizeSet.layeredBitmapSizes[1];
                nn::fs::SubStorage storageBitmapL1BExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.layeredBitmapSizes[1]
                );
                offsetInfo += sizeSet.layeredBitmapSizes[1];

                // 拡張前のストレージから内容をコピー
                CopySubStorage(&storageBitmapL1AExpand, &storageBitmapL1A);
                CopySubStorage(&storageBitmapL1BExpand, &storageBitmapL1B);

                // データ本体の情報
                infoDuplexExpand.info[2].offset = offsetInfo;
                infoDuplexExpand.info[2].size = sizeSet.bodySize;
                infoDuplexExpand.info[2].orderBlock = blockL2Order;

                // データ本体のストレージ
                nn::fs::SubStorage storageDataAExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.bodySize
                );
                offsetInfo += sizeSet.bodySize;
                nn::fs::SubStorage storageDataBExpand(
                    &baseStorageBodyExpand, offsetInfo, sizeSet.bodySize
                );
                offsetInfo += sizeSet.bodySize;

                // 拡張前のストレージから内容をコピー
                CopySubStorage(&storageDataAExpand, &storageDataA);
                CopySubStorage(&storageDataBExpand, &storageDataB);

                // 二重化ビットマップを拡張
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::fssystem::save::HierarchicalDuplexStorage::Expand(
                        infoDuplex,
                        infoDuplexExpand,
                        storageBitmapMasterAExpand,
                        storageBitmapMasterBExpand,
                        storageBitmapL1AExpand,
                        storageBitmapL1BExpand,
                        &cacheBuffer
                    )
                );

                // 拡張した二重化ファイルをマウント
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage1;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage1.Initialize(
                            infoDuplexExpand,
                            storageBitmapMasterAExpand,
                            storageBitmapMasterBExpand,
                            storageBitmapL1AExpand,
                            storageBitmapL1BExpand,
                            storageDataAExpand,
                            storageDataBExpand,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    // 拡張前と同様にビットマップを反転させデータが変わっていないことを確認
                    NNT_ASSERT_RESULT_SUCCESS(storage1.SwapDuplexBitmap());
                    NNT_ASSERT_RESULT_SUCCESS(storage1.Read(0, pReadBuf, sizeOriginalBodyExpand));
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(pReadBuf2, pReadBuf, sizeOriginalBody);

                    // 次の読み書きのためにビットマップを元に戻す
                    NNT_ASSERT_RESULT_SUCCESS(storage1.SwapDuplexBitmap());

                    // ランダムなオフセット、サイズで読み書きを行う
                    RandomWriteVerify(
                        &storage1, sizeOriginalBodyExpand, &mt, pReadBuf, pWriteBuf
                    );

                    //  元のデータが書き換わらないことを確認するために保存
                    NNT_ASSERT_RESULT_SUCCESS(storage1.Read(0, pReadBuf2, sizeOriginalBodyExpand));
                }

                // 書き込み先を示すビット変えて再度二重化ファイルをマウント
                {
                    nn::fssystem::save::HierarchicalDuplexStorage storage2;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage2.Initialize(
                            infoDuplexExpand,
                            storageBitmapMasterAExpand,
                            storageBitmapMasterBExpand,
                            storageBitmapL1AExpand,
                            storageBitmapL1BExpand,
                            storageDataAExpand,
                            storageDataBExpand,
                            !isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    // ランダムなオフセット、サイズで読み書きを行う
                    RandomWriteVerify(
                        &storage2, sizeOriginalBodyExpand, &mt, pReadBuf, pWriteBuf
                    );
                }

                {
                    // 書き込み先を示すビットを戻して、以前のデータが
                    // 書き換わっていないことを確認
                    nn::fssystem::save::HierarchicalDuplexStorage storage3;
                    NNT_ASSERT_RESULT_SUCCESS(
                        storage3.Initialize(
                            infoDuplexExpand,
                            storageBitmapMasterAExpand,
                            storageBitmapMasterBExpand,
                            storageBitmapL1AExpand,
                            storageBitmapL1BExpand,
                            storageDataAExpand,
                            storageDataBExpand,
                            isSelectBit,
                            &cacheBuffer,
                            &locker
                        )
                    );

                    NNT_ASSERT_RESULT_SUCCESS(storage3.SwapDuplexBitmap());

                    NNT_ASSERT_RESULT_SUCCESS(storage3.Read(0, pReadBuf, sizeOriginalBodyExpand));
                    NNT_FS_UTIL_ASSERT_MEMCMPEQ(pReadBuf2, pReadBuf, sizeOriginalBodyExpand);
                }
            }
        }
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

// ストレージ拡張時のメタデータ更新の失敗テスト
TEST_F(FsHierarchicalDuplexStorageTest, ExpandControlAreaFailure)
{
    // Expand に渡すための仮の引数（実際にデータは作成しない）
    static const int64_t SizeData = 1024;

    const size_t sizeControl = sizeof(nn::fssystem::save::HierarchicalDuplexMetaInformation);

    // 管理領域のストレージサイズが小さい
    {
        const size_t sizeControlSmall = sizeControl - 1;
        nnt::fs::util::SafeMemoryStorage baseStorageControl(sizeControlSmall);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::Expand(
                nn::fs::SubStorage(&baseStorageControl, 0, sizeControlSmall),
                SizeData
            )
        );
    }

    // メタデータとしてフォーマットされていない
    {
        nnt::fs::util::SafeMemoryStorage baseStorageControl(sizeControl);
        baseStorageControl.FillBuffer(0, sizeControl, 0);
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultIncorrectDuplexMagicCode,
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::Expand(
                nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
                SizeData
            )
        );
    }

    // 既存のメタデータのバージョンが未来のものになっている
    {
        // バージョンを変更するため手動でメタデータを作成
        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
        inputParam.sizeBlockLevel[0] = 4;
        inputParam.sizeBlockLevel[1] = 4;

        nn::fssystem::save::HierarchicalDuplexMetaInformation meta;
        NNT_ASSERT_RESULT_SUCCESS(CreateMetaInformation(&meta, inputParam, SizeData));

        // バージョンを先のものに変更
        meta.version = 0xFFFFFFFF;

        // メタデータを書き込む
        nnt::fs::util::SafeMemoryStorage baseStorageControl(sizeControl);
        nn::fs::SubStorage subStorageControl(&baseStorageControl, 0, sizeControl);
        subStorageControl.Write(0, &meta, sizeof(meta));

        // 拡張
        NNT_ASSERT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedVersion,
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::Expand(
                nn::fs::SubStorage(&baseStorageControl, 0, sizeControl),
                SizeData
            )
        );
    }
}

TEST_F(FsHierarchicalDuplexStorageTest, TestPowerInterruption)
{
    const int64_t SizeOriginalBody = 64 * 1024;
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParamDuplex;
    inputParamDuplex.sizeBlockLevel[0] = 32;
    inputParamDuplex.sizeBlockLevel[1] = 64;
    nn::fssystem::save::HierarchicalDuplexMetaInformation meta;

    // メタ作り
    NNT_ASSERT_RESULT_SUCCESS(
        CreateMetaInformation(
            &meta,
            inputParamDuplex,
            SizeOriginalBody
        )
    );
    const auto SizeBaseStorage
        = meta.infoDuplex.info[0].size * 2
        + meta.infoDuplex.info[1].size * 2
        + meta.infoDuplex.info[2].size * 2;

    // 中間バッファ作り
    static const size_t CacheSize = 128 * 1024;
    nnt::fs::util::Vector<char> bufCache(CacheSize);
    std::pair<uintptr_t, size_t> memoryRange;
    memoryRange.first = reinterpret_cast<uintptr_t>(&bufCache[0]);
    memoryRange.second = CacheSize;
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    const int MaxCacheCount = 1024;
    const size_t CacheBufferPageSize = 4096;
    cacheBuffer.Initialize(
        MaxCacheCount,
        memoryRange.first,
        memoryRange.second,
        CacheBufferPageSize
    );

    // 排他制御オブジェクト
    nn::os::Mutex locker(true);

    nn::Result result;
    for( int writableCount = 0;; ++writableCount )
    {
        // メモリファイルの生成
        nnt::fs::util::SafeMemoryStorage baseStorage;
        baseStorage.Initialize(static_cast<size_t>(SizeBaseStorage));
        std::memset(baseStorage.GetBuffer(), 0, static_cast<size_t>(SizeBaseStorage));
        nnt::fs::util::PowerInterruptionStorage storage;
        storage.Initialize(
            nn::fs::SubStorage(&baseStorage, 0, SizeBaseStorage)
        );
        storage.StartCounting(writableCount);

        // ストレージ
        int64_t offset = 0;
        nn::fs::SubStorage storageBitmapMasterA(&storage, offset, meta.infoDuplex.info[0].size);
        offset += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapMasterB(&storage, offset, meta.infoDuplex.info[0].size);
        offset += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapL1A(&storage, offset, meta.infoDuplex.info[1].size);
        offset += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageBitmapL1B(&storage, offset, meta.infoDuplex.info[1].size);
        offset += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageDataA(&storage, offset, meta.infoDuplex.info[2].size);
        offset += meta.infoDuplex.info[2].size;
        nn::fs::SubStorage storageDataB(&storage, offset, meta.infoDuplex.info[2].size);
        offset += meta.infoDuplex.info[2].size;

        // 現在のストレージAの中身のバッファ
        std::unique_ptr<char[]> currentBufferA(new char[SizeOriginalBody]);
        memset(currentBufferA.get(), 0x00, SizeOriginalBody);

        // 現在のストレージBの中身のバッファ
        std::unique_ptr<char[]> currentBufferB(new char[SizeOriginalBody]);
        memset(currentBufferB.get(), 0x00, SizeOriginalBody);

        // 書き込み用のバッファ
        std::unique_ptr<char[]> writeBuffer(new char[SizeOriginalBody]);
        memset(writeBuffer.get(), 0xFF, SizeOriginalBody);

        // 本体はマウント
        result = nn::fssystem::save::HierarchicalDuplexStorage::Format(
            meta.infoDuplex,
            storageBitmapMasterA,
            storageBitmapMasterB,
            storageBitmapL1A,
            storageBitmapL1B,
            &cacheBuffer
        );
        if( result.IsFailure() )
        {
            ASSERT_TRUE(storage.IsPowerInterruptionOccurred());
            continue;
        }
        nn::fssystem::save::HierarchicalDuplexStorage storageDuplex;
        NNT_ASSERT_RESULT_SUCCESS(
            storageDuplex.Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                false,
                &cacheBuffer,
                &locker
            )
        );

        // ストレージ全体に書き込みし、フラッシュ
        {
            result = storageDuplex.Write(0, writeBuffer.get(), SizeOriginalBody);
            if( result.IsSuccess() )
            {
                result = storageDuplex.Flush();
            }

            // 再マウント
            storageDuplex.Finalize();
            cacheBuffer.Finalize();
            cacheBuffer.Initialize(
                MaxCacheCount,
                memoryRange.first,
                memoryRange.second,
                CacheBufferPageSize
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Initialize(
                    meta.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    storageDataA,
                    storageDataB,
                    false,
                    &cacheBuffer,
                    &locker
                )
            );
            if( result.IsSuccess() )
            {
                // 書き込めたなら反転する
                result = storageDuplex.SwapDuplexBitmap();
            }
            if( result.IsFailure() )
            {
                ASSERT_TRUE(
                    storage.IsPowerInterruptionOccurred()
                );
                storage.StopCounting();
                std::unique_ptr<char[]> readBuffer(new char[SizeOriginalBody]);
                NNT_ASSERT_RESULT_SUCCESS(
                    storageDuplex.Read(0, readBuffer.get(), SizeOriginalBody)
                );
                ASSERT_EQ(0, memcmp(currentBufferA.get(), readBuffer.get(), SizeOriginalBody));
                ASSERT_EQ(0, memcmp(currentBufferB.get(), readBuffer.get(), SizeOriginalBody));
                continue;
            }
        }

        // バリデーション
        {
            // 仕様上、電源断されても書き込み成功となることがあるため、
            // 電源断が発生しているなら continueする
            if( storage.IsPowerInterruptionOccurred() )
            {
                continue;
            }

            // 書き込めたか確認
            std::unique_ptr<char[]> readBuffer(new char[SizeOriginalBody]);
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Read(0, readBuffer.get(), SizeOriginalBody)
            );
            ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), SizeOriginalBody));
        }

        // バッファ更新
        memcpy(currentBufferA.get(), writeBuffer.get(), SizeOriginalBody);
        memset(writeBuffer.get(), 0x00, SizeOriginalBody);

        // ストレージサイズの 1/8 ずつ書き込み、フラッシュ
        {
            for( int i = 0; i < 8; ++i )
            {
                result = storageDuplex.Write(
                    i * SizeOriginalBody / 8,
                    writeBuffer.get(),
                    SizeOriginalBody / 8
                );
                if( result.IsFailure() )
                {
                    break;
                }
            }
            if( result.IsSuccess() )
            {
                result = storageDuplex.Flush();
            }

            // 再マウント
            storageDuplex.Finalize();
            cacheBuffer.Finalize();
            cacheBuffer.Initialize(
                MaxCacheCount,
                memoryRange.first,
                memoryRange.second,
                CacheBufferPageSize
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Initialize(
                    meta.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    storageDataA,
                    storageDataB,
                    true,
                    &cacheBuffer,
                    &locker
                )
            );

            if( result.IsSuccess() )
            {
                // 書き込めたなら反転する
                result = storageDuplex.SwapDuplexBitmap();
            }
            if( result.IsFailure() )
            {
                ASSERT_TRUE(
                    storage.IsPowerInterruptionOccurred()
                );
                storage.StopCounting();
                std::unique_ptr<char[]> readBuffer(new char[SizeOriginalBody]);
                NNT_ASSERT_RESULT_SUCCESS(
                    storageDuplex.Read(0, readBuffer.get(), SizeOriginalBody)
                );
                ASSERT_EQ(0, memcmp(currentBufferA.get(), readBuffer.get(), SizeOriginalBody));
                ASSERT_EQ(0, memcmp(currentBufferB.get(), writeBuffer.get(), SizeOriginalBody));
                continue;
            }
        }

        // バリデーション
        {
            // 電源断状態を解除する
            storage.StopCounting();

            // 書き込めたか確認
            std::unique_ptr<char[]> readBuffer(new char[SizeOriginalBody]);
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Read(0, readBuffer.get(), SizeOriginalBody)
            );
            ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), SizeOriginalBody));
        }

        // 最後まで処理できたので終了。
        break;
    }
} // NOLINT(impl/function_size)

// 拡張した状態で電源断のテスト
TEST_F(FsHierarchicalDuplexStorageTest, TestPowerInterruptionExpand)
{
    const int64_t OriginalSizeBefore = 64 * 1024;
    const int64_t OriginalSizeAfter = 256 * 1024;
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParamDuplexBefore;
    inputParamDuplexBefore.sizeBlockLevel[0] = 32;
    inputParamDuplexBefore.sizeBlockLevel[1] = 64;
    nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParamDuplexAfter;
    inputParamDuplexAfter.sizeBlockLevel[0] = 32;
    inputParamDuplexAfter.sizeBlockLevel[1] = 64;
    nn::fssystem::save::HierarchicalDuplexMetaInformation metaBefore;
    nn::fssystem::save::HierarchicalDuplexMetaInformation metaAfter;

    // メタ作り
    NNT_ASSERT_RESULT_SUCCESS(
        CreateMetaInformation(
            &metaBefore,
            inputParamDuplexBefore,
            OriginalSizeBefore
        )
    );
    NNT_ASSERT_RESULT_SUCCESS(
        CreateMetaInformation(
            &metaAfter,
            inputParamDuplexAfter,
            OriginalSizeAfter
        )
    );
    const auto SizeStorageAfter
        = metaAfter.infoDuplex.info[0].size * 2
        + metaAfter.infoDuplex.info[1].size * 2
        + metaAfter.infoDuplex.info[2].size * 2;

    // 中間バッファ作り
    static const size_t CacheSize = 128 * 1024;
    nnt::fs::util::Vector<uint8_t> bufCache(CacheSize);
    std::pair<uintptr_t, size_t> memoryRange;
    memoryRange.first = reinterpret_cast<uintptr_t>(&bufCache[0]);
    memoryRange.second = CacheSize;
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    const int MaxCacheCount = 1024;
    const size_t CacheBufferPageSize = 4096;
    cacheBuffer.Initialize(
        MaxCacheCount,
        memoryRange.first,
        memoryRange.second,
        CacheBufferPageSize
    );

    // 排他制御オブジェクト
    nn::os::Mutex locker(true);

    nn::Result result;
    for( int writableCount = 0;; ++writableCount )
    {
        // メモリファイルの生成
        nnt::fs::util::SafeMemoryStorage baseStorage;
        baseStorage.Initialize(static_cast<size_t>(SizeStorageAfter));
        std::memset(baseStorage.GetBuffer(), 0, static_cast<size_t>(SizeStorageAfter));
        nnt::fs::util::PowerInterruptionStorage storage;
        storage.Initialize(
            nn::fs::SubStorage(&baseStorage, 0, SizeStorageAfter)
        );
        storage.StartCounting(writableCount);

        // ストレージ
        int64_t offset = 0;
        nn::fs::SubStorage storageBitmapMasterA(&storage, offset, metaAfter.infoDuplex.info[0].size);
        offset += metaAfter.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapMasterB(&storage, offset, metaAfter.infoDuplex.info[0].size);
        offset += metaAfter.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapL1A(&storage, offset, metaAfter.infoDuplex.info[1].size);
        offset += metaAfter.infoDuplex.info[1].size;
        nn::fs::SubStorage storageBitmapL1B(&storage, offset, metaAfter.infoDuplex.info[1].size);
        offset += metaAfter.infoDuplex.info[1].size;
        nn::fs::SubStorage storageDataA(&storage, offset, metaAfter.infoDuplex.info[2].size);
        offset += metaAfter.infoDuplex.info[2].size;
        nn::fs::SubStorage storageDataB(&storage, offset, metaAfter.infoDuplex.info[2].size);
        offset += metaAfter.infoDuplex.info[2].size;

        // 現在のストレージの中身のバッファ
        std::unique_ptr<char[]> currentBufferA(new char[OriginalSizeAfter]);
        memset(currentBufferA.get(), 0x00, OriginalSizeAfter);
        std::unique_ptr<char[]> currentBufferB(new char[OriginalSizeAfter]);
        memset(currentBufferB.get(), 0x00, OriginalSizeAfter);

        // 書き込み用のバッファ
        std::unique_ptr<char[]> writeBuffer(new char[OriginalSizeAfter]);
        memset(writeBuffer.get(), 0xFF, OriginalSizeAfter);

        // 初期化
        result = nn::fssystem::save::HierarchicalDuplexStorage::Format(
            metaBefore.infoDuplex,
            storageBitmapMasterA,
            storageBitmapMasterB,
            storageBitmapL1A,
            storageBitmapL1B,
            &cacheBuffer
        );
        if( result.IsFailure() )
        {
            ASSERT_TRUE(storage.IsPowerInterruptionOccurred());
            continue;
        }

        // 拡張
        result = nn::fssystem::save::HierarchicalDuplexStorage::Expand
        (
            metaBefore.infoDuplex,
            metaAfter.infoDuplex,
            storageBitmapMasterA,
            storageBitmapMasterB,
            storageBitmapL1A,
            storageBitmapL1B,
            &cacheBuffer
        );
        if( result.IsFailure() )
        {
            ASSERT_TRUE(storage.IsPowerInterruptionOccurred());
            continue;
        }

        // マウント
        nn::fssystem::save::HierarchicalDuplexStorage storageDuplex;
        NNT_ASSERT_RESULT_SUCCESS(
            storageDuplex.Initialize(
                metaAfter.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                false,
                &cacheBuffer,
                &locker
            )
        );

        // ストレージ全体に書き込みし、フラッシュ
        {
            result = storageDuplex.Write(0, writeBuffer.get(), OriginalSizeAfter);
            if( result.IsSuccess() )
            {
                result = storageDuplex.Flush();
            }

            // 再マウント
            storageDuplex.Finalize();
            cacheBuffer.Finalize();
            cacheBuffer.Initialize(
                MaxCacheCount,
                memoryRange.first,
                memoryRange.second,
                CacheBufferPageSize
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Initialize(
                    metaAfter.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    storageDataA,
                    storageDataB,
                    false,
                    &cacheBuffer,
                    &locker
                )
            );
            if( result.IsSuccess() )
            {
                // 書き込めたなら反転する
                result = storageDuplex.SwapDuplexBitmap();
            }
            if( result.IsFailure() )
            {
                ASSERT_TRUE(
                    storage.IsPowerInterruptionOccurred()
                );
                storage.StopCounting();
                std::unique_ptr<char[]> readBuffer(new char[OriginalSizeAfter]);
                NNT_ASSERT_RESULT_SUCCESS(
                    storageDuplex.Read(0, readBuffer.get(), OriginalSizeAfter)
                );
                ASSERT_EQ(0, memcmp(currentBufferA.get(), readBuffer.get(), OriginalSizeAfter));
                ASSERT_EQ(0, memcmp(currentBufferB.get(), readBuffer.get(), OriginalSizeAfter));
                continue;
            }
        }

        // バリデーション
        {
            // 仕様上、電源断が発生してもここに来ることがある
            // 電源断されている場合は continueする
            if( storage.IsPowerInterruptionOccurred() )
            {
                continue;
            }

            // 書き込めたか確認
            std::unique_ptr<char[]> readBuffer(new char[OriginalSizeAfter]);
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Read(0, readBuffer.get(), OriginalSizeAfter)
            );
            for( int i = 0; i < OriginalSizeAfter; i += 4096 )
            {
                if( writeBuffer.get()[i] != readBuffer.get()[i] )
                {
                    printf("%d\n", i);
                }

            }
            ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), OriginalSizeAfter));
        }

        // バッファ更新
        memcpy(currentBufferA.get(), writeBuffer.get(), OriginalSizeAfter);
        memset(writeBuffer.get(), 0x00, OriginalSizeAfter);

        // ストレージサイズの 1/8 ずつ書き込み、フラッシュ
        {
            for( int i = 0; i < 8; ++i )
            {
                result = storageDuplex.Write(
                    i * OriginalSizeAfter / 8,
                    writeBuffer.get(),
                    OriginalSizeAfter / 8
                );
                if( result.IsFailure() )
                {
                    break;
                }
            }
            if( result.IsSuccess() )
            {
                result = storageDuplex.Flush();
            }

            // 再マウント
            storageDuplex.Finalize();
            cacheBuffer.Finalize();
            cacheBuffer.Initialize(
                MaxCacheCount,
                memoryRange.first,
                memoryRange.second,
                CacheBufferPageSize
            );
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Initialize(
                    metaAfter.infoDuplex,
                    storageBitmapMasterA,
                    storageBitmapMasterB,
                    storageBitmapL1A,
                    storageBitmapL1B,
                    storageDataA,
                    storageDataB,
                    true,
                    &cacheBuffer,
                    &locker
                )
            );
            if( result.IsSuccess() )
            {
                // 書き込めたなら反転する
                result = storageDuplex.SwapDuplexBitmap();
            }
            if( result.IsFailure() )
            {
                ASSERT_TRUE(
                    storage.IsPowerInterruptionOccurred()
                );
                storage.StopCounting();
                std::unique_ptr<char[]> readBuffer(new char[OriginalSizeAfter]);
                NNT_ASSERT_RESULT_SUCCESS(
                    storageDuplex.Read(0, readBuffer.get(), OriginalSizeAfter)
                );
                ASSERT_EQ(0, memcmp(currentBufferA.get(), readBuffer.get(), OriginalSizeAfter));
                ASSERT_EQ(0, memcmp(currentBufferB.get(), writeBuffer.get(), OriginalSizeAfter));
                continue;
            }
        }
        // バリデーション
        {
            storage.StopCounting();

            // 書き込めたか確認
            std::unique_ptr<char[]> readBuffer(new char[OriginalSizeAfter]);
            NNT_ASSERT_RESULT_SUCCESS(
                storageDuplex.Read(0, readBuffer.get(), OriginalSizeAfter)
            );
            ASSERT_EQ(0, memcmp(writeBuffer.get(), readBuffer.get(), OriginalSizeAfter));
        }

        // 最後まで処理できたので終了。
        break;
    }
} // NOLINT(impl/function_size)

class HierarchicalDuplexStorageTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 16 * 1024;

public:
    // コンストラクタです。
    HierarchicalDuplexStorageTestSetup() NN_NOEXCEPT
        : m_Lock(true)
        , m_isSelectBit(false)
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeBlock);

        static const int CacheCountMax = 1024;
        static const size_t CacheBlockSize = 4 * 1024;
        static const size_t CacheBufferSize = 2 * 1024 * 1024;

        // キャッシュバッファマネージャを初期化
        m_BufCache.resize(CacheBufferSize);

        NNT_ASSERT_RESULT_SUCCESS(
            m_CacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&m_BufCache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );

        size_t blockL1 = GenerateRandomBlockSize(32, MaxBlockSize);
        size_t blockL2 = GenerateRandomBlockSize(32, MaxBlockSize);

        // ストレージサイズをブロックサイズ単位に合わせる
        sizeStorage = nn::util::align_up(sizeStorage, blockL2);

        nn::fssystem::save::HierarchicalDuplexStorageControlArea::InputParam inputParam;
        inputParam.sizeBlockLevel[0] = blockL1;
        inputParam.sizeBlockLevel[1] = blockL2;

        nn::fssystem::save::HierarchicalDuplexSizeSet sizes = {};

        // メタ作り
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::QuerySize(
                &sizes,
                inputParam,
                sizeStorage
            )
        );

        const int64_t sizeDuplexControl =
            sizeof(nn::fssystem::save::HierarchicalDuplexMetaInformation);
        const int64_t sizeDuplexBody = sizes.layeredBitmapSizes[0] * 2 +
                                       sizes.layeredBitmapSizes[1] * 2 +
                                       sizes.bodySize * 2;

        // 管理領域に使うメモリファイルの作成
        m_BaseStorageDuplexControl.reset(new nnt::fs::util::SafeMemoryStorage());
        m_BaseStorageDuplexControl->Initialize(sizeDuplexControl + offset * 2);

        // 本体に使うメモリファイルの作成
        m_BaseStorageDuplexBody.reset(new nnt::fs::util::SafeMemoryStorage());
        m_BaseStorageDuplexBody->Initialize(sizeDuplexBody + offset * 2);

        // 管理領域をフォーマット
        nn::fs::SubStorage
            storageControlArea(m_BaseStorageDuplexControl.get(), offset, sizeDuplexControl);
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorageControlArea::Format(
                storageControlArea,
                inputParam,
                sizeStorage
            )
        );

        // 管理領域をマウントしてメタデータを取得
        nn::fssystem::save::HierarchicalDuplexStorageControlArea controlAreaHieDuplex;
        NNT_ASSERT_RESULT_SUCCESS(controlAreaHieDuplex.Initialize(storageControlArea));

        const nn::fssystem::save::HierarchicalDuplexMetaInformation& meta =
            controlAreaHieDuplex.GetMeta();

        // ストレージ
        int64_t offsetStorage = 0;
        nn::fs::SubStorage storageBitmapMasterA(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapMasterB(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapL1A(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageBitmapL1B(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageDataA(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;
        nn::fs::SubStorage storageDataB(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;

        // 二重化ビットマップをフォーマットします。
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::HierarchicalDuplexStorage::Format(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                &m_CacheBuffer
            )
        );

        controlAreaHieDuplex.Finalize();
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(sizeStorage);
        NN_UNUSED(sizeBlock);

        int64_t sizeDuplexControl;
        NNT_ASSERT_RESULT_SUCCESS(m_BaseStorageDuplexControl->GetSize(&sizeDuplexControl));

        int64_t sizeDuplexBody;
        NNT_ASSERT_RESULT_SUCCESS(m_BaseStorageDuplexBody->GetSize(&sizeDuplexBody));

        // 管理領域をマウントしてメタデータを取得
        nn::fssystem::save::HierarchicalDuplexStorageControlArea controlAreaHieDuplex;
        nn::fs::SubStorage
            storageControlArea(m_BaseStorageDuplexControl.get(), offset, sizeDuplexControl);
        NNT_ASSERT_RESULT_SUCCESS(controlAreaHieDuplex.Initialize(storageControlArea));

        const nn::fssystem::save::HierarchicalDuplexMetaInformation& meta =
            controlAreaHieDuplex.GetMeta();

        m_isSelectBit = !m_isSelectBit;

        // ストレージ
        int64_t offsetStorage = 0;
        nn::fs::SubStorage storageBitmapMasterA(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapMasterB(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[0].size);
        offsetStorage += meta.infoDuplex.info[0].size;
        nn::fs::SubStorage storageBitmapL1A(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageBitmapL1B(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[1].size);
        offsetStorage += meta.infoDuplex.info[1].size;
        nn::fs::SubStorage storageDataA(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;
        nn::fs::SubStorage storageDataB(
            m_BaseStorageDuplexBody.get(), offsetStorage, meta.infoDuplex.info[2].size);
        offsetStorage += meta.infoDuplex.info[2].size;

        // 二重化ファイルをマウントします。
        m_StorageDuplex.reset(new nn::fssystem::save::HierarchicalDuplexStorage());
        NNT_ASSERT_RESULT_SUCCESS(
            m_StorageDuplex->Initialize(
                meta.infoDuplex,
                storageBitmapMasterA,
                storageBitmapMasterB,
                storageBitmapL1A,
                storageBitmapL1B,
                storageDataA,
                storageDataB,
                m_isSelectBit,
                &m_CacheBuffer,
                &m_Lock
            )
        );

        SetStorage(m_StorageDuplex.get());

        controlAreaHieDuplex.Finalize();
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageDuplex.reset(nullptr);
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_StorageDuplex.reset(nullptr);
        m_BaseStorageDuplexControl.reset(nullptr);
        m_BaseStorageDuplexBody.reset(nullptr);
        m_CacheBuffer.Finalize();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageDuplexControl;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_BaseStorageDuplexBody;
    std::unique_ptr<nn::fssystem::save::HierarchicalDuplexStorage> m_StorageDuplex;
    nn::os::Mutex m_Lock;
    nnt::fs::util::Vector<char> m_BufCache;
    nn::fssystem::FileSystemBufferManager m_CacheBuffer;
    bool m_isSelectBit;
};

// 共通テストを行います。
TEST_F(FsHierarchicalDuplexStorageTest, StorageCommonTest)
{
    StorageTest::RunTest<HierarchicalDuplexStorageTestSetup>();
}
