﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/fssystem/save/fs_DuplexStorage.h>
#include <nn/fssystem/save/fs_DuplexBitmapHolder.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_util_CommonStorageTests.h"

// テストクラス
class FsDuplexStorageTest : public testing::Test
{
public:
    // ２重化ストレージのラッパークラスです。
    template< typename TStorage >
    class DuplexStorageWrapper
    {
    public:
        // コンストラクタです。
        DuplexStorageWrapper(
            int cacheCountMax,
            size_t cacheBufferSize,
            size_t cacheBlockSize
        ) NN_NOEXCEPT
            : m_CacheBuffer(cacheBufferSize)
            , m_Buffer()
            , m_OriginalStorage()
            , m_ModifiedStorage()
            , m_Bitmap()
            , m_RealStorage1()
            , m_RealStorage2()
            , m_DuplexStorage()
        {
            nn::Result result = m_Buffer.Initialize(
                                    cacheCountMax,
                                    reinterpret_cast<uintptr_t>(&m_CacheBuffer[0]),
                                    cacheBufferSize,
                                    cacheBlockSize
                                );
            NN_UNUSED(result);
            NN_SDK_ASSERT(result.IsSuccess());
        }

        // 初期化します。
        nn::Result Initialize(
                       uint32_t bitmapBlockCount,
                       int64_t bitmapStorageOffset,
                       int64_t bitmapStorageSize,
                       int64_t bitmapSubStorageSize,
                       int64_t realStorageBlockSize,
                       int64_t realStorageOffset,
                       int64_t realStorageSize,
                       int64_t realSubStorageSize
                   ) NN_NOEXCEPT
        {
            m_OriginalStorage.Initialize(bitmapStorageSize);
            m_ModifiedStorage.Initialize(bitmapStorageSize);

            nn::Result result = nn::fssystem::save::DuplexBitmapHolder::Format(
                bitmapBlockCount,
                nn::fs::SubStorage(
                    &m_ModifiedStorage,
                    bitmapStorageOffset,
                    bitmapSubStorageSize
                ),
                nn::fs::SubStorage(
                    &m_OriginalStorage,
                    bitmapStorageOffset,
                    bitmapSubStorageSize
                )
            );

            if( result.IsSuccess() )
            {
                // ２重化ビットマップを初期化
                m_Bitmap.Initialize(
                    bitmapBlockCount,
                    nn::fs::SubStorage(
                        &m_ModifiedStorage,
                        bitmapStorageOffset,
                        bitmapSubStorageSize
                    ),
                    nn::fs::SubStorage(
                        &m_OriginalStorage,
                        bitmapStorageOffset,
                        bitmapSubStorageSize
                    )
                );

                // 実データ領域を初期化
                m_RealStorage1.Initialize(realStorageSize);
                m_RealStorage2.Initialize(realStorageSize);

                m_DuplexStorage.Initialize(
                    &m_Bitmap,
                    nn::fs::SubStorage(
                        &m_RealStorage1,
                        realStorageOffset,
                        realSubStorageSize
                    ),
                    nn::fs::SubStorage(
                        &m_RealStorage2,
                        realStorageOffset,
                        realSubStorageSize
                    ),
                    realStorageBlockSize,
                    &m_Buffer
                );
            }

            return result;
        }

        // 初期化します。
        nn::Result Initialize(
                       uint32_t bitmapBlockCount,
                       int64_t bitmapStorageSize,
                       int64_t realStorageBlockSize,
                       int64_t realStorageSize
                   ) NN_NOEXCEPT
        {
            return Initialize(
                bitmapBlockCount,
                0,
                bitmapStorageSize,
                bitmapStorageSize,
                realStorageBlockSize,
                0,
                realStorageSize,
                realStorageSize
            );
        }

        // 初期化します。
        nn::Result Initialize(uint32_t blockCount, int64_t blockSize) NN_NOEXCEPT
        {
            return Initialize(
                blockCount,
                nn::fssystem::dbm::DuplexBitmap::QuerySize(blockCount),
                blockSize,
                blockCount * blockSize
            );
        }

        // バッファを取得します。
        nn::fssystem::FileSystemBufferManager& GetBuffer() NN_NOEXCEPT
        {
            return m_Buffer;
        }

        // バッファを取得します。
        const nn::fssystem::FileSystemBufferManager& GetBuffer() const NN_NOEXCEPT
        {
            return m_Buffer;
        }

        // オリジナルメタデータ用ストレージを取得します。
        TStorage& GetOriginalStorage() NN_NOEXCEPT
        {
            return m_OriginalStorage;
        }

        // オリジナルメタデータ用ストレージを取得します。
        const TStorage& GetOriginalStorage() const NN_NOEXCEPT
        {
            return m_OriginalStorage;
        }

        // メタデータ用ストレージを取得します。
        TStorage& GetModifiedStorage() NN_NOEXCEPT
        {
            return m_ModifiedStorage;
        }

        // メタデータ用ストレージを取得します。
        const TStorage& GetModifiedStorage() const NN_NOEXCEPT
        {
            return m_ModifiedStorage;
        }

        // 実データストレージ１を取得します。
        TStorage& GetStorage1() NN_NOEXCEPT
        {
            return m_RealStorage1;
        }

        // 実データストレージ１を取得します。
        const TStorage& GetStorage1() const NN_NOEXCEPT
        {
            return m_RealStorage1;
        }

        // 実データストレージ２を取得します。
        TStorage& GetStorage2() NN_NOEXCEPT
        {
            return m_RealStorage2;
        }

        // 実データストレージ２を取得します。
        const TStorage& GetStorage2() const NN_NOEXCEPT
        {
            return m_RealStorage2;
        }

        // ２重化ストレージを取得します。
        nn::fssystem::save::DuplexStorage& Get() NN_NOEXCEPT
        {
            return m_DuplexStorage;
        }

        // ２重化ストレージを取得します。
        const nn::fssystem::save::DuplexStorage& Get() const NN_NOEXCEPT
        {
            return m_DuplexStorage;
        }

    private:
        nnt::fs::util::Vector<char> m_CacheBuffer;
        nn::fssystem::FileSystemBufferManager m_Buffer;
        TStorage m_OriginalStorage;
        TStorage m_ModifiedStorage;
        nn::fssystem::save::DuplexBitmapHolder m_Bitmap;
        TStorage m_RealStorage1;
        TStorage m_RealStorage2;
        nn::fssystem::save::DuplexStorage m_DuplexStorage;
    };

public:
    // ビットマップデータを更新します。
    static nn::Result UpdateModifiedBits(
                          nn::fssystem::save::DuplexStorage* pStorage,
                          int64_t offset,
                          int64_t offsetEnd
                      ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        return pStorage->UpdateModifiedBits(offset, offsetEnd);
    }

    // ビットの値を読み込みます。
    static nn::Result ReadDuplexBits(
                          nn::fssystem::save::DuplexStorage* pStorage,
                          uint32_t* pOriginalBits,
                          uint32_t* pModifiedBits,
                          int64_t offset,
                          size_t bitCount
                      ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);
        return pStorage->ReadDuplexBits(pOriginalBits, pModifiedBits, offset, bitCount);
    }

    // 2 つのストレージの内容を比較します。
    static void VerifyStorage(
                    nn::fs::IStorage* pStorage1,
                    nn::fs::IStorage* pStorage2
                ) NN_NOEXCEPT
    {
        ASSERT_NE(nullptr, pStorage1);
        ASSERT_NE(nullptr, pStorage2);

        int64_t size1;
        int64_t size2;
        NNT_ASSERT_RESULT_SUCCESS(pStorage1->GetSize(&size1));
        NNT_ASSERT_RESULT_SUCCESS(pStorage2->GetSize(&size2));
        ASSERT_EQ(size1, size2);

        char buf1[512];
        char buf2[512];
        int64_t offset = 0;
        while( offset < size1 )
        {
            size_t sizeRead = sizeof(buf1);
            if( size1 - offset < sizeof(buf1) )
            {
                sizeRead = static_cast<size_t>(size1 - offset);
            }

            NNT_ASSERT_RESULT_SUCCESS(pStorage1->Read(offset, buf1, sizeRead));
            NNT_ASSERT_RESULT_SUCCESS(pStorage2->Read(offset, buf2, sizeRead));

            ASSERT_EQ(0, std::memcmp(buf1, buf2, sizeRead));

            offset += sizeRead;
        }
    }

    // シャッフルバーストテスト
    static void BurstShuffle(
                    DuplexStorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>* pStorage,
                    size_t blockSize,
                    size_t storageSize,
                    void* buf
                ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        size_t size = blockSize * 8;

        if( size <= storageSize )
        {
            // アクセス回数リセット
            pStorage->GetOriginalStorage().ResetAccessCounter();
            pStorage->GetModifiedStorage().ResetAccessCounter();
            pStorage->GetStorage1().ResetAccessCounter();
            pStorage->GetStorage2().ResetAccessCounter();

            // 書き込み先はブロック境界丁度のアクセスなので
            // ・リードは発生しません
            // ・ライトは発生します
            std::memset(buf, 1, size);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Get().Write(0, buf, size));

            // アクセスカウンタを確認します。期待値としては

            // ・ビットマップ部分はそれぞれ1回のみリードされます
            // ・更新ビットマップ部分は1回のみライトされます
#ifndef NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK
            ASSERT_EQ(1 + 1 + 2, pStorage->GetOriginalStorage().GetReadTimes()); // DuplexBitmap::MarkModfied で論理和のために1回読み込み
                                                                               // ビットマップ書き込み検証のために2回読み込み
            ASSERT_EQ(1 + 1 + 2, pStorage->GetModifiedStorage().GetReadTimes()); // DuplexBitmap::MarkModfied で論理和のために1回読み込み
                                                                               // ビットマップ書き込み検証のために2回読み込み
#else
            ASSERT_EQ(1 + 1, pStorage->GetOriginalStorage().GetReadTimes()); // DuplexBitmap::MarkModfied で論理和のために1回読み込み
            ASSERT_EQ(1 + 1, pStorage->GetModifiedStorage().GetReadTimes()); // DuplexBitmap::MarkModfied で論理和のために1回読み込み
#endif
            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(1, pStorage->GetModifiedStorage().GetWriteTimes());

            // 8ブロック書き込んだはずで
            // ・合計 5 回書き込まれ、2 回フラッシュされることを期待します(0x99で初期化しているので "1 00 11 00 1" -> 5 回の書き込みになるはず)
            // ・リードは発生しません
            ASSERT_EQ(
                5,
                pStorage->GetStorage1().GetWriteTimes() + pStorage->GetStorage2().GetWriteTimes()
            );
            // TODO: Flush 回数を最適化
            // NN_TEST_ASSERT_EQUAL(1, pStorage->GetStorage1().GetFlushTimes());
            // NN_TEST_ASSERT_EQUAL(1, pStorage->GetStorage2().GetFlushTimes());
            ASSERT_EQ(
                0, pStorage->GetStorage1().GetReadTimes() + pStorage->GetStorage2().GetReadTimes()
            );

            // アクセス回数リセット
            pStorage->GetOriginalStorage().ResetAccessCounter();
            pStorage->GetModifiedStorage().ResetAccessCounter();
            pStorage->GetStorage1().ResetAccessCounter();
            pStorage->GetStorage2().ResetAccessCounter();

            // 同じ範囲に再度書き込みします
            std::memset(buf, 2, size);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Get().Write(0, buf, size));

            // ・ビットマップ部分はそれぞれ 1 回のみリードされます
            // ・更新ビットマップ部分は書き込まれません
            ASSERT_EQ(1, pStorage->GetOriginalStorage().GetReadTimes());
            ASSERT_EQ(1, pStorage->GetModifiedStorage().GetReadTimes());
            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(0, pStorage->GetModifiedStorage().GetWriteTimes());

            // 8ブロック書き込んだはずで
            // ・合計 5 回書き込まれることを期待します
            // ・フラッシュ、リードは発生しません
            ASSERT_EQ(
                5, // 1 00 11 00 1 の 5回
                pStorage->GetStorage1().GetWriteTimes() + pStorage->GetStorage2().GetWriteTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetFlushTimes() + pStorage->GetStorage2().GetFlushTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetReadTimes() + pStorage->GetStorage2().GetReadTimes()
            );
        }
    }

    // 通常バーストテスト
    static void BurstNormal(
                    DuplexStorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>* pStorage,
                    size_t blockSize,
                    size_t storageSize,
                    void* buf
                ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pStorage);

        size_t size1 = blockSize * 4;
        size_t size2 = blockSize * 8;

        if( size2 <= storageSize )
        {
            // アクセス回数リセット
            pStorage->GetOriginalStorage().ResetAccessCounter();
            pStorage->GetModifiedStorage().ResetAccessCounter();
            pStorage->GetStorage1().ResetAccessCounter();
            pStorage->GetStorage2().ResetAccessCounter();

            // 書き込み先はブロック境界丁度のアクセスなので
            // ・リードは発生しません
            // ・ライトは発生します
            std::memset(buf, 1, size1);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Get().Write(0, buf, size1));

            // アクセスカウンタを確認します。期待値としては

            // ・ビットマップ部分はそれぞれ1回のみリードされます
            // ・更新ビットマップ部分は1回のみライトされます

            // DuplexBitmap::MarkModfied で論理和のために1回読み込み
            // ビットマップ書き込み検証のために2回読み込み
            ASSERT_EQ(1 + 1 + 2, pStorage->GetOriginalStorage().GetReadTimes());

            // DuplexBitmap::MarkModfied で論理和のために1回読み込み
            // ビットマップ書き込み検証のために2回読み込み
            ASSERT_EQ(1 + 1 + 2, pStorage->GetModifiedStorage().GetReadTimes());

            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(1, pStorage->GetModifiedStorage().GetWriteTimes());
            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetFlushTimes());
            ASSERT_EQ(0, pStorage->GetModifiedStorage().GetFlushTimes());

            // 4ブロック書き込んだはずで
            // ・1回書き込まれることを期待します。
            // ・フラッシュはまだ発生していません。
            // ・リードは発生しません
            ASSERT_EQ(
                1,
                pStorage->GetStorage1().GetWriteTimes() + pStorage->GetStorage2().GetWriteTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetFlushTimes() + pStorage->GetStorage2().GetFlushTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetReadTimes() + pStorage->GetStorage2().GetReadTimes()
            );

            // アクセス回数リセット
            pStorage->GetOriginalStorage().ResetAccessCounter();
            pStorage->GetModifiedStorage().ResetAccessCounter();
            pStorage->GetStorage1().ResetAccessCounter();
            pStorage->GetStorage2().ResetAccessCounter();

            // フラッシュを実行します。
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Get().Flush());
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetWriteTimes() + pStorage->GetStorage2().GetWriteTimes()
            );
            ASSERT_EQ(
                2,
                pStorage->GetStorage1().GetFlushTimes() + pStorage->GetStorage2().GetFlushTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetReadTimes() + pStorage->GetStorage2().GetReadTimes()
            );

            // アクセス回数リセット
            pStorage->GetOriginalStorage().ResetAccessCounter();
            pStorage->GetModifiedStorage().ResetAccessCounter();
            pStorage->GetStorage1().ResetAccessCounter();
            pStorage->GetStorage2().ResetAccessCounter();

            // 同じ範囲に再度書き込みします。サイズを若干増やします。
            std::memset(buf, 2, size2);
            NNT_ASSERT_RESULT_SUCCESS(pStorage->Get().Write(0, buf, size2));

            // ・ビットマップ部分はそれぞれ1回のみリードされます
            // ・更新ビットマップ部分は1回書き込まれます

            // DuplexBitmap::MarkModfied で論理和のために1回読み込み
            // ビットマップ書き込み検証のために2回読み込み
            ASSERT_EQ(1 + 1 + 2, pStorage->GetOriginalStorage().GetReadTimes());

            // DuplexBitmap::MarkModfied で論理和のために1回読み込み
            // ビットマップ書き込み検証のために2回読み込み
            ASSERT_EQ(1 + 1 + 2, pStorage->GetModifiedStorage().GetReadTimes());

            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(1, pStorage->GetModifiedStorage().GetWriteTimes());
            ASSERT_EQ(0, pStorage->GetOriginalStorage().GetFlushTimes());
            ASSERT_EQ(0, pStorage->GetModifiedStorage().GetFlushTimes());

            // 8ブロック書き込んだはずで
            // ・1回書き込まれることを期待します
            // ・フラッシュ、リードは発生しません
            ASSERT_EQ(
                1,
                pStorage->GetStorage1().GetWriteTimes() + pStorage->GetStorage2().GetWriteTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetFlushTimes() + pStorage->GetStorage2().GetFlushTimes()
            );
            ASSERT_EQ(
                0,
                pStorage->GetStorage1().GetReadTimes() + pStorage->GetStorage2().GetReadTimes()
            );
        }
    }
};

// 二重化ビットマップ操作のテストを行います。
TEST_F(FsDuplexStorageTest, Simple)
{
    static const int CacheCountMax = 1024;
    static const size_t CacheBufferSize = 64 * 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const uint32_t BlockCount = 32;
    static const int64_t BlockSize = 512;

    DuplexStorageWrapper<nnt::fs::util::SafeMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(BlockCount, BlockSize));

    char* pModifiedBuf = reinterpret_cast<char*>(storage.GetModifiedStorage().GetBuffer());
    uint32_t originalBits;
    uint32_t modifiedBits;

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0, originalBits);
    ASSERT_EQ(0, modifiedBits);

    std::iota(pModifiedBuf, pModifiedBuf + 1, '\x80'); // 適当な箇所のビットを書き換え

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0x00000000, originalBits);
    ASSERT_EQ(0x00000080, modifiedBits);

    std::iota(pModifiedBuf + 2, pModifiedBuf + 3, '\xCD'); // 適当な箇所のビットを書き換え

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0x00000000, originalBits);
    ASSERT_EQ(0x00CD0080, modifiedBits);
}

// 二重化ビットマップの読み込みテスト
TEST_F(FsDuplexStorageTest, ReadBits)
{
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 64 * 1024;
    static const uint32_t BlockCount = 40;
    static const int64_t BlockSize = 512;

    DuplexStorageWrapper<nnt::fs::util::SafeMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(BlockCount, BlockSize));

    char* pModifiedBuf = reinterpret_cast<char*>(storage.GetModifiedStorage().GetBuffer());
    uint32_t originalBits;
    uint32_t modifiedBits;

    // 現在の二重化選択ビットを取得
    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0, originalBits);
    ASSERT_EQ(0, modifiedBits);

    std::iota(pModifiedBuf, pModifiedBuf + 1, '\x80'); // 適当な箇所のビットを書き換え

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0x00000000, originalBits);
    ASSERT_EQ(0x00000080, modifiedBits);

    std::iota(pModifiedBuf + 2, pModifiedBuf + 3, '\xCD');

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            0,
            NN_BITSIZEOF(uint32_t)
        )
    );

    ASSERT_EQ(0x00000000, originalBits);
    ASSERT_EQ(0x00CD0080, modifiedBits);

    std::iota(pModifiedBuf + 7, pModifiedBuf + 8, '\x32');

    NNT_ASSERT_RESULT_SUCCESS(
        ReadDuplexBits(
            &storage.Get(),
            &originalBits,
            &modifiedBits,
            BlockSize * 32,
            BlockCount - 32
        )
    );

    ASSERT_EQ(0x00000000, originalBits);
    ASSERT_EQ(0x32000000, modifiedBits);
}

// 二重化ビットマップの書き込みテスト
TEST_F(FsDuplexStorageTest, WriteBits)
{
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 64 * 1024;
    static const uint32_t BlockCount = 32;
    static const int64_t BlockSize = 512;

    DuplexStorageWrapper<nnt::fs::util::SafeMemoryStorage>
        storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(BlockCount, BlockSize));

    char* pModifiedBuf = reinterpret_cast<char*>(storage.GetModifiedStorage().GetBuffer());

    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[3]));

    NNT_ASSERT_RESULT_SUCCESS(
        UpdateModifiedBits(
            &storage.Get(),
            0,
            BlockSize * 1
        )
    );

    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[3]));

    NNT_ASSERT_RESULT_SUCCESS(
        UpdateModifiedBits(
            &storage.Get(),
            BlockSize * 16,
            BlockSize * 24
        )
    );

    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0xFF, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x00, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[3]));

    NNT_ASSERT_RESULT_SUCCESS(
        UpdateModifiedBits(
            &storage.Get(),
            BlockSize * 15,
            BlockSize * 25
        )
    );

    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0xFF, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x01, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[3]));

    NNT_ASSERT_RESULT_SUCCESS(
        UpdateModifiedBits(
            &storage.Get(),
            BlockSize * 24,
            BlockSize * 26
        )
    );

    ASSERT_EQ(0xC0, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0xFF, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x01, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[3]));

    NNT_ASSERT_RESULT_SUCCESS(
        UpdateModifiedBits(
            &storage.Get(),
            BlockSize * 4,
            BlockSize * 4
        )
    );

    ASSERT_EQ(0xC0, static_cast<uint8_t>(pModifiedBuf[0]));
    ASSERT_EQ(0xFF, static_cast<uint8_t>(pModifiedBuf[1]));
    ASSERT_EQ(0x01, static_cast<uint8_t>(pModifiedBuf[2]));
    ASSERT_EQ(0x80, static_cast<uint8_t>(pModifiedBuf[3]));
}

#if 0
// 二重化選択ビットのランダムな位置を書き換え、ベリファイします。
TEST_F(FsDuplexStorageTest, Test4)
{
    nn::fssystem::save::Random rnd;

    for( size_t sizeBlock = 8; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_LOG(".");
        for (size_t loop = 0; loop < 20; loop++)
        {
            size_t bitCount = rnd.Get32(1024) + 1;
            size_t bitSize = nn::fssystem::dbm::DuplexBitmap::QuerySize(bitCount);
            size_t sizeStorage = sizeBlock * bitCount;

            // 二重化ビットマップを初期化します。
            nn::fssystem::save::MemoryStorage storageOriginalBitmap;
            nn::fssystem::save::MemoryStorage storageModifiedBitmap;
            storageOriginalBitmap.Create(bitSize);
            storageModifiedBitmap.Create(bitSize);
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fssystem::save::DuplexBitmapHolder::Format(
                    bitCount,
                    &storageModifiedBitmap,
                    0,
                    &storageOriginalBitmap,
                    0
                )
            );
            nn::fssystem::save::DuplexBitmapHolder duplexBitmap;
            duplexBitmap.Initialize(
                bitCount,
                &storageModifiedBitmap,
                0,
                &storageOriginalBitmap,
                0
            );

            // 実データ領域を初期化します。
            // (ビット操作のテストなので適当な数値)
            nn::fssystem::save::MemoryStorage storage1;
            nn::fssystem::save::MemoryStorage storage2;
            NNT_ASSERT_RESULT_SUCCESS(storage1.Create(sizeStorage));
            NNT_ASSERT_RESULT_SUCCESS(storage2.Create(sizeStorage));

            nnt::fs::util::Vector<uint8_t> bufCache(65536);
            nn::fssystem::save::BlockCacheBufferManager  bufMgr;
            nn::fnd::MemoryRange range(reinterpret_cast<uptr>(&bufCache[0]), reinterpret_cast<uptr>(&bufCache[0] + bufCache.size()));
            bufMgr.Initialize(range);

            nn::fssystem::save::DuplexStorage storageDup;
            nn::fssystem::save::SubStorageSet storageData1(&storage1, 0, sizeStorage);
            nn::fssystem::save::SubStorageSet storageData2(&storage2, 0, sizeStorage);
            storageDup.Initialize(&duplexBitmap, storageData1, storageData2, sizeBlock, &bufMgr);

            for (int32_t i = 0; i < 5; i++)
            {
#if 0
                // 変更後のビットをオリジナル側に反映します。
                if ((i + 1) % 100 == 0)
                {
                    u8* pOriginalBuf = reinterpret_cast<u8 *>(storageOriginalBitmap.GetBuffer());
                    u8* pModifiedBuf = reinterpret_cast<u8 *>(storageModifiedBitmap.GetBuffer());
                    std::memcpy(pOriginalBuf, pModifiedBuf, bitSize);
                }
#endif

                uint32_t originalBits;
                uint32_t modifiedBits;
                uint32_t offset = rnd.Get32(sizeStorage) & ~(sizeBlock - 1);
                uint32_t bits = 32;
                if (bits + ((offset + sizeBlock - 1) / sizeBlock) > bitCount)
                {
                    bits = bitCount - ((offset + sizeBlock - 1) / sizeBlock);
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    ReadDuplexBits(
                        storageDup,
                        &originalBits,
                        &modifiedBits,
                        offset,
                        bits
                    )
                );

                uint32_t newModifiedBits = modifiedBits;
                uint32_t rndCnt = 0;
                for (int32_t j = 31; j >= 0; j--)
                {
                    if (rnd.Get32(2) == 1)
                    {
                        newModifiedBits |= (1 << j);
                    }
                    if (++rndCnt >= bits)
                    {
                        break;
                    }
                }

                NNT_ASSERT_RESULT_SUCCESS(
                    UpdateModifiedBits(
                        storageDup,
                        offset,
                        originalBits ^ newModifiedBits,
                        nn::fs::WriteOption(true)
                    )
                );

                uint32_t originalBits2;
                uint32_t modifiedBits2;
                NNT_ASSERT_RESULT_SUCCESS(
                    ReadDuplexBits(
                        storageDup,
                        &originalBits2,
                        &modifiedBits2,
                        offset,
                        bits
                    )
                );

                ASSERT_EQ(originalBits, originalBits2);
                ASSERT_EQ(newModifiedBits, modifiedBits2);
            }
        }
    }
    NN_LOG("\n");
} // NOLINT(impl/function_size)
#endif

// 二重化ビットマップの読み書きテスト
TEST_F(FsDuplexStorageTest, ReadWrite)
{
    static const int TestLoop = 10;
    static const size_t TestSize = 256 * 1024;
    static const int CacheCountMax = 1024;
    static const size_t CacheBlockSize = 4 * 1024;
    static const size_t CacheBufferSize = 64 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> workBuf(TestSize);
    nnt::fs::util::Vector<char> readBuf1(TestSize);
    nnt::fs::util::Vector<char> readBuf2(TestSize);
    nnt::fs::util::Vector<char> writeBuf(TestSize);
    char* pWorkBuf = &workBuf[0];
    char* pReadBuf1 = &readBuf1[0];
    char* pReadBuf2 = &readBuf2[0];
    char* pWriteBuf = &writeBuf[0];

    for( size_t sizeBlock = 8; sizeBlock <= MaxBlockSize; sizeBlock *= 2 )
    {
        NN_SDK_LOG(".");

        for( int loop = 0; loop < TestLoop; ++loop )
        {
            size_t blockSize = GenerateRandomBlockSize(8, MaxBlockSize);
            size_t storageSize = nn::util::align_up(
                std::uniform_int_distribution<size_t>(1, TestSize)(mt), blockSize);
            uint32_t blockCount = static_cast<uint32_t>(storageSize / blockSize);
            int64_t querySize = nn::fssystem::dbm::DuplexBitmap::QuerySize(blockCount);

            DuplexStorageWrapper<nnt::fs::util::SafeMemoryStorage>
                storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

            NNT_ASSERT_RESULT_SUCCESS(
                storage.Initialize(blockCount, querySize, blockSize, storageSize));

            nnt::fs::util::SafeMemoryStorage storageOriginal;
            nnt::fs::util::SafeMemoryStorage storageVerify;
            storageOriginal.Initialize(storageSize);
            storageVerify.Initialize(storageSize);

            // バッファを全て同じ内容に初期化
            std::memset(storageOriginal.GetBuffer(), 0, storageSize);
            std::memset(storage.GetStorage1().GetBuffer(), 0, storageSize);
            std::memset(storage.GetStorage2().GetBuffer(), 0, storageSize);
            std::memset(storageVerify.GetBuffer(), 0, storageSize);
            std::memset(pWriteBuf, 0, TestSize);

            // オリジナルデータを指し続ける二重化ファイルを初期化
            // この二重化ファイルは常に同じデータを読み取れます。
            nn::fssystem::save::DuplexBitmapHolder readonlyBitmap;
            readonlyBitmap.Initialize(
                blockCount,
                nn::fs::SubStorage(&storage.GetOriginalStorage(), 0, querySize),
                nn::fs::SubStorage(&storage.GetOriginalStorage(), 0, querySize)
            );
            nn::fssystem::save::DuplexStorage readonlyStorage;
            readonlyStorage.Initialize(
                &readonlyBitmap,
                nn::fs::SubStorage(&storage.GetStorage1(), 0, storageSize),
                nn::fs::SubStorage(&storage.GetStorage2(), 0, storageSize),
                blockSize,
                &storage.GetBuffer()
            );
            readonlyStorage.SetReadOnly(true);

            for( int j = 0; j < 3; ++j )
            {
                // ストレージに書き込み
                for( int k = 0; k <= 3; ++k )
                {
                    int64_t offset =
                        std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);
                    size_t size =
                        std::uniform_int_distribution<size_t>(
                            1, storageSize - static_cast<size_t>(offset))(mt);

                    std::memset(pWorkBuf, k + 1, size);

                    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, pWorkBuf, size));
                    NNT_ASSERT_RESULT_SUCCESS(storageVerify.Write(offset, pWorkBuf, size));

                    std::memcpy(pWriteBuf + offset, pWorkBuf, size);
                }

                // ストレージを読み込んでベリファイ
                for( int k = 0; k <= 3; ++k )
                {
                    int64_t offset =
                        std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);
                    size_t size =
                        std::uniform_int_distribution<size_t>(
                            1, storageSize - static_cast<size_t>(offset))(mt);

                    NNT_ASSERT_RESULT_SUCCESS(storage.Get().Read(offset, pReadBuf1, size));
                    NNT_ASSERT_RESULT_SUCCESS(storageVerify.Read(offset, pReadBuf2, size));

                    ASSERT_EQ(0, std::memcmp(pReadBuf1, pReadBuf2, size));
                    ASSERT_EQ(0, std::memcmp(pWriteBuf + offset, pReadBuf1, size));
                }

                // オリジナルファイル側が書き換わっていないことを確認
                VerifyStorage(&storageOriginal, &readonlyStorage);

                // 二重化ファイルとベリファイ用データを比較
                VerifyStorage(&storage.Get(), &storageVerify);

                NNT_ASSERT_RESULT_SUCCESS(
                    storage.Get().Read(0, pReadBuf1, storageSize));
                ASSERT_EQ(0, std::memcmp(pWriteBuf, pReadBuf1, storageSize));
            }
        }
    }

    NN_SDK_LOG("\n");
}

// 二重化ビットマップのアクセス回数テスト
TEST_F(FsDuplexStorageTest, AccessCount)
{
    static const int TestLoop = 10;
    static const size_t TestSize = 1024 * 1024;
    static const int CacheCountMax = 2048;
    static const size_t CacheBlockSize = 4 * 1024;
    static const int32_t CacheBufferSize = 64 * 1024;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];

    for( int i = 0; i < TestLoop; ++i )
    {
        NN_SDK_LOG(".");

        int64_t storageOffset = std::uniform_int_distribution<int64_t>(0, 127)(mt);
        size_t storageSize = std::uniform_int_distribution<size_t>(1, TestSize)(mt);
        size_t blockSize = GenerateRandomBlockSize(8, MaxBlockSize);
        uint32_t blockCount = static_cast<uint32_t>((storageSize + blockSize - 1) / blockSize);
        int64_t querySize = nn::fssystem::dbm::DuplexBitmap::QuerySize(blockCount);

        DuplexStorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
            storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(
                blockCount,
                storageOffset,
                querySize + storageOffset * 2,
                querySize,
                blockSize,
                storageOffset,
                storageSize + storageOffset * 2,
                storageSize
            )
        );

        // バッファを全て同じ内容に初期化
        std::memset(storage.GetStorage1().GetBuffer(), 0, storageSize);
        std::memset(storage.GetStorage2().GetBuffer(), 0, storageSize);

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 適当な位置に書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);
            size_t size =
                std::uniform_int_distribution<size_t>(
                    1, storageSize - static_cast<size_t>(offset))(mt);

            std::memset(ptr, j + 1, size);

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr, size));
        }

        // 編集ビットマップにのみ書き込みが発生したことを確認
        ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
        ASSERT_LT(0, storage.GetModifiedStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage1().GetFlushTimes());
        ASSERT_EQ(0, storage.GetStorage2().GetFlushTimes());

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 編集ビットマップにのみフラッシュが発生することを確認
        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Flush());
        ASSERT_EQ(0, storage.GetOriginalStorage().GetFlushTimes());
        ASSERT_EQ(1, storage.GetModifiedStorage().GetFlushTimes());
        ASSERT_EQ(1, storage.GetStorage1().GetFlushTimes());
        ASSERT_EQ(1, storage.GetStorage2().GetFlushTimes());

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 0 バイト書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);

            // アクセス回数リセット
            storage.GetOriginalStorage().ResetAccessCounter();
            storage.GetModifiedStorage().ResetAccessCounter();
            storage.GetStorage1().ResetAccessCounter();
            storage.GetStorage2().ResetAccessCounter();

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr, 0));

            // 書き込みは発生しない
            ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(0, storage.GetModifiedStorage().GetWriteTimes());
            ASSERT_EQ(0, storage.GetStorage1().GetWriteTimes());
            ASSERT_EQ(0, storage.GetStorage2().GetWriteTimes());
        }

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 範囲外に書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(storageSize + 1, storageSize * 2)(mt);
            size_t size =
                std::uniform_int_distribution<size_t>(1, storageSize)(mt);

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                storage.Get().Write(offset, ptr, size)
            );
        }

        // 範囲外エラー発生時に書き込みは行われない
        ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetModifiedStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage1().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage2().GetWriteTimes());
    }

    NN_SDK_LOG("\n");
}

// 二重化ビットマップのバーストテスト
TEST_F(FsDuplexStorageTest, Burst)
{
    static const size_t TestSize = 16 * 1024;
    static const int TestLoop = 10;
    static const int CacheCountMax = 1024;
    static const size_t CacheBufferSize = 512 * 1024;
    static const size_t CacheBlockSize = 4096;

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    nnt::fs::util::Vector<char> buf(TestSize);
    char* ptr = &buf[0];

    for( int i = 0; i < TestLoop; ++i )
    {
        NN_SDK_LOG(".");

        size_t storageSize = std::uniform_int_distribution<size_t>(1, TestSize)(mt);
        size_t blockSize = GenerateRandomBlockSize(8, MaxBlockSize);
        uint32_t blockCount = static_cast<uint32_t>((storageSize + blockSize - 1) / blockSize);
        int64_t querySize = nn::fssystem::dbm::DuplexBitmap::QuerySize(blockCount);
        bool isShuffleBits = (i & 1) ? false : true;

        DuplexStorageWrapper<nnt::fs::util::AccessCountedMemoryStorage>
            storage(CacheCountMax, CacheBufferSize, CacheBlockSize);

        NNT_ASSERT_RESULT_SUCCESS(
            storage.Initialize(blockCount, querySize, blockSize, storageSize));

        // バッファを全て同じ内容に初期化
        std::memset(storage.GetStorage1().GetBuffer(), 0, storageSize);
        std::memset(storage.GetStorage2().GetBuffer(), 0, storageSize);

        if( isShuffleBits )
        {
            // バーストテスト用にビットマップをまだら模様にしておく
            std::memset(
                storage.GetModifiedStorage().GetBuffer(), 0x99, static_cast<size_t>(querySize));
            std::memset(
                storage.GetOriginalStorage().GetBuffer(), 0x99, static_cast<size_t>(querySize));

            BurstShuffle(&storage, blockSize, storageSize, ptr);
        }
        else
        {
            BurstNormal(&storage, blockSize, storageSize, ptr);
        }

        // 全てが Modified になったら、それ以上に書き込みは発生しない
        // このため、このイテレーションはスキップする
        {
            bool isAllFilled = false;
            for( int64_t j = 0; j < querySize; ++j )
            {
                if( reinterpret_cast<const uint8_t*>(
                    storage.GetModifiedStorage().GetBuffer())[j] != 0xFF )
                {
                    isAllFilled = true;
                    break;
                }
            }
            if( isAllFilled )
            {
                continue;
            }
        }

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 適当な位置に書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);
            size_t size =
                std::uniform_int_distribution<size_t>(
                    1, storageSize - static_cast<size_t>(offset))(mt);

            std::memset(ptr, j + 1, size);

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr, size));
        }

        // Burst と同じ領域への書き込みの可能性がある
        // 書き込みがあるまで適当に書き込み
        while( storage.GetModifiedStorage().GetWriteTimes() == 0 )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);
            size_t size =
                std::uniform_int_distribution<size_t>(
                    1, storageSize - static_cast<size_t>(offset))(mt);
            int data = std::uniform_int_distribution<int>(0, 255)(mt);

            std::memset(ptr, data, size);

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr, size));
        }

        // 編集ビットマップ以外には書き込みが発生しなかったことを確認します。
        ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage1().GetFlushTimes());
        ASSERT_EQ(0, storage.GetStorage2().GetFlushTimes());

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 編集ビットマップにのみフラッシュが発生することを確認します。
        NNT_ASSERT_RESULT_SUCCESS(storage.Get().Flush());
        ASSERT_EQ(0, storage.GetOriginalStorage().GetFlushTimes());
        ASSERT_EQ(1, storage.GetModifiedStorage().GetFlushTimes());
        ASSERT_LE(0, storage.GetStorage1().GetFlushTimes());
        ASSERT_LE(0, storage.GetStorage2().GetFlushTimes());

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 0 バイト書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset = std::uniform_int_distribution<int64_t>(0, storageSize - 1)(mt);

            // アクセス回数リセット
            storage.GetOriginalStorage().ResetAccessCounter();
            storage.GetModifiedStorage().ResetAccessCounter();
            storage.GetStorage1().ResetAccessCounter();
            storage.GetStorage2().ResetAccessCounter();

            NNT_ASSERT_RESULT_SUCCESS(storage.Get().Write(offset, ptr, 0));

            // 書き込みは発生しない
            ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
            ASSERT_EQ(0, storage.GetModifiedStorage().GetWriteTimes());
            ASSERT_EQ(0, storage.GetStorage1().GetWriteTimes());
            ASSERT_EQ(0, storage.GetStorage2().GetWriteTimes());
        }

        // アクセス回数リセット
        storage.GetOriginalStorage().ResetAccessCounter();
        storage.GetModifiedStorage().ResetAccessCounter();
        storage.GetStorage1().ResetAccessCounter();
        storage.GetStorage2().ResetAccessCounter();

        // 範囲外に書き込み
        for( int j = 0; j <= 3; ++j )
        {
            int64_t offset =
                std::uniform_int_distribution<int64_t>(storageSize + 1, storageSize * 2)(mt);
            size_t size =
                std::uniform_int_distribution<size_t>(1, storageSize)(mt);

            NNT_ASSERT_RESULT_FAILURE(
                nn::fs::ResultInvalidOffset,
                storage.Get().Write(offset, ptr, size)
            );
        }

        // 範囲外エラー発生時に書き込みは行われない
        ASSERT_EQ(0, storage.GetOriginalStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetModifiedStorage().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage1().GetWriteTimes());
        ASSERT_EQ(0, storage.GetStorage2().GetWriteTimes());
    }

    NN_SDK_LOG("\n");
} // NOLINT(impl/function_size)

class DuplexStorageTestSetup : public TestStorageSetup
{
public:
    static const int64_t MinFileSize = 1;

public:
    // コンストラクタです。
    DuplexStorageTestSetup() NN_NOEXCEPT
    {
    }

    // ファイルオブジェクトをフォーマットします。
    virtual void Format(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        m_BitCount = static_cast<uint32_t>((sizeStorage + sizeBlock - 1) / sizeBlock);

        int64_t bitSize = nn::fssystem::dbm::DuplexBitmap::QuerySize(m_BitCount);

        // 二重化ビットマップを初期化します。
        m_pStorageOriginalBitmap.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pStorageOriginalBitmap->Initialize(bitSize + offset * 2);

        m_pStorageModifiedBitmap.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pStorageModifiedBitmap->Initialize(bitSize + offset * 2);

        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::save::DuplexBitmapHolder::Format(
                m_BitCount,
                nn::fs::SubStorage(
                    m_pStorageModifiedBitmap.get(),
                    offset,
                    bitSize + offset * 2
                ),
                nn::fs::SubStorage(
                    m_pStorageOriginalBitmap.get(),
                    offset,
                    bitSize + offset * 2
                )
            )
        );

        // 元データ、二重化データ、ベリファイ用領域を初期化します。
        m_pStorageData1.reset(new nnt::fs::util::SafeMemoryStorage());
        m_pStorageData1->Initialize(sizeStorage + offset * 2);

        m_pStorageData2.reset(new nnt::fs::util::SafeMemoryStorage);
        m_pStorageData2->Initialize(sizeStorage + offset * 2);

        // 破壊チェック
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageOriginalBitmap->Write(
                0, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageOriginalBitmap->Write(
                bitSize + offset, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageModifiedBitmap->Write(
                0, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageModifiedBitmap->Write(
                bitSize + offset, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageData1->Write(
                0, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageData1->Write(
                sizeStorage + offset, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageData2->Write(
                0, "AAAAAAAA", static_cast<size_t>(offset)));
        NNT_ASSERT_RESULT_SUCCESS(
            m_pStorageData2->Write(
                sizeStorage + offset, "AAAAAAAA", static_cast<size_t>(offset)));

        // バッファを全て同じ内容に初期化
        std::memset(m_pStorageData1->GetBuffer(), 0, static_cast<size_t>(sizeStorage));
        std::memset(m_pStorageData2->GetBuffer(), 0, static_cast<size_t>(sizeStorage));
    }

    // ファイルオブジェクトを初期化します。
    virtual void Initialize(
                     int64_t sizeStorage,
                     int64_t offset,
                     size_t sizeBlock
                 ) NN_NOEXCEPT NN_OVERRIDE
    {
        static const int CacheCountMax = 1024;
        static const size_t CacheBlockSize = 4096;
        static const size_t CacheBufferSize = 64 * 1024;

        // キャッシュバッファマネージャを初期化します。
        m_BufCache.resize(CacheBufferSize);
        NNT_ASSERT_RESULT_SUCCESS(
            m_CacheBuffer.Initialize(
                CacheCountMax,
                reinterpret_cast<uintptr_t>(&m_BufCache[0]),
                CacheBufferSize,
                CacheBlockSize
            )
        );

        // 二重化ファイルをマウントします。
        {
            int64_t sizeModified;
            int64_t sizeOriginal;
            m_pStorageModifiedBitmap->GetSize(&sizeModified);
            m_pStorageOriginalBitmap->GetSize(&sizeOriginal);

            m_pDuplexBitmap.reset(new nn::fssystem::save::DuplexBitmapHolder());
            m_pDuplexBitmap->Initialize(
                m_BitCount,
                nn::fs::SubStorage(m_pStorageModifiedBitmap.get(), offset, sizeModified),
                nn::fs::SubStorage(m_pStorageOriginalBitmap.get(), offset, sizeOriginal)
            );
        }

        // 更新系
        m_pDuplexStorage.reset(new nn::fssystem::save::DuplexStorage());
        nn::fs::SubStorage storageData1(m_pStorageData1.get(), offset, sizeStorage);
        nn::fs::SubStorage storageData2(m_pStorageData2.get(), offset, sizeStorage);
        m_pDuplexStorage->Initialize(
            m_pDuplexBitmap.get(),
            storageData1,
            storageData2,
            sizeBlock,
            &m_CacheBuffer
            );

        // テスト対象のファイルオブジェクトを設定します。
        SetStorage(m_pDuplexStorage.get());
    }

    // ファイルオブジェクトをアンマウントします。
    virtual void Unmount() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pDuplexStorage.reset(nullptr);
        m_pDuplexBitmap->Finalize();
        m_pDuplexBitmap.reset(nullptr);
        m_CacheBuffer.Finalize();
    }

    // ファイルオブジェクトを破棄します。
    virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pDuplexBitmap->Finalize();
        m_CacheBuffer.Finalize();
    }

    // 読み込み専用かどうか
    virtual bool IsReadOnly() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルへのデータ書き込み時に自動伸長するかどうか
    virtual bool IsAutoExtendSizeOnWrite() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // ファイルのサイズ変更が可能かどうか
    virtual bool IsEnableResize() const NN_NOEXCEPT NN_OVERRIDE
    {
        return false;
    }

    // サイズ 0 のファイルを作成可能かどうか
    virtual bool IsAcceptsSizeZero() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

private:
    uint32_t m_BitCount;
    nnt::fs::util::Vector<char> m_BufCache;
    nn::fssystem::FileSystemBufferManager m_CacheBuffer;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pStorageOriginalBitmap;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pStorageModifiedBitmap;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pStorageData1;
    std::unique_ptr<nnt::fs::util::SafeMemoryStorage> m_pStorageData2;
    std::unique_ptr<nn::fssystem::save::DuplexBitmapHolder> m_pDuplexBitmap;
    std::unique_ptr<nn::fssystem::save::DuplexStorage> m_pDuplexStorage;
};

// 共通テストを行います。
TEST_F(FsDuplexStorageTest, StorageCommonTest)
{
    StorageTest::RunTest<DuplexStorageTestSetup>();
}
