﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/fs_BucketTreeBuilder.h>
#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#endif

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nn { namespace fssystem {

/**
 * @brief   IndirectStorage のテストクラスです。
 */
class IndirectStorageTest : public ::testing::Test
{
public:
    class StorageWrapperBase;
    class StorageWrapper;
    class LargeStorageWrapper;

    typedef nn::fssystem::IndirectStorage ImplType;
    typedef ImplType::Entry TableEntry;
    typedef ImplType::ContinuousReadingEntry ContinuousReadingEntry;

public:
    static const size_t NodeSize = ImplType::NodeSize;
    static const size_t EntrySize = sizeof(ImplType::Entry);
    static const int StorageCount = 2;

public:
    static Result GetVisitor(
        BucketTree::Visitor* pOutVisitor,
        const ImplType& storage,
        int64_t offset
    ) NN_NOEXCEPT
    {
        return storage.m_Table.Find(pOutVisitor, offset);
    }
};

NN_DEFINE_STATIC_CONSTANT(const size_t IndirectStorageTest::NodeSize);
NN_DEFINE_STATIC_CONSTANT(const size_t IndirectStorageTest::EntrySize);
NN_DEFINE_STATIC_CONSTANT(const int IndirectStorageTest::StorageCount);

/**
 * @brief   IndirectStorage のラッパークラスです。
 */
class IndirectStorageTest::StorageWrapperBase
{
public:
    //! コンストラクタです。
    StorageWrapperBase() NN_NOEXCEPT
        : m_TableStorage()
        //, m_DataStorage()
        , m_Impl()
        , m_Entries()
    {
    }

    void Initialize() NN_NOEXCEPT
    {
        typedef nn::fssystem::BucketTreeBuilder TableBuilder;

        // エントリデータを作成
        InitializeEntryData(&m_Entries, &m_EndOffset);

        const auto entryCount = static_cast<int>(m_Entries.size());
        const auto headerStorageSize =
            TableBuilder::QueryHeaderStorageSize();
        const auto nodeStorageSize =
            TableBuilder::QueryNodeStorageSize(NodeSize, EntrySize, entryCount);
        const auto entryStorageSize =
            TableBuilder::QueryEntryStorageSize(NodeSize, EntrySize, entryCount);
        const auto headerStorageOffset = 0;
        const auto nodeStorageOffset = headerStorageSize;
        const auto entryStorageOffset = headerStorageSize + nodeStorageSize;

        m_TableStorage.Initialize(headerStorageSize + nodeStorageSize + entryStorageSize);

        // テーブルデータを作成
        {
            TableBuilder builder;
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                builder.Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    nn::fs::SubStorage(&m_TableStorage, headerStorageOffset, headerStorageSize),
                    nn::fs::SubStorage(&m_TableStorage, nodeStorageOffset, nodeStorageSize),
                    nn::fs::SubStorage(&m_TableStorage, entryStorageOffset, entryStorageSize),
                    NodeSize,
                    EntrySize,
                    entryCount
                )
            );

            for( auto& entry : m_Entries )
            {
                ImplType::Entry data;
                data.SetVirtualOffset(entry.virtualOffset);
                data.SetPhysicalOffset(entry.physicalOffset);
                data.storageIndex = entry.storageIndex;

                NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Write(data));
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Commit(m_EndOffset));
        }

        // ストレージを初期化
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_Impl.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&m_TableStorage, 0, m_TableStorage.GetSize())
            )
        );

        for( int i = 0; i < StorageCount; ++i )
        {
            nn::fs::IStorage* pStorage = GetDataStorage(i);
            int64_t storageSize = 0;
            NN_ABORT_UNLESS_RESULT_SUCCESS(pStorage->GetSize(&storageSize));
            m_Impl.SetStorage(i, pStorage, 0, storageSize);
        }
    } // NOLINT(impl/function_size)

    virtual ~StorageWrapperBase() NN_NOEXCEPT
    {
    }

    //! ストレージを読み込みます。
    nn::Result Read(int64_t offset, char* buffer, size_t size) NN_NOEXCEPT
    {
        return m_Impl.Read(offset, buffer, size);
    }

    //! 読み込んだデータをチェックします。
    bool Verify(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT
    {
        auto pos = std::upper_bound(
            std::begin(m_Entries),
            std::end(m_Entries),
            offset,
            [](int64_t value, const Entry& entry) NN_NOEXCEPT
            {
                return value < entry.virtualOffset;
            }
        );
        NN_ABORT_UNLESS(pos != m_Entries.begin());

        if( pos == m_Entries.end() )
        {
            NN_SDK_ASSERT_RANGE(offset, (pos - 1)->virtualOffset, m_EndOffset);
        }
        else
        {
            NN_SDK_ASSERT_RANGE(offset, (pos - 1)->virtualOffset, pos->virtualOffset);
        }
        --pos;

        int64_t dataOffset = 0;
        int64_t dataSize = size;
        auto readBuffer = nnt::fs::util::AllocateBuffer(size);
        while( 0 < dataSize )
        {
            const auto readOffset = offset + dataOffset - pos->virtualOffset;
            NN_SDK_ASSERT_LESS_EQUAL(readOffset, static_cast<int64_t>(pos->physicalSize));
            const auto readSize = std::min<int64_t>(dataSize, pos->physicalSize - readOffset);

            const auto* const ptr1 = buffer + dataOffset;
            NN_ABORT_UNLESS_RESULT_SUCCESS(GetDataStorage(pos->storageIndex)->Read(
                pos->physicalOffset + readOffset, readBuffer.get(), static_cast<size_t>(readSize)));

            if( std::memcmp(ptr1, readBuffer.get(), static_cast<size_t>(readSize)) != 0 )
            {
                return false;
            }

            dataOffset += readSize;
            dataSize -= readSize;

            ++pos;
        }
        return true;
    }

    //! ストレージを取得します。
    ImplType& GetImpl() NN_NOEXCEPT
    {
        return m_Impl;
    }

    //! ストレージを取得します。
    const ImplType& GetImpl() const NN_NOEXCEPT
    {
        return m_Impl;
    }

    //! 仮想オフセットの最大値を取得します。
    int64_t GetEndOffset() const NN_NOEXCEPT
    {
        return m_EndOffset;
    }

protected:
    struct Entry
    {
        int64_t virtualOffset;
        int64_t physicalOffset;
        int64_t physicalSize;
        int storageIndex;
    };

    typedef nnt::fs::util::Vector<Entry> EntryArray;

protected:
    virtual nn::fs::IStorage* GetDataStorage(int index) NN_NOEXCEPT = 0;
    virtual void InitializeEntryData(EntryArray* pOutEntryArray, int64_t* pEndOffset) NN_NOEXCEPT = 0;

private:
    nnt::fs::util::SafeMemoryStorage m_TableStorage;
    ImplType m_Impl;
    EntryArray m_Entries;
    int64_t m_EndOffset;
};

class IndirectStorageTest::StorageWrapper : public IndirectStorageTest::StorageWrapperBase
{
public:
    static const int EntryCount = 10000;

protected:
    virtual nn::fs::IStorage* GetDataStorage(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_DataStorage[index];
    }

    virtual void InitializeEntryData(EntryArray* pOutEntryArray, int64_t* pOutEndOffset) NN_NOEXCEPT NN_OVERRIDE
    {
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        nnt::fs::util::Vector<EntryArray::iterator> entryIter;

        EntryArray entryArray(EntryCount);

        // 実データのサイズとストレージを選択
        for( auto iter = entryArray.begin(); iter != entryArray.end(); ++iter )
        {
            const int64_t physicalSize =
                nn::util::align_up(
                    std::uniform_int_distribution<int64_t>(64, 1024)(mt),
                    sizeof(int64_t)
                );

            iter->physicalSize = physicalSize;
            iter->storageIndex = std::uniform_int_distribution<>(0, 1)(mt);

            entryIter.push_back(iter);
        }

        {
            std::mt19937 mt2(nnt::fs::util::GetRandomSeed());
            std::shuffle(std::begin(entryIter), std::end(entryIter), mt2);
        }

        // 物理オフセットを付与
        int64_t physicalOffset[StorageCount] = {};
        for( auto iter = entryIter.begin(); iter != entryIter.end(); ++iter )
        {
            // 直前のイテレータと同じ領域を付与
            if( iter != std::begin(entryIter) &&
                std::uniform_int_distribution<>(0, 20)(mt) == 0 )
            {
                **iter = **(iter - 1);
            }
            else
            {
                const auto index = (*iter)->storageIndex;

                (*iter)->physicalOffset = physicalOffset[index];
                physicalOffset[index] += (*iter)->physicalSize;
            }
        }

        // 仮想オフセットを付与
        int64_t virtualOffset = 0;
        for( auto& entry : entryArray )
        {
            entry.virtualOffset = virtualOffset;
            virtualOffset += entry.physicalSize;
        }

        // エントリに従ってデータを生成
        {
            for( int i = 0; i < StorageCount; ++i )
            {
                m_DataStorage[i].Initialize(physicalOffset[i]);
            }

            for( const auto& entry : entryArray )
            {
                const auto index = entry.storageIndex;

                char* buffer = reinterpret_cast<char*>(m_DataStorage[index].GetBuffer());
                buffer += entry.physicalOffset;

                std::iota(buffer, buffer + entry.physicalSize, '\x0');
            }
        }

        *pOutEntryArray = std::move(entryArray);
        *pOutEndOffset = virtualOffset;
    }

private:
    nnt::fs::util::SafeMemoryStorage m_DataStorage[StorageCount];
};

class IndirectStorageTest::LargeStorageWrapper : public IndirectStorageTest::StorageWrapperBase
{
public:
    static const int BufferSize = 1024;

public:
    LargeStorageWrapper() NN_NOEXCEPT
    {
    }

    virtual ~LargeStorageWrapper() NN_NOEXCEPT NN_OVERRIDE
    {
        for( auto& dataStorage : m_DataStorage )
        {
            dataStorage.Finalize();
        }
    }

protected:
    virtual nn::fs::IStorage* GetDataStorage(int index) NN_NOEXCEPT NN_OVERRIDE
    {
        return &m_DataStorage[index];
    }

    virtual void InitializeEntryData(EntryArray* pOutEntryArray, int64_t* pOutEndOffset) NN_NOEXCEPT NN_OVERRIDE
    {
        // 実際にデータを書き込むオフセット
        const int64_t offsetList[] =
        {
            0,
            static_cast<int64_t>(4) * 1024 * 1024 * 1024,
            static_cast<int64_t>(8) * 1024 * 1024 * 1024,
            static_cast<int64_t>(16) * 1024 * 1024 * 1024,
            static_cast<int64_t>(32) * 1024 * 1024 * 1024,
        };
        const auto offsetListLength = sizeof(offsetList) / sizeof(int64_t);

        EntryArray entryArray;

        // 物理オフセットを付与
        for( size_t i = 0; i < offsetListLength; ++i )
        {
            for( int storageIndex = 0; storageIndex < StorageCount; ++storageIndex )
            {
                Entry entry;
                entry.storageIndex = storageIndex;
                entry.physicalOffset = offsetList[i];
                entry.physicalSize = BufferSize;
                entryArray.push_back(entry);

                if( i + 1 < offsetListLength )
                {
                    // 実際にはデータを書き込まない、間の部分のエントリ
                    Entry fillEntry;
                    fillEntry.storageIndex = storageIndex;
                    fillEntry.physicalOffset = entry.physicalOffset + entry.physicalSize;
                    fillEntry.physicalSize = offsetList[i + 1] - fillEntry.physicalOffset;
                    entryArray.push_back(fillEntry);
                }
            }
        }

        // 仮想オフセットを付与
        int64_t virtualOffset = 0;
        for( auto& entry : entryArray )
        {
            entry.virtualOffset = virtualOffset;
            virtualOffset += entry.physicalSize;
        }

        // エントリに従ってデータを生成
        {
            for( int i = 0; i < StorageCount; ++i )
            {
                m_DataStorage[i].Initialize(offsetList[offsetListLength - 1] + BufferSize);
            }

            auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
            for( const auto& entry : entryArray )
            {
                // offsetList のエントリだけ書き込み（他は 0 フィル扱い）
                if( entry.physicalSize == BufferSize )
                {
                    nnt::fs::util::FillBufferWithRandomValue(buffer.get(), BufferSize);

                    NNT_ASSERT_RESULT_SUCCESS(m_DataStorage[entry.storageIndex].Write(
                        entry.physicalOffset, buffer.get(), BufferSize));
                }
            }
        }

        *pOutEntryArray = std::move(entryArray);
        *pOutEndOffset = virtualOffset;
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_DataStorage[StorageCount];
};

}}

namespace {

typedef nn::fssystem::IndirectStorageTest IndirectStorageTest;
typedef nn::fssystem::IndirectStorageTest IndirectStorageDeathTest;

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証の範囲外のテストをします。
 */
TEST_F(IndirectStorageDeathTest, Precondition)
{
    static const int EntryCount = 10000;

    nn::fs::SubStorage subStorage;
    nn::fssystem::IndirectStorage storage;

    // static 関数の調査
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fssystem::IndirectStorage::QueryNodeStorageSize(-1),
        ""
    );
    EXPECT_DEATH_IF_SUPPORTED(
        nn::fssystem::IndirectStorage::QueryEntryStorageSize(-1),
        ""
    );

    // pAllocator 不正
    EXPECT_DEATH_IF_SUPPORTED(
        storage.Initialize(nullptr, subStorage, subStorage, EntryCount),
        ""
    );

    // offset 不正
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(-1, nullptr, 1), "");

    // buffer 不正
    EXPECT_DEATH_IF_SUPPORTED(storage.Read(0, nullptr, 1), "");

    // 再初期化
    StorageWrapper storageWrapper;
    storageWrapper.Initialize();
    EXPECT_DEATH_IF_SUPPORTED(
        storageWrapper.GetImpl().Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            subStorage,
            subStorage,
            EntryCount
        ),
        ""
    );
}
#endif

/**
 * @brief   想定したリザルトを返すかをテストします。
 */
TEST_F(IndirectStorageTest, ReturnResult)
{
    static const int BlockSize = 64;

    nn::fs::SubStorage subStorage;
    StorageWrapper storage;
    storage.Initialize();
    nnt::fs::util::Vector<char> buffer(BlockSize);
    const auto offsetMax = storage.GetEndOffset();

    // 0 バイト読み込み
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(0, nullptr, 0));

    // ストレージ範囲内のアクセス
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(0, buffer.data(), BlockSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(offsetMax - BlockSize, buffer.data(), BlockSize));
    NNT_EXPECT_RESULT_SUCCESS(storage.Read(offsetMax, buffer.data(), 0));

    // ストレージ範囲外のアクセス
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        storage.Read(offsetMax - BlockSize + 4, buffer.data(), BlockSize)
    );
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultOutOfRange,
        storage.Read(offsetMax - BlockSize, buffer.data(), BlockSize + 4)
    );

    // サイズ取得
    int64_t size = 0;
    NNT_EXPECT_RESULT_SUCCESS(storage.GetImpl().GetSize(&size));
}

/**
 * @brief   全てのデータを順に読み込みます。
 */
TEST_F(IndirectStorageTest, ReadAll)
{
    static const size_t BlockSize = 512;

    nnt::fs::util::Vector<char> buffer(BlockSize);
    StorageWrapper storage;
    storage.Initialize();
    const auto offsetMax = storage.GetEndOffset();

    for( int64_t offset = 0; offset < offsetMax; offset += BlockSize )
    {
        const auto readSize =
            std::min(static_cast<size_t>(offsetMax - offset), BlockSize);

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), readSize));

        EXPECT_TRUE(storage.Verify(offset, buffer.data(), readSize));
    }
}

/**
 * @brief   ランダムなサイズとオフセットでストレージを読み込みます。
 */
TEST_F(IndirectStorageTest, ReadRandom)
{
    static const size_t BlockSize = 1024;

    nnt::fs::util::Vector<char> buffer(BlockSize);
    StorageWrapper storage;
    storage.Initialize();
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    const auto offsetMax = storage.GetEndOffset();

    for( auto count = 0; count < 1000; ++count )
    {
        const auto offset = std::uniform_int_distribution<int64_t>(0, offsetMax - 1)(mt);
        const auto size = std::uniform_int_distribution<size_t>(
                              1, std::min(BlockSize, static_cast<size_t>(offsetMax - offset))
                          )(mt);

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), size));

        EXPECT_TRUE(storage.Verify(offset, buffer.data(), size));
    }
}

/**
 * @brief   マルチスレッドでのストレージの読み込みをテストします。
 */
TEST_F(IndirectStorageTest, MultiThread)
{
    static const int ThreadCount = 8;
    static const size_t StackSize = 32 * 1024;
    static const size_t BlockSize = 1024;

    NN_OS_ALIGNAS_THREAD_STACK static char s_Stack[StackSize * ThreadCount] = {};

    StorageWrapper storage;
    storage.Initialize();
    nn::os::ThreadType threads[ThreadCount];
    const auto func = [](void* pArg) NN_NOEXCEPT
    {
        const auto pStorage = reinterpret_cast<StorageWrapper*>(pArg);
        const auto offsetMax = pStorage->GetEndOffset();

        nnt::fs::util::Vector<char> buffer(BlockSize);
        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        for( auto count = 0; count < 500; ++count )
        {
            const auto offset =
                std::uniform_int_distribution<int64_t>(0, offsetMax - 1)(mt);
            const auto size =
                std::uniform_int_distribution<size_t>(
                    1, std::min(BlockSize, static_cast<size_t>(offsetMax - offset))
                )(mt);

            NNT_ASSERT_RESULT_SUCCESS(
                pStorage->Read(offset, buffer.data(), size));

            EXPECT_TRUE(pStorage->Verify(offset, buffer.data(), size));
        }
    };

    // スレッド起動
    for( auto& thread : threads )
    {
        nn::os::CreateThread(
            &thread,
            func,
            &storage,
            s_Stack + StackSize * (&thread - threads),
            StackSize,
            nn::os::DefaultThreadPriority
        );
        nn::os::StartThread(&thread);
    }

    // スレッド削除
    for( auto& thread : threads )
    {
        nn::os::WaitThread(&thread);
        nn::os::DestroyThread(&thread);
    }
}

/**
 * @brief   直接作成したテーブルを使用してテストします。
 */
TEST_F(IndirectStorageTest, RawData)
{
    typedef nn::fssystem::BucketTree Table;

    struct NodeData
    {
        int32_t index;
        int32_t count;
        int64_t end;
        int64_t begin;
    };
    NN_STATIC_ASSERT(sizeof(NodeData) == (sizeof(Table::NodeHeader) + sizeof(int64_t)));

    static const int EntryCount2 = 10;

    struct EntryData
    {
        Table::NodeHeader header;
        TableEntry entries[EntryCount2];

        static const TableEntry Make(
                                    int64_t virtualOffset,
                                    int64_t physicalOffset,
                                    int storageIndex
                                ) NN_NOEXCEPT
        {
            TableEntry entry;
            entry.SetVirtualOffset(virtualOffset);
            entry.SetPhysicalOffset(physicalOffset);
            entry.storageIndex = storageIndex;
            return entry;
        }
    };

    static const size_t DataSize = 32;
    static const char SampleData[] =
    {
        "GVE64YZB0UBH4BBCXH85LETW20JWQH8Q"
        "X5AVN9A2HX7BEK7READ8133MIH4LF2MB"
        "OIJNMAC4RMI4V27L616ON4LUGMCNOSFE"
        "IC4T8WV6EWV4QVZU2FRDZV0M0QNGXFCE"
        "CW3XL6VACM9SNFNDM338N6M3L37NVXWL"
        "SSJPZLMW84JEV4THC2RND8QAMGWASPEX"
        "ZMJNB31HKGPMAN74TPEY16UHU50HKGXM"
        "G1KWEOHSTIZHPIID8I8MY1JAF9NN7DP8"
        "C439SQYX26E4L4CSPI8T5OKGQ7GCO7N4"
        "Y7UGM20B3RCQ7I7AGOO20BOTI98QXJBY"
    };
    NN_STATIC_ASSERT(sizeof(SampleData) == DataSize * EntryCount2 + 1);

    // L1 ノード
    static const NodeData NodeL1Data =
    {
        0,
        1,
        DataSize * EntryCount2,
        0
    };
    // エントリデータ
    static const EntryData EntriesData =
    {
        { 0, EntryCount2, DataSize * EntryCount2 },
        {
            EntryData::Make(DataSize *  0, DataSize * 0, 0),
            EntryData::Make(DataSize *  1, DataSize * 3, 0),
            EntryData::Make(DataSize *  2, DataSize * 2, 1),
            EntryData::Make(DataSize *  3, DataSize * 2, 0),
            EntryData::Make(DataSize *  4, DataSize * 0, 1),
            EntryData::Make(DataSize *  5, DataSize * 1, 1),
            EntryData::Make(DataSize *  6, DataSize * 4, 1),
            EntryData::Make(DataSize *  7, DataSize * 4, 0),
            EntryData::Make(DataSize *  8, DataSize * 1, 0),
            EntryData::Make(DataSize *  9, DataSize * 3, 1),
        }
    };

    static const int TableStorageSize = NodeSize * 2;
    static const int DataStorageSize = sizeof(SampleData) - 1; // ヌル文字分引く

    ASSERT_EQ(
        Table::QueryNodeStorageSize(NodeSize, sizeof(TableEntry), EntryCount2),
        NodeSize
    );
    ASSERT_EQ(
        Table::QueryEntryStorageSize(NodeSize, sizeof(TableEntry), EntryCount2),
        NodeSize
    );

    // テーブルの設定
    nnt::fs::util::SafeMemoryStorage tableStorage(TableStorageSize);
    ASSERT_TRUE(
        nn::util::is_aligned(
            reinterpret_cast<uintptr_t>(tableStorage.GetBuffer()),
            nn::DefaultAlignment
        )
    );
    {
        nn::util::BytePtr buffer(tableStorage.GetBuffer());
        std::fill(buffer.Get<char>(), buffer.Get<char>() + TableStorageSize, '\x0');

        // L1 部分のコピー
        std::memcpy(buffer.Get(), &NodeL1Data, sizeof(NodeL1Data));

        buffer.Advance(NodeSize);

        // エントリのコピー
        std::memcpy(buffer.Get(), &EntriesData, sizeof(EntriesData));
    }

    // データの設定
    nnt::fs::util::SafeMemoryStorage dataStorage(DataStorageSize);
    std::memcpy(dataStorage.GetBuffer(), SampleData, DataStorageSize);

    nn::fssystem::IndirectStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(
        storage.Initialize(
            nnt::fs::util::GetTestLibraryAllocator(),
            nn::fs::SubStorage(&tableStorage, 0, NodeSize),
            nn::fs::SubStorage(&tableStorage, NodeSize, NodeSize),
            EntryCount2 + 1
        )
    );

    storage.SetStorage(0, &dataStorage, 0, DataStorageSize / 2);
    storage.SetStorage(1, &dataStorage, DataStorageSize / 2, DataStorageSize / 2);

    nnt::fs::util::Vector<char> buffer(DataStorageSize);
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // データ読み込み
    for( int loop = 0; loop < 200; ++loop )
    {
        auto offset = std::uniform_int_distribution<int64_t>(0, buffer.size() - 1)(mt);
        const auto size = std::uniform_int_distribution<size_t>(
                    1, DataSize - static_cast<size_t>(offset % DataSize))(mt);

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.data(), size));

        auto pos = std::upper_bound(
            std::begin(EntriesData.entries),
            std::end(EntriesData.entries),
            offset,
            [](int64_t value, const TableEntry& entry) NN_NOEXCEPT
            {
                return value < entry.GetVirtualOffset();
            }
        );
        ASSERT_NE(pos, std::begin(EntriesData.entries));

        if( pos == std::end(EntriesData.entries) )
        {
            const int64_t endOffset = DataSize * EntryCount2;
            ASSERT_LE((pos - 1)->GetVirtualOffset(), offset);
            ASSERT_LT(offset, endOffset);
        }
        else
        {
            ASSERT_LE((pos - 1)->GetVirtualOffset(), offset);
            ASSERT_LT(offset, pos->GetVirtualOffset());
        }
        --pos;

        offset -= pos->GetVirtualOffset();
        offset += pos->GetPhysicalOffset() + pos->storageIndex * (DataStorageSize / 2);

        NNT_FS_UTIL_ASSERT_MEMCMPEQ(&SampleData[offset], buffer.data(), size);
    }
} // NOLINT(impl/function_size)

#if defined(NN_BUILD_CONFIG_OS_WIN)
/**
 * @brief   まとめ読みのテストします。
 */
TEST_F(IndirectStorageTest, ContinuousReading)
{
    static const size_t BlockSize = 16;
    static const int BlockRound = 128;
    static const size_t RegionSize = BlockSize * BlockRound;
    static const size_t SegmentSize = RegionSize * 4;
    static const size_t DataSize = SegmentSize * 4;

    std::unique_ptr<char[]> originalData(new char[DataSize]);
    NN_ABORT_UNLESS_NOT_NULL(originalData);
    std::unique_ptr<char[]> currentData(new char[DataSize]);
    NN_ABORT_UNLESS_NOT_NULL(currentData);
    char patchData[RegionSize * 2];

    for( int i = 0; i < DataSize / RegionSize; ++i )
    {
        std::memset(originalData.get() + RegionSize * i, i, RegionSize);
    }

    std::memcpy(currentData.get(), originalData.get(), DataSize);
    std::memset(currentData.get() + SegmentSize + RegionSize, 127, RegionSize);
    std::memset(currentData.get() + SegmentSize * 2 + RegionSize * 3, 126, RegionSize);
    std::memset(currentData.get() + DataSize - RegionSize, 0, RegionSize);

    std::memset(patchData + RegionSize * 0, 127, RegionSize);
    std::memset(patchData + RegionSize * 1, 126, RegionSize);

    nn::fs::MemoryStorage oldStorage(originalData.get(), DataSize);
    nn::fs::MemoryStorage newStorage(currentData.get(), DataSize);

    nn::fssystem::utilTool::IndirectStorageBuilder builder;
    NNT_ASSERT_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage, BlockSize));
    NNT_ASSERT_RESULT_SUCCESS(builder.Build(BlockSize, RegionSize, 0));

    ASSERT_EQ(sizeof(nn::fssystem::BucketTree::Header), builder.QueryTableHeaderStorageSize());
    ASSERT_EQ(nn::fssystem::IndirectStorage::NodeSize, builder.QueryTableNodeStorageSize());
    ASSERT_EQ(nn::fssystem::IndirectStorage::NodeSize, builder.QueryTableEntryStorageSize());
    ASSERT_EQ(RegionSize * 2, builder.QueryDataStorageSize());

    nn::fssystem::BucketTree::Header header;
    std::unique_ptr<char[]> nodeBuffer(
        new char[static_cast<size_t>(builder.QueryTableNodeStorageSize())]);
    NN_ABORT_UNLESS_NOT_NULL(nodeBuffer);
    std::unique_ptr<char[]> entryBuffer(
        new char[static_cast<size_t>(builder.QueryTableEntryStorageSize())]);
    NN_ABORT_UNLESS_NOT_NULL(entryBuffer);

    nn::fs::MemoryStorage headerStorage(&header, sizeof(header));
    nn::fs::MemoryStorage nodeStorage(nodeBuffer.get(), builder.QueryTableNodeStorageSize());
    nn::fs::MemoryStorage entryStorage(entryBuffer.get(), builder.QueryTableEntryStorageSize());

    NNT_ASSERT_RESULT_SUCCESS(builder.WriteTable(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&headerStorage, 0, sizeof(header)),
        nn::fs::SubStorage(&nodeStorage, 0, builder.QueryTableNodeStorageSize()),
        nn::fs::SubStorage(&entryStorage, 0, builder.QueryTableEntryStorageSize())
    ));
    ASSERT_EQ(6, header.entryCount);
    {
        char patchBuffer[sizeof(patchData)];
        NNT_ASSERT_RESULT_SUCCESS(builder.ReadData(0, patchBuffer, sizeof(patchBuffer)));
        NNT_FS_UTIL_ASSERT_MEMCMPEQ(patchData, patchBuffer, sizeof(patchBuffer));
    }
    nn::fs::MemoryStorage dataStorage(patchData, sizeof(patchData));

    nn::fssystem::IndirectStorage storage;
    NNT_ASSERT_RESULT_SUCCESS(storage.Initialize(
        nnt::fs::util::GetTestLibraryAllocator(),
        nn::fs::SubStorage(&nodeStorage, 0, builder.QueryTableNodeStorageSize()),
        nn::fs::SubStorage(&entryStorage, 0, builder.QueryTableEntryStorageSize()),
        header.entryCount
    ));
    storage.SetStorage(0, &oldStorage, 0, DataSize);
    storage.SetStorage(1, &dataStorage, 0, sizeof(patchData));

    std::unique_ptr<char[]> buffer(new char[DataSize]);
    NN_ABORT_UNLESS_NOT_NULL(buffer);

    NNT_ASSERT_RESULT_SUCCESS(storage.Read(0, buffer.get(), DataSize));
    NNT_FS_UTIL_ASSERT_MEMCMPEQ(currentData.get(), buffer.get(), DataSize);

    const auto check = [&](
        int64_t offset,
        size_t size,
        size_t expectedReadSize,
        int expectedSkipCount,
        bool expectedCanDo
    ) NN_NOEXCEPT -> nn::Result
    {
        nn::fssystem::BucketTree::Visitor visitor;
        NN_RESULT_DO(GetVisitor(&visitor, storage, offset));
        nn::fssystem::BucketTree::ContinuousReadingInfo info;
        NN_RESULT_DO(visitor.ScanContinuousReading<ContinuousReadingEntry>(
            &info, offset, size
        ));
        EXPECT_EQ(expectedReadSize, info.GetReadSize());
        EXPECT_EQ(expectedSkipCount, info.GetSkipCount());
        EXPECT_EQ(expectedCanDo, info.CanDo());
        NN_RESULT_DO(storage.Read(offset, buffer.get(), size));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(currentData.get() + offset, buffer.get(), size);
        NN_RESULT_SUCCESS;
    };

    // 元ロムの領域に収まる読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(0, RegionSize, 0, 1, false));

    // 元ロムとパッチをまたぐ読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize, RegionSize + RegionSize / 2, 0, 1, false));

    // 元ロムとパッチの境界まで読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize, RegionSize * 2, 0, 1, false));

    // パッチをはさんで元ロムを読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize, RegionSize * 3, RegionSize * 3, 3, true));

    // パッチに収まる読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize + RegionSize, RegionSize, 0, 0, false));

    // パッチと元ロムをまたぐ読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize + RegionSize, RegionSize * 2, 0, 0, false));

    // パッチをはさんで元ロムの境界まで読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize * 2, SegmentSize * 2 - RegionSize, SegmentSize * 2 - RegionSize, 3, true));

    // 不連続な元ロムを読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize * 3, SegmentSize, 0, 1, false));

    // 不連続な元ロムを含む読み込み
    NNT_EXPECT_RESULT_SUCCESS(
        check(SegmentSize * 2, SegmentSize * 2, SegmentSize * 2 - RegionSize, 3, true));
} // NOLINT(impl/function_size)

/**
 * @brief   QueryRange をテストします。
 */
TEST_F(IndirectStorageTest, QueryRange)
{
    static const size_t BlockSize = 1024;

    nnt::fs::util::Vector<char> buffer(BlockSize);
    StorageWrapper storage;
    storage.Initialize();
    std::mt19937 mt;

    int64_t size = 0;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetImpl().GetSize(&size));

    nn::fs::QueryRangeInfo info;
    NNT_ASSERT_RESULT_SUCCESS(storage.GetImpl().OperateRange(
        &info, sizeof(info), nn::fs::OperationId::QueryRange, 0, size, nullptr, 0));
    EXPECT_EQ(0, info.aesCtrKeyTypeFlag);
    EXPECT_EQ(0, info.speedEmulationTypeFlag);
}

/**
 * @brief   巨大なサイズでの読み込みをテストします。
 */
TEST_F(IndirectStorageTest, LargeFile)
{
    LargeStorageWrapper storage;
    storage.Initialize();
    const auto offsetMax = storage.GetEndOffset();

    // LargeStorageWrapper で実際に書き込んだデータがあるオフセット
    const int64_t offsetList[] =
    {
        0,
        static_cast<int64_t>(4) * 1024 * 1024 * 1024,
        static_cast<int64_t>(8) * 1024 * 1024 * 1024,
        static_cast<int64_t>(12) * 1024 * 1024 * 1024,
        static_cast<int64_t>(16) * 1024 * 1024 * 1024,
        static_cast<int64_t>(24) * 1024 * 1024 * 1024,
        static_cast<int64_t>(32) * 1024 * 1024 * 1024,
        static_cast<int64_t>(48) * 1024 * 1024 * 1024,
        static_cast<int64_t>(64) * 1024 * 1024 * 1024,
    };

    static const auto BufferSize = LargeStorageWrapper::BufferSize;
    auto buffer = nnt::fs::util::AllocateBuffer(BufferSize);
    auto zeroBuffer = nnt::fs::util::AllocateBuffer(BufferSize);
    std::memset(zeroBuffer.get(), 0, BufferSize);
    for( auto offset : offsetList )
    {
        ASSERT_LT(offset + BufferSize, offsetMax);

        NNT_ASSERT_RESULT_SUCCESS(storage.Read(offset, buffer.get(), BufferSize));

        // データのある場所を読み込んでいるか
        EXPECT_NE(0, std::memcmp(buffer.get(), zeroBuffer.get(), BufferSize));

        // データの内容が正しいか
        EXPECT_TRUE(storage.Verify(offset, buffer.get(), BufferSize));
    }
}

#endif
