﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/fs_BucketTreeUtility.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nnt/nnt_Argument.h>
#include <nnt/nnt_Compiler.h>
#include <nnt/base/testBase_Exit.h>
#include "testFs_Unit_BucketTreeUtil.h"

namespace {

typedef nn::fssystem::BucketTree BucketTree;
typedef BucketTree::Header Header;
typedef BucketTree::NodeHeader NodeHeader;
typedef nn::fssystem::detail::SafeValue ValueProxy;
typedef nn::fssystem::BucketTreeTest BucketTreeTest;
typedef BucketTreeTest BucketTreeDeathTest;

/**
 * @brief   BucketTree を作成するクラスです。
 */
template< typename T, typename TMemoryStorage = nnt::fs::util::SafeMemoryStorage >
class BucketTreeCreator : public BucketTreeData<T, TMemoryStorage>
{
public:
    //! コンストラクタです。
    BucketTreeCreator(size_t nodeSize, uint32_t entryCount) NN_NOEXCEPT
        : BaseType(nodeSize, entryCount)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            m_BucketTree.Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                nn::fs::SubStorage(&this->m_NodeStorage, 0, this->m_NodeStorageSize),
                nn::fs::SubStorage(&this->m_EntryStorage, 0, this->m_EntryStorageSize),
                this->m_NodeSize,
                this->m_EntrySize,
                this->m_EntryCount
            )
        );
    }

    //! BucketTree を取得します。
    nn::fssystem::BucketTree& Get() NN_NOEXCEPT
    {
        return m_BucketTree;
    }

    //! BucketTree を取得します。
    const nn::fssystem::BucketTree& Get() const NN_NOEXCEPT
    {
        return m_BucketTree;
    }

protected:
    nn::fssystem::BucketTree m_BucketTree;

private:
    typedef BucketTreeData<T, TMemoryStorage> BaseType;
};

}

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   事前検証のデステストをします。
 */
TEST_F(BucketTreeDeathTest, Precondition)
{
    // GetNodeStorageSize() のテスト
    {
        // entrySize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(
                BucketTree::NodeSizeMin, sizeof(int64_t) - 1, 8
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(
                BucketTree::NodeSizeMin,
                BucketTree::NodeSizeMin - sizeof(NodeHeader) + 1,
                8
            ),
            ""
        );

        // nodeSize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin - 1, 128, 8), "");
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMax + 1, 128, 8), "");

        // nodeSize が２の累乗以外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin + 1, 128, 8), "");

        // entryCount の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin, 128, -1), "");
    }

    // GetEntryStorageSize() のテスト
    {
        // entrySize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(
                BucketTree::NodeSizeMin, sizeof(int64_t) - 1, 8
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(
                BucketTree::NodeSizeMin,
                BucketTree::NodeSizeMin - sizeof(NodeHeader) + 1,
                8
            ),
            ""
        );

        // nodeSize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin - 1, 128, 8), "");
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMax + 1, 128, 8), "");

        // nodeSize が２の累乗以外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin + 1, 128, 8), "");

        // entryCount の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin, 128, -1), "");
    }

    // Initialize() のテスト
    {
        nn::fs::SubStorage storage;

        // pAllocator が nullptr
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nullptr,
                storage,
                storage,
                BucketTree::NodeSizeMin,
                sizeof(int64_t),
                1
            ),
            ""
        );

        // entrySize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                storage,
                storage,
                BucketTree::NodeSizeMin,
                sizeof(int64_t) - 1,
                8
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                storage,
                storage,
                BucketTree::NodeSizeMin,
                BucketTree::NodeSizeMin - sizeof(NodeHeader) + 1,
                8
            ),
            ""
        );

        // nodeSize の境界値外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                storage,
                storage,
                BucketTree::NodeSizeMin - 1,
                128,
                8
            ),
            ""
        );
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                storage,
                storage,
                BucketTree::NodeSizeMax + 1,
                128,
                8
            ),
            ""
        );

        // nodeSize が２の累乗以外
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                storage,
                storage,
                BucketTree::NodeSizeMin + 1,
                128,
                8
            ),
            ""
        );

        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Initialize(BucketTree::NodeSizeMin / 2, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Initialize(BucketTree::NodeSizeMax * 2, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Initialize(BucketTree::NodeSizeMin + 1, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Initialize(BucketTree::NodeSizeMin, -1), "");
    }

    // Find() のテスト
    {
        // pIter が nullptr
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Find(nullptr, 0), "");

        // virtualOffset が不正
        BucketTree::Visitor visitor;
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Find(&visitor, -1), "");

        // 未初期化
        EXPECT_DEATH_IF_SUPPORTED(BucketTree().Find(&visitor, 0), "");
    }

    // Visitor のテスト
    {
        // IsValid() がエラーを返す時は事前検証に失敗する
        EXPECT_FALSE(BucketTree::Visitor().IsValid());
        EXPECT_DEATH_IF_SUPPORTED(BucketTree::Visitor().Get(), "");
    }
} // NOLINT(impl/function_size)
#endif

/**
 * @brief   事前検証のテストをします。
 */
TEST_F(BucketTreeTest, Precondition)
{
    // GetNodeStorageSize() のテスト
    {
        // entrySize の境界値内
        BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin, sizeof(int64_t), 8);
        BucketTree::QueryNodeStorageSize(
            BucketTree::NodeSizeMin, BucketTree::NodeSizeMin - sizeof(NodeHeader), 8);

        // nodeSize の境界値内
        BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin, 128, 8);
        BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMax, 128, 8);

        // nodeSize が２の累乗
        for( size_t nodeSize = BucketTree::NodeSizeMin;
             nodeSize <= BucketTree::NodeSizeMax;
             nodeSize <<= 1 )
        {
            BucketTree::QueryNodeStorageSize(nodeSize, 128, 8);
        }

        // entryCount の境界値内
        BucketTree::QueryNodeStorageSize(BucketTree::NodeSizeMin, 128, 0);
    }

    // GetEntryStorageSize() のテスト
    {
        // entrySize の境界値内
        BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin, sizeof(int64_t), 8);
        BucketTree::QueryEntryStorageSize(
            BucketTree::NodeSizeMin, BucketTree::NodeSizeMin - sizeof(NodeHeader), 8);

        // nodeSize の境界値内
        BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin, 128, 8);
        BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMax, 128, 8);

        // nodeSize が２の累乗
        for( size_t nodeSize = BucketTree::NodeSizeMin;
             nodeSize <= BucketTree::NodeSizeMax;
             nodeSize <<= 1 )
        {
            BucketTree::QueryEntryStorageSize(nodeSize, 128, 8);
        }

        // entryCount の境界値内
        BucketTree::QueryEntryStorageSize(BucketTree::NodeSizeMin, 128, 0);
    }

    // Initialize() のテスト
    // NOTE: 全て事前検証はパスするがストレージのアクセスで失敗する
    {
        nnt::fs::util::SafeMemoryStorage storage(4);
        nn::fs::SubStorage subStorage(&storage, 0, 4);

        // entrySize の境界値内
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                sizeof(int64_t),
                8
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                BucketTree::NodeSizeMin - sizeof(NodeHeader),
                8
            )
        );

        // nodeSize の境界値内
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                128,
                8
            )
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMax,
                128,
                8
            )
        );

        // nodeSize が２の累乗
        for( size_t nodeSize = BucketTree::NodeSizeMin;
             nodeSize <= BucketTree::NodeSizeMax;
             nodeSize <<= 1 )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                BucketTree().Initialize(
                    nnt::fs::util::GetTestLibraryAllocator(),
                    subStorage,
                    subStorage,
                    nodeSize,
                    128,
                    8
                )
            );
        }

        // entryCount の境界値外
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                128,
                -1
            )
        );

        // entryCount の境界値内
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                128,
                0
            )
        );

        // entryCount の境界値内
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange,
            BucketTree().Initialize(
                nnt::fs::util::GetTestLibraryAllocator(),
                subStorage,
                subStorage,
                BucketTree::NodeSizeMin,
                128,
                1
            )
        );
    }
} // NOLINT(impl/function_size)

#if !defined(NN_SDK_BUILD_RELEASE)
/**
 * @brief   関数が想定した通り abort するかをテストします。
 */
TEST_F(BucketTreeDeathTest, StorageSize)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    for( int i = 0; i < 100; ++i )
    {
        int entryCount = 8;
        const auto nodeSize = BucketTree::NodeSizeMin;
        const auto entrySize = std::uniform_int_distribution<size_t>(
                                    16, (BucketTree::NodeSizeMin - 1) / entryCount)(mt);

        const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);
        const auto offsetCountPerNode = GetOffsetCount(nodeSize);

        // エントリ数の最大値
        entryCount = offsetCountPerNode * offsetCountPerNode * entryCountPerNode;

        // エントリ数の最大値 + 1 は abort
        ++entryCount;
        EXPECT_DEATH_IF_SUPPORTED(
            BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount), "");
    }
}
#endif

/**
 * @brief   関数が想定した値を返すかをテストします。
 */
TEST_F(BucketTreeTest, ReturnValue)
{
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // BucketTree::GetNodeStorageSize() のテスト
    for( int i = 0; i < 100; ++i )
    {
        int entryCount = 8;
        const auto nodeSize = BucketTree::NodeSizeMin;
        const auto entrySize = std::uniform_int_distribution<size_t>(
                                    16, (BucketTree::NodeSizeMin - 1) / entryCount)(mt);

        // L1 １ノードに収まる数
        ASSERT_EQ(nodeSize, BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount));

        const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);
        const auto offsetCountPerNode = GetOffsetCount(nodeSize);

        // L1 １ノードに収まる最大数
        entryCount = entryCountPerNode * offsetCountPerNode;
        ASSERT_EQ(nodeSize, BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount));

        for( int j = 1; j <= offsetCountPerNode; ++j )
        {
            const auto nodeCount = j + 1;

            // L2 が j ノード必要なエントリ数の最小値
            ++entryCount;
            ASSERT_EQ(
                nodeCount * static_cast<int64_t>(nodeSize),
                BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount)
            );

            // L2 が j ノード必要なエントリ数の最大値
            entryCount = (offsetCountPerNode - j) * entryCountPerNode + // L1 ノードに詰めるエントリの数
                         offsetCountPerNode * j * entryCountPerNode;    // L2 に設定するエントリの数
            ASSERT_EQ(
                nodeCount * static_cast<int64_t>(nodeSize),
                BucketTree::QueryNodeStorageSize(nodeSize, entrySize, entryCount)
            );
        }
    }

    // BucketTree::GetEntryStorageSize() のテスト
    for( size_t nodeSize = BucketTree::NodeSizeMin;
         nodeSize <= BucketTree::NodeSizeMax;
         nodeSize <<= 1 )
    {
        for( int i = 0; i < 500; ++i )
        {
            const auto entrySize = std::uniform_int_distribution<size_t>(16, 256)(mt);
            const auto entryCount = std::uniform_int_distribution<>(10, 10000)(mt);
            const auto entrySetCount = GetEntrySetCount(nodeSize, entrySize, entryCount);

            ASSERT_EQ(
                entrySetCount * nodeSize,
                BucketTree::QueryEntryStorageSize(nodeSize, entrySize, entryCount)
            );
        }
    }

    // IsInclude() のテスト
    {
        // offset < 0
        EXPECT_FALSE(BucketTree().IsInclude(-1));

        EXPECT_FALSE(BucketTree().IsInclude(0));

        // offset < 0
        EXPECT_FALSE(BucketTree().IsInclude(-1, 0));

        // size <= 0
        EXPECT_FALSE(BucketTree().IsInclude(0, 0));

        EXPECT_FALSE(BucketTree().IsInclude(0, 1));

        const auto max = std::numeric_limits<int64_t>::max();
        EXPECT_FALSE(BucketTree().IsInclude(max, max));
    }
}

/**
 * @brief   関数が想定したリザルトを返すかをテストします。
 */
TEST_F(BucketTreeTest, ReturnResult)
{
    // Header::Verify() のチェック
    {
        Header header = { 0, BucketTree::Version + 4, 1 };

        // signature 不正
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidBucketTreeSignature, header.Verify());
        header.signature = BucketTree::Signature;

        // version 不正
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultUnsupportedVersion, header.Verify());
        header.version = BucketTree::Version;

        // entryCount 不正
        header.entryCount = -1;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultInvalidBucketTreeEntryCount, header.Verify());
    }

    // NodeHeader::Verify() のチェック
    {
        NodeHeader header = { 1, 2, -1 };

        // nodeIndex 不正
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidArgument,
            header.Verify(0, BucketTree::NodeSizeMin, sizeof(int64_t))
        );

        // サイズ不正
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            header.Verify(1, BucketTree::NodeSizeMin, 0)
        );
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidSize,
            header.Verify(1, sizeof(NodeHeader), sizeof(NodeHeader))
        );

        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidBucketTreeNodeEntryCount,
            header.Verify(1, BucketTree::NodeSizeMin, BucketTree::NodeSizeMin - sizeof(NodeHeader))
        );

        // count 不正
        header.count = 0;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidBucketTreeNodeEntryCount,
            header.Verify(1, BucketTree::NodeSizeMin, BucketTree::NodeSizeMin - sizeof(NodeHeader))
        );
        header.count = 1;

        // offset 不正
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidBucketTreeNodeOffset,
            header.Verify(1, BucketTree::NodeSizeMin, sizeof(int64_t))
        );
    }

    // BucketTree::Find() のテスト
    {
        nn::fssystem::BucketTree bucketTree;
        bucketTree.Initialize(BucketTree::NodeSizeMin, 256);
        BucketTree::Visitor visitor;
        NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, bucketTree.Find(&visitor, 0));
    }
    {
        BucketTreeCreator<IndirectStorageData> data(BucketTree::NodeSizeMin, 4);
        BucketTree::Visitor visitor;

        // オフセット範囲外
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange, data.Get().Find(&visitor, 0)
        );

        // オフセット範囲外
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultOutOfRange, data.Get().Find(&visitor, data.GetEndOffset())
        );

        // エントリデータを破壊
        std::memset(data.GetEntryStorage().GetBuffer(), 0, sizeof(NodeHeader));

        auto offset = ValueProxy::GetInt64(data.GetEntry(0).virtualOffset);
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidBucketTreeNodeEntryCount, data.Get().Find(&visitor, offset)
        );
    }

    // BucketTree::Visitor::MoveNext() のテスト
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, BucketTree::Visitor().MoveNext());

    // BucketTree::Visitor::MovePrevious() のテスト
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultOutOfRange, BucketTree::Visitor().MovePrevious());
}

/**
 * @brief   ツリーの範囲をテストします。
 */
TEST_F(BucketTreeTest, CheckRange)
{
    typedef IndirectStorageData StorageData;

    // 初期化前は常に 0 を返す
    EXPECT_EQ(0, BucketTree().GetSize());

    const auto nodeSize = BucketTree::NodeSizeMin;
    const auto entrySize = sizeof(StorageData);
    const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);

    // L1 のみで収まるツリーを作成
    {
        BucketTreeCreator<StorageData> data(nodeSize, entryCountPerNode);

        const auto size = data.GetEndOffset() - data.GetEntry()[0].GetOffset();
        EXPECT_EQ(size, data.Get().GetSize());

        EXPECT_EQ(data.GetEntry()[0].GetOffset(), data.Get().GetBegin());
        EXPECT_EQ(data.GetEndOffset(), data.Get().GetEnd());
    }

    // L2 を含むツリーを作成
    {
        const auto offsetCountPerNode = GetOffsetCount(nodeSize);
        const auto entryCount = entryCountPerNode * offsetCountPerNode * 8;

        BucketTreeCreator<StorageData> data(nodeSize, entryCount);

        const auto size = data.GetEndOffset() - data.GetEntry()[0].GetOffset();
        EXPECT_EQ(size, data.Get().GetSize());

        EXPECT_EQ(data.GetEntry()[0].GetOffset(), data.Get().GetBegin());
        EXPECT_EQ(data.GetEndOffset(), data.Get().GetEnd());
    }
}

NNT_DISABLE_OPTIMIZATION
/**
 * @brief   Find() をテストします。
 */
TEST_F(BucketTreeTest, Find)
{
    typedef IndirectStorageData StorageData;

    const auto nodeSize = BucketTree::NodeSizeMin;
    const auto entrySize = sizeof(StorageData);
    const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);

    // L1 のみで収まるツリーを作成
    {
        BucketTreeCreator<StorageData> data(nodeSize, entryCountPerNode);
        BucketTree::Visitor visitor;

        // ツリー範囲の最小値
        const auto* pEntry = data.GetEntry();
        auto offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        // ツリー範囲の最大値
        pEntry = data.GetEntry() + (data.GetEntryCount() - 1);
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, data.GetEndOffset() - 1));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
    }

    // L2 を含むツリーを作成
    {
        const auto offsetCountPerNode = GetOffsetCount(nodeSize);
        const auto entryCount = entryCountPerNode * offsetCountPerNode * 8;
        const auto nodeCountL2 = GetNodeL2Count(nodeSize, entrySize, entryCount);
        const auto entryCountL2OnL1 = (offsetCountPerNode - nodeCountL2) * entryCountPerNode;

        BucketTreeCreator<StorageData> data(nodeSize, entryCount);
        BucketTree::Visitor visitor;

        // ツリー範囲の最小値
        const auto* pEntry = data.GetEntry();
        auto offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        // L1 ノード上の L2 データの最大値
        pEntry = data.GetEntry() + (entryCountL2OnL1 - 1);
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        // L2 ノードの最小値
        pEntry = data.GetEntry() + entryCountL2OnL1;
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
        {
            nn::util::BytePtr pNode(data.GetNodeStorage().GetBuffer());
            pNode.Advance(nodeSize + sizeof(NodeHeader));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(pNode.Get(), visitor.Get(), sizeof(int64_t));
        }

        // L2 最小ノードの最大値
        pEntry = data.GetEntry() + (entryCountL2OnL1 + offsetCountPerNode * entryCountPerNode - 1);
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        // ツリー範囲の最大値
        pEntry = data.GetEntry() + (data.GetEntryCount() - 1);
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, data.GetEndOffset() - 1));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        // ランダム
        for( int i = 0; i < 500; ++i )
        {
            // TODO: 最適化を off にしないと Win64_VS2017(ver 15.5) で int64_t(index) すると化ける（コンパイラのバグ？）
            const auto index = std::uniform_int_distribution<>(0, entryCount - 1)(mt);

            pEntry = data.GetEntry() + index;
            offset = ValueProxy::GetInt64(pEntry->virtualOffset);

            NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);

            NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset + 2));
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
        }
    }
}

/**
 * @brief   MoveNext(), MovePrevious() をテストします。
 */
TEST_F(BucketTreeTest, Move)
{
    typedef IndirectStorageData StorageData;

    const auto nodeSize = BucketTree::NodeSizeMin;
    const auto entrySize = sizeof(StorageData);
    const auto entryCountPerNode = GetEntryCount(nodeSize, entrySize);
    const auto offsetCountPerNode = GetOffsetCount(nodeSize);
    const auto entryCount = entryCountPerNode * offsetCountPerNode * 8;
    const auto nodeCountL2 = GetNodeL2Count(nodeSize, entrySize, entryCount);
    const auto entryCountL2OnL1 = (offsetCountPerNode - nodeCountL2) * entryCountPerNode;

    BucketTreeCreator<StorageData> data(nodeSize, entryCount);
    BucketTree::Visitor visitor;

    // ツリー範囲の最小値
    const auto* pEntry = data.GetEntry();
    auto offset = ValueProxy::GetInt64(pEntry->virtualOffset);
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    // ツリー範囲外には進めない
    EXPECT_FALSE(visitor.CanMovePrevious());
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, visitor.MovePrevious());

    // Move でエラーが起きた後は無効になっている事があるので再検索
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    EXPECT_TRUE(visitor.CanMoveNext());
    NNT_EXPECT_RESULT_SUCCESS(visitor.MoveNext());
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry + 1, visitor.Get<StorageData>(), entrySize);

    // L1 ノード上の L2 データの最大値
    pEntry = data.GetEntry() + (entryCountL2OnL1 - 1);
    offset = ValueProxy::GetInt64(pEntry->virtualOffset);
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    // L1 ノードと L2 ノードを移動
    {
        EXPECT_TRUE(visitor.CanMoveNext());
        NNT_EXPECT_RESULT_SUCCESS(visitor.MoveNext());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry + 1, visitor.Get<StorageData>(), entrySize);

        EXPECT_TRUE(visitor.CanMovePrevious());
        NNT_EXPECT_RESULT_SUCCESS(visitor.MovePrevious());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
    }

    // L2 最小ノードの最大値
    pEntry = data.GetEntry() + (entryCountL2OnL1 + offsetCountPerNode * entryCountPerNode - 1);
    offset = ValueProxy::GetInt64(pEntry->virtualOffset);
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    // L2 ノードの境界を移動
    {
        EXPECT_TRUE(visitor.CanMoveNext());
        NNT_EXPECT_RESULT_SUCCESS(visitor.MoveNext());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry + 1, visitor.Get<StorageData>(), entrySize);

        EXPECT_TRUE(visitor.CanMovePrevious());
        NNT_EXPECT_RESULT_SUCCESS(visitor.MovePrevious());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
    }

    // ツリー範囲の最大値
    pEntry = data.GetEntry() + (data.GetEntryCount() - 1);
    offset = ValueProxy::GetInt64(pEntry->virtualOffset);
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    // ツリー範囲外には進めない
    EXPECT_FALSE(visitor.CanMoveNext());
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultOutOfRange, visitor.MoveNext());

    // Move でエラーが起きた後は無効になっている事があるので再検索
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    NNT_EXPECT_RESULT_SUCCESS(visitor.MovePrevious());
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry - 1, visitor.Get<StorageData>(), entrySize);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    // ランダム
    for( int i = 0; i < 500; ++i )
    {
        // TODO: 最適化を off にしないと Win64_VS2017(ver 15.5) で int64_t(index) すると化ける（コンパイラのバグ？）
        const auto index = std::uniform_int_distribution<>(0, entryCount - 2)(mt);

        pEntry = data.GetEntry() + index;
        offset = ValueProxy::GetInt64(pEntry->virtualOffset);
        NNT_ASSERT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

        EXPECT_TRUE(visitor.CanMoveNext());
        NNT_ASSERT_RESULT_SUCCESS(visitor.MoveNext());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry + 1, visitor.Get<StorageData>(), entrySize);
        EXPECT_TRUE(visitor.CanMovePrevious());
        NNT_ASSERT_RESULT_SUCCESS(visitor.MovePrevious());
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(pEntry, visitor.Get<StorageData>(), entrySize);
    }
}
NNT_RESTORE_OPTIMIZATION

/**
 * @brief   巡回オブジェクトをテストします。
 */
TEST_F(BucketTreeTest, Visitor)
{
    typedef AesCtrCounterExtendedStorageData StorageData;

    const auto nodeSize = BucketTree::NodeSizeMin;
    const auto entryCount = 1000;

    BucketTreeCreator<StorageData> data(nodeSize, entryCount);
    BucketTree::Visitor visitor;

    // 未初期化でも呼び出せるが常に false
    EXPECT_FALSE(visitor.CanMoveNext());
    EXPECT_FALSE(visitor.CanMovePrevious());

    // Find() より前は無効な値を返す
    {
        EXPECT_FALSE(visitor.IsValid());
        EXPECT_EQ(visitor.GetTree(), nullptr);
    }

    const auto* pEntry = data.GetEntry();
    auto offset = ValueProxy::GetInt64(pEntry->offset);
    NNT_EXPECT_RESULT_SUCCESS(data.Get().Find(&visitor, offset));

    // Find() より後は有効な値を返す
    {
        EXPECT_TRUE(visitor.IsValid());
        EXPECT_NE(visitor.Get(), nullptr);
        EXPECT_EQ(visitor.GetTree(), &data.Get());
    }

    EXPECT_TRUE(visitor.CanMoveNext());
    EXPECT_FALSE(visitor.CanMovePrevious());

    for( int i = 1; i < entryCount - 1; ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(visitor.MoveNext());
        EXPECT_TRUE(visitor.CanMoveNext());
        EXPECT_TRUE(visitor.CanMovePrevious());
    }

    NNT_ASSERT_RESULT_SUCCESS(visitor.MoveNext());
    EXPECT_FALSE(visitor.CanMoveNext());
    EXPECT_TRUE(visitor.CanMovePrevious());
}
