﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#if !defined(NNT_FS_NCA_FILE_SYSTEM_DRIVER_WITH_NCA_FILES)

#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_AesXtsStorage.h>
#include <nn/fssystem/fs_BucketTree.h>
#include <nn/fssystem/fs_NcaFileSystemDriver.h>
#include <nn/fssystem/fs_Utility.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>
#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include "testFs_NcaFileSystemDriverUtil.h"

namespace nn { namespace fssystem {

class NcaFsHeaderReaderTest
{
public:
    static void Fill(NcaFsHeaderReader* pOutValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        nnt::fs::util::FillBufferWith8BitCount(&pOutValue->m_Data, sizeof(pOutValue->m_Data), 0);
        pOutValue->m_FsIndex = 0;
    }
};

}}

namespace
{

typedef nn::fssystem::NcaHeader Header;
typedef nn::fssystem::NcaReader Reader;
typedef nn::fssystem::NcaFsHeader FsHeader;
typedef nn::fssystem::NcaFsHeaderReader FsHeaderReader;
typedef nn::fssystem::NcaCryptoConfiguration CryptoConfiguration;

const int AesBlockSize = 16;
const int AesKeySize = 16;

void EncryptNcaHeader(void* headerBuffer, const CryptoConfiguration& configuration) NN_NOEXCEPT
{
    // テスト用なのでバッファサイズは渡さない
    NN_SDK_REQUIRES_NOT_NULL(headerBuffer);

    char zeroIv[nn::fssystem::AesXtsStorage::IvSize] = {0};

    char xtsKey[2][AesBlockSize];
    configuration.pGenerateKey(
        xtsKey[0], AesBlockSize, configuration.headerEncryptedEncryptionKey[0], AesBlockSize, int(nn::fssystem::KeyType::NcaHeaderKey), configuration);
    configuration.pGenerateKey(
        xtsKey[1], AesBlockSize, configuration.headerEncryptedEncryptionKey[1], AesBlockSize, int(nn::fssystem::KeyType::NcaHeaderKey), configuration);

    NN_ABORT_UNLESS(nn::util::is_aligned(Header::Size, Header::XtsBlockSize));
    for( int i = 0; i < Header::Size / Header::XtsBlockSize; ++i )
    {
        auto srcDstBuffer = reinterpret_cast<char*>(headerBuffer) + Header::XtsBlockSize * i;
        auto encryptedSize = nn::crypto::EncryptAes128Xts(
            srcDstBuffer,
            Header::XtsBlockSize,
            xtsKey[0],
            xtsKey[1],
            nn::fssystem::AesXtsStorage::KeySize,
            zeroIv,
            nn::fssystem::AesXtsStorage::IvSize,
            srcDstBuffer,
            Header::XtsBlockSize
        );
        NN_ABORT_UNLESS_EQUAL(size_t(Header::XtsBlockSize), encryptedSize);

        nn::fssystem::AddCounter(zeroIv, sizeof(zeroIv), 1);
    }
}

}

#if !defined(NN_SDK_BUILD_RELEASE)
// 事前検証
TEST(NcaReaderDeathTest, Precondition)
{
    nnt::fs::util::NcaReaderBuilder builder;

    EXPECT_DEATH_IF_SUPPORTED(Reader().GetSignature(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetDistributionType(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetContentType(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetKeyGeneration(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetKeyIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetContentSize(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetProgramId(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetContentIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetSdkAddonVersion(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsCount(), "");
    {
        uint8_t buffer[Header::RightIdSize];
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetRightsId(nullptr, Header::RightIdSize), "");
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetRightsId(buffer, sizeof(buffer) - 1), "");
    }
    EXPECT_DEATH_IF_SUPPORTED(Reader().HasFsInfo(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().HasFsInfo(Header::FsCountMax), "");
    {
        nn::fssystem::Hash hash;
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsHeaderHash(0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsHeaderHash(-1), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsHeaderHash(Header::FsCountMax), "");
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsHeaderHash(&hash, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsHeaderHash(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsHeaderHash(&hash, -1), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsHeaderHash(&hash, Header::FsCountMax), "");
    }
    {
        Header::FsInfo info;
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsInfo(&info, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsInfo(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsInfo(&info, -1), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsInfo(&info, Header::FsCountMax), "");
    }
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsOffset(0), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsOffset(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsOffset(Header::FsCountMax), "");

    EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsEndOffset(0), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsEndOffset(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsEndOffset(Header::FsCountMax), "");

    EXPECT_DEATH_IF_SUPPORTED(Reader().GetFsSize(0), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsSize(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetFsSize(Header::FsCountMax), "");
    {
        char encryptedKey[Header::EncryptedKeyAreaSize];
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetEncryptedKey(encryptedKey, sizeof(encryptedKey)), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetEncryptedKey(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetEncryptedKey(encryptedKey, sizeof(encryptedKey) - 1), "");
    }
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetDecryptionKey(0), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetDecryptionKey(-1), "");
    EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetDecryptionKey(Header::DecryptionKey_Count), "");
    {
        char decryptionKey[CryptoConfiguration::Aes128KeySize];
        EXPECT_DEATH_IF_SUPPORTED(Reader().SetExternalDecryptionKey(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(Reader().SetExternalDecryptionKey(decryptionKey, sizeof(decryptionKey) - 1), "");
    }
    {
        Header header;
        EXPECT_DEATH_IF_SUPPORTED(Reader().GetRawData(&header, sizeof(header)), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetRawData(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->GetRawData(&header, sizeof(header) - 1), "");
    }
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetExternalDecryptAesCtrFunction(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().GetExternalDecryptAesCtrFunctionForExternalKey(), "");
    {
        nn::fssystem::NcaFsHeader fsHeader;
        EXPECT_DEATH_IF_SUPPORTED(Reader().ReadHeader(nullptr, 0), "");
        EXPECT_DEATH_IF_SUPPORTED(Reader().ReadHeader(&fsHeader, -1), "");
        EXPECT_DEATH_IF_SUPPORTED(Reader().ReadHeader(&fsHeader, Header::FsCountMax), "");
    }
    EXPECT_DEATH_IF_SUPPORTED(Reader().Verify(), "");
    EXPECT_DEATH_IF_SUPPORTED(Reader().VerifyHeaderSign2(nullptr, 0), "");
    {
        nn::fs::MemoryStorage storage(nullptr, 0);
        EXPECT_DEATH_IF_SUPPORTED(Reader().Initialize(nullptr, nnt::fs::util::CryptoConfiguration), "");
        EXPECT_DEATH_IF_SUPPORTED(builder.Make()->Initialize(&storage, nnt::fs::util::CryptoConfiguration), "");
    }
}
#endif

// 各フィールドが正しく読めるかチェック
TEST(NcaReaderTest, GetApis)
{
    // ヘッダを用意
    Header header;
    nnt::fs::util::FillBufferWith8BitCount(&header, sizeof(header), 0);
    header.signature = Header::Signature;
    const auto contentSize = Header::Size + FsHeader::Size * Header::FsCountMax;
    header.contentSize = contentSize;
    header.keyIndex = 1;      // KeyIndex <= 2
    header.keyGeneration = 2; // KeyGeneration <= 2
    header.keyGeneration2 = 0;
    std::memset(header.fsInfo, 0, sizeof(header.fsInfo));
    nnt::fs::util::SignNcaHeader(header.headerSign1, sizeof(header.headerSign1), header);
    EncryptNcaHeader(&header, nnt::fs::util::CryptoConfiguration);

    nn::fs::MemoryStorage storage(&header, sizeof(header));

    Reader reader;
    NNT_ASSERT_RESULT_SUCCESS(reader.Initialize(&storage, nnt::fs::util::CryptoConfiguration));

    EXPECT_EQ(Header::Signature, reader.GetSignature());
    EXPECT_EQ(0x04, int(reader.GetDistributionType()));
    EXPECT_EQ(0x05, int(reader.GetContentType()));
    EXPECT_EQ(0x02, reader.GetKeyGeneration());
    EXPECT_EQ(0x01, reader.GetKeyIndex());

    EXPECT_EQ(contentSize, reader.GetContentSize());
    EXPECT_EQ(0x1716151413121110, reader.GetProgramId());
    EXPECT_EQ(0x1B1A1918, reader.GetContentIndex());
    EXPECT_EQ(0x1F1E1D1C, reader.GetSdkAddonVersion());
    // padding 0x10

    uint8_t rightsId[Header::RightIdSize];
    reader.GetRightsId(rightsId, sizeof(rightsId));
    EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(rightsId, sizeof(rightsId), 0x30));

    Header::FsInfo fsInfo;
    reader.GetFsInfo(&fsInfo, 0);
    EXPECT_EQ(0, fsInfo.startSector);
    EXPECT_EQ(0, fsInfo.endSector);
    EXPECT_EQ(0, fsInfo.hashTargetSize);
    // padding 0x04

    reader.GetFsInfo(&fsInfo, 1);
    EXPECT_EQ(0, fsInfo.startSector);
    EXPECT_EQ(0, fsInfo.endSector);
    EXPECT_EQ(0, fsInfo.hashTargetSize);
    // padding 0x04

    // (FsInfo[2, 3])

    nn::fssystem::Hash hash;
    nn::fssystem::Hash expectedHash;
    nnt::fs::util::FillBufferWith8BitCount(&expectedHash, sizeof(expectedHash), 0x80);
    reader.GetFsHeaderHash(&hash, 0);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&expectedHash, &hash, sizeof(expectedHash));

    nnt::fs::util::FillBufferWith8BitCount(&expectedHash, sizeof(expectedHash), 0xA0);
    reader.GetFsHeaderHash(&hash, 1);
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&expectedHash, &hash, sizeof(expectedHash));

    // (FsHeaderHash[2, 3])

    char encryptedKey[Header::EncryptedKeyAreaSize];
    char expectedEncryptedKey[Header::EncryptedKeyAreaSize];
    nnt::fs::util::FillBufferWith8BitCount(&expectedEncryptedKey, sizeof(expectedEncryptedKey), 0);
    reader.GetEncryptedKey(&encryptedKey, sizeof(encryptedKey));
    NNT_FS_UTIL_EXPECT_MEMCMPEQ(&expectedEncryptedKey, &encryptedKey, sizeof(expectedEncryptedKey));
}

// シグネチャのチェック
TEST(NcaReaderTest, Signature)
{
    // 不正なシグネチャ
    {
        char ncaHeaderBuffer[Header::Size] = {};
        nn::fs::MemoryStorage storage(ncaHeaderBuffer, sizeof(ncaHeaderBuffer));

        Reader reader;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultInvalidNcaSignature,
            reader.Initialize(&storage, nnt::fs::util::CryptoConfiguration)
        );
    }

    static const uint32_t OldSignature[] =
    {
        NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '0'),
        NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '1'),
        NN_UTIL_CREATE_SIGNATURE_4('N', 'C', 'A', '2'),
    };
    const size_t KeySize = 16;

    // 古いバージョン
    for( auto signature : OldSignature )
    {
        char ncaHeaderBuffer[Header::Size] = {};
        std::memcpy(&ncaHeaderBuffer[Header::HeaderSignSize * 2], &signature, 4);

        char key[2][AesKeySize];
        nnt::fs::util::CryptoConfiguration.pGenerateKey(
            key[0],
            AesKeySize,
            nnt::fs::util::CryptoConfiguration.headerEncryptedEncryptionKey[0],
            AesKeySize,
            static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
            nnt::fs::util::CryptoConfiguration
        );
        nnt::fs::util::CryptoConfiguration.pGenerateKey(
            key[1],
            AesKeySize,
            nnt::fs::util::CryptoConfiguration.headerEncryptedEncryptionKey[1],
            AesKeySize,
            static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
            nnt::fs::util::CryptoConfiguration
        );

        char iv[KeySize] = {};
        nn::crypto::EncryptAes128Xts(
            ncaHeaderBuffer,
            Header::XtsBlockSize,
            key[0],
            key[1],
            KeySize,
            iv,
            KeySize,
            ncaHeaderBuffer,
            Header::XtsBlockSize
        );

        iv[KeySize - 1] = 1;
        nn::crypto::EncryptAes128Xts(
            ncaHeaderBuffer + Header::XtsBlockSize,
            Header::XtsBlockSize,
            key[0],
            key[1],
            KeySize,
            iv,
            KeySize,
            ncaHeaderBuffer + Header::XtsBlockSize,
            Header::XtsBlockSize
        );

        nn::fs::MemoryStorage storage(ncaHeaderBuffer, sizeof(ncaHeaderBuffer));

        Reader reader;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultUnsupportedSdkVersion,
            reader.Initialize(&storage, nnt::fs::util::CryptoConfiguration)
        );
    }
}

#if !defined(NN_SDK_BUILD_RELEASE)
// 事前検証
TEST(NcaFsReaderDeathTest, Precondition)
{
    nnt::fs::util::NcaReaderBuilder builder;
    auto pReader = builder.Make();

    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().Initialize(*pReader, -1), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().Initialize(*pReader, Header::FsCountMax), "");

    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetHashData(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetVersion(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetFsIndex(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetFsType(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetHashType(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetEncryptionType(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetPatchInfo(), "");
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetAesCtrUpperIv(), "");
    {
        FsHeader header;
        EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().GetRawData(&header, sizeof(header)), "");
        {
            FsHeaderReader reader;
            NNT_ASSERT_RESULT_SUCCESS(reader.Initialize(*pReader, 0));
            EXPECT_DEATH_IF_SUPPORTED(reader.GetRawData(nullptr, 0), "");
        }
        {
            FsHeaderReader reader;
            NNT_ASSERT_RESULT_SUCCESS(reader.Initialize(*pReader, 0));
            EXPECT_DEATH_IF_SUPPORTED(reader.GetRawData(&header, sizeof(header) - 1), "");
        }
    }
    EXPECT_DEATH_IF_SUPPORTED(FsHeaderReader().Verify(Header::ContentType::Program), "");
    {
        FsHeaderReader reader;
        NNT_ASSERT_RESULT_SUCCESS(reader.Initialize(*pReader, 0));
        EXPECT_DEATH_IF_SUPPORTED(reader.Verify(Header::ContentType(-1)), "");
    }
    {
        FsHeaderReader reader;
        NNT_ASSERT_RESULT_SUCCESS(reader.Initialize(*pReader, 0));
        EXPECT_DEATH_IF_SUPPORTED(reader.Verify(Header::ContentType(int(Header::ContentType::PublicData) + 1)), "");
    }
}
#endif

// 各フィールドが正しく読めるかチェック
TEST(NcaFsReaderTest, GetApis)
{
    FsHeaderReader reader;
    nn::fssystem::NcaFsHeaderReaderTest::Fill(&reader);
    {
        FsHeader::HashData data;
        nnt::fs::util::FillBufferWith8BitCount(&data, sizeof(data), offsetof(FsHeader, hashData));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&data, &reader.GetHashData(), sizeof(data));
    }
    EXPECT_EQ(0, reader.GetFsIndex());
    EXPECT_EQ(0x0100, reader.GetVersion());
    EXPECT_EQ(0x02, int(reader.GetFsType()));
    EXPECT_EQ(0x03, int(reader.GetHashType()));
    EXPECT_EQ(0x04, int(reader.GetEncryptionType()));
    {
        nn::fssystem::NcaPatchInfo info;
        nnt::fs::util::FillBufferWith8BitCount(&info, sizeof(info), offsetof(FsHeader, patchInfo));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&info, &reader.GetPatchInfo(), sizeof(info));
    }
    {
        nn::fssystem::NcaAesCtrUpperIv iv;
        nnt::fs::util::FillBufferWith8BitCount(&iv, sizeof(iv), offsetof(FsHeader, aesCtrUpperIv));
        EXPECT_EQ(iv.value, reader.GetAesCtrUpperIv().value);
        EXPECT_EQ(0x43424140, reader.GetAesCtrUpperIv().part.generation);
        EXPECT_EQ(0x47464544, reader.GetAesCtrUpperIv().part.secureValue);
    }
    {
        nn::fssystem::NcaSparseInfo info;
        nnt::fs::util::FillBufferWith8BitCount(&info, sizeof(info), offsetof(FsHeader, sparseInfo));
        NNT_FS_UTIL_EXPECT_MEMCMPEQ(&info, &reader.GetSparseInfo(), sizeof(info));
    }
}

#endif
