﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto.h>
#include <nn/nn_Version.h>
#include <nn/fssystem/fs_AesCtrStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorage.h>
#include <nn/fssystem/fs_AllocatorUtility.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>
#include <nn/fssystem/utilTool/fs_IndirectStorageBuilder.h>
#include <nnt/fsUtil/testFs_util.h>
#include "testFs_NcaFileSystemDriverUtil.h"

NN_DEFINE_STATIC_CONSTANT(const int nnt::fs::util::NcaReaderBuilder::BlockSize);

namespace
{

typedef nn::fssystem::AesCtrStorage AesCtrStorage;
typedef nn::fssystem::AesXtsStorage AesXtsStorage;
typedef nn::fssystem::NcaHeader NcaHeader;
typedef nn::fssystem::NcaFsHeader NcaFsHeader;
typedef nn::fssystem::NcaReader Reader;
typedef nn::fssystem::NcaFsHeaderReader FsHeaderReader;

const int AesKeySize = 16;
const int AesBlockSize = 16;
const int XtsBlockSize = 512;
const int Fs0HashSize = 512; // 適当なパディングを含む
const int Fs1HashSize = 4 * 1024; // 適当なパディングを含む
const int Fs0DataSize = 1024 * 1024;
const int Fs1DataSize = 1024 * 1024 + 512 * 1024;

// fs ヘッダのオフセット（ncaの先頭からの値
const int FsHeaderOffset[2] = { NcaHeader::Size, NcaHeader::Size + NcaFsHeader::Size };

// ハッシュ領域のオフセット
// fs0 は 2MB 地点からとする
// fs1 は 3KB 地点から
const int FsHashOffset[2] = { 2 * 1024 * 1024, 3 * 1024 };

// データ領域のオフセット（同上
const int FsDataOffset[2] =
{
    FsHashOffset[0] + Fs0HashSize,
    FsHashOffset[1] + Fs1HashSize,
};

const int StorageSize = 4 * 1024 * 1024;
const int PatchStorageSize = StorageSize + 2 * 1024 * 1024;

void DecryptNcaContentKeyTest(
         void* pOutValue,
         size_t outSize,
         const void* pEncryptedKey,
         size_t encryptedKeySize,
         int keyTypeValue,
         const nn::fssystem::NcaCryptoConfiguration& cryptoConfiguration
     ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_NOT_NULL(pEncryptedKey);

    if( keyTypeValue <= nn::fssystem::GetKeyTypeValue(2, 1) )
    {
        nn::crypto::AesDecryptor128 aesDecryptor;
        aesDecryptor.Initialize(cryptoConfiguration.keyAreaEncryptionKeySource[keyTypeValue], cryptoConfiguration.Aes128KeySize);
        aesDecryptor.DecryptBlock(pOutValue, outSize, pEncryptedKey, encryptedKeySize);
    }
    else
    {
        NN_SDK_REQUIRES_EQUAL(keyTypeValue, static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey));

        if( std::memcmp(pEncryptedKey, nnt::fs::util::CryptoConfiguration.headerEncryptedEncryptionKey[0], encryptedKeySize) == 0 )
        {
            uint8_t key[] =
            {
                0xcb, 0x9a, 0x93, 0x9f, 0x82, 0x72, 0x54, 0x4a, 0x74, 0x5d, 0x28, 0x46, 0x9d, 0xcc, 0x38, 0x12
            };
            std::memcpy(pOutValue, key, outSize);
        }
        else if( std::memcmp(pEncryptedKey, nnt::fs::util::CryptoConfiguration.headerEncryptedEncryptionKey[1], encryptedKeySize) == 0 )
        {
            uint8_t key[] =
            {
                0x06, 0x31, 0x27, 0x06, 0xae, 0x62, 0x56, 0x8c, 0x5b, 0x7e, 0xe6, 0x9f, 0x7e, 0x01, 0x02, 0x24
            };
            std::memcpy(pOutValue, key, outSize);
        }
        else
        {
            NN_ABORT();
        }
    }
}

void DecryptAesCtrForHardwareAes(
         void* pDst,
         size_t dstSize,
         int keyTypeValue,
         const void* pEncryptedKey,
         size_t encryptedKeySize,
         const void* pIv,
         size_t ivSize,
         const void* pSrc,
         size_t srcSize
     ) NN_NOEXCEPT
{
    NN_UNUSED(keyTypeValue);
    // 事前検証は DecryptAes128Ctr に任せる
    nn::crypto::DecryptAes128Ctr(pDst, dstSize, pEncryptedKey, encryptedKeySize, pIv, ivSize, pSrc, srcSize);
}

}

namespace nnt { namespace fs { namespace util {

const uint8_t BodyAesKey[4][AesXtsStorage::KeySize] =
{
    {
        0x5a, 0x0d, 0x5b, 0xaf, 0xdf, 0x3d, 0x78, 0x35,
        0xbc, 0xe9, 0xe8, 0x15, 0x77, 0x7b, 0x98, 0x57,
    },
    {
        0x7c, 0x05, 0xdd, 0xd6, 0x15, 0x67, 0x9a, 0x31,
        0x51, 0x71, 0xb8, 0xcb, 0xb8, 0x2c, 0xb9, 0xd0,
    },
    {
        0xbc, 0xe9, 0xe8, 0x15, 0x77, 0x7b, 0x98, 0x57,
        0x5a, 0x0d, 0x5b, 0xaf, 0xdf, 0x3d, 0x78, 0x35,
    },
    {
        0x51, 0x71, 0xb8, 0xcb, 0xb8, 0x2c, 0xb9, 0xd0,
        0x7c, 0x05, 0xdd, 0xd6, 0x15, 0x67, 0x9a, 0x31,
    },
};

NcaReaderBuilder::NcaReaderBuilder() NN_NOEXCEPT
    : m_DataBuffer(new char[Fs0HashSize + Fs0DataSize + Fs1HashSize + Fs1DataSize])
    , m_NewDataBuffer()
    , m_StorageBuffer(new char[StorageSize])
    , m_PatchStorageBuffer()
{
    BuildStorage();
}

NcaReaderBuilder::NcaReaderBuilder(bool isPatch) NN_NOEXCEPT
    : m_DataBuffer(new char[Fs0HashSize + Fs0DataSize + Fs1HashSize + Fs1DataSize])
    , m_NewDataBuffer()
    , m_StorageBuffer(new char[StorageSize])
    , m_PatchStorageBuffer(isPatch ? new char[PatchStorageSize] : nullptr)
{
    BuildStorage();
    if( isPatch )
    {
        BuildPatchStorage();
    }
}

void NcaReaderBuilder::BuildStorage() NN_NOEXCEPT
{
    // メモリが確保できないならテストできないので abort する
    NN_ABORT_UNLESS_NOT_NULL(m_DataBuffer);
    NN_ABORT_UNLESS_NOT_NULL(m_StorageBuffer);

    const auto fs0HashBuffer = m_DataBuffer.get();
    const auto fs0DataBuffer = m_DataBuffer.get() + Fs0HashSize;
    const auto fs1HashBuffer = m_DataBuffer.get() + Fs0HashSize + Fs0DataSize;
    const auto fs1DataBuffer = m_DataBuffer.get() + Fs0HashSize + Fs0DataSize + Fs1HashSize;

    std::memset(m_StorageBuffer.get(), 0x00, StorageSize);
    nn::fs::MemoryStorage base(m_StorageBuffer.get(), StorageSize); // 全体

    FillBufferWith32BitCount(fs0DataBuffer, Fs0DataSize, 0);          // fs0: 1MB, 0x00000000～
    FillBufferWith32BitCount(fs1DataBuffer, Fs1DataSize, 0x10000000); // fs1: 1.5MB, 0x10000000～

    nn::fssystem::Hash masterHash[2]; // マスターハッシュ
    nn::fssystem::Hash hash[2][3]; // ハッシュ1段目計算（手動）

    // fs0 データ書き込み
    {
        std::memset(fs0HashBuffer, 0, Fs0HashSize);

        char iv[AesCtrStorage::IvSize] = {};
        AesCtrStorage::MakeIv(iv, sizeof(iv), 0, FsHashOffset[0]);

        nn::fs::SubStorage bodyRaw(&base, FsHashOffset[0], Fs0HashSize + Fs0DataSize);
        AesCtrStorage body(&bodyRaw, BodyAesKey[2], AesCtrStorage::KeySize, iv, sizeof(iv));

        // 1MB
        nn::crypto::GenerateSha256Hash(&hash[0][0], sizeof(Hash), fs0DataBuffer, BlockSize);
        std::memcpy(fs0HashBuffer, &hash[0][0], sizeof(Hash));

        nn::crypto::GenerateSha256Hash(&hash[0][1], sizeof(Hash), fs0DataBuffer + BlockSize, BlockSize);
        std::memcpy(fs0HashBuffer + sizeof(Hash), &hash[0][1], sizeof(Hash));

        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(0, fs0HashBuffer, sizeof(Hash) * 2));
        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(Fs0HashSize, fs0DataBuffer, Fs0DataSize));
    }

    // fs0 マスターハッシュ
    nn::crypto::GenerateSha256Hash(&masterHash[0], sizeof(hash), hash[0], nn::crypto::Sha256Generator::HashSize * 2);

    // fs1 データ書き込み
    {
        std::memset(fs1HashBuffer, 0, Fs1HashSize);

        char iv[AesCtrStorage::IvSize] = {};
        AesCtrStorage::MakeIv(iv, sizeof(iv), 0, FsHashOffset[1]);

        nn::fs::SubStorage bodyRaw(&base, FsHashOffset[1], Fs1HashSize + Fs1DataSize);
        AesCtrStorage body(&bodyRaw, BodyAesKey[2], AesCtrStorage::KeySize, iv, sizeof(iv));

        // 1MB + 1B
        nn::crypto::GenerateSha256Hash(&hash[1][0], sizeof(Hash), fs1DataBuffer, BlockSize);
        std::memcpy(fs1HashBuffer, &hash[1][0], sizeof(Hash));

        nn::crypto::GenerateSha256Hash(&hash[1][1], sizeof(Hash), fs1DataBuffer + BlockSize, BlockSize);
        std::memcpy(fs1HashBuffer + sizeof(Hash), &hash[1][1], sizeof(Hash));

        nn::crypto::GenerateSha256Hash(&hash[1][2], sizeof(Hash), fs1DataBuffer + BlockSize * 2, BlockSize);
        std::memcpy(fs1HashBuffer + sizeof(Hash) * 2, &hash[1][2], sizeof(Hash));

        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(0, fs1HashBuffer, sizeof(Hash) * 3));
        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(Fs1HashSize, fs1DataBuffer, Fs1DataSize));
    }

    // fs1 マスターハッシュ
    nn::crypto::GenerateSha256Hash(&masterHash[1], sizeof(hash), hash[1], nn::crypto::Sha256Generator::HashSize * 3);

    // fsヘッダ作成
    NcaFsHeader fsHeader[2] = {};
    fsHeader[0].version = 0x0002;
    fsHeader[0].fsType = NcaFsHeader::FsType::RomFs;
    fsHeader[0].hashType = NcaFsHeader::HashType::HierarchicalSha256Hash;
    fsHeader[0].encryptionType = NcaFsHeader::EncryptionType::AesCtr;
    {
        auto& data = fsHeader[0].hashData.hierarchicalSha256Data;
        data.fsDataMasterHash = masterHash[0];
        data.hashBlockSize = BlockSize;
        data.hashLayerCount = 2;
        data.hashLayerRegion[0].offset = 0;
        data.hashLayerRegion[0].size = 2 * sizeof(Hash);
        data.hashLayerRegion[1].offset = Fs0HashSize;
        data.hashLayerRegion[1].size = Fs0DataSize;
    }

    fsHeader[1].version = 0x0002;
    fsHeader[1].fsType = NcaFsHeader::FsType::RomFs;
    fsHeader[1].hashType = NcaFsHeader::HashType::HierarchicalSha256Hash;
    fsHeader[1].encryptionType = NcaFsHeader::EncryptionType::AesCtr;
    {
        auto& data = fsHeader[1].hashData.hierarchicalSha256Data;
        data.fsDataMasterHash = masterHash[1];
        data.hashBlockSize = BlockSize;
        data.hashLayerCount = 2;
        data.hashLayerRegion[0].offset = 0;
        data.hashLayerRegion[0].size = 3 * sizeof(Hash);
        data.hashLayerRegion[1].offset = Fs1HashSize;
        data.hashLayerRegion[1].size = Fs1DataSize;
    }

    // fs ヘッダハッシュ
    nn::fssystem::Hash fsHeaderHash[2];
    nn::crypto::GenerateSha256Hash(&fsHeaderHash[0], sizeof(hash), &fsHeader[0], sizeof(NcaFsHeader));
    nn::crypto::GenerateSha256Hash(&fsHeaderHash[1], sizeof(hash), &fsHeader[1], sizeof(NcaFsHeader));

    const uint8_t keyIndex = 1;

    // nca ヘッダ作成
    NcaHeader ncaHeader = {};
    ncaHeader.signature = NcaHeader::Signature;
    ncaHeader.keyGeneration = 0;
    ncaHeader.keyIndex = keyIndex;
    ncaHeader.contentType = NcaHeader::ContentType::Program;
    ncaHeader.contentSize = StorageSize;
    ncaHeader.sdkAddonVersion = NN_SDK_CURRENT_VERSION_NUMBER;
    ncaHeader.fsInfo[0].startSector = NcaHeader::ByteToSector(FsHashOffset[0]);
    ncaHeader.fsInfo[0].endSector = NcaHeader::ByteToSector(FsDataOffset[0] + Fs0DataSize);
    ncaHeader.fsInfo[0].hashTargetSize = NcaFsHeader::Size;
    ncaHeader.fsHeaderHash[0] = fsHeaderHash[0];
    ncaHeader.fsInfo[1].startSector = NcaHeader::ByteToSector(FsHashOffset[1]);
    ncaHeader.fsInfo[1].endSector = NcaHeader::ByteToSector(FsDataOffset[1] + Fs1DataSize);
    ncaHeader.fsInfo[1].hashTargetSize = NcaFsHeader::Size;
    ncaHeader.fsHeaderHash[1] = fsHeaderHash[1];
    {
        // AES の鍵を AES 暗号化して配置
        nn::crypto::AesEncryptor128 aesEncryptor;
        aesEncryptor.Initialize(CryptoConfiguration.keyAreaEncryptionKeySource[keyIndex], AesKeySize);

        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey,                  AesKeySize, BodyAesKey[0], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize,     AesKeySize, BodyAesKey[1], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize * 2, AesKeySize, BodyAesKey[2], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize * 3, AesKeySize, BodyAesKey[3], AesKeySize);
    }

    SignNcaHeader(ncaHeader.headerSign1, sizeof(ncaHeader.headerSign1), ncaHeader); // 署名 1
    SignNcaHeader(ncaHeader.headerSign2, sizeof(ncaHeader.headerSign2), ncaHeader); // 署名 2 (1 と鍵共通)

    // TODO: content id 等のフィールド

    // ヘッダに暗号化かぶせる
    char xtsKey[2][AesKeySize];
    CryptoConfiguration.pGenerateKey(
        xtsKey[0],
        AesKeySize,
        CryptoConfiguration.headerEncryptedEncryptionKey[0],
        AesKeySize,
        static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
        CryptoConfiguration
    );
    CryptoConfiguration.pGenerateKey(
        xtsKey[1],
        AesKeySize,
        CryptoConfiguration.headerEncryptedEncryptionKey[1],
        AesKeySize,
        static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
        CryptoConfiguration
    );

    // fs0,1 ヘッダ書き込み
    {
        char iv[AesXtsStorage::IvSize] = {};
        MakeAesXtsIv(iv, sizeof(iv), sizeof(NcaHeader));

        nn::fs::SubStorage bodyRaw(&base, FsHeaderOffset[0], NcaFsHeader::Size * 2);
        AesXtsStorage body(&bodyRaw, xtsKey[0], xtsKey[1], AesXtsStorage::KeySize, iv, sizeof(iv), XtsBlockSize );

        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(0, fsHeader, sizeof(NcaFsHeader) * 2));
    }

    // fs ヘッダ書き込み
    {
        const char iv[AesXtsStorage::IvSize] = {};
        AesXtsStorage header(&base, xtsKey[0], xtsKey[1], AesXtsStorage::KeySize, iv, sizeof(iv), XtsBlockSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(header.Write(0, &ncaHeader, sizeof(ncaHeader)));
    }
} // NOLINT(impl/function_size)

void NcaReaderBuilder::BuildPatchStorage() NN_NOEXCEPT
{
    typedef nn::fssystem::NcaPatchInfo PatchInfo;
    typedef nn::fssystem::NcaBucketInfo BucketInfo;
    typedef nn::fssystem::NcaFileSystemDriver NcaFileSystemDriver;

    // メモリが確保できないならテストできないので abort する
    NN_ABORT_UNLESS_NOT_NULL(m_PatchStorageBuffer);

    std::memcpy(m_PatchStorageBuffer.get(), m_StorageBuffer.get(), StorageSize);
    std::memset(m_PatchStorageBuffer.get() + StorageSize, 0, PatchStorageSize - StorageSize);
    std::memset(m_PatchStorageBuffer.get() + FsDataOffset[1], 0, Fs1DataSize);

    nn::fs::MemoryStorage base(m_PatchStorageBuffer.get(), PatchStorageSize);

    auto pNcaReader = nn::fssystem::AllocateShared<Reader>();
    NN_ABORT_UNLESS_RESULT_SUCCESS(pNcaReader->Initialize(&base, CryptoConfiguration));

    // NcaFsHeaderReader::Get 系が正しく動くこと
    {
        FsHeaderReader fsReader;
        NN_ABORT_UNLESS_RESULT_SUCCESS(fsReader.Initialize(*pNcaReader.get(), 0));
        NN_SDK_ASSERT_EQUAL(NcaFsHeader::EncryptionType::AesCtr, fsReader.GetEncryptionType());
        NN_SDK_ASSERT_EQUAL(NcaFsHeader::FsType::RomFs, fsReader.GetFsType());
        NN_SDK_ASSERT_EQUAL(NcaFsHeader::HashType::HierarchicalSha256Hash, fsReader.GetHashType());
        NN_SDK_ASSERT_EQUAL(0x0002, fsReader.GetVersion());
    }

    const auto hashBuffer = m_DataBuffer.get() + Fs0HashSize + Fs0DataSize;
    const auto dataBuffer = m_DataBuffer.get() + Fs0HashSize + Fs0DataSize + Fs1HashSize;
    NcaFileSystemDriver ncafs(pNcaReader, GetTestLibraryAllocator(), GetBufferManager());

    NcaFsHeader fsHeader[2];
    nnt::fs::util::Vector<char> oldBuffer(Fs1HashSize + Fs1DataSize);
    std::memcpy(oldBuffer.data(), hashBuffer, Fs1HashSize);
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        FsHeaderReader reader;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ncafs.OpenStorage(&storage, &reader, 0));
        reader.GetRawData(&fsHeader[0], sizeof(fsHeader[0]));
    }
    {
        std::shared_ptr<nn::fs::IStorage> storage;
        FsHeaderReader reader;
        NN_ABORT_UNLESS_RESULT_SUCCESS(ncafs.OpenStorage(&storage, &reader, 1));
        reader.GetRawData(&fsHeader[1], sizeof(fsHeader[1]));

        std::memcpy(oldBuffer.data() + Fs1HashSize, dataBuffer, Fs1DataSize);
    }

    BuildPatchData();
    auto& newData = m_NewDataBuffer;

    nnt::fs::util::Vector<char> newBuffer(Fs1HashSize + newData.size());
    std::memset(newBuffer.data(), 0, Fs1HashSize);
    std::memcpy(newBuffer.data() + Fs1HashSize, newData.data(), newData.size());

    // fs1 のハッシュ再計算
    int hashCount = 0;
    Hash hash[8];
    for( int i = 0; i < 8; ++i )
    {
        const auto offset = i * BlockSize;
        const auto size = std::min<size_t>(BlockSize, newData.size() - offset);
        if( size == 0 )
        {
            break;
        }

        nn::crypto::GenerateSha256Hash(&hash[i], sizeof(Hash), newData.data() + offset, size);
        std::memcpy(newBuffer.data() + sizeof(Hash) * i, &hash[i], sizeof(Hash));

        ++hashCount;

        if( size != BlockSize )
        {
            break;
        }
    }
    NN_ABORT_UNLESS_LESS(hashCount, 8);

    nn::fs::MemoryStorage oldStorage(oldBuffer.data(), oldBuffer.size());
    nn::fs::MemoryStorage newStorage(newBuffer.data(), newBuffer.size());
    nn::fssystem::utilTool::IndirectStorageBuilder builder;
    NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Initialize(&oldStorage, &newStorage, 16));
    NN_ABORT_UNLESS_RESULT_SUCCESS(builder.Build(16, 2 * 1024, 1));

    // body 生
    const size_t storageSize = nn::util::align_up(StorageSize - oldBuffer.size() + newBuffer.size(), XtsBlockSize);

    // fs1 作成
    {
        nn::fs::SubStorage bodyRaw(&base, FsHashOffset[1], PatchStorageSize - FsHashOffset[1]);

        char iv[AesCtrStorage::IvSize] = {};
        AesCtrStorage::MakeIv(iv, sizeof(iv), 0, FsHashOffset[1]);

        AesCtrStorage bodyUnaligned(&bodyRaw, BodyAesKey[2], AesCtrStorage::KeySize, iv, sizeof(iv));
        nn::fssystem::AlignmentMatchingStorage<XtsBlockSize, 4> body(&bodyUnaligned);

        // fs1 マスターハッシュ
        nn::fssystem::Hash masterHash;
        nn::crypto::GenerateSha256Hash(&masterHash, sizeof(hash), hash, sizeof(Hash) * hashCount);

        // ハッシュ情報の書き換え
        {
            auto& data = fsHeader[1].hashData.hierarchicalSha256Data;
            data.fsDataMasterHash = masterHash;
            data.hashLayerRegion[0].size = hashCount * sizeof(Hash);
            data.hashLayerRegion[1].size = newData.size();
        }

        // データ書き出し
        {
            char buffer[256];
            NN_ABORT_UNLESS_RESULT_SUCCESS(builder.WriteData(&body, buffer, sizeof(buffer)));
        }

        auto& patchInfo = *nn::util::BytePtr(&fsHeader[1]).Advance(PatchInfo::Offset).Get<PatchInfo>();

        const auto offset = nn::util::align_up(builder.QueryDataStorageSize(), XtsBlockSize);
        patchInfo.indirectOffset = offset;

        const auto size = nn::util::align_up(builder.QueryTableNodeStorageSize() + builder.QueryTableEntryStorageSize(), XtsBlockSize);
        patchInfo.indirectSize = size;

        nn::fs::MemoryStorage headerStorage(patchInfo.indirectHeader, BucketInfo::HeaderSize);
        const auto nodeOffset = offset;
        const auto nodeSize = builder.QueryTableNodeStorageSize();
        const auto entryOffset = nodeOffset + nodeSize;
        const auto entrySize = builder.QueryTableEntryStorageSize();

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            builder.WriteTable(
                GetTestLibraryAllocator(),
                nn::fs::SubStorage(&headerStorage, 0, BucketInfo::HeaderSize),
                nn::fs::SubStorage(&body, nodeOffset, nodeSize),
                nn::fs::SubStorage(&body, entryOffset, entrySize)
            )
        );
    }

    // fs ヘッダハッシュ
    nn::fssystem::Hash fsHeaderHash[2];
    nn::crypto::GenerateSha256Hash(&fsHeaderHash[0], sizeof(nn::fssystem::Hash), &fsHeader[0], sizeof(NcaFsHeader));
    nn::crypto::GenerateSha256Hash(&fsHeaderHash[1], sizeof(nn::fssystem::Hash), &fsHeader[1], sizeof(NcaFsHeader));

    const uint8_t keyIndex = 1;

    // nca ヘッダ作成
    NcaHeader ncaHeader = {};
    ncaHeader.signature = NcaHeader::Signature;
    ncaHeader.keyGeneration = 0;
    ncaHeader.keyIndex = keyIndex;
    ncaHeader.contentType = NcaHeader::ContentType::Program;
    ncaHeader.contentSize = storageSize;
    ncaHeader.sdkAddonVersion = NN_SDK_CURRENT_VERSION_NUMBER;
    ncaHeader.fsInfo[0].startSector = NcaHeader::ByteToSector(FsHashOffset[0]);
    ncaHeader.fsInfo[0].endSector = NcaHeader::ByteToSector(FsDataOffset[0] + Fs0DataSize);
    ncaHeader.fsInfo[0].hashTargetSize = NcaFsHeader::Size;
    ncaHeader.fsHeaderHash[0] = fsHeaderHash[0];
    ncaHeader.fsInfo[1].startSector = NcaHeader::ByteToSector(FsHashOffset[1]);
    ncaHeader.fsInfo[1].endSector = NcaHeader::ByteToSector(FsDataOffset[1] + newData.size() + NcaHeader::XtsBlockSize - 1);
    ncaHeader.fsInfo[1].hashTargetSize = NcaFsHeader::Size;
    ncaHeader.fsHeaderHash[1] = fsHeaderHash[1];
    {
        // AES の鍵を AES 暗号化して配置
        nn::crypto::AesEncryptor128 aesEncryptor;
        aesEncryptor.Initialize(CryptoConfiguration.keyAreaEncryptionKeySource[keyIndex], AesKeySize);

        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey,                  AesKeySize, BodyAesKey[0], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize,     AesKeySize, BodyAesKey[1], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize * 2, AesKeySize, BodyAesKey[2], AesKeySize);
        aesEncryptor.EncryptBlock(ncaHeader.encryptedKey + AesKeySize * 3, AesKeySize, BodyAesKey[3], AesKeySize);
    }

    SignNcaHeader(ncaHeader.headerSign1, sizeof(ncaHeader.headerSign1), ncaHeader); // 署名 1
    SignNcaHeader(ncaHeader.headerSign2, sizeof(ncaHeader.headerSign2), ncaHeader); // 署名 2 (1 と鍵共通)

    // ヘッダに暗号化かぶせる
    char xtsKey[2][AesKeySize];
    CryptoConfiguration.pGenerateKey(
        xtsKey[0],
        AesKeySize,
        CryptoConfiguration.headerEncryptedEncryptionKey[0],
        AesKeySize,
        static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
        CryptoConfiguration
    );
    CryptoConfiguration.pGenerateKey(
        xtsKey[1],
        AesKeySize,
        CryptoConfiguration.headerEncryptedEncryptionKey[1],
        AesKeySize,
        static_cast<int>(nn::fssystem::KeyType::NcaHeaderKey),
        CryptoConfiguration
    );

    // fs0,1 ヘッダ書き込み
    {
        char iv[AesXtsStorage::IvSize] = {};
        MakeAesXtsIv(iv, sizeof(iv), sizeof(NcaHeader));

        nn::fs::SubStorage bodyRaw(&base, FsHeaderOffset[0], NcaFsHeader::Size * 2);
        AesXtsStorage body(&bodyRaw, xtsKey[0], xtsKey[1], AesXtsStorage::KeySize, iv, sizeof(iv), XtsBlockSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(body.Write(0, fsHeader, sizeof(NcaFsHeader) * 2));
    }

    // fs ヘッダ書き込み
    {
        const char iv[AesXtsStorage::IvSize] = {};
        AesXtsStorage header(&base, xtsKey[0], xtsKey[1], AesXtsStorage::KeySize, iv, sizeof(iv), XtsBlockSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(header.Write(0, &ncaHeader, sizeof(ncaHeader)));
    }
} // NOLINT(impl/function_size)

void NcaReaderBuilder::BuildPatchData() NN_NOEXCEPT
{
    typedef nnt::fs::util::Vector<char> File;

    const char* pOldData = m_DataBuffer.get() + Fs0HashSize + Fs0DataSize + Fs1HashSize;

    nnt::fs::util::Vector<File> files;
    std::mt19937 mt(GetRandomSeed());

    size_t newSize = 0;
    size_t oldSize = 0;
    for( int i = 0; oldSize < Fs1DataSize; ++i )
    {
        size_t oldFileSize = std::uniform_int_distribution<>(4 * 1024, 8 * 1024)(mt);
        oldFileSize = nn::util::align_down(std::min(oldFileSize, Fs1DataSize - oldSize), AesBlockSize);

        // ファイル伸縮
        size_t newFileSize = oldFileSize;
        if( std::uniform_int_distribution<>(0, 20)(mt) == 0 )
        {
            const auto width = std::uniform_int_distribution<>(1024, 2 * 1024)(mt);
            const auto difference = nn::util::align_down(width, AesBlockSize) - 1024;
            if( 0 <= difference || static_cast<size_t>(-difference) < newFileSize )
            {
                newFileSize += difference;
            }
        }

        const auto copySize = std::min(oldFileSize, newFileSize);
        File newFile(newFileSize);
        std::memcpy(newFile.data(), pOldData + oldSize, copySize);

        // ファイル書き換え
        size_t offset = 0;
        while( offset < copySize )
        {
            size_t size = std::uniform_int_distribution<>(512, 2 * 1024)(mt);
            size = std::min(nn::util::align_down(size, sizeof(int)), copySize - offset);

            if( std::uniform_int_distribution<>(0, 20)(mt) == 0 )
            {
                FillBufferWithRandomValue(newFile.data() + offset, size);
            }

            offset += size;
        }

        // 伸びた部分を埋める
        if( oldFileSize < newFileSize )
        {
            FillBufferWithRandomValue(newFile.data() + oldFileSize, newFileSize - oldFileSize);
        }

        files.push_back(std::move(newFile));

        oldSize += oldFileSize;
        newSize += newFileSize;
    }

    m_NewDataBuffer.resize(newSize);

    size_t offset = 0;
    for( auto& file : files )
    {
        std::memcpy(m_NewDataBuffer.data() + offset, file.data(), file.size());

        offset += file.size();
    }
}

std::shared_ptr<nn::fssystem::NcaReader> NcaReaderBuilder::Make() NN_NOEXCEPT
{
    auto pBaseStorage = nn::fssystem::AllocateShared<nn::fs::MemoryStorage>(m_StorageBuffer.get(), StorageSize);
    NN_ABORT_UNLESS_NOT_NULL(pBaseStorage);
    auto pNcaReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NN_ABORT_UNLESS_NOT_NULL(pNcaReader);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pNcaReader->Initialize(std::move(pBaseStorage), CryptoConfiguration));
    return pNcaReader;
}

std::shared_ptr<nn::fssystem::NcaReader> NcaReaderBuilder::MakePatch() NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(m_PatchStorageBuffer);
    auto pBaseStorage = nn::fssystem::AllocateShared<nn::fs::MemoryStorage>(m_PatchStorageBuffer.get(), PatchStorageSize);
    NN_ABORT_UNLESS_NOT_NULL(pBaseStorage);
    auto pNcaReader = nn::fssystem::AllocateShared<nn::fssystem::NcaReader>();
    NN_ABORT_UNLESS_NOT_NULL(pNcaReader);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pNcaReader->Initialize(std::move(pBaseStorage), CryptoConfiguration));
    return pNcaReader;
}

const NcaReaderBuilder::Buffer NcaReaderBuilder::GetStorageBuffer() NN_NOEXCEPT
{
    Buffer data = { m_StorageBuffer.get(), StorageSize };
    return data;
}

const NcaReaderBuilder::Buffer NcaReaderBuilder::GetPatchStorageBuffer() NN_NOEXCEPT
{
    Buffer data = { m_PatchStorageBuffer.get(), PatchStorageSize };
    return data;
}

const NcaReaderBuilder::Buffer NcaReaderBuilder::GetFs0Data() NN_NOEXCEPT
{
    Buffer data = { m_DataBuffer.get() + Fs0HashSize, Fs0DataSize };
    return data;
}

const NcaReaderBuilder::Buffer NcaReaderBuilder::GetFs1Data() NN_NOEXCEPT
{
    Buffer data = { m_DataBuffer.get() + Fs0HashSize + Fs0DataSize + Fs1HashSize, Fs1DataSize };
    return data;
}

const NcaReaderBuilder::Buffer NcaReaderBuilder::GetNewData() NN_NOEXCEPT
{
    Buffer data = { m_NewDataBuffer.data(), m_NewDataBuffer.size() };
    return data;
}

nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT
{
    static nn::fssystem::IBufferManager* s_pBufferManager = nullptr;

    if( s_pBufferManager == nullptr )
    {
        static const auto MaxCacheCount = 1024;
        static const auto SizeBlock = 16 * 1024;
        static const auto BufferManagerHeapSize = 8 * 1024 * 1024;
        static NN_ALIGNAS(4096) char s_BufferManagerHeap[BufferManagerHeapSize];
        static nn::fssystem::FileSystemBufferManager s_BufferManagerInstance;

        s_BufferManagerInstance.Initialize(
            MaxCacheCount,
            reinterpret_cast<uintptr_t>(s_BufferManagerHeap),
            sizeof(s_BufferManagerHeap),
            SizeBlock
        );
        s_pBufferManager = &s_BufferManagerInstance;
    }

    return s_pBufferManager;
}

void SignNcaHeader(void* pOutValue, size_t outSize, const NcaHeader& ncaHeader) NN_NOEXCEPT
{
    // 事前検証は SignRsa2048PssSha256 に任せる

    char rsaInterimSalt[nn::crypto::Sha256Generator::HashSize];
    nn::crypto::GenerateCryptographicallyRandomBytes(rsaInterimSalt, sizeof(rsaInterimSalt));

    nn::crypto::SignRsa2048PssSha256(
        pOutValue,
        outSize,
        CryptoConfiguration.headerSign1KeyModulus,
        CryptoConfiguration.Rsa2048KeyModulusSize,
        Rsa2048KeyPrivateExponent,
        sizeof(Rsa2048KeyPrivateExponent),
        &ncaHeader.signature,
        NcaHeader::Size - NcaHeader::HeaderSignSize * 2,
        rsaInterimSalt,
        sizeof(rsaInterimSalt)
    );
}

void MakeAesXtsIv(void* pOutValue, size_t outSize, int64_t baseOffset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutValue);
    NN_SDK_REQUIRES_GREATER_EQUAL(outSize, size_t(AesKeySize));
    NN_SDK_REQUIRES_GREATER_EQUAL(baseOffset, 0);
    NN_UNUSED(outSize);

    nn::util::BytePtr ptr(pOutValue);
    ptr.Advance(sizeof(int64_t));

    nn::util::StoreBigEndian<int64_t>(ptr.Get<int64_t>(), baseOffset / NcaHeader::XtsBlockSize);
}

const uint8_t Rsa2048KeyPrivateExponent[] =
{
    0x0c, 0x05, 0xb5, 0x6d, 0xe9, 0x0f, 0xe6, 0x41, 0x55, 0x6d, 0x52, 0x36, 0xc8, 0x57, 0xb3, 0x60,
    0x57, 0xdb, 0xcd, 0xb3, 0x03, 0x0f, 0x57, 0xf1, 0x17, 0x8a, 0x30, 0x33, 0x8a, 0x68, 0x92, 0xfb,
    0x73, 0x57, 0x04, 0x8a, 0xcb, 0xe3, 0xf4, 0x8a, 0xbf, 0xe3, 0xf2, 0xac, 0x38, 0x23, 0x30, 0x26,
    0x95, 0x42, 0x3d, 0x50, 0xfa, 0xb4, 0xaf, 0x60, 0x21, 0x75, 0xdc, 0xd9, 0x57, 0xb4, 0xc3, 0x6c,
    0xe5, 0xf6, 0xe5, 0xe0, 0x55, 0x65, 0x77, 0x4b, 0xc7, 0xa6, 0x7e, 0x0a, 0xfe, 0xdd, 0x80, 0x42,
    0x4f, 0x0d, 0x7e, 0x15, 0x8d, 0xf4, 0x27, 0x37, 0x24, 0x99, 0xf2, 0x12, 0x31, 0xdb, 0xd7, 0x7f,
    0x1e, 0x92, 0x21, 0x14, 0xca, 0x21, 0xf6, 0x50, 0x08, 0x92, 0xae, 0x31, 0xde, 0xf4, 0x29, 0x24,
    0xd6, 0x41, 0xb3, 0x47, 0x18, 0x37, 0x14, 0xf9, 0x8d, 0x5d, 0x95, 0xf4, 0xf5, 0x7f, 0x99, 0xfb,
    0x86, 0xda, 0x65, 0xe9, 0x72, 0xa9, 0x77, 0x65, 0xc8, 0xc5, 0x29, 0x5a, 0x19, 0x2b, 0x51, 0x1c,
    0x72, 0xeb, 0x49, 0xd1, 0x0b, 0x73, 0x8b, 0x3e, 0x2e, 0xc8, 0x7e, 0xff, 0xd8, 0xfe, 0xf4, 0xf4,
    0xf6, 0x92, 0x27, 0x7f, 0xa0, 0xdb, 0xc1, 0x25, 0xbc, 0xec, 0x5f, 0x0b, 0x2d, 0x99, 0xeb, 0xdd,
    0x9e, 0x5d, 0x42, 0x75, 0xb5, 0xe3, 0x24, 0xcb, 0xe9, 0xeb, 0xd9, 0x00, 0x4b, 0x12, 0x5d, 0xa3,
    0xa6, 0x25, 0xac, 0x20, 0x82, 0x25, 0x53, 0x1f, 0xc6, 0x2f, 0x27, 0xf1, 0x99, 0x7a, 0x99, 0xdc,
    0xa5, 0xc0, 0x5e, 0x63, 0x0f, 0x78, 0x03, 0x2a, 0x18, 0xd9, 0xe1, 0x06, 0x3b, 0xdf, 0xb2, 0x95,
    0x19, 0x32, 0xb4, 0x65, 0xd2, 0xd0, 0xfe, 0x18, 0xc7, 0x54, 0x5c, 0xa4, 0xf6, 0xd8, 0xfd, 0xdb,
    0x6d, 0xd8, 0xda, 0xf2, 0x9a, 0x55, 0x5c, 0x3e, 0xec, 0x17, 0x72, 0x09, 0xa3, 0x1a, 0x0a, 0xc1
};

const nn::fssystem::NcaCryptoConfiguration CryptoConfiguration =
{
    // ヘッダー署名鍵1 (Modulus)
    {
        0xd8, 0xf1, 0x18, 0xef, 0x32, 0x72, 0x4c, 0xa7, 0x47, 0x4c, 0xb9, 0xea, 0xb3, 0x04, 0xa8, 0xa4,
        0xac, 0x99, 0x08, 0x08, 0x04, 0xbf, 0x68, 0x57, 0xb8, 0x43, 0x94, 0x2b, 0xc7, 0xb9, 0x66, 0x49,
        0x85, 0xe5, 0x8a, 0x9b, 0xc1, 0x00, 0x9a, 0x6a, 0x8d, 0xd0, 0xef, 0xce, 0xff, 0x86, 0xc8, 0x5c,
        0x5d, 0xe9, 0x53, 0x7b, 0x19, 0x2a, 0xa8, 0xc0, 0x22, 0xd1, 0xf3, 0x22, 0x0a, 0x50, 0xf2, 0x2b,
        0x65, 0x05, 0x1b, 0x9e, 0xec, 0x61, 0xb5, 0x63, 0xa3, 0x6f, 0x3b, 0xba, 0x63, 0x3a, 0x53, 0xf4,
        0x49, 0x2f, 0xcf, 0x03, 0xcc, 0xd7, 0x50, 0x82, 0x1b, 0x29, 0x4f, 0x08, 0xde, 0x1b, 0x6d, 0x47,
        0x4f, 0xa8, 0xb6, 0x6a, 0x26, 0xa0, 0x83, 0x3f, 0x1a, 0xaf, 0x83, 0x8f, 0x0e, 0x17, 0x3f, 0xfe,
        0x44, 0x1c, 0x56, 0x94, 0x2e, 0x49, 0x83, 0x83, 0x03, 0xe9, 0xb6, 0xad, 0xd5, 0xde, 0xe3, 0x2d,
        0xa1, 0xd9, 0x66, 0x20, 0x5d, 0x1f, 0x5e, 0x96, 0x5d, 0x5b, 0x55, 0x0d, 0xd4, 0xb4, 0x77, 0x6e,
        0xae, 0x1b, 0x69, 0xf3, 0xa6, 0x61, 0x0e, 0x51, 0x62, 0x39, 0x28, 0x63, 0x75, 0x76, 0xbf, 0xb0,
        0xd2, 0x22, 0xef, 0x98, 0x25, 0x02, 0x05, 0xc0, 0xd7, 0x6a, 0x06, 0x2c, 0xa5, 0xd8, 0x5a, 0x9d,
        0x7a, 0xa4, 0x21, 0x55, 0x9f, 0xf9, 0x3e, 0xbf, 0x16, 0xf6, 0x07, 0xc2, 0xb9, 0x6e, 0x87, 0x9e,
        0xb5, 0x1c, 0xbe, 0x97, 0xfa, 0x82, 0x7e, 0xed, 0x30, 0xd4, 0x66, 0x3f, 0xde, 0xd8, 0x1b, 0x4b,
        0x15, 0xd9, 0xfb, 0x2f, 0x50, 0xf0, 0x9d, 0x1d, 0x52, 0x4c, 0x1c, 0x4d, 0x8d, 0xae, 0x85, 0x1e,
        0xea, 0x7f, 0x86, 0xf3, 0x0b, 0x7b, 0x87, 0x81, 0x98, 0x23, 0x80, 0x63, 0x4f, 0x2f, 0xb0, 0x62,
        0xcc, 0x6e, 0xd2, 0x46, 0x13, 0x65, 0x2b, 0xd6, 0x44, 0x33, 0x59, 0xb5, 0x8f, 0xb9, 0x4a, 0xa9
    },

    // ヘッダー署名鍵1 (PublicExponent)
    { 0x01, 0x00, 0x01 },

    // aes key for key area (とりあえず 0, 1 世代の鍵を入れておく)
    {
        { 0x3a, 0x7c, 0x3e, 0x38, 0x4a, 0x8f, 0x22, 0xff, 0x4b, 0x21, 0x57, 0x19, 0xb7, 0x81, 0xad, 0x0c },
        { 0xc0, 0x26, 0xfc, 0x7e, 0xea, 0xea, 0x81, 0x50, 0xc6, 0x6e, 0x57, 0x99, 0xac, 0x3c, 0x57, 0xc3 },
        { 0x47, 0xbb, 0x5d, 0x75, 0xc1, 0xfb, 0x7f, 0xe6, 0xf0, 0xb2, 0x26, 0x61, 0x74, 0xdc, 0x63, 0x56 },
    },

    // aes key for header (Kek は使わず一致判定でデコードする)
    { 0 },
    {
        { 0x5a, 0x3e, 0xd8, 0x4f, 0xde, 0xc0, 0xd8, 0x26, 0x31, 0xf7, 0xe2, 0x5d, 0x19, 0x7b, 0xf5, 0xd0 },
        { 0x1c, 0x9b, 0x7b, 0xfa, 0xf6, 0x28, 0x18, 0x3d, 0x71, 0xf6, 0x4d, 0x73, 0xf1, 0x50, 0xb9, 0xd2 },
    },

    DecryptNcaContentKeyTest,
    DecryptAesCtrForHardwareAes,
    DecryptAesCtrForHardwareAes
};

}}}
