﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

typedef uint8_t bit8;
#define NN_TEST_ASSERT_EQUAL(a, b) NN_SDK_ASSERT((a) == (b))

//! バッファオーバーラン検知機能つきバッファ
class TestBuffer
{
private:
#if     0
    static const size_t OFFSET_BYTES = 16;    //!< オフセット用領域サイズ
    static const size_t OVERRUN_BYTES = 16;   //!< オーバーランチェック用領域サイズ
#else
    static const size_t OFFSET_BYTES = 4;    //!< オフセット用領域サイズ
    static const size_t OVERRUN_BYTES = 4;   //!< オーバーランチェック用領域サイズ
#endif
    static const bit8 FillChar = 0xFF;

private:
    nnt::fs::util::Vector<bit8> m_Buffer;
    size_t m_Size;
    char reserved[4];

public:
    void AssertValid() const
    {
        NN_SDK_ASSERT(m_Buffer.size() > 0);
        NN_SDK_ASSERT(m_Size > 0);
        // チェック領域の値が書き換わっていないかどうかをチェックします。
        for (size_t i = 0; i < OFFSET_BYTES; i++)
        {
            NN_TEST_ASSERT_EQUAL(m_Buffer[i], FillChar);
        }
        for (size_t i = 0; i < OVERRUN_BYTES; i++)
        {
            NN_TEST_ASSERT_EQUAL(m_Buffer[OFFSET_BYTES + m_Size + i], FillChar);
        }
    }

public:
    explicit TestBuffer(size_t size)
        : m_Size(size)
    {
        // オフセット、オーバーランチェック用に余分に確保します。
        m_Buffer.resize(size + OFFSET_BYTES + OVERRUN_BYTES);

        // 初期化抜けチェック&破壊チェック用に0xFFで全体を埋めます。
        std::memset(&m_Buffer[0], FillChar, size + OFFSET_BYTES + OVERRUN_BYTES);
    }

    ~TestBuffer()
    {
        if (m_Size != 0)
        {
            AssertValid();
        }
    }

    //!< バッファへのポインタを取得します。
    inline const void* GetBuffer() const
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< バッファへのポインタを取得します。
    inline void* GetBuffer()
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< バッファへのポインタを取得します。
    inline operator const bit8*() const
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< バッファへのポインタを取得します。
    inline operator bit8*()
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< バッファへのポインタを取得します。
    inline operator const void*() const
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< バッファへのポインタを取得します。
    inline operator void*()
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return &m_Buffer[0];
    }

    //!< 要素数を取得します。
    inline size_t GetSize() const
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return m_Size;
    }

    //!< バッファサイズを取得します。
    inline size_t GetBufferSize() const
    {
        NN_SDK_ASSERT(m_Buffer.size());
        NN_SDK_ASSERT(m_Size >= 0);
        return (m_Size + OFFSET_BYTES + OVERRUN_BYTES);
    }

    //!< オフセットサイズを取得します。
    static inline size_t GetOffset()
    {
        return OFFSET_BYTES;
    }

    //!< 引数で与えたバッファの内容を自身にコピーします。
    void Copy(const TestBuffer& ref)
    {
        NN_SDK_ASSERT(this->GetOffset() == ref.GetOffset());
        NN_SDK_ASSERT(this->GetSize() == ref.GetSize());
        std::memcpy(this->operator bit8 *() + OFFSET_BYTES, ref.operator const bit8 *() + OFFSET_BYTES, m_Size);
    }
};
