﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_IStorage.h>

// $(SigloRoot)\Programs\Chris\Sources\Libraries\fs を追加インクルードディレクトリに足す必要あり
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>

#include "prfile2/pf_api.h"

//#define USE_FILE

namespace nn { namespace fs { namespace detail {


// アクセスログのためのプロクシ
class AccessLogStorage : public IStorage, public Newable
{
    NN_DISALLOW_COPY(AccessLogStorage);

private:
    IStorage* const m_pBaseStorage;

public:
    explicit AccessLogStorage(IStorage* pBaseStorage) NN_NOEXCEPT
        : m_pBaseStorage(pBaseStorage)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG("Read: %lld %d\n", offset, size);
        return m_pBaseStorage->Read(offset, buffer, size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_LOG("Write: %lld %d\n", offset, size);
        return m_pBaseStorage->Write(offset, buffer, size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->GetSize(outValue);
    }
};


TEST(FatFileSystem, CreateImageSample)
{

    // イメージを構築する領域を用意：
#ifdef USE_FILE
    // Host PC ファイル上に構築する版

    MountHostRoot();

    // @pre 所望のストレージサイズのファイルを予め用意しておく（中身は問わない）
    // (ここで CreateFile してもよい)
    FileHandle baseHandle;
    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&baseHandle, "x:/fat.img", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));

    FileHandleStorage baseStorage(baseHandle);
#else
    // メモリ上に構築する版
    const int BufferSize = 8 * 1024 * 1024;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    ASSERT_NE(buffer, nullptr);
    memset(buffer.get(), 0xCD, BufferSize);

    nn::fs::MemoryStorage baseStorage(buffer.get(), BufferSize);
#endif

    // ログ取得のレイヤを被せる
    AccessLogStorage logStorage(&baseStorage);

    // FatFs を生成
    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    ASSERT_NE(fatFs, nullptr);

    // キャッシュバッファ・IStorage を割り当て
    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&logStorage, cacheBuffer.get(), cacheBufferSize));

    // FAT にフォーマット
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());

    // fatfs 内部のマウント処理
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

    // fsa に登録 (MountXxx() に相当)
    NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));



    {
        // ディレクトリ・ファイルを作成する等
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory"));
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/file", 1024));

        // long file name
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/longfilename________________________________", 1024));

        // 8.3 大文字エントリ名
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/SHORTF", 1024));
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory/SHORTD"));

        FileHandle handle;
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("fat:/directory/file2", 0));
        NNT_EXPECT_RESULT_SUCCESS(OpenFile(&handle, "fat:/directory/file2", static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

        const int fileSize = 1024 * 1024;
        std::unique_ptr<char[]> writeBuffer(new char[fileSize]);
        ASSERT_NE(writeBuffer, nullptr);

        memset(writeBuffer.get(), 'A', fileSize);
        NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle,        0, writeBuffer.get(), fileSize, WriteOption()));
        memset(writeBuffer.get(), 'B', fileSize);
        NNT_EXPECT_RESULT_SUCCESS(WriteFile(handle, fileSize, writeBuffer.get(), fileSize, WriteOption()));

        NNT_EXPECT_RESULT_SUCCESS(FlushFile(handle));
        CloseFile(handle);

        // 作成したエントリを列挙
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));
    }




    Unmount("fat");

    // (logStorage, baseStorage はここまで生存している必要がある)

#ifdef USE_FILE
    NNT_EXPECT_RESULT_SUCCESS(FlushFile(baseHandle));
    CloseFile(baseHandle); // (本来は baseStorage が破棄された後に Close すべき)
#endif

    // (構築したイメージを SD に焼けば内容を確認できる)
}


namespace{
    nnt::fs::util::String GenerateMountName(int index)
    {
        char tmp[4];
        nn::util::SNPrintf(tmp, sizeof(tmp), "%d", index);
        nnt::fs::util::String mountName = "fat";
        mountName += tmp;
        return mountName;
    }
};


// FatSafe 有効版のマウント
TEST(FatFileSystem, MountWithFatSafe)
{
    const int64_t StorageSize = 2ULL * 1024 * 1024 * 1024; // must be FAT32
    nnt::fs::util::VirtualMemoryStorage baseStorage(StorageSize);

    nn::fat::FatFileSystem::SetAllocatorForFat(nnt::fs::util::GetTestLibraryAllocator());

    std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
    ASSERT_NE(fatFs, nullptr);
    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char[]> cacheBuffer(new char[cacheBufferSize]);
    nn::fat::FatAttribute attrs = {true, false, false};
    std::unique_ptr<nn::fat::FatErrorInfoSetter> pFatErrorInfoSetter(new nn::fat::FatErrorInfoSetter());
    ASSERT_NE(pFatErrorInfoSetter, nullptr);

    NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize, &attrs, std::move(pFatErrorInfoSetter), ResultInvalidFatFormat(), ResultUsableSpaceNotEnough()));
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
    NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
    NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));

    {
        NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("fat:/directory"));
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));
    }

    Unmount("fat");
}






// 複数の別インスタンス（別ストレージ）が互いに影響しないこと
TEST(FatFileSystem, MultiInstanceOrthogonality)
{
    const int StorageSize = 1 * 1024 * 1024;
    const int BufferSize  = 1 * 1024 * 1024;
    const int DriveNum = 5;

    // ストレージを複数作成
    std::unique_ptr<nn::fat::FatFileSystem> pInstance[DriveNum];
    std::unique_ptr<char>                   storageBuffer[DriveNum];
    std::unique_ptr<IStorage>               storage[DriveNum];
    std::unique_ptr<char>                   cacheBuffer[DriveNum];

    for (int i = 0; i < DriveNum; i++)
    {
        // ストレージ
        storageBuffer[i].reset(new char[StorageSize]);
        ASSERT_NE(storageBuffer[i], nullptr);
        memset(storageBuffer[i].get(), 0xCD, StorageSize);
        storage[i].reset(new MemoryStorage(storageBuffer[i].get(), StorageSize));

        // バッファ
        cacheBuffer[i].reset(new char[BufferSize]);

        // fatfs 生成、フォーマット
        auto& pFatFs = pInstance[i];
        pFatFs.reset(new nn::fat::FatFileSystem());
        ASSERT_NE(pFatFs, nullptr);

        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        ASSERT_GE(static_cast<size_t>(BufferSize), cacheBufferSize);

        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Initialize(storage[i].get(), cacheBuffer[i].get(), cacheBufferSize));

        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Format());
    }

    // それぞれマウント
    for (int i = 0; i < DriveNum; i++)
    {
        auto& pFatFs = pInstance[i];

        NNT_ASSERT_RESULT_SUCCESS(pFatFs->Mount());

        auto mountName = GenerateMountName(i);
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register(mountName.c_str(), std::move(pFatFs)));

        // ファイル作成
        auto filePath = mountName + ":/file_" + mountName;
        NNT_EXPECT_RESULT_SUCCESS(CreateFile(filePath.c_str(), i));

        // 作ったファイルのみが存在することを確認
        NNT_EXPECT_RESULT_SUCCESS(
            nnt::fs::util::IterateDirectoryRecursive(
                (mountName + ":/").c_str(),
                [](const char* path, const DirectoryEntry& entry){
                    NN_UNUSED(entry);
                    NN_UNUSED(path);
                    return nn::fs::ResultDataCorrupted(); // ディレクトリは無いはず
                },
                nullptr,
                [&](const char* path, const DirectoryEntry& entry) -> Result {
                    NN_UNUSED(entry);
                    if(strncmp(path, filePath.c_str(), nn::fs::EntryNameLengthMax) == 0)
                    {
                        NN_RESULT_SUCCESS;
                    }
                    else
                    {
                        return nn::fs::ResultDataCorrupted(); // ファイルは filePath のみのはず
                    }
                }
            )
        );

        // (目視確認用)
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive((mountName + ":/").c_str()));
    }

    for (int i = 0; i < DriveNum; i++)
    {
        Unmount(GenerateMountName(i).c_str());
    }
}


TEST(FatFileSystem, MultiDrive)
{
    // メモリ上に構築する版
    const int BufferSize = 1 * 1024 * 1024;
    const int DriveNum = 5;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    ASSERT_NE(buffer, nullptr);

    MemoryStorage baseStorage(buffer.get(), BufferSize);

    std::unique_ptr<char[]> cacheBuffer[DriveNum];

    // 同じドライブに割り当て
    for (int i = 0; i < DriveNum; i++)
    {
        memset(buffer.get(), 0xCD, BufferSize);

        // FatFs を生成
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        // キャッシュバッファ・IStorage を割り当て
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        cacheBuffer[i].reset(new char[cacheBufferSize]);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&baseStorage, cacheBuffer[i].get(), cacheBufferSize));

        // FAT にフォーマット
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
    }

    // 別々のドライブに割り当て
    std::unique_ptr<nn::fat::FatFileSystem> pInstance[DriveNum];
    for (int i = 0; i < DriveNum; i++)
    {
        memset(buffer.get(), 0xCD, BufferSize);
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        cacheBuffer[i].reset(new char[cacheBufferSize]);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&baseStorage, cacheBuffer[i].get(), cacheBufferSize));

        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());

        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

        pInstance[i] = std::move(fatFs);
    }

    {
        // ドライブ数が限界
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        std::unique_ptr<char[]> extraCacheBuffer(new char[cacheBufferSize]);
        NNT_ASSERT_RESULT_FAILURE(ResultFatFsUnexpectedSystemError, fatFs->Initialize(&baseStorage, extraCacheBuffer.get(), cacheBufferSize));
    }

    // 全ドライブをリリース
    for (int i = 0; i < DriveNum; i++)
    {
        pInstance[i].reset(nullptr);
    }

    // Initialize せずに終了
    for (int i = 0; i < DriveNum; i++)
    {
        memset(buffer.get(), 0xCD, BufferSize);

        // FatFs を生成
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);
    }

    // Mount せずにデストラクト
    for (int i = 0; i < DriveNum; i++)
    {
        memset(buffer.get(), 0xCD, BufferSize);

        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        cacheBuffer[i].reset(new char[cacheBufferSize]);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&baseStorage, cacheBuffer[i].get(), cacheBufferSize));

        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
    }
}

TEST(FatFileSystem, ReadOnlyEntry)
{
    // メモリ上に構築する版
    const int BufferSize = 1 * 1024 * 1024;
    std::unique_ptr<char[]> buffer(new char[BufferSize]);
    ASSERT_NE(buffer, nullptr);

    MemoryStorage baseStorage(buffer.get(), BufferSize);

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    std::unique_ptr<char> cacheBuffer(new char[cacheBufferSize]);

    {
        memset(buffer.get(), 0xCD, BufferSize);
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(&baseStorage, cacheBuffer.get(), cacheBufferSize));
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register("fat", std::move(fatFs)));

        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory("fat:/readonlydir"));
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("fat:/readonlyfile", 1024));
        NNT_ASSERT_RESULT_SUCCESS(CreateFile("fat:/readonlydir/file", 1024));

        ASSERT_EQ(nne::prfile2::pf_chmod("A:\\readonlyfile", ATTR_RDONLY), 0);
        ASSERT_EQ(nne::prfile2::pf_chdmod("A:\\readonlydir", ATTR_RDONLY), 0);

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("fat:/"));

        FileHandle file;
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, OpenFile(&file, "fat:/readonlyfile", OpenMode_Read | OpenMode_Write)); // Todo: Result 修正
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&file, "fat:/readonlyfile", OpenMode_Read));
        {
            char readBuffer[1024];
            NNT_EXPECT_RESULT_SUCCESS(ReadFile(file, 0, readBuffer, 1024));
        }
        CloseFile(file);
    }
}

}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
