﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>

#include <nn/fssystem/dbm/fs_KeyValueListTemplate.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>

#include <nnt/nnt_Compiler.h>
#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace nnt { namespace fs {
    nn::fssystem::IBufferManager* GetBufferManager() NN_NOEXCEPT;
}}

//! キー型(非テンプレート版と一緒)
struct DefaultKey
{
    char key[8];

    //! キーの比較
    bool operator==(const struct DefaultKey& rKey) const NN_NOEXCEPT
    {
        return ( 0 == std::memcmp(key, rKey.key, sizeof(key)) );
    }
};

//! 値型(非テンプレート版と一緒)
typedef uint32_t DefaultValue;

//! テストに使用するテンプレート
typedef nn::fssystem::dbm::KeyValueListTemplate<DefaultKey, DefaultValue> KeyValueListTmpl;

class KeyValueListTest : public ::testing::Test, public KeyValueListTmpl
{
public:
    typedef KeyValueListTmpl::Key Key;
    typedef KeyValueListTmpl::Value Value;
};

// キーの追加に使うテストデータ
static const KeyValueListTest::Key Name1 = { "TESTKY1" };
static const KeyValueListTest::Key Name2 = { "TESTKY2" };
static const KeyValueListTest::Key Name3 = { "TESTKY3" };
static const KeyValueListTest::Key Name4 = { "KY4TEST" };
static const KeyValueListTest::Key Name5 = { "KY5TEST" };
static const KeyValueListTest::Key Name6 = { "6TESTKY" };
static const KeyValueListTest::Key Name7 = { "7TESTKY" };
static const KeyValueListTest::Key Name8 = { "8TSET" };
static const KeyValueListTest::Key Name9 = { "TSET9" };
static const KeyValueListTest::Key Name0 = { "0000000" };

static const uint32_t ValueName1 = 0x12345678;
static const uint32_t ValueName2 = 0x87654321;
static const uint32_t ValueName3 = 0xAAAAAAAA;
static const uint32_t ValueName4 = 0xBBBBBBBB;
static const uint32_t ValueName5 = 0xBCBCBCBC;
static const uint32_t ValueName6 = 0x12341234;
static const uint32_t ValueName7 = 0x98769876;
static const uint32_t ValueName8 = 0x67584930;
static const uint32_t ValueName9 = 0x54637281;
static const uint32_t ValueName0 = 0xABCDEF01;

//! KeyValueList とそれが使用するストレージを初期化して保持するクラスです。
class KeyValueListSetup
{
public:
    KeyValueListSetup(
        uint32_t blockCount, uint32_t blockCountMax, uint32_t entryCountMax, uint32_t blockSize
    ) NN_NOEXCEPT
    : m_BodySize(KeyValueListTmpl::QueryStorageSize(entryCountMax)),
      m_BareTableStorage(nn::fssystem::dbm::AllocationTable::QuerySize(blockCountMax)),
      m_BareBodyStorage(KeyValueListTmpl::QueryStorageSize(entryCountMax)),
      m_TableStorage(),
      m_BodyStorage(),
      m_AllocationTable(),
      m_AllocationTableStorage(),
      m_KeyValueList(),
      m_AllocationTableStorageIndex()
    {
        NN_SDK_ASSERT_NOT_EQUAL(blockSize, static_cast<uint32_t>(0));

        // チェックテストストレージを挟み込みます
        nn::Result result = m_TableStorage.Initialize(&m_BareTableStorage);
        NN_ASSERT(result.IsSuccess());
        result = m_BodyStorage.Initialize(&m_BareBodyStorage);
        NN_ASSERT(result.IsSuccess());

        int64_t tableSize;
        result = m_TableStorage.GetSize(&tableSize);
        NN_ASSERT(result.IsSuccess());
        nn::fs::SubStorage tableSubStorage(&m_TableStorage, 0, tableSize);

        nn::fs::SubStorage bodySubStorage(&m_BodyStorage, 0, m_BodySize);

        // tableStorage 上にアロケーションテーブルを作成
        result = m_AllocationTable.Format(tableSubStorage, blockCount);
        NN_ASSERT(result.IsSuccess());
        m_AllocationTable.Initialize(tableSubStorage, blockCount);

        // アロケーションテーブルストレージ用に bodyStorage 確保
        result = m_AllocationTable.Allocate(&m_AllocationTableStorageIndex, blockCount);
        NN_ASSERT(result.IsSuccess());

        // アロケーションテーブルを用いたストレージの作成
        result = m_AllocationTableStorage.InitializeBuffered(
                     &m_AllocationTable,
                     m_AllocationTableStorageIndex,
                     blockSize,
                     bodySubStorage,
                     nnt::fs::GetBufferManager()
                 );
        NN_ASSERT(result.IsSuccess());

        // アロケーションテーブルストレージ上にキーバリューストレージを作成。
        result = KeyValueListTmpl::Format(&m_AllocationTableStorage);
        NN_ASSERT(result.IsSuccess());
        m_KeyValueList.Initialize(&m_AllocationTableStorage);
    }

    //! マウントします。
    void Mount(uint32_t blockCount, uint32_t blockSize) NN_NOEXCEPT
    {
        int64_t tableSize;
        NNT_ASSERT_RESULT_SUCCESS(m_TableStorage.GetSize(&tableSize));
        nn::fs::SubStorage tableSubStorage(&m_TableStorage, 0, tableSize);

        m_AllocationTable.Initialize(tableSubStorage, blockCount);

        nn::fs::SubStorage bodySubStorage(&m_BodyStorage, 0, m_BodySize);

        NNT_ASSERT_RESULT_SUCCESS(
            m_AllocationTableStorage.InitializeBuffered(
                &m_AllocationTable,
                m_AllocationTableStorageIndex,
                blockSize,
                bodySubStorage,
                nnt::fs::GetBufferManager()
            )
        );
        m_KeyValueList.Initialize(&m_AllocationTableStorage);
    }

    //! アンマウントします。
    void Unmount() NN_NOEXCEPT
    {
        m_KeyValueList.Finalize();
        m_AllocationTableStorage.Finalize();
        m_AllocationTable.Finalize();
    }

public:
    nnt::fs::util::WriteCheckStorage& GetTableStorage() NN_NOEXCEPT
    {
        return m_TableStorage;
    }

    KeyValueListTmpl& GetKeyValueList() NN_NOEXCEPT
    {
        return m_KeyValueList;
    }

private:
    uint32_t m_BodySize;
    nnt::fs::util::SafeMemoryStorage m_BareTableStorage;
    nnt::fs::util::SafeMemoryStorage m_BareBodyStorage;
    nnt::fs::util::WriteCheckStorage m_TableStorage;
    nnt::fs::util::WriteCheckStorage m_BodyStorage;
    nn::fssystem::dbm::AllocationTable m_AllocationTable;
    nn::fssystem::dbm::BufferedAllocationTableStorage m_AllocationTableStorage;
    KeyValueListTmpl m_KeyValueList;
    uint32_t m_AllocationTableStorageIndex;
};

//! キーバリューストレージのサイズ計算をテストします。
TEST(KeyValueListTest, QuerySize)
{
    static const auto BlockSize = 16; // sizeof(Key) + sizeof(Value) + sizeof(uint32_t)
    static const auto ReserveCount = 2;

    // KeyValue サイズ取得テスト
    // 必要なサイズ = （ほしいブロック + 管理ブロック） * 先頭ブロックのサイズ
    static const uint32_t CountKeyValues[] = {0, 10, 9999};
    for( uint32_t countKeyValue : CountKeyValues )
    {
        ASSERT_EQ(
            BlockSize * (countKeyValue + ReserveCount),
            KeyValueListTmpl::QueryStorageSize(countKeyValue)
        );
    }

    // 渡されたサイズ内に確保できるブロック数 = 作成可能ブロック数 - 管理ブロック数
    // 作成可能ブロック数は 全体サイズ / ブロックのサイズ
    static const int64_t StorageSizes[] = {32, 63, 64, 65, 2047, 160016, 0x0123456789ABCDEF};
    for( int64_t storageSize : StorageSizes )
    {
        ASSERT_EQ(
            static_cast<uint32_t>(storageSize / BlockSize) - ReserveCount,
            KeyValueListTmpl::QueryEntryCount(storageSize)
        );
    }
}

//! 1ブロックだけのアロケーションテーブルと1ブロックだけのキーバリューストレージを作成し、
//! ブロックが 1 つも作成できない状態を作るテストをします。
TEST(KeyValueListTest, Initialize)
{
    // エントリー可能数
    static const size_t EntryCountMax = 0;
    const uint32_t bodySize = KeyValueListTmpl::QueryStorageSize(EntryCountMax);

    // AllocationTable のブロック数
    static const uint32_t AllocationBlockCount = 2;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        AllocationBlockCount,
        AllocationBlockCount,
        EntryCountMax,
        bodySize / 2
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // ストレージはサイズが 0 なので、追加に失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        keyValueList.Add(Name1, ValueName1)
    );
}

//! 未登録のキーを取得しようとして見つからないことをテストします。
TEST(KeyValueListTest, TestNotFound)
{
    static const size_t AllocationBlockCount = 2;
    static const uint32_t BlockSize = 64;

    // エントリー可能数
    static const size_t EntryCountMax = 10;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        AllocationBlockCount,
        AllocationBlockCount,
        EntryCountMax,
        BlockSize
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // 未登録な状態に対してキーのサーチをします。
    // もちろん失敗します。
    uint32_t value;
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name1));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name2));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name3));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name4));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name5));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name6));
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name7));
}

//! キーの追加をテストします。
TEST(KeyValueListTest, TestAddKey)
{
    static const size_t AllocationBlockCount = 2;
    static const uint32_t BlockSize = 64;

    // エントリー可能数
    static const size_t EntryCountMax = 10;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        AllocationBlockCount,
        AllocationBlockCount,
        EntryCountMax,
        BlockSize
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // 1つ登録します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name1, ValueName1));
    // 登録状態に対してキーのサーチをします。
    uint32_t value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name1));
    ASSERT_EQ(ValueName1, value);

    // 未登録のキーをサーチします
    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, keyValueList.Get(&value, Name2));

    // もうひとつ登録します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name2, ValueName2));

    // 登録状態に対してキーのサーチをします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name1));
    ASSERT_EQ(ValueName1, value);

    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name2));
    ASSERT_EQ(ValueName2, value);

    // 登録しているキーを書き換えます。
    const uint32_t newValueName1 = 0x23456789;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name1, newValueName1));
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name1));
    ASSERT_EQ(newValueName1, value);
}

//! 限界までキーを追加して削除するテストをします。
TEST(KeyValueListTest, TestAddLimit)
{
    // エントリー可能数
    static const size_t EntryCountMax = 6;

    // エントリー可能数に十分なブロック数を求めます。
    static const uint32_t BlockSize = 64;
    const uint32_t allocationBlockCount
        = (KeyValueListTest::QueryStorageSize(EntryCountMax) + BlockSize - 1) / BlockSize;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        allocationBlockCount,
        allocationBlockCount,
        EntryCountMax,
        BlockSize
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // キーバリューストレージの要素数取得のテスト
    {
        uint32_t count = 0;
        NNT_ASSERT_RESULT_SUCCESS(keyValueList.GetMaxCount(&count));
        // 要素数には予約領域が含まれるためエントリー可能数より大きくなります。
        // またエントリー可能数によってはアロケーションテーブルのサイズが余るため
        // 実際のエントリー可能数が大きくなります。
        ASSERT_LT(EntryCountMax, count);
    }

    // キーを登録します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name1, ValueName1));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name2, ValueName2));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name3, ValueName3));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name4, ValueName4));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name5, ValueName5));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name6, ValueName6));

    // もう keyValueList の容量が足りない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        keyValueList.Add(Name7, ValueName7)
    );

    // 存在しないキーは keyValueList から消せません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Remove(Name7)
    );

    // 存在するキーを消します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Remove(Name1));

    // 消したキーは取得できません。
    uint32_t value;
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Get(&value, Name1)
    );

    // 新たにキーを足します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name7, ValueName7));

    // 書かれているはずのキーをチェックします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name2));
    ASSERT_EQ(ValueName2, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name3));
    ASSERT_EQ(ValueName3, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name4));
    ASSERT_EQ(ValueName4, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name5));
    ASSERT_EQ(ValueName5, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name6));
    ASSERT_EQ(ValueName6, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name7));
    ASSERT_EQ(ValueName7, value);
}

//! イテレーションのテストをします。
TEST(KeyValueListTest, TestIterate)
{
    static const size_t AllocationBlockCount = 2;
    static const uint32_t BlockSize = 64;

    // エントリー可能数
    static const size_t EntryCountMax = 10;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        AllocationBlockCount,
        AllocationBlockCount,
        EntryCountMax,
        BlockSize
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // キーを追加します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name2, ValueName2));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name3, ValueName3));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name4, ValueName4));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name5, ValueName5));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name6, ValueName6));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name7, ValueName7));

    // イテレーションし続ければそのうち終わることのチェック
    KeyValueListTmpl::FindIndex findIndex;
    keyValueList.FindOpen(&findIndex);
    for( ; ; )
    {
        KeyValueListTest::Key key;
        KeyValueListTest::Value val;
        bool isFinished;
        nn::Result result = keyValueList.FindNext(&key, &val, &isFinished, &findIndex);
        NNT_ASSERT_RESULT_SUCCESS(result);

        if( isFinished )
        {
            break;
        }
        //NN_TLOG_("%s %08X\n", key.key, val);
    }
}

//! AllocationTable を拡張してキーを追加するテストをします。
TEST(KeyValueListTest, TestExpand)
{
    //                                                 拡張前       拡張後
    // KVL の使える AllocationTableStoragte のサイズ  128(64*2) → 192(64*3)
    // KVL の使うサイズ                               128(16*8) → 192(16*12)
    // KVL のエントリー数                             6         → 10

    static const size_t AllocationBlockCountBeforeExpand = 2;
    static const size_t AllocationBlockCountAfterExpand = 3;

    // エントリー可能数
    static const size_t EntryCountMax = 10;

    // ブロックサイズ
    static const uint32_t BlockSize = 64;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        AllocationBlockCountBeforeExpand,
        AllocationBlockCountAfterExpand,
        EntryCountMax,
        BlockSize
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // キーを追加します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name2, ValueName2));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name3, ValueName3));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name4, ValueName4));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name5, ValueName5));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name6, ValueName6));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name7, ValueName7));

    // もう keyValueList の容量が足りない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        keyValueList.Add(Name1, ValueName1)
    );

    // 書かれているはずのキーをチェックします。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    uint32_t value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name2));
    ASSERT_EQ(ValueName2, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name3));
    ASSERT_EQ(ValueName3, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name4));
    ASSERT_EQ(ValueName4, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name5));
    ASSERT_EQ(ValueName5, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name6));
    ASSERT_EQ(ValueName6, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name7));
    ASSERT_EQ(ValueName7, value);

    // AllocationTable を拡張します。
    {
        nn::fs::IStorage& tableStorage = keyValueListSetup.GetTableStorage();

        // アンマウントします。
        keyValueListSetup.Unmount();

        // AllocationTable を拡張します
        int64_t tableSize;
        NNT_ASSERT_RESULT_SUCCESS(tableStorage.GetSize(&tableSize));
        nn::fs::SubStorage tableSubStorage(&tableStorage, 0, tableSize);
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fssystem::dbm::AllocationTable::Expand(
                tableSubStorage,
                AllocationBlockCountBeforeExpand,
                AllocationBlockCountAfterExpand
            )
        );

        // 再度マウントします
        keyValueListSetup.Mount(AllocationBlockCountAfterExpand, BlockSize);
    }

    // 書かれているはずのキーを再度チェックします。
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name2));
    ASSERT_EQ(ValueName2, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name3));
    ASSERT_EQ(ValueName3, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name4));
    ASSERT_EQ(ValueName4, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name5));
    ASSERT_EQ(ValueName5, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name6));
    ASSERT_EQ(ValueName6, value);
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name7));
    ASSERT_EQ(ValueName7, value);

    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Get(&value, Name8)
    );
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Get(&value, Name9)
    );
    // 新たにキーを足します
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name1, ValueName1));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name8, ValueName8));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name9, ValueName9));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name0, ValueName0));

    static const uint32_t ValueNameX = 0x990099FF;
    static const KeyValueListTest::Key NameX = { "1234567" };

    // もう keyValueList の容量が足りない
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultAllocationTableFull,
        keyValueList.Add(NameX, ValueNameX)
    );

    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name1));
    ASSERT_EQ(ValueName1, value);

    // イテレーションが終わることの確認
    {
        KeyValueListTmpl::FindIndex findIndex;
        keyValueList.FindOpen(&findIndex);
        for( ; ; )
        {
            KeyValueListTest::Key key;
            KeyValueListTest::Value val;
            bool isFinished;
            nn::Result result = keyValueList.FindNext(&key, &val, &isFinished, &findIndex);
            NNT_ASSERT_RESULT_SUCCESS(result);
            if( isFinished )
            {
                break;
            }
            //NN_TLOG_("%s %08X\n", key.key, val);
        }
    }
} // NOLINT(impl/function_size)

//! キーの書き換えをテストします。
TEST(KeyValueListTest, TestModifyKey)
{
    static const size_t allocationBlockCount = 2;

    // エントリー可能数
    static const size_t entryCountMax = 10;

    // アロケーションテーブルストレージ上にキーバリューストレージを作成。
    KeyValueListSetup keyValueListSetup(
        allocationBlockCount,
        allocationBlockCount,
        entryCountMax,
        64
    );
    KeyValueListTmpl& keyValueList = keyValueListSetup.GetKeyValueList();

    // キーを登録します。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name1, ValueName1));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name2, ValueName2));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name3, ValueName3));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name4, ValueName4));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name5, ValueName5));
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Add(Name6, ValueName6));

    // 存在しないキーの書き換えは失敗します。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.ModifyKey(Name0, Name7)
    );

    // キーを書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.ModifyKey(Name7, Name1));

    // 新しいキーで値を取得します。
    KeyValueListTmpl::Value value;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name7));
    ASSERT_EQ(ValueName1, value);

    // 古いキーでは取得できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Get(&value, Name1)
    );

    // 既に存在するキーに書き換えます。
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.ModifyKey(Name3, Name2));

    // 値は書き換わっていません。
    // 出力用引数が書き換わることをテストするために一度クリアします。
    value = 0;
    NNT_ASSERT_RESULT_SUCCESS(keyValueList.Get(&value, Name3));
    ASSERT_EQ(ValueName3, value);

    // 古いキーでは取得できません。
    NNT_ASSERT_RESULT_FAILURE(
        nn::fs::ResultDatabaseKeyNotFound,
        keyValueList.Get(&value, Name2)
    );
}

NNT_DISABLE_OPTIMIZATION
//! ランダムにキーバリューリストへ以下の操作を行います。
//!     - キーの追加
//!     - キーの削除
//!     - キーの取得と内容の突き合わせ
//!     - テーブルの拡張
//!     - 全キーの取得と内容の突き合わせ
//!     - 登録済みのキーの付け替え
TEST(KeyValueListTest, TestRandom)
{
    // KeyValueList のエントリー数とエントリーサイズ
    // 一度も拡張していない場合の最大エントリー数
    uint32_t countKeyValue = 30;
    // 最大値まで拡張したときの最大エントリー数
    static const uint32_t CountKeyValueMax = 50;
    // 1つのブロックのサイズ
    static const uint32_t SizeBlock = 512;

    // テストデータを生成します。
    static const int RandomCount = 1024;
    struct
    {
        KeyValueListTest::Key k;
        KeyValueListTest::Value v;

        bool isRegistered;
    } randomData[RandomCount];

    int randomSequence;
    {
        for( randomSequence = 0; randomSequence < RandomCount; ++randomSequence )
        {
            std::sprintf(
                reinterpret_cast<char*>( &randomData[randomSequence].k.key ),
                "%u",
                randomSequence
            );
            randomData[randomSequence].v = randomSequence;
            randomData[randomSequence].isRegistered = false;
        }
    }

    // ストレージ用意
    nnt::fs::util::WriteCheckStorage tableStorage;
    nnt::fs::util::WriteCheckStorage bodyStorage;

    // 初期化
    uint32_t maxBodySize = KeyValueListTmpl::QueryStorageSize(CountKeyValueMax);
    if( maxBodySize % SizeBlock != 0 )
    {
        maxBodySize += SizeBlock - maxBodySize % SizeBlock;
    }

    // アロケーションテーブルのブロック数の拡張できる最大値
    const uint32_t blockCountMax = maxBodySize / SizeBlock;
    const int64_t tableSize = nn::fssystem::dbm::AllocationTable::QuerySize(blockCountMax);
    nnt::fs::util::SafeMemoryStorage bareTableStorage(tableSize);
    nnt::fs::util::SafeMemoryStorage bareBodyStorage(maxBodySize);
    NNT_ASSERT_RESULT_SUCCESS(
        tableStorage.Initialize(&bareTableStorage)
    );
    NNT_ASSERT_RESULT_SUCCESS(
        bodyStorage.Initialize(&bareBodyStorage)
    );

    // ブロック数が変わらない範囲で最大エントリー数、ストレージサイズを大きくする
    uint32_t defaultBodySize = KeyValueListTmpl::QueryStorageSize(countKeyValue);
    while( KeyValueListTmpl::QueryStorageSize(countKeyValue + 1) == defaultBodySize )
    {
        ++countKeyValue;
    }
    if( defaultBodySize % SizeBlock != 0 )
    {
        defaultBodySize += SizeBlock - defaultBodySize % SizeBlock;
    }

    // アロケーションテーブルのブロック数の現在の最大値
    uint32_t currentBlockCountMax = defaultBodySize / SizeBlock;

    nn::fs::SubStorage tableSubStorage(&tableStorage, 0, tableSize);
    nn::fs::SubStorage bodySubStorage(&bodyStorage ,0, defaultBodySize);

    nn::fssystem::dbm::AllocationTable allocationTable;
    NNT_ASSERT_RESULT_SUCCESS(
        allocationTable.Format(
            tableSubStorage,
            currentBlockCountMax
        )
    );
    allocationTable.Initialize(
        tableSubStorage,
        currentBlockCountMax
    );

    // アロケーションテーブルストレージの開始インデックス
    uint32_t storageIndex;
    NNT_ASSERT_RESULT_SUCCESS(
        allocationTable.Allocate(&storageIndex, 1)
    );
    nn::fssystem::dbm::BufferedAllocationTableStorage allocationTableStorage;
    NNT_ASSERT_RESULT_SUCCESS(
        allocationTableStorage.InitializeBuffered(
            &allocationTable,
            storageIndex,
            SizeBlock,
            bodySubStorage,
            nnt::fs::GetBufferManager()
        )
    );
    // キーバリューリスト
    KeyValueListTmpl keyValueList;
    NNT_ASSERT_RESULT_SUCCESS(
        keyValueList.Format(&allocationTableStorage)
    );

    keyValueList.Initialize(&allocationTableStorage);

    const int MaxLoop = 400;
    uint32_t registoredCount = 0;
    std::mt19937 mt(nnt::fs::util::GetRandomSeed());
    for( int i = 0; i < MaxLoop; ++i )
    {
        switch( std::uniform_int_distribution<>(0, 5)(mt) )
        {
        case 0:
            {
                // キーの追加
                const int index = std::uniform_int_distribution<int>(0, RandomCount - 1)(mt);
                const nn::Result result = keyValueList.Add(
                                              randomData[index].k, randomData[index].v
                                          );

                // キーが既に使われているか否か
                if( !randomData[index].isRegistered )
                {
                    // エントリー数が上限に達しているか否か
                    if( registoredCount == countKeyValue )
                    {
                        // ストレージが溢れていたので追加に失敗します
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultAllocationTableFull, result);
                    }
                    else
                    {
                        // まだ上限ではないので追加に成功
                        ++registoredCount;
                        randomData[index].isRegistered = true;
                        NNT_ASSERT_RESULT_SUCCESS(result);
                    }
                }
                else
                {
                    // 既に使われているキーなので、バリューが更新される
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }
            }
            break;

        case 1:
            {
                // キーの削除
                const int index = std::uniform_int_distribution<int>(0, RandomCount - 1)(mt);
                const nn::Result result = keyValueList.Remove(randomData[index].k);

                // 既に登録してあるキーかを判定します
                if( !randomData[index].isRegistered )
                {
                    // 登録していないので削除できません
                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, result);
                }
                else
                {
                    // 登録していたので成功します
                    registoredCount--;
                    randomData[index].isRegistered = false;
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }
            }
            break;

        case 2:
            {
                // キーの取得、突合せ
                const int index = std::uniform_int_distribution<int>(0, RandomCount - 1)(mt);
                uint32_t value = 0;
                const nn::Result result = keyValueList.Get(&value, randomData[index].k);

                // 既に登録してあるキー?
                if( !randomData[index].isRegistered )
                {
                    // 登録していないので見つかりません

                    NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, result);
                }
                else
                {
                    // 登録しているので成功します
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    ASSERT_EQ(randomData[index].v, value);
                }
            }
            break;

        case 3:
            {
                // テーブル拡張
                if( registoredCount > countKeyValue * 8 / 10 )
                {
                    // 使用率が80%を超える場合、ストレージを拡張します
                    // 拡張するサイズは1～2
                    uint32_t newBlockCountMax
                        = currentBlockCountMax + std::uniform_int_distribution<uint32_t>(1, 2)(mt);
                    if( newBlockCountMax > blockCountMax )
                    {
                        newBlockCountMax = blockCountMax;
                    }

                    if( newBlockCountMax > currentBlockCountMax )
                    {
                        keyValueList.Finalize();
                        allocationTableStorage.Finalize();
                        allocationTable.Finalize();

                        NNT_ASSERT_RESULT_SUCCESS(
                            nn::fssystem::dbm::AllocationTable::Expand(
                                tableSubStorage,
                                currentBlockCountMax,
                                newBlockCountMax
                            )
                        );
                        const int64_t newStoragteSize = newBlockCountMax * SizeBlock;

                        //エントリー数の上限も上がる
                        countKeyValue = KeyValueListTmpl::QueryEntryCount(newStoragteSize);

                        allocationTable.Initialize(
                            tableSubStorage,
                            newBlockCountMax
                        );
                        currentBlockCountMax = newBlockCountMax;
                        nn::fs::SubStorage newBodySubStorage(
                            &bodyStorage,
                            0,
                            KeyValueListTmpl::QueryStorageSize(countKeyValue)
                        );
                        NNT_ASSERT_RESULT_SUCCESS(
                            allocationTableStorage.InitializeBuffered(
                                &allocationTable,
                                storageIndex,
                                SizeBlock,
                                newBodySubStorage,
                                nnt::fs::GetBufferManager()
                            )
                        );
                        keyValueList.Initialize(&allocationTableStorage);
                    }
                }
            }
            break;

        case 4:
            // 768 分の 1 の確率で、全キーを舐めて問題が無いか確認する
            if( 0 == (std::uniform_int_distribution<>(0, 127)(mt)) )
            {
                for( auto randomDataEntry : randomData )
                {
                    uint32_t value = 0;
                    const nn::Result result = keyValueList.Get(&value, randomDataEntry.k);
                    if( randomDataEntry.isRegistered )
                    {
                        // 登録済み
                        NNT_ASSERT_RESULT_SUCCESS(result);
                        ASSERT_EQ(randomDataEntry.v, value);
                    }
                    else
                    {
                        // 未登録
                        NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, result);
                    }
                }
            }
            break;

        case 5:
            // 登録済みのキーの付け替え
            {
                const int index = std::uniform_int_distribution<int>(0, RandomCount - 1)(mt);
                if( randomData[index].isRegistered )
                {
                    KeyValueListTest::Key newKey;

                    std::sprintf(reinterpret_cast<char*>( &newKey ), "%u", randomSequence);
                    ++randomSequence;
                    NNT_ASSERT_RESULT_SUCCESS(
                        keyValueList.ModifyKey(newKey, randomData[index].k)
                    );
                    randomData[index].k = newKey;
                }
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }

        // 最後に全体チェック
        for( auto randomDataEntry : randomData )
        {
            uint32_t value = 0;
            const nn::Result result = keyValueList.Get(&value, randomDataEntry.k);
            if( randomDataEntry.isRegistered )
            {
                // 登録済み
                NNT_ASSERT_RESULT_SUCCESS(result);
                ASSERT_EQ(randomDataEntry.v, value);
            }
            else
            {
                // 未登録
                NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultDatabaseKeyNotFound, result);
            }
        }
    }
} // NOLINT(impl/function_size)
NNT_RESTORE_OPTIMIZATION
