﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_SubdirectoryFileSystem.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fs_Result.h>
#include <nn/fssystem/fs_ConcatenationFileSystem.h>
#include <nn/fs/fs_FileSystemPrivate.h>


using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nn::fs::detail;
using namespace nn::fssystem;
using namespace nnt::fs::util;

namespace {


int64_t GetDirectoryEntryCount(const char* path)
{
    DirectoryHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenDirectory(&handle, path, OpenDirectoryMode_All));

    int64_t count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(GetDirectoryEntryCount(&count, handle));

    CloseDirectory(handle);
    return count;
}

//! @pre path 下に対象ファイルが1つだけ存在する
int64_t GetFileSizeByReadDirectory(const char* path)
{
    DirectoryHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(OpenDirectory(&handle, path, OpenDirectoryMode_All));

    int64_t count;
    DirectoryEntry entry;
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadDirectory(&count, &entry, handle, 1));

    CloseDirectory(handle);
    return entry.fileSize;
}

const int64_t StorageSize = 80ULL * 1024 * 1024 * 1024;

}

class ConcatenationFileSystemTestBase : public ::testing::Test
{
protected:
    void SetUpImpl(nn::fs::IStorage* pBaseStorage) NN_NOEXCEPT
    {
        // FatFs を生成
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        // キャッシュバッファ・IStorage を割り当て
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        m_CacheBuffer.reset(new char[cacheBufferSize]);
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(pBaseStorage, m_CacheBuffer.get(), cacheBufferSize));

        // FAT にフォーマット
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

        // shared_ptr代わり
        std::unique_ptr<IFileSystem> pFatFsCopy(new SubdirectoryFileSystem(fatFs.get(), "/"));

        {
            std::unique_ptr<ConcatenationFileSystem> catFs(new ConcatenationFileSystem(std::move(fatFs)));
            m_pCatFs = catFs.get();

            // mount
            NNT_ASSERT_RESULT_SUCCESS(fsa::Register("cat", std::move(catFs)));
        }

        // fat 直も mount
        NNT_ASSERT_RESULT_SUCCESS(fsa::Register("raw", std::move(pFatFsCopy)));
    }

    void TearDownImpl() NN_NOEXCEPT
    {
        Unmount("raw");
        Unmount("cat");
    }

    nn::fssystem::ConcatenationFileSystem* GetCatFs() NN_NOEXCEPT
    {
        return m_pCatFs;
    }

private:
    nn::fssystem::ConcatenationFileSystem* m_pCatFs;
    std::unique_ptr<char[]> m_CacheBuffer;
};

class ConcatenationFileSystemTest : public ConcatenationFileSystemTestBase
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        m_BaseStorage.Initialize(StorageSize);

        SetUpImpl(&m_BaseStorage);
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        TearDownImpl();

        m_BaseStorage.Finalize();
    }

private:
    nnt::fs::util::VirtualMemoryStorage m_BaseStorage;
};

class ConcatenationFileSystemTestWithHost : public ConcatenationFileSystemTestBase, public nnt::fs::util::PrepareWorkDirFixture
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        CreateWorkRootPath();
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHost(MountName, GetWorkRootPath().c_str()));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(FilePath, StorageSize));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(
            &m_BaseHandle,
            FilePath,
            static_cast<nn::fs::OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));

        m_pBaseStorage.reset(new nn::fs::FileHandleStorage(m_BaseHandle));

        SetUpImpl(m_pBaseStorage.get());
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        TearDownImpl();

        m_pBaseStorage.reset();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(m_BaseHandle));
        nn::fs::CloseFile(m_BaseHandle);
        nn::fs::DeleteFile(FilePath);

        nn::fs::Unmount(MountName);
        DeleteWorkRootPath();
    }

private:
    static const char MountName[];
    static const char FilePath[];

private:
    nn::fs::FileHandle m_BaseHandle;
    std::unique_ptr<nn::fs::FileHandleStorage> m_pBaseStorage;
};
const char ConcatenationFileSystemTestWithHost::MountName[] = "host";
const char ConcatenationFileSystemTestWithHost::FilePath[] = "host:/fat.img";

TEST_F(ConcatenationFileSystemTest, CreateAndDelete)
{
    struct TestCase
    {
        int64_t fileSize;
        int fileCount;
    } testCases[] =
    {
        {  0xFFFF0000ULL, 1},
        {  0xFFFF0001ULL, 2},
        { 0x200000000ULL, 3},
        {0x1000000000ULL, 17},
    };

    // 適切に分割ファイルが作られること/削除できること
    for(auto testCase : testCases)
    {
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/file", testCase.fileSize, CreateFileOptionFlag_BigFile));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(testCase.fileCount, GetDirectoryEntryCount("raw:/file/"));
        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("cat:/"));

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/file"));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("cat:/"));
    }
}

// CreateFile で容量不足時にロールバック出来ること
TEST_F(ConcatenationFileSystemTest, CreateFileUsableSpaceNotEnough)
{
    struct TestCase
    {
        int64_t fileSize;
        int64_t freeSize;
    } testCases[] =
    {
        { 0xF0000000ULL,   0xE0000000ULL },
        { 0xFFFF0000ULL,   0xF0000000ULL },
        { 0xFFFF0001ULL,   0xF0000000ULL },
        { 0x200000000ULL,  0x20000000ULL },
        { 0x200000000ULL,  0x50000000ULL },
        { 0x200000000ULL,  0x1F0000000ULL },
        { 0x1000000000ULL, 0x1F0000000ULL },
    };

    // 適切に分割ファイルが作られること/削除できること
    for (auto testCase : testCases)
    {
        const int64_t PaddingSize = StorageSize - testCase.freeSize;
        NNT_EXPECT_RESULT_SUCCESS(CreateFile("cat:/padding", PaddingSize, CreateFileOptionFlag_BigFile));

        NNT_EXPECT_RESULT_FAILURE(ResultUsableSpaceNotEnough, GetCatFs()->CreateFile("/file", testCase.fileSize, nn::fs::CreateFileOptionFlag_BigFile));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(1, GetDirectoryEntryCount("raw:/"));
        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("cat:/"));

        EXPECT_EQ(1, GetDirectoryEntryCount("raw:/"));
        EXPECT_EQ(1, GetDirectoryEntryCount("cat:/"));

        NNT_EXPECT_RESULT_SUCCESS(DeleteFile("cat:/padding"));
    }
}


TEST_F(ConcatenationFileSystemTest, DeleteArchiveDirectoryIncludingUnknownEntry)
{
    // アーカイブディレクトリ内に不明なファイルがあっても強制削除する
    {
        const int64_t FileSize = 4ULL * 1024 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/file", FileSize, CreateFileOptionFlag_BigFile));
        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));

        {
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/file/!", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/file/#"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/file/#/$", 0));

            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/file/001", 0));

            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/file/x", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/file/y"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/file/y/z", 0));
        }

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/file"));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("cat:/"));
    }
}


TEST_F(ConcatenationFileSystemTest, DeleteDirectoryRecursively)
{
    // ディレクトリ再起削除
    {
        const int64_t FileSize = 4ULL * 1024 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));

        {
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/b", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir/dir/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/b", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/c", 0));
        }

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/dir/file", FileSize, CreateFileOptionFlag_BigFile));

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteDirectoryRecursively("/dir"));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("cat:/"));

        // ルートディレクトリは削除出来ない
        NNT_EXPECT_RESULT_FAILURE(ResultDirectoryUndeletable, GetCatFs()->DeleteDirectoryRecursively("/"));
    }
}


TEST_F(ConcatenationFileSystemTest, CleanDirectoryRecursively)
{
    // ディレクトリ再起削除
    {
        const int64_t FileSize = 4ULL * 1024 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));

        {
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/b", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateDirectory("raw:/dir/dir/dir"));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/a", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/b", 0));
            NNT_EXPECT_RESULT_SUCCESS(CreateFile("raw:/dir/dir/dir/c", 0));
        }

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/dir/file", FileSize, CreateFileOptionFlag_BigFile));

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CleanDirectoryRecursively("/dir"));
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteDirectory("/dir"));

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("raw:/"));
        EXPECT_EQ(0, GetDirectoryEntryCount("cat:/"));
    }
}


TEST_F(ConcatenationFileSystemTest, BoundaryReadWrite)
{
    // 分割境界をまたいで write/read
    {
        const int64_t FileSize = 4ULL * 1024 * 1024 * 1024;
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/4GB", FileSize, CreateFileOptionFlag_BigFile));
        std::unique_ptr<IFile> file;

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->OpenFile(&file, "/4GB", static_cast<OpenMode>(OpenMode_Write | OpenMode_Read)));

        const size_t Size = 128 * 1024;
        int64_t offset = FileSize - Size;

        std::unique_ptr<char[]> buffer(new char[Size]);
        FillBufferWith32BitCount(buffer.get(), Size, 0);
        NNT_EXPECT_RESULT_SUCCESS(file->Write(offset, buffer.get(), Size, WriteOption()));

        std::unique_ptr<char[]> buffer2(new char[Size]);
        size_t readSize;
        InvalidateVariable(buffer2.get(), Size);
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, offset, buffer2.get(), Size, ReadOption()));

        EXPECT_EQ(Size, readSize);

        NNT_FS_UTIL_EXPECT_MEMCMPEQ(buffer.get(), buffer2.get(), Size);

        file.reset(); // close

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/4GB"));
    }
}

TEST_F(ConcatenationFileSystemTest, BoundaryAppendWrite)
{
    // 分割境界をまたいで append write
    const int64_t Offsets[] = {
        0xFFFF0000 - 128 * 1024,
        0xFFFF0000 - 64 * 1024,
        0xFFFF0000
    };

    for(auto offset : Offsets)
    {
        const int64_t FileSize = offset;
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/1file", FileSize, CreateFileOptionFlag_BigFile));
        std::unique_ptr<IFile> file;

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->OpenFile(&file, "/1file", static_cast<OpenMode>(OpenMode_Write | OpenMode_Read | OpenMode_AllowAppend)));

        const size_t Size = 128 * 1024;

        std::unique_ptr<char[]> buffer(new char[Size]);
        FillBufferWith32BitCount(buffer.get(), Size, 0);
        NNT_EXPECT_RESULT_SUCCESS(file->Write(offset, buffer.get(), Size, WriteOption()));

        int64_t gotSize = 0;
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&gotSize));
        EXPECT_EQ(offset + Size, gotSize);

        std::unique_ptr<char[]> buffer2(new char[Size]);
        size_t readSize;
        InvalidateVariable(buffer2.get(), Size);
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, offset, buffer2.get(), Size, ReadOption()));

        EXPECT_TRUE(IsFilledWith32BitCount(buffer2.get(), Size, 0));

        file.reset(); // close
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/1file"));
    }
}


TEST_F(ConcatenationFileSystemTest, BoundarySetSize)
{
    // 分割境界をまたいで SetSize
    {
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/file", 0, CreateFileOptionFlag_BigFile));
        std::unique_ptr<IFile> file;

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->OpenFile(&file, "/file", OpenMode_Write));
        int64_t sizes[] = {
            0,
            1,
            0xFFFEFFFF,
            0xFFFF0000,     // (4GB - 64KB)
            0xFFFF0001,

            0x100000000ULL, // 4GB

            0x1FFFDFFFFULL,
            0x1FFFE0000ULL, // (4GB - 64KB) * 2
            0x1FFFE0001ULL,

            0x200000000ULL, // 8GB

            0x400000000ULL, // 16GB
        };

        for(auto from : sizes)
        {
            for(auto to : sizes)
            {
                int64_t gotSize;

                NNT_EXPECT_RESULT_SUCCESS(file->SetSize(from));
                NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&gotSize));
                EXPECT_EQ(from, gotSize);

                NN_SDK_LOG("size: %llx -> %llx\n", from, to);

                NNT_EXPECT_RESULT_SUCCESS(file->SetSize(to));
                NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&gotSize));
                EXPECT_EQ(to, gotSize);

                EXPECT_EQ(to, GetFileSizeByReadDirectory("cat:/"));
            }
        }

        file.reset();

        NNT_EXPECT_RESULT_SUCCESS(ListDirectoryRecursive("raw:/"));
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/file"));
    }
}


TEST_F(ConcatenationFileSystemTestWithHost, WriteReadHeavy)
{
    // 12GB を書き込み・ベリファイ
    {
        const int64_t FileSize = 0x300000000ULL;
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->CreateFile("/1file", FileSize, CreateFileOptionFlag_BigFile));
        std::unique_ptr<IFile> file;

        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->OpenFile(&file, "/1file", static_cast<OpenMode>(OpenMode_Write | OpenMode_Read)));

        const size_t  Size = 32 * 1024 * 1024;

        for(int i=0; i<FileSize / Size; ++i)
        {
            int64_t offset = Size * i;

            std::unique_ptr<char[]> buffer(new char[Size]);
            FillBufferWith32BitCount(buffer.get(), Size, Size * i);
            NNT_EXPECT_RESULT_SUCCESS(file->Write(offset, buffer.get(), Size, WriteOption()));
        }

        for(int i=0; i<FileSize / Size; ++i)
        {
            int64_t offset = Size * i;

            std::unique_ptr<char[]> buffer2(new char[Size]);
            size_t readSize;
            InvalidateVariable(buffer2.get(), Size);
            NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, offset, buffer2.get(), Size, ReadOption()));
            EXPECT_TRUE(IsFilledWith32BitCount(buffer2.get(), Size, Size * i));
        }

        file.reset(); // close
        NNT_EXPECT_RESULT_SUCCESS(GetCatFs()->DeleteFile("/1file"));
    }
}

// 4 GB を超えるオフセットへのアクセステスト
TEST_F(ConcatenationFileSystemTest, LargeOffset)
{
    nnt::fs::util::TestFileSystemAccessWithLargeOffset(GetCatFs(), "/file");
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    ::testing::InitGoogleTest(&argc, argv);
    int result = RUN_ALL_TESTS();

    nnt::Exit(result);
}
