﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>
#include <numeric>
#include <mutex>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_Mutex.h>
#include <nn/os/os_ThreadApi.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/os/os_ThreadCommon.h>

#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Compiler.h>
#include <nnt/fsUtil/testFs_util.h>

//! バッファ確保のテストです。
TEST(FileSystemBufferManagerTest, Allocation)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle;

    struct Entry
    {
        std::pair<uintptr_t, size_t> range;
        CacheHandle handle;
    };

    static const auto EntryCountMax = 3;
    static const auto CacheCountMax = 2;
    static const auto LoopCountMax = 100;

    Entry entries[EntryCountMax];

    std::mt19937 mt(1);

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        if( ((loopCount + 1) % (LoopCountMax / 10)) == 0 )
        {
            NN_LOG(".");
        }

        const auto cacheSize = std::uniform_int_distribution<size_t>(128, 255)(mt) * 1024;

        // キャッシュ用バッファを初期化します。
        nnt::fs::util::Vector<char> buf(cacheSize);

        const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());
        const auto blockSize = (sizeof(char) << std::uniform_int_distribution<>(5, 10)(mt));

        // 引数を省略すると最大オーダーは全領域を格納できる値に自動設定されます。
        BufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, blockSize)
        );

        // 全エントリを初期化します。
        for( auto& entry : entries )
        {
            entry.range.first = 0;
            entry.range.second = 0;
            entry.handle = 0;
        }

        // バッファからメモリを確保します。
        // ブロックサイズ単位で確保されるはずです。
        entries[0].range = cacheBuffer.AllocateBuffer(1);
        ASSERT_EQ(blockSize, entries[0].range.second);
        entries[1].range = cacheBuffer.AllocateBuffer(blockSize);
        ASSERT_EQ(blockSize, entries[1].range.second);
        entries[2].range = cacheBuffer.AllocateBuffer(blockSize + 1);
        ASSERT_EQ(blockSize * 2, entries[2].range.second);

        // キャッシュの後始末
        for( auto& entry : entries )
        {
            // キャッシュは取得済みである点に注意
            entry.handle = 0;
            cacheBuffer.DeallocateBuffer(entry.range.first, entry.range.second);
        }

        cacheBuffer.Finalize();
    }

    NN_LOG("\n");
}

//! キャッシュが引き剥がされない挙動のテストです。
void AllocateNone(
         nn::fssystem::FileSystemBufferManager& bufManager,
         size_t pageSize,
         int cacheCountMax
     ) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    nnt::fs::util::Vector<BufferManager::CacheHandle> handles;

    // キャッシュを最大数まで登録します。
    for( auto count = 0; count < cacheCountMax; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(pageSize);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(count % 3)
                            );
        handles.push_back(handle);
    }

    // 全てのキャッシュが有効です。
    for( const auto handle : handles )
    {
        const auto memRange = bufManager.AcquireCache(handle);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }
}

//! 最古のキャッシュが引き剥がされる挙動のテストです。
void AllocateOldest(
         nn::fssystem::FileSystemBufferManager& bufManager,
         size_t pageSize,
         int cacheCountMax
     ) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    nnt::fs::util::Vector<BufferManager::CacheHandle> handles;

    // 最大数 + 3 個のキャッシュを登録します。
    for( auto count = 0; count < cacheCountMax + 3; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(pageSize);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(count % 3)
                            );
        handles.push_back(handle);
    }

    // 最初の 3 個のキャッシュが引き剥がされています。
    for( auto count = 0; count < 3; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_EQ(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_EQ(static_cast<size_t>(0), memRange.second);
    }

    // 残りのキャッシュは有効です。
    for( auto count = 3; count < cacheCountMax + 3; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }
}

// 登録数が多い属性のキャッシュから引き剥がされる挙動のテストです。
void AllocateFew(
         nn::fssystem::FileSystemBufferManager& bufManager,
         size_t pageSize,
         size_t bufferSize,
         int cacheCountMax
     ) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    nnt::fs::util::Vector<BufferManager::CacheHandle> handles;

    // 属性 0 で 2 個だけキャッシュを登録します。
    for( auto count = 0; count < 2; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(bufferSize / 4);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(0)
                            );
        handles.push_back(handle);
    }

    // 属性 1 で最大数をオーバーする個数を登録します。
    for( auto count = 2; count < cacheCountMax + 1; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(pageSize);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(1)
                            );
        handles.push_back(handle);
    }

    // 2 個だけ登録した属性のキャッシュは全て有効です。
    for( auto count = 0; count < 2; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }

    // 属性 1 のキャッシュは最初のひとつだけ引き剥がされ他は有効です。
    for( auto count = 2; count < 3; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_EQ(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_EQ(static_cast<size_t>(0), memRange.second);
    }

    for( auto count = 3; count < cacheCountMax + 1; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }
}

// 登録サイズが大きい属性のキャッシュから引き剥がされる挙動のテストです。
void AllocateSmall(
         nn::fssystem::FileSystemBufferManager& bufManager,
         size_t pageSize,
         size_t blockSize,
         int cacheCountMax
     ) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    nnt::fs::util::Vector<BufferManager::CacheHandle> handles;

    // ブロックサイズと同じサイズで最大数 - 100 個のキャッシュを登録します。
    for( auto count = 0; count < cacheCountMax - 100; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(blockSize);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(1)
                            );
        handles.push_back(handle);
    }

    // ページサイズと同じサイズで 101個のキャッシュを登録します。
    for( auto count = cacheCountMax - 100; count < cacheCountMax + 1; ++count )
    {
        const auto memRange = bufManager.AllocateBuffer(pageSize);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        const auto handle = bufManager.RegisterCache(
                                memRange.first,
                                memRange.second,
                                BufferManager::BufferAttribute(0)
                            );
        handles.push_back(handle);
    }

    // ブロックサイズで登録したキャッシュは全て有効です。
    for( auto count = 0; count < cacheCountMax - 100; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }

    // ページサイズで登録したキャッシュは最初のひとつだけ引き剥がされて他は有効です。
    for( auto count = cacheCountMax - 100; count < cacheCountMax - 99; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_EQ(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_EQ(static_cast<size_t>(0), memRange.second);
    }

    for( auto count = cacheCountMax - 99; count < cacheCountMax + 1; ++count )
    {
        const auto memRange = bufManager.AcquireCache(handles[count]);
        ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_NE(static_cast<size_t>(0), memRange.second);
        bufManager.DeallocateBuffer(memRange.first, memRange.second);
    }
}

// キャッシュが引き剥がされる挙動をテストします。
TEST(FileSystemBufferManagerTest, Cache)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle NN_IS_UNUSED_MEMBER;
    typedef BufferManager::BufferAttribute Attribute NN_IS_UNUSED_MEMBER;

    static const auto BlockSize = 16;
    static const auto PageSize = 256;
    static const auto BufferSize = 1024 * 1024;
    static const auto CacheCountMax = 1000;

    BufferManager bufManager;
    nnt::fs::util::Vector<char> buf(BufferSize);
    const auto bufAddress = reinterpret_cast<uintptr_t>(buf.data());
    const auto bufSize = buf.size();

    // キャッシュが引き剥がされないテスト
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(CacheCountMax, bufAddress, bufSize, BlockSize)
    );
    AllocateNone(bufManager, PageSize, CacheCountMax);
    bufManager.Finalize();

    // 古いキャッシュが引き剥がされるテスト
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(CacheCountMax, bufAddress, bufSize, BlockSize)
    );
    AllocateOldest(bufManager, PageSize, CacheCountMax);
    bufManager.Finalize();

    // 登録個数が多い属性のキャッシュが引き剥がされるテスト
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(CacheCountMax, bufAddress, bufSize, BlockSize)
    );
    AllocateFew(bufManager, PageSize, BufferSize, CacheCountMax);
    bufManager.Finalize();

    // 登録サイズが大きい属性のキャッシュが引き剥がされるテスト
    NNT_ASSERT_RESULT_SUCCESS(
        bufManager.Initialize(CacheCountMax, bufAddress, bufSize, BlockSize)
    );
    AllocateSmall(bufManager, PageSize, BlockSize, CacheCountMax);
    bufManager.Finalize();
}

//! ランダムテストです。
struct RandomEntry
{
    std::pair<uintptr_t, size_t> range;
    nn::fssystem::FileSystemBufferManager::CacheHandle handle;
    int loopCount;
    bool isValid;
};

void RandomOperation(
         nn::fssystem::FileSystemBufferManager* pCacheBuffer,
         RandomEntry* pEntry,
         size_t blockSize,
         int loop,
         std::mt19937* pMt
     ) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    // バッファを取得してキャッシュに登録する関数です。
    const auto RegisterCache = [=]() NN_NOEXCEPT
    {
        const auto size = std::uniform_int_distribution<size_t>(0, blockSize - 1)(*pMt);
        pEntry->range = pCacheBuffer->AllocateBuffer(size);

        // 登録するバッファにデータを書き込みます。
        // 書き込むデータはエントリーに保存して照合できるようにします。
        const auto dst = reinterpret_cast<char*>(pEntry->range.first);
        pEntry->loopCount = loop & 0xFF;
        std::iota(dst, dst + pEntry->range.second, static_cast<char>(pEntry->loopCount));

        // キャッシュに登録します。
        const auto attr = BufferManager::BufferAttribute(loop & 0x01);
        pEntry->handle = pCacheBuffer->RegisterCache(
                             pEntry->range.first,
                             pEntry->range.second,
                             attr
                         );
        pEntry->isValid = true;
    };

    switch( std::uniform_int_distribution<>(0, 3)(*pMt) )
    {
    case 0:
        {
            // バッファを取得してキャッシュに登録します。
            if( !pEntry->isValid )
            {
                RegisterCache();
            }
        }
        break;

    case 1:
        {
            // メモリを確保して解放します。
            // (メモリ確保時に引き剥がしが発生することがあります。)
            const auto size = std::uniform_int_distribution<size_t>(0, blockSize - 1)(*pMt);
            const auto range = pCacheBuffer->AllocateBuffer(size);
            if( 0 < range.second )
            {
                pCacheBuffer->DeallocateBuffer(range.first, range.second);
            }
        }
        break;

    case 2:
        {
            // キャッシュからデータを取得して解放します。
            if( pEntry->isValid )
            {
                const auto range = pCacheBuffer->AcquireCache(pEntry->handle);
                ASSERT_NE(static_cast<uintptr_t>(0), range.first);
                ASSERT_NE(static_cast<size_t>(0), range.second);
                pEntry->isValid = false;

                // キャッシュの内容が壊れていないことを確認します。
                const auto src = reinterpret_cast<const char*>(range.first);
                for( uint32_t offset = 0; offset < range.second; ++offset )
                {
                    ASSERT_EQ(
                        static_cast<char>((pEntry->loopCount + offset) & 0xFF),
                        src[offset]
                    );
                }

                if( 0 < range.second )
                {
                    pCacheBuffer->DeallocateBuffer(range.first, range.second);
                }
            }
            else // 代わりに新しいエントリを追加します。
            {
                RegisterCache();
            }
        }
        break;

    case 3:
        {
            // キャッシュからデータを取り出して再度登録します。
            if( pEntry->isValid )
            {
                const auto range = pCacheBuffer->AcquireCache(pEntry->handle);
                ASSERT_NE(static_cast<uintptr_t>(0), range.first);
                ASSERT_NE(static_cast<size_t>(0), range.second);

                // キャッシュの内容が壊れていないことを確認します。
                const auto src = reinterpret_cast<const char*>(range.first);
                for( uint32_t offset = 0; offset < range.second; ++offset )
                {
                    ASSERT_EQ(
                        static_cast<char>((pEntry->loopCount + offset) & 0xFF),
                        src[offset]
                    );
                }

                const auto attr = BufferManager::BufferAttribute(loop & 0x01);
                pEntry->handle = pCacheBuffer->RegisterCache(
                                     pEntry->range.first,
                                     pEntry->range.second,
                                     attr
                                 );
            }
            else // 代わりに新しいエントリを追加します。
            {
                RegisterCache();
            }
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

// ランダムテストです。
TEST(FileSystemBufferManagerTest, Random)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle NN_IS_UNUSED_MEMBER;
    typedef BufferManager::BufferAttribute Attribute NN_IS_UNUSED_MEMBER;

    static const auto EntryCountMax = 128;
    static const auto LoopCountMax = 50;
    static const auto InnerLoopCountMax = 1000;
    static const auto CacheCountMax = InnerLoopCountMax;

    RandomEntry entries[EntryCountMax];

    std::mt19937 mt(1);

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        static const auto CacheBaseSize = 1024;
        const auto cacheSize = std::uniform_int_distribution<size_t>(128, 255)(mt) * CacheBaseSize;

        // キャッシュ用バッファを初期化します。
        nnt::fs::util::Vector<char> buf(cacheSize);
        const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());

        const auto blockSize = (sizeof(char) << std::uniform_int_distribution<size_t>(5, 10)(mt));
        const auto orderMax = BufferManager::BuddyHeap::QueryOrderMax(cacheSize, blockSize);

        BufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, blockSize, orderMax)
        );

        // 全エントリを初期化します。
        for( auto& entry : entries )
        {
            entry.range.first = 0;
            entry.range.second = 0;
            entry.handle = 0;
            entry.loopCount = 0;
            entry.isValid = false;
        }

        // ランダムな操作を行います。
        for( auto loop = 0; loop < InnerLoopCountMax; ++loop )
        {
            if( ((loop + 1) % (InnerLoopCountMax / 10)) == 0 )
            {
                NN_LOG(".");
            }

            const auto entryIndex = std::uniform_int_distribution<int>(0, EntryCountMax - 1)(mt);
            RandomOperation(&cacheBuffer, entries + entryIndex, blockSize, loop, &mt);
        }

        // キャッシュの後始末
        for( auto& entry : entries )
        {
            if( entry.isValid )
            {
                const auto memRange = cacheBuffer.AcquireCache(entry.handle);
                ASSERT_NE(static_cast<uintptr_t>(0), memRange.first);
                ASSERT_NE(static_cast<size_t>(0), memRange.second);
                entry.handle = 0;
                cacheBuffer.DeallocateBuffer(memRange.first, memRange.second);
            }
        }

        cacheBuffer.Finalize();
    }

    NN_LOG("\n");
}

// 複数のスレッドからバッファにアクセスするテストでのスレッド数です。
static const auto EntryCountMaxPerThread = 32;

// 各スレッドの情報です。
struct ThreadEntry
{
    std::pair<uintptr_t, size_t> range;
    nn::fssystem::FileSystemBufferManager::CacheHandle handle;
    int loopCount;
    bool isValid;
};

// 各スレッドに渡す情報です。
struct ThreadArg
{
    nn::fssystem::FileSystemBufferManager* pCacheBuffer;
    size_t blockSize;
    int threadIndex;
    ThreadEntry (*pEntries)[EntryCountMaxPerThread];
    nn::os::Mutex* pMutex;
};

NNT_DISABLE_OPTIMIZATION
// 各スレッドが実行するランダムテストです。
void ThreadFunc(void* pArg) NN_NOEXCEPT
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;

    static const auto InnerLoopCountMax = 1000;

    const auto pThreadArg = reinterpret_cast<const ThreadArg*>(pArg);
    auto& cacheBuffer = *pThreadArg->pCacheBuffer;
    auto blockSize = pThreadArg->blockSize;
    auto threadIndex = pThreadArg->threadIndex;
    auto& entries = *pThreadArg->pEntries;
    auto& mutex = *pThreadArg->pMutex;

    const auto seed = static_cast<int>(123456789L * (threadIndex + 1));
    std::mt19937 mt(seed);

    for( auto loop = 0; loop < InnerLoopCountMax; ++loop )
    {
        if( (threadIndex == 0) && ((loop + 1) % 2000 == 0) )
        {
            NN_LOG(".");
        }

        const auto entryIndex
            = std::uniform_int_distribution<>(0, EntryCountMaxPerThread - 1)(mt);
        auto& entry = entries[entryIndex];

        switch( std::uniform_int_distribution<>(0, 2)(mt) )
        {
        case 0:
            {
                nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(1000));
            }
            break;

        case 1:
            {
                // バッファを取得してキャッシュに登録します。
                if( !entry.isValid )
                {
                    std::lock_guard<nn::os::Mutex> lock(mutex);

                    const auto size
                        = std::uniform_int_distribution<size_t>(0, blockSize - 1)(mt);
                    entry.range = cacheBuffer.AllocateBuffer(size);

                    // 登録するバッファにデータを書き込みます。
                    const auto dst = reinterpret_cast<char*>(entry.range.first);
                    entry.loopCount = loop & 0xFF;
                    std::iota(dst, dst + entry.range.second, static_cast<char>(entry.loopCount));

                    // キャッシュに登録します。
                    const auto attr = BufferManager::BufferAttribute(loop & 0x01);
                    entry.handle = cacheBuffer.RegisterCache(
                                       entry.range.first,
                                       entry.range.second,
                                       attr
                                   );
                    entry.isValid = true;
                }
            }
            break;

        case 2:
            {
                std::lock_guard<nn::os::Mutex> lock(mutex);

                // メモリを確保して解放します。
                // (メモリ確保時に引き剥がしが発生することがあります。)
                const auto size
                    = std::uniform_int_distribution<size_t>(0, blockSize - 1)(mt);
                const auto range = cacheBuffer.AllocateBuffer(size);
                if( 0 < range.second )
                {
                    cacheBuffer.DeallocateBuffer(range.first, range.second);
                }
            }
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}
NNT_RESTORE_OPTIMIZATION

// 複数のスレッドからバッファにアクセスするテストです。
TEST(FileSystemBufferManagerTest, MultiThread)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle NN_IS_UNUSED_MEMBER;
    typedef BufferManager::BufferAttribute Attribute NN_IS_UNUSED_MEMBER;

    static const auto ThreadCount = 4;
    static const auto LoopCountMax = 50;
    static const auto CacheCountMax = LoopCountMax;

    nn::os::Mutex mutex(false);

    nn::os::ThreadType threads[ThreadCount] = {};
    ThreadEntry entries[ThreadCount][EntryCountMaxPerThread];

    std::mt19937 mt(1);

    for( auto loopCount = 0; loopCount < LoopCountMax; ++loopCount )
    {
        NN_LOG(".");

        const auto cacheSize = std::uniform_int_distribution<size_t>(128, 255)(mt) * 1024;

        // キャッシュ用バッファを初期化します。
        nnt::fs::util::Vector<char> buf(cacheSize);
        const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());

        const auto blockSize = (sizeof(char) << std::uniform_int_distribution<>(5, 10)(mt));
        const auto orderMax = BufferManager::BuddyHeap::QueryOrderMax(cacheSize, blockSize);

        BufferManager cacheBuffer;
        NNT_ASSERT_RESULT_SUCCESS(
            cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, blockSize, orderMax)
        );

        // 全エントリを初期化します。
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            for( auto& entry : entries[threadIndex] )
            {
                entry.range.first = 0;
                entry.range.second = 0;
                entry.handle = 0;
                entry.loopCount = 0;
                entry.isValid = false;
            }
        }

        // 複数のスレッドから同時にファイルシステムにアクセスします。
        static const size_t StackSize = 32 * 1024;
        nnt::fs::util::Vector<char> stack(StackSize * ThreadCount + nn::os::ThreadStackAlignment);
        const auto alignedStack
            = reinterpret_cast<char*>(nn::util::align_up(
                                          reinterpret_cast<uintptr_t>(stack.data()),
                                          nn::os::ThreadStackAlignment
                                      ));

        ThreadArg args[ThreadCount] = {};

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            // 引数を準備します。
            args[threadIndex].pCacheBuffer = &cacheBuffer;
            args[threadIndex].blockSize = blockSize;
            args[threadIndex].threadIndex = threadIndex;
            args[threadIndex].pEntries = &entries[threadIndex];
            args[threadIndex].pMutex = &mutex;

            // スレッドを起動します。
            const auto prio = nn::os::DefaultThreadPriority / 2
                + std::uniform_int_distribution<>(0, nn::os::DefaultThreadPriority - 1)(mt);

            const nn::Result result = nn::os::CreateThread(
                                         threads + threadIndex,
                                         ThreadFunc,
                                         args + threadIndex,
                                         alignedStack + StackSize * threadIndex,
                                         StackSize,
                                         prio
                                     );
            NNT_ASSERT_RESULT_SUCCESS(result);

            nn::os::StartThread(threads + threadIndex);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            nn::os::WaitThread(threads + threadIndex);
            nn::os::DestroyThread(threads + threadIndex);
        }

        // キャッシュの後始末をします。
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            for( auto& entry : entries[threadIndex] )
            {
                if( entry.isValid )
                {
                    const auto memRange
                        = cacheBuffer.AcquireCache(entry.handle);
                    if( 0 < memRange.second )
                    {
                        EXPECT_NE(memRange.first, static_cast<uintptr_t>(0));
                        cacheBuffer.DeallocateBuffer(memRange.first, memRange.second);
                    }
                    else
                    {
                        EXPECT_EQ(memRange.first, static_cast<uintptr_t>(0));
                    }
                    entry.handle = 0;
                }
            }
        }

        cacheBuffer.Finalize();
    }

    NN_LOG("\n");
}

// 限界まで確保するテストです。
TEST(FileSystemBufferManagerTest, Limit)
{
    for( auto bufSize = 4 * 1024; bufSize <= 64 * 1024; bufSize *= 4 )
    {
        for( auto blockSize = 64; blockSize <= 1024; blockSize *= 4 )
        {
            for( auto pageSize = blockSize; pageSize <= blockSize * 4; pageSize *= 2 )
            {
                static const auto CacheCountMax = 4;

                // 初期化します。
                nnt::fs::util::Vector<char> buf(bufSize);
                const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());
                const auto cacheSize = buf.size();
                nn::fssystem::FileSystemBufferManager cacheBuffer;
                NNT_ASSERT_RESULT_SUCCESS(
                    cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, blockSize)
                );

                // 限界まで確保できることを確認します。
                nnt::fs::util::Vector<std::pair<uintptr_t, size_t>> ranges;
                for( auto allocatedSize = 0; allocatedSize < bufSize; allocatedSize += pageSize )
                {
                    // 失敗すると無限ループに陥ります。
                    const auto range = cacheBuffer.AllocateBuffer(pageSize);
                    EXPECT_EQ(pageSize, range.second);
                    ranges.emplace_back(range);
                }

                // 後始末をします。
                for( const auto& range : ranges )
                {
                    cacheBuffer.DeallocateBuffer(range.first, range.second);
                }
            }
        }
    }
}

// キャッシュ最大数を超える失敗時のテストです。
TEST(FileSystemBufferManagerTest, CacheCountFailure)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle;
    typedef BufferManager::BufferAttribute Attribute;

    static const auto BufSize = 64 * 1024;
    static const auto CacheCountMax = 32;
    static const auto BlockSize = 1024;

    // 初期化します。
    nnt::fs::util::Vector<char> buf(BufSize);
    const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());
    const auto cacheSize = buf.size();
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, BlockSize)
    );

    nnt::fs::util::Vector<std::pair<uintptr_t, size_t>> memRanges;
    nnt::fs::util::Vector<CacheHandle> cacheHandles;

    // 登録を失敗させます。
    // キャッシュ最大数を超える個数のバッファを確保、登録します。
    for( auto cacheIndex = 0; cacheIndex < CacheCountMax + 1; ++cacheIndex )
    {
        memRanges.push_back(cacheBuffer.AllocateBuffer(BlockSize));
        ASSERT_NE(static_cast<uintptr_t>(0), memRanges.back().first);
        ASSERT_NE(static_cast<size_t>(0), memRanges.back().second);
        cacheHandles.push_back(cacheBuffer.RegisterCache(
                                   memRanges[cacheIndex].first,
                                   memRanges[cacheIndex].second,
                                   Attribute(cacheIndex)
                               ));
    }

    // 最大数を超えたので最初のキャッシュは引き剥がされています。
    {
        const auto memRange = cacheBuffer.AcquireCache(cacheHandles.front());
        ASSERT_EQ(static_cast<uintptr_t>(0), memRange.first);
        ASSERT_EQ(static_cast<size_t>(0), memRange.second);
    }

    // 登録成功したキャッシュは取得、解放できます。
    for( auto cacheIndex = 1; cacheIndex < CacheCountMax + 1; ++cacheIndex )
    {
        const auto memRange = cacheBuffer.AcquireCache(cacheHandles[cacheIndex]);
        ASSERT_EQ(memRanges[cacheIndex].first, memRange.first);
        ASSERT_EQ(memRanges[cacheIndex].second, memRange.second);
        cacheBuffer.DeallocateBuffer(memRange.first, memRange.second);
    }

    // バッファが解放されたのでキャッシュの取得に失敗します。
    {
        const auto memRange = cacheBuffer.AcquireCache(cacheHandles.back());
        EXPECT_EQ(static_cast<uintptr_t>(0), memRange.first);
        EXPECT_EQ(static_cast<size_t>(0), memRange.second);
    }
}

// キャッシュ最大サイズを超える失敗時のテストです。
TEST(FileSystemBufferManagerTest, CacheSizeFailure)
{
    typedef nn::fssystem::FileSystemBufferManager BufferManager;
    typedef BufferManager::CacheHandle CacheHandle;
    typedef BufferManager::BufferAttribute Attribute;

    static const auto BufSize = 64 * 1024;
    static const auto CacheCountMax = 32;
    static const auto BlockSize = 1024;

    // 初期化します。
    nnt::fs::util::Vector<char> buf(BufSize);
    const auto cacheAddress = reinterpret_cast<uintptr_t>(buf.data());
    const auto cacheSize = buf.size();
    nn::fssystem::FileSystemBufferManager cacheBuffer;
    NNT_ASSERT_RESULT_SUCCESS(
        cacheBuffer.Initialize(CacheCountMax, cacheAddress, cacheSize, BlockSize)
    );

    nnt::fs::util::Vector<std::pair<uintptr_t, size_t>> memRanges;
    nnt::fs::util::Vector<CacheHandle> cacheHandles;

    // 再度バッファとキャッシュを用意します。
    // 1 / 8 サイズのバッファを 9 個確保しようとするので最初のひとつが引き剥がされます。
    for( auto cacheCount = 0; cacheCount < 9; ++cacheCount )
    {
        memRanges.push_back(cacheBuffer.AllocateBuffer(BufSize / 8));
        ASSERT_NE(static_cast<uintptr_t>(0), memRanges.back().first);
        ASSERT_NE(static_cast<size_t>(0), memRanges.back().second);

        cacheHandles.push_back(cacheBuffer.RegisterCache(
            memRanges.back().first,
            memRanges.back().second,
            Attribute(0)
        ));
    }

    auto failureCount = 0;

    // キャッシュを取得します。最初のひとつだけ失敗します。
    for( auto cacheHandle : cacheHandles )
    {
        const auto memRange = cacheBuffer.AcquireCache(cacheHandle);
        if( 0 < memRange.second )
        {
            cacheBuffer.DeallocateBuffer(memRange.first, memRange.second);
        }
        else
        {
            EXPECT_EQ(static_cast<uintptr_t>(0), memRange.first);
            ++failureCount;
        }
    }

    EXPECT_EQ(1, failureCount);
}
