﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include "detail/fssrv_AccessControl.cpp"

using namespace nn::fssrv;
using namespace nn::fssrv::detail;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>
#endif

namespace {

    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];
    nn::lmem::HeapHandle g_HeapHandle = nullptr;
    bool g_HeapAllocateAvalilable = true;
    nn::os::Mutex g_HeapMutex(false);

    void InitializeMyHeap()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
        if( g_HeapHandle == nullptr )
        {
            g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapStack, HeapSize, nn::lmem::CreationOption_DebugFill);
            NN_ASSERT_NOT_NULL(g_HeapHandle);
        }
    }

    void SetAllocateAvailable(bool value)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
        g_HeapAllocateAvalilable = value;
    }

    void* MyAllocate(size_t size)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
        if (!g_HeapAllocateAvalilable)
        {
            return nullptr;
        }
        return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
    }

    void MyDeallocate(void* p, size_t size)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_HeapMutex);
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_HeapHandle, p);
    }

    typedef nnt::fs::util::CheckGlobalNewDeleteFlagTestFixture AccessControlTestBase;

    class DeathTest : public AccessControlTestBase
    {
    protected:
        DeathTest()
        {
            CheckGlobalNewDeleteFlagTestFixture::DisableCheck();
        }
    };

    typedef DeathTest AccessControlDeathTest;

    // データ確認用クラス
    class AccessControlForTest : public AccessControl
    {
    public:
        AccessControlForTest(const void* fsAccessControlData, int64_t dataSize, const void* fsAccessControlDesc, int64_t descSize) NN_NOEXCEPT : AccessControl(fsAccessControlData, dataSize, fsAccessControlDesc, descSize)
        {
        }
        const nn::Bit64 GetRawFlagBits() NN_NOEXCEPT
        {
            return AccessControl::GetRawFlagBits();
        }
    };

} // namespace

// 設定した情報が正しく取得できること
TEST_F(AccessControlTestBase, Basic)
{
    static const unsigned char fac[] =
    {
        0x01, 0x00, 0x00, 0x00, // version + padding

        0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, // flag

        0x1C, 0x00, 0x00, 0x00, // offset of content owner Infos start
        0x14, 0x00, 0x00, 0x00, // size of content owner Infos
        0x30, 0x00, 0x00, 0x00, // offset of savedata owner Infos start
        0x18, 0x00, 0x00, 0x00, // size of savedata owner Infos

        0x02, 0x00, 0x00, 0x00, // num of content owner Ids
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // content owner Id1
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, // content owner Id2

        0x02, 0x00, 0x00, 0x00, // num of savedata owner Ids
        0x02, 0x01, 0x00, 0x00, // savedata access for each Id
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // savedata owner Id1
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, // savedata owner Id2
    };

    static const unsigned char facd[] =
    {
        0x01, 0x00, 0x00, 0x00, // version + padding

        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, // flag

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // content owner Id begin
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, // content owner Id end
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // savedata owner Id begin
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, // savedata owner Id end
    };

    AccessControlForTest ac(fac, sizeof(fac), facd, sizeof(facd));

    nn::Bit64 flag = 0x8877665544332211ULL;
    EXPECT_EQ(ac.GetRawFlagBits(), flag);

    nn::Bit64 id1 = 0x0000000000000001ULL;
    nn::Bit64 id2 = 0x8000000000000001ULL;

    EXPECT_TRUE(ac.HasContentOwnerId(id1));
    EXPECT_TRUE(ac.HasContentOwnerId(id2));

    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id1).CanRead());
    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id1).CanWrite());
    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id2).CanRead());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id2).CanWrite());
}

// フラグビット・ID が desc でマスクされること
TEST_F(AccessControlTestBase, Verify)
{
    static const unsigned char fac[] =
    {
        0x01, 0x00, 0x00, 0x00,

        0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

        0x1C, 0x00, 0x00, 0x00,
        0x14, 0x00, 0x00, 0x00,
        0x30, 0x00, 0x00, 0x00,
        0x34, 0x00, 0x00, 0x00,

        0x02, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

        0x05, 0x00, 0x00, 0x00,
        0x03, 0x01, 0x02, 0x03, 0x01, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7F,
    };

    static const unsigned char facd[] =
    {
        0x01, 0x00, 0x00, 0x00,

        0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
    };

    static const unsigned char facdIdsSpecified[] =
    {
        0x01, 0x01, 0x03, 0x00,

        0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    };

    AccessControlForTest ac(fac, sizeof(fac), facd, sizeof(facd));

    nn::Bit64 flag = 0x5A5A5A5A00000000ULL;
    EXPECT_EQ(ac.GetRawFlagBits(), flag);

    nn::Bit64 id1 = 0x0000000000000001ULL;
    nn::Bit64 id2 = 0x8000000000000001ULL;
    nn::Bit64 id3 = 0x8000000000000000ULL;
    nn::Bit64 id4 = 0xFFFFFFFFFFFFFFFFULL;
    nn::Bit64 id5 = 0x7F00000000000000ULL;

    EXPECT_FALSE(ac.HasContentOwnerId(id1));
    EXPECT_TRUE(ac.HasContentOwnerId(id2));

    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id1).CanRead());
    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id1).CanWrite());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id2).CanRead());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id2).CanWrite());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id3).CanRead());
    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id3).CanWrite());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id4).CanRead());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id4).CanWrite());
    EXPECT_TRUE(ac.GetAccessibilitySaveDataOwnedBy(id5).CanRead());
    EXPECT_FALSE(ac.GetAccessibilitySaveDataOwnedBy(id5).CanWrite());

    // IdBegin/End の設定は無視される
    AccessControlForTest ac2(fac, sizeof(fac), facdIdsSpecified, sizeof(facdIdsSpecified));

    EXPECT_TRUE(ac2.HasContentOwnerId(id1));
    EXPECT_FALSE(ac2.HasContentOwnerId(id2));

    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id1).CanRead());
    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id1).CanWrite());
    EXPECT_TRUE(ac2.GetAccessibilitySaveDataOwnedBy(id2).CanRead());
    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id2).CanWrite());
    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id3).CanRead());
    EXPECT_TRUE(ac2.GetAccessibilitySaveDataOwnedBy(id3).CanWrite());
    EXPECT_TRUE(ac2.GetAccessibilitySaveDataOwnedBy(id4).CanRead());
    EXPECT_TRUE(ac2.GetAccessibilitySaveDataOwnedBy(id4).CanWrite());
    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id5).CanRead());
    EXPECT_FALSE(ac2.GetAccessibilitySaveDataOwnedBy(id5).CanWrite());
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)
// 不正なデータフォーマットが入力されたときにアボートすること
TEST_F(AccessControlDeathTest, Illegal)
{
    {
        static const unsigned char facV0[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
        };

        static const unsigned char facV1[] =
        {
            0x02, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
        };

        static const unsigned char facdV0[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        };

        static const unsigned char facdV1[] =
        {
            0x02, 0x00, 0x00, 0x00,

            0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facdInvalidNumContentOwnerId[] =
        {
            0x01, 0x01, 0x00, 0x00,

            0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facdInvalidNumSaveDataOwnerId[] =
        {
            0x01, 0x01, 0x01, 0x00,

            0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        // バージョン違い
        {
            AccessControlForTest ac(facV0, sizeof(facV0), facdV1, sizeof(facdV1));
            nn::Bit64 flag = 0x0000000000000000ULL;
            EXPECT_EQ(ac.GetRawFlagBits(), flag);
        }
        {
             // AccessControlForTest ac(facV1, sizeof(facV1), facdV1, sizeof(facdV1));
             // nn::Bit64 flag = 0x0000000000000000ULL;
             // EXPECT_EQ(ac.GetRawFlagBits(), flag);
        }

        // ACI ヘッダが不正
        EXPECT_DEATH(AccessControlForTest ac(facV0, sizeof(facV0) - 1, facdV0, sizeof(facdV0)), "");

        // ACID ヘッダが不正
        EXPECT_DEATH(AccessControlForTest ac(facV0, sizeof(facV0), facdV0, sizeof(facdV0) - 1), "");

        // ACID ContentOwnerId の数が不正
        EXPECT_DEATH(AccessControlForTest ac(facV0, sizeof(facV0), facdInvalidNumContentOwnerId, sizeof(facdInvalidNumContentOwnerId)), "");

        // ACID SaveDataOwnerId の数が不正
        EXPECT_DEATH(AccessControlForTest ac(facV0, sizeof(facV0), facdInvalidNumSaveDataOwnerId, sizeof(facdInvalidNumSaveDataOwnerId)), "");

        // OK
        AccessControlForTest ac(facV0, sizeof(facV0), facdV0, sizeof(facdV0));
    }

    {
        static const unsigned char facInvalidNumContentOwnerId[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x14, 0x00, 0x00, 0x00,
            0x30, 0x00, 0x00, 0x00,
            0x18, 0x00, 0x00, 0x00,

            0x03, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

            0x02, 0x00, 0x00, 0x00,
            0x03, 0x01, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facInvalidNumSaveDataOwnerId[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x14, 0x00, 0x00, 0x00,
            0x30, 0x00, 0x00, 0x00,
            0x18, 0x00, 0x00, 0x00,

            0x02, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

            0x03, 0x00, 0x00, 0x00,
            0x03, 0x01, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facInvalidOffset[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x14, 0x00, 0x00, 0x00,
            0x31, 0x00, 0x00, 0x00,
            0x18, 0x00, 0x00, 0x00,

            0x02, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

            0x02, 0x00, 0x00, 0x00,
            0x03, 0x01, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facInvalidSize[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0xA5, 0xA5, 0xA5, 0xA5, 0x5A, 0x5A, 0x5A, 0x5A,

            0x1C, 0x00, 0x00, 0x00,
            0x14, 0x00, 0x00, 0x00,
            0x30, 0x00, 0x00, 0x00,
            0x19, 0x00, 0x00, 0x00,

            0x02, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,

            0x02, 0x00, 0x00, 0x00,
            0x03, 0x01, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        static const unsigned char facd[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A, 0x5A,

            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
            0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80,
        };

        // ContentOwnerId の数が不正
        EXPECT_DEATH(AccessControlForTest ac(facInvalidNumContentOwnerId, sizeof(facInvalidNumContentOwnerId), facd, sizeof(facd)), "");

        // SaveDataOwnerId の数が不正
        EXPECT_DEATH(AccessControlForTest ac(facInvalidNumContentOwnerId, sizeof(facInvalidNumContentOwnerId), facd, sizeof(facd)), "");

        // 終端オフセットが不正
        EXPECT_DEATH(AccessControlForTest ac(facInvalidOffset, sizeof(facInvalidOffset), facd, sizeof(facd)), "");

        // 終端サイズが不正
        EXPECT_DEATH(AccessControlForTest ac(facInvalidSize, sizeof(facInvalidSize), facd, sizeof(facd)), "");
    }
} // NOLINT(impl/function_size)
#endif

// SetDebugFlagEnabled で Debug Flag、FullPermission が落とされること
TEST_F(AccessControlTestBase, DisableDebug)
{
    static const unsigned char fac[] =
    {
        0x01, 0x00, 0x00, 0x00,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xC0,

        0x1C, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
        0x1C, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00,
    };

    static const unsigned char facd[] =
    {
        0x01, 0x00, 0x00, 0x00,

        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    };

    {
        SetDebugFlagEnabled(false);
        AccessControlForTest ac(fac, sizeof(fac), facd, sizeof(facd));
        nn::Bit64 flag = 0x0000000000000000ULL;
        EXPECT_EQ(ac.GetRawFlagBits(), flag);
        EXPECT_FALSE(ac.CanCall(AccessControl::OperationType::FillBis));
    }

    {
        SetDebugFlagEnabled(true);
        AccessControlForTest ac(fac, sizeof(fac), facd, sizeof(facd));
        nn::Bit64 flag = 0xC000000000000000ULL;
        EXPECT_EQ(ac.GetRawFlagBits(), flag);
        EXPECT_TRUE(ac.CanCall(AccessControl::OperationType::FillBis));
    }
}

// 権限フラグが正しく管理されていること
TEST_F(AccessControlTestBase, Callability)
{
    static const unsigned char facd[] =
    {
        0x01, 0x00, 0x00, 0x00,

        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,

        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    };

    // アプリケーション権限でセーブデータを拡張することはできない
    {
        static const unsigned char fac[] =
        {
            0x01, 0x00, 0x00, 0x00,

            0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
            0x1C, 0x00, 0x00, 0x00,
            0x00, 0x00, 0x00, 0x00,
        };

        AccessControl ac(fac, sizeof(fac), facd, sizeof(facd));

        EXPECT_FALSE(ac.CanCall(nn::fssrv::detail::AccessControl::OperationType::ExtendSaveData));
        EXPECT_FALSE(ac.CanCall(nn::fssrv::detail::AccessControl::OperationType::ExtendSystemSaveData));
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    InitializeMyHeap();
    nn::fs::SetAllocator(MyAllocate, MyDeallocate);
    SetDebugFlagEnabled(true);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
