﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn;
using namespace nn::fs;
using namespace nnt::fs::util;

TEST(GameCard, WriteSampleData)
{
    const int KeyAreaSize = 0x8 * 512;
    const int CardHeaderSize = 0x78 * 512; // header + cert + reserved
    const size_t BufferSize = 8 * 1024 * 1024; // 4MB の倍数が望ましい
    const size_t SecureAreaStartAddr = 256 * 1024 * 1024;

    const int64_t CardSize = 1ULL * 1024 * 1024 * 1024;

    {
        GameCardHandle handle;
        NNT_ASSERT_RESULT_SUCCESS(GetGameCardHandle(&handle));
        std::unique_ptr<nn::fs::IStorage> storage;
        NNT_ASSERT_RESULT_SUCCESS(OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::RootWriteOnly));

        std::unique_ptr<char[]> buffer(new char[BufferSize]);


        NNT_ASSERT_RESULT_SUCCESS(EraseGameCard(GameCardSize::Size2GB, SecureAreaStartAddr));

        {
            size_t mountRomCacheBufferSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&mountRomCacheBufferSize));
            auto mountRomCacheBuffer = AllocateBuffer(mountRomCacheBufferSize);
            NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", mountRomCacheBuffer.get(), mountRomCacheBufferSize));

            size_t readSize;
            FileHandle handle;
            NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, "rom:/KeyArea.bin", OpenMode_Read));
            NNT_ASSERT_RESULT_SUCCESS(ReadFile(&readSize, handle, 0, buffer.get(), BufferSize));
            NNT_ASSERT_RESULT_SUCCESS(storage->Write(0, buffer.get(), readSize));

            CloseFile(handle);
            Unmount("rom");
        }

        int64_t restSize = CardSize;
        int64_t base = KeyAreaSize + CardHeaderSize;
        int64_t offset = 0;
        auto totalStartTick = nn::os::GetSystemTick();
        while( restSize > 0 )
        {
            auto startTick = nn::os::GetSystemTick();
            FillBufferWith32BitCount(buffer.get(), BufferSize, offset);
            auto writeSize = static_cast<size_t>( std::min<int64_t>(
                restSize,
                BufferSize - ((base + offset) % BufferSize)) // 書き込みのアラインをバッファサイズに合わせる
            );
            NNT_ASSERT_RESULT_SUCCESS(storage->Write(base + offset, buffer.get(), BufferSize));
            auto endTick = nn::os::GetSystemTick();

            NN_LOG("offset: %lld. rest size: %lld. (%lld ms, %.4lf MB/s)\n",
                offset,
                restSize,
                nn::os::ConvertToTimeSpan(endTick - startTick).GetMilliSeconds(),
                static_cast<double>(writeSize) / 1024 / 1024
                / (static_cast<double>(nn::os::ConvertToTimeSpan(endTick - startTick).GetMicroSeconds()) / 1000 / 1000)
                );

            offset   += writeSize;
            restSize -= writeSize;
        }

        auto totalEndTick = nn::os::GetSystemTick();
        NN_LOG("total: %lld ms, %.4lf MB/s.\n",
            nn::os::ConvertToTimeSpan(totalEndTick - totalStartTick).GetMilliSeconds(),
            static_cast<double>(CardSize) / 1024 / 1024
            / (static_cast<double>(nn::os::ConvertToTimeSpan(totalEndTick - totalStartTick).GetMicroSeconds()) / 1000 / 1000)
            );

    }
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nnt::fs::util::ResetAllocateCount();

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
