﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/fs.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_DeviceSimulation.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_IStorage.h>

#include <nn/os.h>

using namespace nn;
using namespace nn::fs;

namespace {

    class Time
    {
    public:
        Time() : m_StartTime(0), m_EndTime(0) {}
        ~Time() {}

        void Start()
        {
            m_StartTime = nn::os::GetSystemTick();
        }

        int64_t GetElapsedTimeMsec()
        {
            m_EndTime = nn::os::GetSystemTick();
            return (m_EndTime - m_StartTime).ToTimeSpan().GetMilliSeconds();
        }

        int64_t GetElapsedTimeSec()
        {
            return GetElapsedTimeMsec() / 1000;
        }

    private:
        nn::os::Tick m_StartTime;
        nn::os::Tick m_EndTime;
    };

    // 挿入されたゲームカードの normal area 先頭 512 Bytes を読み込む
    Result ReadNormalArea()
    {
        NN_LOG("Reading normal area.\n");

        const int NormalAreaOffset = 0x78 * 512; // card header, cert area, reserved area を読み飛ばす
        const size_t BufferSize = 512;

        GameCardHandle handle;
        Result result;

        // ゲームカードのハンドルを取得
        result = GetGameCardHandle(&handle);
        if (result.IsFailure())
        {
            NN_LOG("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        // normal area をオープン
        std::unique_ptr<nn::fs::IStorage> storage;
        result = OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::NormalReadOnly);
        if (result.IsFailure())
        {
            NN_LOG("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        // normal area の先頭 512 バイトを読み込み
        // アクセスのオフセット・サイズは 512 バイト単位である必要がある
        char buffer[BufferSize];
        memset(buffer, 0xAB, BufferSize);
        result = (storage->Read(NormalAreaOffset, buffer, BufferSize));
        if (result.IsFailure())
        {
            NN_LOG("failed to read normal area. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        NN_LOG("# Succeeded normal read\n");

        // CUP バージョンを取得 & 表示

        nn::fs::GameCardUpdatePartitionInfo info;
        nn::fs::GetGameCardUpdatePartitionInfo(&info, handle);
        NN_LOG("CupVersion = %x\n", info.version);
        NN_LOG("CupId      = %x\n", info.id);
        NN_RESULT_SUCCESS;
    }

    // 挿入されたゲームカードの secure area 先頭 512 bytes を読み込む
    Result ReadSecureArea()
    {
        NN_LOG("Reading secure area.\n");

        const size_t BufferSize = 512;

        GameCardHandle handle;
        Result result;

        // ゲームカードのハンドルを取得
        result = GetGameCardHandle(&handle);
        if (result.IsFailure())
        {
            NN_LOG("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        // secure area をオープン
        std::unique_ptr<nn::fs::IStorage> storage;
        result = OpenGameCardPartition(&storage, handle, GameCardPartitionRaw::SecureReadOnly);
        if (result.IsFailure())
        {
            NN_LOG("failed to open handle. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        // secure area の先頭 512 バイトを読み込み
        // アクセスのオフセット・サイズは 512 バイト単位である必要がある
        char buffer[BufferSize];
        memset(buffer, 0xAB, BufferSize);
        result = (storage->Read(0, buffer, BufferSize));
        if (result.IsFailure())
        {
            NN_LOG("failed to read secure area. (%d:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }

        NN_LOG("# Succeeded secure read\n");
        NN_RESULT_SUCCESS;
    }

    void CheckTimeout(int64_t elapsedTime, int64_t timeout, bool shouldOccur)
    {
        NN_LOG("# (elapsed time: %d msec)\n", elapsedTime);
        if (shouldOccur)
        {
            NN_ABORT_UNLESS(timeout <= elapsedTime, "timeout must happen\n");
            NN_LOG("# (intended timeout)\n");
        }
        else
        {
            NN_ABORT_UNLESS(elapsedTime < timeout, "timeout must not happen\n");
            NN_LOG("# (no timeout)\n");
        }
    }
    void CheckTimeoutOccurred(int64_t elapsedTime, int64_t timeout)
    {
        CheckTimeout(elapsedTime, timeout, true);
    }
    void CheckTimeoutNotOccurred(int64_t elapsedTime, int64_t timeout)
    {
        CheckTimeout(elapsedTime, timeout, false);
    }


    void WaitGameCardInsertionEvent(nn::os::SystemEventType* deviceDetectionEvent, bool isOnce)
    {
        while (isOnce || !IsGameCardInserted())
        {
            NN_LOG("# Waiting for card insertion...\n");
            nn::os::WaitSystemEvent(deviceDetectionEvent);
            nn::os::ClearSystemEvent(deviceDetectionEvent);
            if (isOnce)
            {
                break;
            }
        }
    }
    void WaitGameCardInsertionEvent(nn::os::SystemEventType* deviceDetectionEvent)
    {
        WaitGameCardInsertionEvent(deviceDetectionEvent, false);
    }

    void WaitGameCardRemovalEvent(nn::os::SystemEventType* deviceDetectionEvent, bool isOnce)
    {
        while (isOnce || IsGameCardInserted())
        {
            NN_LOG("# Waiting for card removal...\n");
            nn::os::WaitSystemEvent(deviceDetectionEvent);
            nn::os::ClearSystemEvent(deviceDetectionEvent);
            if (isOnce)
            {
                break;
            }
        }
    }
    void WaitGameCardRemovalEvent(nn::os::SystemEventType* deviceDetectionEvent)
    {
        WaitGameCardRemovalEvent(deviceDetectionEvent, false);
    }

    void WaitGameCardDetectionEvent(nn::os::SystemEventType* deviceDetectionEvent, bool shouldOccur)
    {
        // イベント通知を待つ
        bool isEventOccurred = nn::os::TimedWaitSystemEvent(deviceDetectionEvent, nn::TimeSpan::FromMilliSeconds(1000));
        NN_ABORT_UNLESS(isEventOccurred == shouldOccur, "Device detection event must %s\n", shouldOccur ? "happen" : "not happen");
        nn::os::ClearSystemEvent(deviceDetectionEvent);
        NN_LOG("# %s device detection event\n", shouldOccur ? "Successfully caught the " : "Expected timeout for ");
    }
}

#define NN_DETAIL_FS_CHECK_RESULT(r1, r2) \
    { \
        const auto& _nn_result_do_temporary1((r1)); \
        const auto& _nn_result_do_temporary2((r2)); \
        NN_ABORT_UNLESS(_nn_result_do_temporary1.GetDescription() == _nn_result_do_temporary2.GetDescription()); \
    }


extern "C" void nnMain()
{
    Result result;
    Time timer;

    NN_LOG("# Device event simulation test\n");

    // 連続で走らせることを考慮し、シミュレーションを明示的にクリア
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::NoSimulation, false));
    NN_ABORT_UNLESS_RESULT_SUCCESS(ClearGameCardSimulationEvent());
    fs::SimulatingDeviceTargetOperation target = SimulatingDeviceTargetOperation_Read;

    // 挿抜イベント通知をオープン
    std::unique_ptr<nn::fs::IEventNotifier> deviceDetectionEventNotifier;
    nn::os::SystemEventType deviceDetectionEvent;
    {
        result = OpenGameCardDetectionEventNotifier(&deviceDetectionEventNotifier);
        if (result.IsFailure())
        {
            NN_ABORT("failed to open notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }

        // システムイベントと紐づけ
        result = deviceDetectionEventNotifier->BindEvent(&deviceDetectionEvent, nn::os::EventClearMode_ManualClear);
        if (result.IsFailure())
        {
            NN_ABORT("failed to bind system event to notifier. (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
    }

    // 抜去を待つ
    WaitGameCardRemovalEvent(&deviceDetectionEvent);

    // 挿入を待つ
    WaitGameCardInsertionEvent(&deviceDetectionEvent);

    // 読み込み
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadNormalArea() );

    // 挿抜イベントを起こす
    NN_ABORT_UNLESS_RESULT_SUCCESS( SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::NoSimulation, true) );
    NN_LOG("# Triggered a device detection event\n");

    // イベント通知を待つ
    WaitGameCardDetectionEvent(&deviceDetectionEvent, true);

    // 読み込み
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadNormalArea() );
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadSecureArea() );

    // タイムアウトエラーのシミュレートの登録
    timer.Start();
    NN_LOG("# *** Access Failure Simulation\n");
    NN_LOG("# Register access error simulation event\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS( SetGameCardSimulationEvent(target, nn::fs::SimulatingDeviceAccessFailureEventType::AccessTimeoutFailure, true) );
    NN_ABORT_UNLESS(ReadSecureArea().IsFailure());
    CheckTimeoutOccurred(timer.GetElapsedTimeMsec(), 900);

    timer.Start();
    NN_LOG("# Persistent simulation (timeout error, persistent)\n");
    NN_ABORT_UNLESS(ReadSecureArea().IsFailure());
    CheckTimeoutOccurred(timer.GetElapsedTimeMsec(), 900);

    // 継続シミュレーションのクリア
    timer.Start();
    NN_ABORT_UNLESS_RESULT_SUCCESS( ClearGameCardSimulationEvent() );
    NN_LOG("# Clear simulation event (timeout error)\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS( ReadSecureArea() );
    CheckTimeoutNotOccurred(timer.GetElapsedTimeMsec(), 100);

    // 一度限りのシミュレーション（fs レイヤのリトライで隠蔽されない）
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetGameCardSimulationEvent(target, nn::fs::SimulatingDeviceAccessFailureEventType::AccessFailure, false));
    NN_LOG("# One-time AccessFailure simulation\n");
    ReadSecureArea();   // （今後挙動が変わるかもしれないので IsFailure のチェックはしない）

    // 一度限りのシミュレーション（fs レイヤのリトライで隠蔽される）
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetGameCardSimulationEvent(target, nn::fs::SimulatingDeviceAccessFailureEventType::DataCorruption, false));
    NN_LOG("# One-time DataCorruption simulation (error won't appear because of retry on fs layer)\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadSecureArea());

    // データ壊れのエラー（Result のみでデータは正しい）
    NN_ABORT_UNLESS_RESULT_SUCCESS( SetGameCardSimulationEvent(target, nn::fs::SimulatingDeviceAccessFailureEventType::DataCorruption, true) );
    NN_LOG("# Data corruption simulation\n");
    NN_DETAIL_FS_CHECK_RESULT(ReadSecureArea(), fs::ResultDataCorrupted());
    NN_ABORT_UNLESS_RESULT_SUCCESS( ClearGameCardSimulationEvent() );

    // Result の指定
    NN_ABORT_UNLESS_RESULT_SUCCESS( SetGameCardSimulationEvent(target, fs::ResultUsableSpaceNotEnough(), true) );
    NN_LOG("# Specified result simulation\n");
    NN_DETAIL_FS_CHECK_RESULT(ReadSecureArea(), fs::ResultUsableSpaceNotEnough());

    // 異なるオペレーションでは反応しない
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetGameCardSimulationEvent(SimulatingDeviceTargetOperation_Write, fs::ResultUsableSpaceNotEnough(), true));
    NN_LOG("# Not target operation\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadSecureArea());

    // オペレーションに0 のフラグを渡しても反応しない
    NN_ABORT_UNLESS_RESULT_SUCCESS(SetGameCardSimulationEvent(static_cast<SimulatingDeviceTargetOperation>(0), fs::ResultUsableSpaceNotEnough(), true));
    NN_LOG("# Specified all zero operation flag\n");
    NN_ABORT_UNLESS_RESULT_SUCCESS(ReadSecureArea());

    // 抜去しっぱなしのテスト
    NN_LOG("# *** Device Detection Simulation\n");
    nn::os::ClearSystemEvent(&deviceDetectionEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::DeviceRemoved, true));
    NN_LOG("# Set to removed\n");
    // 抜去状態になる
    NN_ABORT_UNLESS(!IsGameCardInserted());
    // イベントも起きる
    WaitGameCardDetectionEvent(&deviceDetectionEvent, true);
    NN_LOG("# Physically inserted but removed state with event\n");
    // 物理抜去しても抜去状態
    WaitGameCardRemovalEvent(&deviceDetectionEvent, true);
    NN_ABORT_UNLESS(!IsGameCardInserted());
    NN_LOG("# Physically removed and removed state with event\n");
    // 物理挿入しても抜去状態
    WaitGameCardInsertionEvent(&deviceDetectionEvent, true);
    NN_ABORT_UNLESS(!IsGameCardInserted());
    NN_LOG("# Physically inserted but removed state with event\n");

    // シミュレーション状態を解除するテスト１
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::NoSimulation, true));
    NN_LOG("# Reset with one shot event\n");
    NN_ABORT_UNLESS(IsGameCardInserted());
    WaitGameCardDetectionEvent(&deviceDetectionEvent, true);

    // 挿入しっぱなしのテスト
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::DeviceAttached, false));
    NN_LOG("# Set to inserted\n");
    // 挿入状態のまま
    NN_ABORT_UNLESS(IsGameCardInserted());
    // イベントは起きない
    WaitGameCardDetectionEvent(&deviceDetectionEvent, false);
    NN_LOG("# Physically inserted and inserted state without event\n");
    // 物理抜去しても挿入状態
    WaitGameCardRemovalEvent(&deviceDetectionEvent, true);
    NN_ABORT_UNLESS(IsGameCardInserted());
    NN_LOG("# Physically removed and inserted state with event\n");
    // 物理挿入しても挿入状態
    WaitGameCardInsertionEvent(&deviceDetectionEvent, true);
    NN_ABORT_UNLESS(IsGameCardInserted());
    NN_LOG("# Physically inserted but removed state with event\n");

    // シミュレーション状態を解除するテスト２
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::NoSimulation, false));
    NN_ABORT_UNLESS(IsGameCardInserted());
    WaitGameCardDetectionEvent(&deviceDetectionEvent, false);
    NN_LOG("# Reset without event\n");

    // 挿抜イベントが発生しないことのテスト
    nn::os::ClearSystemEvent(&deviceDetectionEvent);
    NN_ABORT_UNLESS_RESULT_SUCCESS(SimulateGameCardDetectionEvent(fs::SimulatingDeviceDetectionMode::NoSimulation, false));
    NN_LOG("# No event with no simulation\n");
    WaitGameCardDetectionEvent(&deviceDetectionEvent, false);

    // イベントを破棄
    NN_ABORT_UNLESS_RESULT_SUCCESS(ClearGameCardSimulationEvent());
    nn::os::DestroySystemEvent(&deviceDetectionEvent);

    NN_LOG("# *** End of the device event simulation test ***\n");
} // NOLINT(impl/function_size)
