﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testFs_Stress_StressTest.h"

namespace {

    char* CreateThreadUniqueFileName(const StressTestMountInfoBase* pTest, int threadIndex)
    {
        NN_ASSERT(pTest != nullptr);
        char* path = new char[nn::fs::EntryNameLengthMax];
        NN_ASSERT(path != nullptr);
        nn::util::SNPrintf(
            path,
            nn::fs::EntryNameLengthMax,
            "%s:/file%d",
            pTest->GetMountName(),
            threadIndex
        );
        return path;
    }

    char* CreateThreadUniqueFileName(const StressTestMountInfoBase* pTest, int threadIndex, int entryIndex)
    {
        NN_ASSERT(pTest != nullptr);
        char* path = new char[nn::fs::EntryNameLengthMax];
        NN_ASSERT(path != nullptr);
        nn::util::SNPrintf(
            path,
            nn::fs::EntryNameLengthMax,
            "%s:/file%d_%d",
            pTest->GetMountName(),
            threadIndex,
            entryIndex
        );
        return path;
    }

}

// 複数スレッドからそれぞれ異なるファイルをオープン、クローズします。
class TestOpenUniqueFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
        nn::util::SNPrintf(path.get(), nn::fs::EntryNameLengthMax, "%s:/file%d", pTest->GetMountName(), threadIndex);

        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        pTest->Fail();

        nn::fs::FileHandle file;
        NNT_EXPECT_RESULT_SUCCESS(
            nn::fs::OpenFile(&file, path.get(), nn::fs::OpenMode_Read)
        );
        nn::fs::CloseFile(file);

        pTest->Succeed();
    }
};

namespace {

    auto runOpenUniqueFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestOpenUniqueFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                std::unique_ptr<char[]> path(
                    CreateThreadUniqueFileName(
                        list[indexTable[threadIndex]],
                        threadIndex
                    )
                );
                (void)nn::fs::DeleteFile(path.get());
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 100));
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::unique_ptr<char[]> path(CreateThreadUniqueFileName(list[indexTable[threadIndex]], threadIndex));
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}
#endif

TEST(HostFsStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}

TEST(RomFsStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("rom");
    StressTestMountRomFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}

TEST(SaveDataFsStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}

TEST(SdCardStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenUniqueFile(list);
}

TEST(MultiFsStressTest, OpenUniqueFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runOpenUniqueFile(list);
}

// 単一のファイルを複数スレッドからオープン、クローズします。
class TestOpenSingleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int) NN_NOEXCEPT NN_OVERRIDE
    {
        const nnt::fs::util::String fileName("/file0");
        std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
        nn::util::SNPrintf(path.get(), nn::fs::EntryNameLengthMax, "%s:/file0", pTest->GetMountName(), fileName.c_str());

        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        pTest->Fail();

        nn::fs::FileHandle file;
        for( auto loop = 0; loop < 100; ++loop )
        {
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(&file, path.get(), nn::fs::OpenMode_Read)
            );
            nn::fs::CloseFile(file);
        }
        pTest->Succeed();
    }
};

namespace {

    auto runOpenSingleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestOpenSingleFile test;

        const nnt::fs::util::String fileName("/file0");

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
                nn::util::SNPrintf(
                    path.get(),
                    nn::fs::EntryNameLengthMax,
                    "%s:%s",
                    list[indexTable[threadIndex]]->GetMountName(),
                    fileName.c_str()
                );
                nn::fs::FileHandle file;
                nn::Result result = nn::fs::OpenFile(&file, path.get(), nn::fs::OpenMode_Read);
                if( !result.IsSuccess() )
                {
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 100));
                }
                else
                {
                    nn::fs::CloseFile(file);
                }
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::unique_ptr<char[]> path(new char[nn::fs::EntryNameLengthMax]);
                    nn::util::SNPrintf(
                        path.get(),
                        nn::fs::EntryNameLengthMax,
                        "%s:%s",
                        list[indexTable[threadIndex]]->GetMountName(),
                        fileName.c_str()
                    );
                    nn::fs::FileHandle file;
                    nn::Result result = nn::fs::OpenFile(&file, path.get(), nn::fs::OpenMode_Read);
                    if( result.IsSuccess() )
                    {
                        nn::fs::CloseFile(file);
                        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                    }
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}
#endif

TEST(HostFsStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}

TEST(RomFsStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("rom");
    StressTestMountRomFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}

TEST(SaveDataFsStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}

TEST(SdCardStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenSingleFile(list);
}

TEST(MultiFsStressTest, OpenSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runOpenSingleFile(list);
}

// 複数スレッドから複数ファイルのオープン、クローズを行います。
class TestOpenMultipleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        pTest->Fail();

        int openedEntryCount = 0;
        nn::fs::FileHandle files[EntryCountMax];
        NN_UTIL_SCOPE_EXIT
        {
            for( auto entryIndex = 0; entryIndex < openedEntryCount; ++entryIndex )
            {
                nn::fs::CloseFile(files[entryIndex]);
            }
        };
        for( auto entryIndex = 0; entryIndex < EntryCountMax; ++entryIndex )
        {
            std::unique_ptr<char[]> path(
                CreateThreadUniqueFileName(
                    pTest->GetTarget(),
                    threadIndex,
                    entryIndex
                )
            );
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &files[entryIndex],
                    path.get(),
                    nn::fs::OpenMode_Read
                )
            );
            ++openedEntryCount;
        }

        pTest->Succeed();
    }

    static const int EntryCountMax = 20;
};

namespace {

    auto runOpenMultipleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestOpenMultipleFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                for( auto fileIndex = 0; fileIndex < TestOpenMultipleFile::EntryCountMax; ++fileIndex )
                {
                    std::unique_ptr<char[]> path(
                        CreateThreadUniqueFileName(
                            list[indexTable[threadIndex]],
                            threadIndex,
                            fileIndex
                        )
                    );
                    (void)nn::fs::DeleteFile(path.get());
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 100));
                }
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    for( auto fileIndex = 0; fileIndex < TestOpenMultipleFile::EntryCountMax; ++fileIndex )
                    {
                        std::unique_ptr<char[]> path(
                            CreateThreadUniqueFileName(
                                list[indexTable[threadIndex]],
                                threadIndex,
                                fileIndex
                            )
                        );
                        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                    }
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}
#endif

TEST(HostFsStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}

TEST(SaveDataFsStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}

TEST(SdCardStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}

TEST(RomFsStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("rom");
    StressTestMountRomFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runOpenMultipleFile(list);
}

TEST(MultiFsStressTest, OpenMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runOpenMultipleFile(list);
}

// 複数のスレッドにてスレッド毎のファイル作成、削除を行う
class TestCreateSingleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        // 優先度を初期化します
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        ASSERT_FALSE(pTest->IsReadOnly());

        // ファイルの作成
        for( auto loop = 0; loop < 100; ++loop )
        {
            std::unique_ptr<char[]> path(
                CreateThreadUniqueFileName(
                    pTest->GetTarget(),
                    threadIndex
                )
            );
            (void)nn::fs::DeleteFile(path.get());
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 100));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
        }
    }
};

namespace {

    auto runCreateSingleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestCreateSingleFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateSingleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateSingleFile(list);
}
#endif

TEST(HostFsStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountBis fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateSingleFile(list);
}

TEST(SaveDataFsStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateSingleFile(list);
}

TEST(SdCardStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateSingleFile(list);
}

TEST(MultiFsStressTest, CreateSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runCreateSingleFile(list);
}

// 複数のスレッドにてスレッド毎の複数ファイル作成、削除を行う
class TestCreateMultipleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        // 優先度を初期化します
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        ASSERT_FALSE(pTest->IsReadOnly());

        // ファイルの作成
        for( auto entryIndex = 0; entryIndex < EntryCountMax; ++entryIndex )
        {
            std::unique_ptr<char[]> path(
                CreateThreadUniqueFileName(
                    pTest->GetTarget(),
                    threadIndex,
                    entryIndex
                )
            );
            (void)nn::fs::DeleteFile(path.get());
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 100));
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
        }
    }

    static const int EntryCountMax = 100;
};

namespace {

    auto runCreateMultipleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestCreateMultipleFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateMultipleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateMultipleFile(list);
}
#endif

TEST(HostFsStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateMultipleFile(list);
}

TEST(SaveDataFsStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateMultipleFile(list);
}

TEST(SdCardStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateMultipleFile(list);
}

TEST(MultiFsStressTest, CreateMultipleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runCreateMultipleFile(list);
}

// 複数の優先度でパラメータで指定されたファイルへのリードアクセスを行う
class TestReadSingleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        // 優先度を初期化します
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        nn::fs::FileHandle fileHandle;
        std::unique_ptr<char[]> path(
            CreateThreadUniqueFileName(
                pTest->GetTarget(),
                threadIndex
            )
        );
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path.get(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        int64_t sizeFile;
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeFile, fileHandle));
        std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(sizeFile)]);
        for( auto loop = 0; loop < 100; ++loop )
        {
            // ファイル内のランダムな範囲に対してリードを行います。
            const auto size = static_cast<size_t>(std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt));
            const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - size - 1)(mt);
            size_t sizeRead;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadFile(&sizeRead, fileHandle, offset, &buffer[0], size));
        }
    }
};

namespace {

    auto runReadSingleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestReadSingleFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                std::unique_ptr<char[]> path(
                    CreateThreadUniqueFileName(
                        list[indexTable[threadIndex]],
                        threadIndex
                    )
                );
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 0x4000));
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::unique_ptr<char[]> path(
                        CreateThreadUniqueFileName(
                            list[indexTable[threadIndex]],
                            threadIndex
                        )
                    );
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("cache");
    StressTestMountCacheStorage fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}
#endif

TEST(HostFsStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("host");
    StressTestMountHostFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}

TEST(SaveDataFsStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("save");
    StressTestMountSaveData fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}

TEST(SdCardStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("sd");
    StressTestMountSdCard fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}

TEST(RomFsStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    const nnt::fs::util::String mountName("rom");
    StressTestMountRomFs fs;
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runReadSingleFile(list);
}

TEST(MultiFsStressTest, ReadSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountRomFs fsRom;
    {
        const nnt::fs::util::String mountName("rom");
        fsRom.SetMountName(mountName.c_str());
        fsRom.Mount();
        list.push_back(&fsRom);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsRom.Unmount();
    };

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runReadSingleFile(list);
}

// 複数の優先度でパラメータで指定されたファイルへのライトアクセスを行う
class TestWriteSingleFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        ASSERT_FALSE(pTest->IsReadOnly());

        // 優先度を初期化します
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        {
            nn::fs::FileHandle fileHandle;
            std::unique_ptr<char[]> path(
                CreateThreadUniqueFileName(
                    pTest->GetTarget(),
                    threadIndex
                )
            );
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &fileHandle,
                    path.get(),
                    nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            int64_t sizeFile;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeFile, fileHandle));
            std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(sizeFile)]);
            for( auto loop = 0; loop < 30; ++loop )
            {
                // ファイル内のランダムな範囲に対してライトを行います。
                const auto size = static_cast<size_t>(std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt));
                const auto offset = std::uniform_int_distribution<int64_t>(0, sizeFile - size - 1)(mt);
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, offset, &buffer[0], size, nn::fs::WriteOption::MakeValue(0)));
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
            }
        }
    }
};

namespace {

    auto runWriteSingleFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestWriteSingleFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            if( !list[indexTable[threadIndex]]->IsReadOnly() )
            {
                std::unique_ptr<char[]> path(
                    CreateThreadUniqueFileName(
                        list[indexTable[threadIndex]],
                        threadIndex
                    )
                );
                (void)nn::fs::DeleteFile(path.get());
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 0x4000));
            }
        }
        NN_UTIL_SCOPE_EXIT
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::unique_ptr<char[]> path(
                        CreateThreadUniqueFileName(
                            list[indexTable[threadIndex]],
                            threadIndex
                        )
                    );
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
                }
            }
        };

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();
    };

}

TEST(BisStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runWriteSingleFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountCacheStorage fs;
    fs.SetMountName("cache");
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runWriteSingleFile(list);
}
#endif

TEST(HostFsStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountHostFs fs;
    fs.SetMountName("host");
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runWriteSingleFile(list);
}

TEST(SaveDataFsStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountSaveData fs;
    fs.SetMountName("save");
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runWriteSingleFile(list);
}

TEST(SdCardStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountSdCard fs;
    fs.SetMountName("sd");
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runWriteSingleFile(list);
}

TEST(MultiFsStressTest, WriteSingleFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase *> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runWriteSingleFile(list);
}

// スレッドごとにファイルの作成、ランダムライトアクセス、削除を繰り返す
class TestCreateAndReadWriteFile : public TestRunner
{
public:
    virtual void RunTest(TestContext* pTest, int threadIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        // 優先度を初期化します
        nn::fs::SetPriorityRawOnCurrentThread(pTest->GetPriority());

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());

        for( auto loopFile = 0; loopFile < 10; ++loopFile )
        {
            std::unique_ptr<char[]> path(
                CreateThreadUniqueFileName(
                    pTest->GetTarget(),
                    threadIndex,
                    loopFile
                )
            );
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(path.get(), 0x4000));
            NN_UTIL_SCOPE_EXIT
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteFile(path.get()));
            };

            nn::fs::FileHandle fileHandle;
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &fileHandle,
                    path.get(),
                    nn::fs::OpenMode_Read | nn::fs::OpenMode_Write
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            int64_t sizeFile;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::GetFileSize(&sizeFile, fileHandle));
            std::unique_ptr<char[]> buffer(new char[static_cast<size_t>(sizeFile)]);
            for( auto loopOperation = 0; loopOperation < 10; ++loopOperation )
            {
                // ファイル内のランダムな範囲に対してリードライトを行います。
                int operation = std::uniform_int_distribution<int>(0, 2 - 1)(mt);
                switch( operation )
                {
                case 0:
                    {
                        int64_t sizeRead = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                        int64_t offsetRead = std::uniform_int_distribution<int64_t>(0, sizeFile - sizeRead - 1)(mt);
                        size_t sizeResult;
                        NNT_EXPECT_RESULT_SUCCESS(
                            nn::fs::ReadFile(
                                &sizeResult,
                                fileHandle,
                                offsetRead,
                                &buffer[0],
                                static_cast<size_t>(sizeRead)
                            )
                        );
                    }
                    break;

                case 1:
                    {
                        int64_t sizeWrite = std::uniform_int_distribution<int64_t>(0, sizeFile - 1)(mt);
                        int64_t offsetWrite = std::uniform_int_distribution<int64_t>(0, sizeFile - sizeWrite - 1)(mt);
                        NNT_EXPECT_RESULT_SUCCESS(
                            nn::fs::WriteFile(
                                fileHandle,
                                offsetWrite,
                                &buffer[0],
                                static_cast<size_t>(sizeWrite),
                                nn::fs::WriteOption::MakeValue(0)
                            )
                        );
                        NNT_EXPECT_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));
                    }
                    break;

                default:
                    NN_UNEXPECTED_DEFAULT;
                }
            }
        }
    }
};

namespace {

    auto runCreateAndReadWriteFile = [](const nnt::fs::util::Vector<StressTestMountInfoBase*>& list) NN_NOEXCEPT
    {
        ASSERT_TRUE(list.size() > 0);

        TestCreateAndReadWriteFile test;

        const auto ThreadCount = StressTestParam::ThreadCountMax;
        int indexTable[ThreadCount];

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            indexTable[threadIndex] = std::uniform_int_distribution<int>(0, static_cast<int>(list.size()) - 1)(mt);
        }

        for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
        {
            auto pContext = new TestContext(list[indexTable[threadIndex]]);
            pContext->SetPriorityRandom();
            test.AddContext(pContext);
        }
        test.RunThreadTest();

        if( test.IsSuccess() )
        {
            for( auto threadIndex = 0; threadIndex < ThreadCount; ++threadIndex )
            {
                if( !list[indexTable[threadIndex]]->IsReadOnly() )
                {
                    std::lock_guard<nn::os::Mutex> lock(list[indexTable[threadIndex]]->GetMutex());
                    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CommitSaveData(list[indexTable[threadIndex]]->GetMountName()));
                }
            }
        }
    };

}

TEST(BisStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountBis fs;
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndReadWriteFile(list);
}

#if !defined(NN_BUILD_CONFIG_OS_WIN)
TEST(CacheStorageStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountCacheStorage fs;
    const nnt::fs::util::String mountName("cache");
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndReadWriteFile(list);
}
#endif

TEST(HostFsStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountHostFs fs;
    const nnt::fs::util::String mountName("host");
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndReadWriteFile(list);
}

TEST(SaveDataFsStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountSaveData fs;
    const nnt::fs::util::String mountName("save");
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndReadWriteFile(list);
}

TEST(SdCardStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;
    StressTestMountSdCard fs;
    const nnt::fs::util::String mountName("sd");
    fs.SetMountName(mountName.c_str());
    fs.Mount();
    NN_UTIL_SCOPE_EXIT
    {
        fs.Unmount();
    };
    list.push_back(&fs);
    runCreateAndReadWriteFile(list);
}

TEST(MultiFsStressTest, CreateAndReadWriteFile)
{
    nnt::fs::util::Vector<StressTestMountInfoBase*> list;

    StressTestMountBis fsBis;
    {
        fsBis.Mount();
        list.push_back(&fsBis);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsBis.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountCacheStorage fsCache;
    {
        const nnt::fs::util::String mountName("cache");
        fsCache.SetMountName(mountName.c_str());
        fsCache.Mount();
        list.push_back(&fsCache);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsCache.Unmount();
    };
#endif

    StressTestMountHostFs fsHost;
    {
        const nnt::fs::util::String mountName("host");
        fsHost.SetMountName(mountName.c_str());
        fsHost.Mount();
        list.push_back(&fsHost);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsHost.Unmount();
    };

#if !defined(NN_BUILD_CONFIG_OS_WIN)
    StressTestMountOtherSaveData fsOtherSave;
    {
        const nnt::fs::util::String mountName("other");
        fsOtherSave.SetMountName(mountName.c_str());
        fsOtherSave.Mount();
        list.push_back(&fsOtherSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsOtherSave.Unmount();
    };
#endif

    StressTestMountSaveData fsSave;
    {
        const nnt::fs::util::String mountName("save");
        fsSave.SetMountName(mountName.c_str());
        fsSave.Mount();
        list.push_back(&fsSave);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSave.Unmount();
    };

    StressTestMountSdCard fsSdCard;
    {
        const nnt::fs::util::String mountName("sd");
        fsSdCard.SetMountName(mountName.c_str());
        fsSdCard.Mount();
        list.push_back(&fsSdCard);
    }
    NN_UTIL_SCOPE_EXIT
    {
        fsSdCard.Unmount();
    };

    runCreateAndReadWriteFile(list);
}

