﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_ENABLE_HTC

#include <random>
#include <algorithm>
#include <nn/init.h>
#include <nn/htc.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/time/time_Api.h>

#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util_allocator.h>
#include <nnt/fsUtil/testFs_util_function.h>

#define NNT_FS_STRESS_MULTIREADWRITELARGE_LOG_ENABLE 0 // NOLINT(preprocessor/const)

#if NNT_FS_STRESS_MULTIREADWRITELARGE_LOG_ENABLE
    #define NNT_FS_STRESS_MULTIREADWRITELARGE_LOG(...) NN_LOG(__VA_ARGS__)
#else
    #define NNT_FS_STRESS_MULTIREADWRITELARGE_LOG(...)
#endif // NNT_FS_STRESS_MULTIREADWRITELARGE_LOG_ENABLE

#if !defined(NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID)
    #define NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID 0x005000C10000000 // NOLINT(preprocessor/const)
#endif // !defined(NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID)

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000002
    #define NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION
#else
    #define NNT_FS_STRESS_MULTIREADWRITELARGE_IS_SYSTEM_PROCESS
#endif

namespace nnt { namespace fs {

namespace {

#if defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
const auto ThreadCountMax = 12;
#else
const auto ThreadCountMax = 6;
#endif

class Deleter
{
public:
    explicit Deleter(size_t size) NN_NOEXCEPT
        : m_Size(size)
    {
    }
    void operator()(void* p) const NN_NOEXCEPT
    {
        if (p != nullptr)
        {
            nnt::fs::util::Deallocate(p, m_Size);
        }
    }

private:
    size_t m_Size;
};

template<typename Array>
std::unique_ptr<Array, Deleter> MakeUnique(size_t size) NN_NOEXCEPT;

class Measurement
{
public:
    Measurement() NN_NOEXCEPT
        : m_SectionAccessCount(0),
          m_SectionTime(0),
          m_SectionSize(0),
          m_SectionAverageSpeedBytesPerSecond(0),
          m_SectionAverageSize(0),
          m_NeedReset(false)
    {
    }

    void Add(nn::TimeSpan elapsed, int size) NN_NOEXCEPT
    {
        if (m_NeedReset)
        {
            m_SectionAccessCount = 0;
            m_SectionSize = 0;
            m_SectionTime = 0;
        }

        ++m_SectionAccessCount;
        m_SectionSize += size;
        m_SectionTime += elapsed;
        m_SectionAverageSize = (m_SectionSize / m_SectionAccessCount);
        if (m_SectionTime.GetMilliSeconds() > 0)
        {
            m_SectionAverageSpeedBytesPerSecond
                = (m_SectionSize * 1000 / m_SectionTime.GetMilliSeconds());
        }
        else
        {
            m_SectionAverageSpeedBytesPerSecond = 0;
        }
    }

    void GetSectionAverageValueAndReset(int64_t* outSpeedBytesPerSecond, int64_t* outSize) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(outSize);
        NN_ABORT_UNLESS_NOT_NULL(outSpeedBytesPerSecond);

        *outSpeedBytesPerSecond = m_SectionAverageSpeedBytesPerSecond;
        *outSize = m_SectionAverageSize;
        m_NeedReset = true;
    }

    void Reset() NN_NOEXCEPT
    {
        m_SectionAccessCount = 0;
        m_SectionTime = 0;
        m_SectionSize = 0;
        m_SectionAverageSpeedBytesPerSecond = 0;
        m_SectionAverageSize = 0;
        m_NeedReset = false;
    }

private:
    int m_SectionAccessCount;
    nn::TimeSpan m_SectionTime;
    int64_t m_SectionSize;
    std::atomic_int64_t m_SectionAverageSpeedBytesPerSecond;
    std::atomic_int64_t m_SectionAverageSize;
    std::atomic_bool m_NeedReset;
};

class MultiReadWriteExecuter
{
public:
    MultiReadWriteExecuter() NN_NOEXCEPT
        : m_pEvent(nullptr),
          m_IdealCore(0),
          m_OffsetMin(0),
          m_OffsetMax(0),
          m_SizeMin(0),
          m_SizeMax(0),
          m_IsContinuous(false),
          m_NeedDestroyThread(false),
          m_NeedUnmount(false),
          m_NeedCloseFile(false),
          m_NeedDeleteFile(false),
          m_NeedVerifyUnmount(false),
          m_NeedVerifyCloseFile(false),
          m_NeedCommit(false),
          m_LastResult(nn::ResultSuccess()),
          m_CacheBuffer(nullptr, Deleter(0)),
          m_pLabel(nullptr)
    {
    }

    nn::Result InitializeAsRomReader(int id, nn::os::Event* pEvent, const char* pVerifyPath) NN_NOEXCEPT;

    nn::Result InitializeAsSdForDebugReaderWriter(int id, nn::os::Event* pEvent) NN_NOEXCEPT;

    nn::Result InitializeAsMmcReaderWriter(int id, nn::os::Event* pEvent) NN_NOEXCEPT;

    nn::Result InitializeAsHostFsReaderWriter(int id, nn::os::Event* pEvent, const char* pHostPath) NN_NOEXCEPT;

    nn::Result InitializeAsContentStorageReaderWriter(int id, nn::os::Event* pEvent, nn::fs::ContentStorageId contentStorageId) NN_NOEXCEPT;

    nn::Result InitializeAsImageDirectoryReaderWriter(int id, nn::os::Event* pEvent, nn::fs::ImageDirectoryId imageDirectoryId) NN_NOEXCEPT;

    nn::Result InitializeAsSystemSaveDataReaderWriter(int id, nn::os::Event* pEvent, nn::fs::SaveDataSpaceId systemSaveDataid, int64_t systemSaveDataId) NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    void Start() NN_NOEXCEPT;

    void Stop() NN_NOEXCEPT;

    const nn::Result& GetLastError() const NN_NOEXCEPT
    {
        return m_LastResult;
    }

    Measurement& GetReadMeasurement() NN_NOEXCEPT
    {
        return m_ReadMeasurement;
    }

    Measurement& GetWriteMeasurement() NN_NOEXCEPT
    {
        return m_WriteMeasurement;
    }

    int GetIdealCore() const NN_NOEXCEPT
    {
        return m_IdealCore;
    }

    const char* GetLabel() const NN_NOEXCEPT
    {
        return m_pLabel;
    }

private:
#ifdef NN_BUILD_CONFIG_OS_WIN
    static const int ThreadStackSize = 32 * 1024;
#else
    static const int ThreadStackSize = 16 * 1024;
#endif
    static const int MountNameLength = 64;

private:
    nn::Result Initialize(int id, nn::os::Event* pEvent, int64_t offsetMin, int64_t offsetMax, int sizeMin, int sizeMax) NN_NOEXCEPT;

    nn::Result InitializeDefaultPath() NN_NOEXCEPT;

    static nn::Result EnsureFile(const char* path, int64_t sizeFile) NN_NOEXCEPT;

    static void Execute(void* pExecuter) NN_NOEXCEPT
    {
        auto* pThis = reinterpret_cast<MultiReadWriteExecuter*>(pExecuter);
        pThis->Execute();
    }

    void Execute() NN_NOEXCEPT;

private:
    nn::os::Event* m_pEvent;
    int m_IdealCore;
    nn::os::ThreadType m_Thread;
    int m_Id;
    int64_t m_OffsetMin;
    int64_t m_OffsetMax;
    int m_SizeMin;
    int m_SizeMax;
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[ThreadStackSize];
    char m_MountName[MountNameLength];
    char m_VerifyMountName[MountNameLength];
    char m_PathBuffer[nn::fs::EntryNameLengthMax + 1];
    nn::fs::FileHandle m_FileHandle;
    nn::fs::FileHandle m_VerifyFileHandle;
    std::atomic_bool m_IsContinuous;
    bool m_NeedDestroyThread;
    bool m_NeedUnmount;
    bool m_NeedCloseFile;
    bool m_NeedDeleteFile;
    bool m_NeedVerifyUnmount;
    bool m_NeedVerifyCloseFile;
    bool m_NeedCommit;
    nn::Result m_LastResult;
    decltype(nnt::fs::MakeUnique<char[]>(0)) m_CacheBuffer;
    Measurement m_ReadMeasurement;
    Measurement m_WriteMeasurement;
    const char* m_pLabel;
};

template<typename Array>
std::unique_ptr<Array, Deleter> MakeUnique(size_t size) NN_NOEXCEPT
{
    typedef typename std::remove_extent<Array>::type Type;
    typedef typename std::add_pointer<Type>::type TypePointer;

    NN_STATIC_ASSERT(std::is_pod<Array>::value);
    NN_STATIC_ASSERT(std::is_array<Array>::value);

    size_t allocationSize = sizeof(Type) * size;
    std::unique_ptr<Array, Deleter> buffer(reinterpret_cast<TypePointer>(util::Allocate(allocationSize)), Deleter(allocationSize));
    if(buffer.get() == nullptr)
    {
        ADD_FAILURE();
        NN_LOG("Memory Allocate Error :Size 0x%x\n", size);
        return buffer;
    }
    memset(buffer.get(), 0xCD, size);
    return buffer;
}

MultiReadWriteExecuter g_Executer[ThreadCountMax];
const char* g_pRomVerifyPath;
nn::TimeSpan g_MultiReadWriteTimeout;

nn::Result MultiReadWriteExecuter::InitializeAsRomReader(int id, nn::os::Event* pEvent, const char* pVerifyPath) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));

    {
        auto size = nn::util::SNPrintf(m_VerifyMountName, sizeof(m_VerifyMountName), "verify%d", m_Id);
        NN_RESULT_THROW_UNLESS(size < MountNameLength, nn::fs::ResultTooLongPath());
        NN_UNUSED(size);

        NN_RESULT_DO(nn::fs::MountHost(m_VerifyMountName, pVerifyPath));
        m_NeedVerifyUnmount = true;
    }

    {
        auto size = nn::util::SNPrintf(m_PathBuffer, sizeof(m_PathBuffer), "%s:/data.bin", m_VerifyMountName);
        NN_ABORT_UNLESS(size < sizeof(m_PathBuffer));
        NN_UNUSED(size);

        NN_RESULT_DO(nn::fs::OpenFile(&m_VerifyFileHandle, m_PathBuffer, nn::fs::OpenMode_Read));
        m_NeedVerifyCloseFile = true;
    }

    NN_RESULT_DO(InitializeDefaultPath());

    // Rom はアプリによらず、ファイル名固定とする
    {
        auto size = nn::util::SNPrintf(m_PathBuffer, sizeof(m_PathBuffer), "%s:/data.bin", m_MountName);
        NN_ABORT_UNLESS(size < sizeof(m_PathBuffer));
        NN_UNUSED(size);
    }

    {
        size_t mountRomCacheBufferSize = 0;
        NN_RESULT_DO(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
        m_CacheBuffer = nnt::fs::MakeUnique<char[]>(mountRomCacheBufferSize);
        NN_RESULT_THROW_UNLESS(m_CacheBuffer.get() != nullptr, nn::fs::ResultAllocationMemoryFailed());

        NN_RESULT_DO(nn::fs::MountRom(m_MountName, m_CacheBuffer.get(), mountRomCacheBufferSize));
        m_NeedUnmount = true;

        NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read));
        m_NeedCloseFile = true;
    }

    m_pLabel = "Rom";

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsSdForDebugReaderWriter(int id, nn::os::Event* pEvent) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    NN_RESULT_DO(nn::fs::MountSdCardForDebug(m_MountName));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    m_pLabel = "Sd";

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsMmcReaderWriter(int id, nn::os::Event* pEvent) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    NN_RESULT_DO(nn::fs::MountBis(m_MountName, nn::fs::BisPartitionId::User));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    m_pLabel = "Bis";

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsHostFsReaderWriter(int id, nn::os::Event* pEvent, const char* pHostPath) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    NN_RESULT_DO(nn::fs::MountHost(m_MountName, pHostPath));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    m_pLabel = "Host";

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsContentStorageReaderWriter(
        int id,
        nn::os::Event* pEvent,
        nn::fs::ContentStorageId contentStorageId
    ) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    NN_RESULT_DO(nn::fs::MountContentStorage(m_MountName, contentStorageId));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    switch (contentStorageId)
    {
    case nn::fs::ContentStorageId::SdCard:
        m_pLabel = "CS(Sd)";
        break;
    case nn::fs::ContentStorageId::System:
        m_pLabel = "CS(Sys)";
        break;
    case nn::fs::ContentStorageId::User:
        m_pLabel = "CS(Usr)";
        break;
    default:
        m_pLabel = "CS(?)";
        break;
    }

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsImageDirectoryReaderWriter(int id, nn::os::Event* pEvent, nn::fs::ImageDirectoryId imageDirectoryId) NN_NOEXCEPT
{
    const int64_t FileSize = 256 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    NN_RESULT_DO(nn::fs::MountImageDirectory(m_MountName, imageDirectoryId));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    switch (imageDirectoryId)
    {
    case nn::fs::ImageDirectoryId::SdCard:
        m_pLabel = "ID(Sd)";
        break;
    case nn::fs::ImageDirectoryId::Nand:
        m_pLabel = "ID(Mmc)";
        break;
    default:
        m_pLabel = "ID(?)";
        break;
    }

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeAsSystemSaveDataReaderWriter(int id, nn::os::Event* pEvent, nn::fs::SaveDataSpaceId spaceId, int64_t systemSaveDataId) NN_NOEXCEPT
{
    const int64_t OwnerIdForTest = NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID;

    const int64_t FileSize = 24 * 1024 * 1024;
    const int SizeMin = 1 * 1024 * 1024;
    const int SizeMax = 8 * 1024 * 1024;
    const int64_t OffsetMin = 0;
    const int64_t OffsetMax = FileSize - SizeMax;

    const int64_t JournalSize = FileSize + 8 * 1024 * 1024;
    const int64_t StorageSize = JournalSize;

    NN_RESULT_DO(Initialize(id, pEvent, OffsetMin, OffsetMax, SizeMin, SizeMax));
    NN_RESULT_DO(InitializeDefaultPath());

    (void) nn::fs::DeleteSaveData(systemSaveDataId);
    NN_RESULT_DO(nn::fs::CreateSystemSaveData(spaceId, systemSaveDataId, OwnerIdForTest, StorageSize, JournalSize, 0));

    NN_RESULT_DO(nn::fs::MountSystemSaveData(m_MountName, spaceId, systemSaveDataId));
    m_NeedUnmount = true;

    NN_RESULT_DO(EnsureFile(m_PathBuffer, FileSize));
    m_NeedDeleteFile = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    m_NeedCloseFile = true;

    switch (spaceId)
    {
    case nn::fs::SaveDataSpaceId::SdSystem:
        m_pLabel = "Sv(Sd)";
        break;
    case nn::fs::SaveDataSpaceId::System:
        m_pLabel = "Sv(Sys)";
        break;
    case nn::fs::SaveDataSpaceId::User:
        m_pLabel = "Sv(Usr)";
        break;
    default:
        m_pLabel = "Sv(?)";
        break;
    }

    m_NeedCommit = true;

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::Initialize(
        int id,
        nn::os::Event* pEvent,
        int64_t offsetMin,
        int64_t offsetMax,
        int sizeMin,
        int sizeMax
    ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(pEvent);
    NN_ABORT_UNLESS_LESS_EQUAL(0, offsetMin);
    NN_ABORT_UNLESS_LESS(0, offsetMax);
    NN_ABORT_UNLESS_LESS(offsetMin, offsetMax);
    NN_ABORT_UNLESS_LESS(0, sizeMin);
    NN_ABORT_UNLESS_LESS(0, sizeMax);
    NN_ABORT_UNLESS_LESS(sizeMin, sizeMax);

    m_Id = id;
    m_pEvent = pEvent;
    m_OffsetMin = offsetMin;
    m_OffsetMax = offsetMax;
    m_SizeMin = sizeMin;
    m_SizeMax = sizeMax;
    m_IsContinuous = false;
    m_NeedDestroyThread = false;
    m_NeedCloseFile = false;
    m_NeedUnmount = false;
    m_NeedDeleteFile = false;
    m_NeedVerifyUnmount = false;
    m_NeedVerifyCloseFile = false;
    m_NeedCommit = false;

    m_IdealCore = (id % 3);

    NN_RESULT_DO(
        nn::os::CreateThread(
            &m_Thread,
            Execute,
            reinterpret_cast<void*>(this),
            m_ThreadStack,
            ThreadStackSize,
            nn::os::DefaultThreadPriority,
            m_IdealCore
        )
    );
    m_NeedDestroyThread = true;

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::InitializeDefaultPath() NN_NOEXCEPT
{
    {
        auto size = nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "mount%d", m_Id);
        NN_RESULT_THROW_UNLESS(size < MountNameLength, nn::fs::ResultTooLongPath());
        NN_UNUSED(size);
    }

    {
        auto size = nn::util::SNPrintf(m_PathBuffer, sizeof(m_PathBuffer), "%s:/data_%016llX.bin", m_MountName, NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID);
        NN_ABORT_UNLESS(size < sizeof(m_PathBuffer));
        NN_UNUSED(size);
    }

    NN_RESULT_SUCCESS;
}

nn::Result MultiReadWriteExecuter::EnsureFile(const char* path, int64_t sizeFile) NN_NOEXCEPT
{
    auto result = nn::fs::CreateFile(path, sizeFile);
    if (nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        nn::fs::DirectoryEntryType type;
        NN_RESULT_DO(nn::fs::GetEntryType(&type, path));

        if (type == nn::fs::DirectoryEntryType_File)
        {
            NN_RESULT_DO(nn::fs::DeleteFile(path));
        }
        else
        {
            NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(path));
        }

        NN_RESULT_DO(nn::fs::CreateFile(path, sizeFile));
    }

    NN_RESULT_SUCCESS;
}

void MultiReadWriteExecuter::Finalize() NN_NOEXCEPT
{
    if (m_NeedCloseFile)
    {
        nn::fs::CloseFile(m_FileHandle);
        m_NeedCloseFile = false;
    }

    if (m_NeedDeleteFile)
    {
        nn::fs::DeleteFile(m_PathBuffer);
        m_NeedDeleteFile = false;
    }

    if (m_NeedUnmount)
    {
        nn::fs::Unmount(m_MountName);
        m_NeedUnmount = false;
    }

    if (m_NeedVerifyCloseFile)
    {
        nn::fs::CloseFile(m_VerifyFileHandle);
        m_NeedVerifyCloseFile = false;
    }

    if (m_NeedVerifyUnmount)
    {
        nn::fs::Unmount(m_VerifyMountName);
        m_NeedVerifyUnmount = false;
    }

    if (m_NeedDestroyThread)
    {
        NN_ABORT_UNLESS(!m_IsContinuous);
        nn::os::DestroyThread(&m_Thread);
        m_NeedDestroyThread = false;
    }

    m_ReadMeasurement.Reset();
    m_WriteMeasurement.Reset();
    m_CacheBuffer.reset();
}

void MultiReadWriteExecuter::Start() NN_NOEXCEPT
{
    m_IsContinuous = true;
    nn::os::StartThread(&m_Thread);
}

void MultiReadWriteExecuter::Stop() NN_NOEXCEPT
{
    m_IsContinuous = false;
    nn::os::WaitThread(&m_Thread);
}

void MultiReadWriteExecuter::Execute() NN_NOEXCEPT
{
    auto seed = nnt::fs::util::GetRandomSeed();
    NN_LOG("[%d] random seed is %d\n", m_Id, seed);

    std::mt19937 mt(seed);
    auto pBuffer = nnt::fs::MakeUnique<char[]>(m_SizeMax);
    if (pBuffer.get() == nullptr)
    {
        m_LastResult = nn::fs::ResultAllocationMemoryFailed();
        m_pEvent->Signal();
        return;
    }
    auto pBufferVerify = nnt::fs::MakeUnique<char[]>(m_SizeMax);
    if (pBufferVerify.get() == nullptr)
    {
        m_LastResult = nn::fs::ResultAllocationMemoryFailed();
        m_pEvent->Signal();
        return;
    }

    std::uniform_int_distribution<int64_t> generateRandomOffset(m_OffsetMin, m_OffsetMax - 1);
    std::uniform_int_distribution<int> generateRandomSize(m_SizeMin, m_SizeMax - 1);
    std::uniform_int_distribution<int> generatePriority(0, 3);
    nn::TimeSpan elapsed;

    int64_t fileSize;
    m_LastResult = nn::fs::GetFileSize(&fileSize, m_FileHandle);
    if (m_LastResult.IsFailure())
    {
        return;
    }

    if (m_NeedVerifyCloseFile)
    {
        while (m_IsContinuous)
        {
            nn::fs::SetPriorityRawOnCurrentThread(
                static_cast<nn::fs::PriorityRaw>(generatePriority(mt))
            );

            auto offset = generateRandomOffset(mt);
            auto size = static_cast<int>(std::min(fileSize - offset, static_cast<int64_t>(generateRandomSize(mt))));

            NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] read %lld %d\n", m_Id, m_IdealCore, offset, size);
            {
                nn::os::Tick tick = nn::os::GetSystemTick();
                m_LastResult = nn::fs::ReadFile(m_FileHandle, offset, pBuffer.get(), size);
                elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan();
            }
            if (m_LastResult.IsFailure())
            {
                NN_LOG("[%3d] [core:%d] read failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                m_pEvent->Signal();
                break;
            }
            m_ReadMeasurement.Add(elapsed, size);

            NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] read %lld %d (verify)\n", m_Id, m_IdealCore, offset, size);
            m_LastResult = nn::fs::ReadFile(m_VerifyFileHandle, offset, pBufferVerify.get(), size);
            if (m_LastResult.IsFailure())
            {
                NN_LOG("[%3d] [core:%d] verify read failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                m_pEvent->Signal();
                break;
            }

            if (std::memcmp(pBuffer.get(), pBufferVerify.get(), size) != 0)
            {
                NN_LOG("[%3d] [core:%d] compare failed\n", m_Id, m_IdealCore, offset, size);
                m_LastResult = nn::fs::ResultInternal();
                m_pEvent->Signal();
                break;
            }
        }
    }
    else
    {
        auto offset = generateRandomOffset(mt);
        auto size = static_cast<int>(std::min(fileSize - offset, static_cast<int64_t>(generateRandomSize(mt))));

        NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] read %lld %d\n", m_Id, m_IdealCore, offset, size);
        {
            nn::os::Tick tick = nn::os::GetSystemTick();
            m_LastResult = nn::fs::ReadFile(m_FileHandle, offset, pBuffer.get(), size);
            elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan();
        }
        if (m_LastResult.IsFailure())
        {
            NN_LOG("[%3d] [core:%d] read failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
            m_pEvent->Signal();
            return;
        }
        m_ReadMeasurement.Add(elapsed, size);

        while (m_IsContinuous)
        {
            // Read Verify
            {
                NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] read %lld %d\n", m_Id, m_IdealCore, offset, size);
                {
                    nn::os::Tick tick = nn::os::GetSystemTick();
                    m_LastResult = nn::fs::ReadFile(m_FileHandle, offset, pBufferVerify.get(), size);
                    elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan();
                }
                if (m_LastResult.IsFailure())
                {
                    NN_LOG("[%3d] [core:%d] read failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                    m_pEvent->Signal();
                    break;
                }
                m_ReadMeasurement.Add(elapsed, size);

                if (std::memcmp(pBuffer.get(), pBufferVerify.get(), size) != 0)
                {
                    NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] compare failed\n", m_Id, m_IdealCore, offset, size);
                    m_LastResult = nn::fs::ResultInternal();
                    m_pEvent->Signal();
                    break;
                }
            }

            // Read Modify Write
            {
                offset = generateRandomOffset(mt);
                size = static_cast<int>(std::min(fileSize - offset, static_cast<int64_t>(generateRandomSize(mt))));

                NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] read %lld %d\n", m_Id, m_IdealCore, offset, size);
                {
                    nn::os::Tick tick = nn::os::GetSystemTick();
                    m_LastResult = nn::fs::ReadFile(m_FileHandle, offset, pBuffer.get(), size);
                    elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan();
                }
                if (m_LastResult.IsFailure())
                {
                    NN_LOG("[%3d] [core:%d] read failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                    m_pEvent->Signal();
                    break;
                }
                m_ReadMeasurement.Add(elapsed, size);

                NNT_FS_STRESS_MULTIREADWRITELARGE_LOG("[%3d] [core:%d] write %lld %d\n", m_Id, m_IdealCore, offset, size);
                {
                    nn::os::Tick tick = nn::os::GetSystemTick();
                    m_LastResult = nn::fs::WriteFile(m_FileHandle, offset, pBuffer.get(), size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                    elapsed = (nn::os::GetSystemTick() - tick).ToTimeSpan();
                }
                if (m_LastResult.IsFailure())
                {
                    NN_LOG("[%3d] [core:%d] write failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                    m_pEvent->Signal();
                    break;
                }
                m_WriteMeasurement.Add(elapsed, size);

                if (m_NeedCommit)
                {
                    nn::fs::CloseFile(m_FileHandle);

                    m_LastResult = nn::fs::CommitSaveData(m_MountName);
                    if (m_LastResult.IsFailure())
                    {
                        NN_LOG("[%3d] [core:%d] commit failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                        m_pEvent->Signal();
                        break;
                    }

                    m_LastResult = nn::fs::OpenFile(&m_FileHandle, m_PathBuffer, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
                    if (m_LastResult.IsFailure())
                    {
                        NN_LOG("[%3d] [core:%d] open failed (0x%08X)\n", m_Id, m_IdealCore, m_LastResult.GetInnerValueForDebug());
                        m_pEvent->Signal();
                        break;
                    }
                }
            }
        }
    }
} // NOLINT(impl/function_size)

}

class MultiReadWriteLarge : public ::testing::Test
{
protected:
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE;
};

void MultiReadWriteLarge::TearDown() NN_NOEXCEPT
{
    for (int i = 0; i < ThreadCountMax; ++i)
    {
        g_Executer[i].Finalize();
    }
}

TEST_F(MultiReadWriteLarge, MultiReadWrite)
{
    const nn::TimeSpan CheckMeasurementInterval = nn::TimeSpan::FromSeconds(20);

    nn::os::Event event(nn::os::EventClearMode_AutoClear);

    // 初期化
    auto executerCount = 0;

    {
        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsRomReader(executerCount, &event, g_pRomVerifyPath)
        );
        ++executerCount;

        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsMmcReaderWriter(executerCount, &event)
        );
        ++executerCount;

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000001 || defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsSdForDebugReaderWriter(executerCount, &event)
        );
        ++executerCount;

        {
            const auto HostPathLengthMax = 260;
            auto pHostPath = MakeUnique<char[]>(HostPathLengthMax);
            auto size = nn::util::SNPrintf(pHostPath.get(), HostPathLengthMax, "%s/..", g_pRomVerifyPath);
            NN_ABORT_UNLESS(size < HostPathLengthMax);
            NN_UNUSED(size);

            NNT_ASSERT_RESULT_SUCCESS(
                g_Executer[executerCount].InitializeAsHostFsReaderWriter(executerCount, &event, pHostPath.get())
            );

            ++executerCount;
        }
#endif

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000000 || defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsContentStorageReaderWriter(executerCount, &event, nn::fs::ContentStorageId::SdCard)
        );
        ++executerCount;

        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsImageDirectoryReaderWriter(executerCount, &event, nn::fs::ImageDirectoryId::SdCard)
        );
        ++executerCount;
#endif

#if defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsContentStorageReaderWriter(executerCount, &event, nn::fs::ContentStorageId::User)
        );
        ++executerCount;

        NNT_ASSERT_RESULT_SUCCESS(
            g_Executer[executerCount].InitializeAsImageDirectoryReaderWriter(executerCount, &event, nn::fs::ImageDirectoryId::Nand)
        );
        ++executerCount;
#endif

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000001 || defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
        {
            const int64_t SystemSaveDataIdForTest = 0x8000000000004000 + (NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID & 0xF);

            NNT_ASSERT_RESULT_SUCCESS(
                g_Executer[executerCount].InitializeAsSystemSaveDataReaderWriter(executerCount, &event, nn::fs::SaveDataSpaceId::SdSystem, SystemSaveDataIdForTest)
            );
            ++executerCount;
        }
#endif

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000000 || defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
        {
            const int64_t SystemSaveDataIdForTest = 0x8000000000004000 + (NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID & 0xF);

            NNT_ASSERT_RESULT_SUCCESS(
                g_Executer[executerCount].InitializeAsSystemSaveDataReaderWriter(executerCount, &event, nn::fs::SaveDataSpaceId::System, SystemSaveDataIdForTest)
            );
            ++executerCount;
        }
#endif
    }

    // 実行
    for (int i = 0; i < executerCount; ++i)
    {
        g_Executer[i].Start();
    }

    auto checkStatus = [&](
            int executerId,
            const char* pAction,
            int64_t speedBytesPerSecond,
            int64_t size
        ) NN_NOEXCEPT -> bool
    {
        if (size != 0)
        {
            auto speedMegaBytesPerSecondTimes1000 = speedBytesPerSecond * 1000 / (1024 * 1024);
            NN_LOG(
                "[appId:%X] [core:%d] [%7s] %s average speed %4d.%03d MiB/s average buffer size %d KiB\n",
                static_cast<int>(NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID & 0xF),
                g_Executer[executerId].GetIdealCore(),
                g_Executer[executerId].GetLabel(),
                pAction,
                static_cast<int>(speedMegaBytesPerSecondTimes1000 / 1000),
                static_cast<int>(speedMegaBytesPerSecondTimes1000 % 1000),
                static_cast<int>(size / 1024)
                );
            return true;
        }
        else
        {
            return false;
        }
    };

    bool isContinue = true;
    nn::os::Tick tickStart = nn::os::GetSystemTick();
    while (isContinue)
    {
        if (g_MultiReadWriteTimeout > 0 && (nn::os::GetSystemTick() - tickStart).ToTimeSpan() > g_MultiReadWriteTimeout)
        {
            break;
        }
        event.TimedWait(CheckMeasurementInterval);

        NN_LOG("---- Section\n");
        for (int executerId = 0; executerId < executerCount; ++executerId)
        {
            const nn::Result& result = g_Executer[executerId].GetLastError();
            if (result.IsFailure())
            {
                isContinue = false;
                break;
            }

            {
                int64_t speedBytesPerSecond = 0;
                int64_t size = 0;
                bool isDead = true;

                g_Executer[executerId].GetReadMeasurement().GetSectionAverageValueAndReset(&speedBytesPerSecond, &size);
                if (checkStatus(executerId, "Read ", speedBytesPerSecond, size))
                {
                    isDead = false;
                }
                g_Executer[executerId].GetWriteMeasurement().GetSectionAverageValueAndReset(&speedBytesPerSecond, &size);
                if (checkStatus(executerId, "Write", speedBytesPerSecond, size))
                {
                    isDead = false;
                }

                if (isDead)
                {
                    isContinue = false;
                    ADD_FAILURE(); // 後始末するため、FAIL はしない
                }
            }
        }
    }

    // 後始末
    for (int i = 0; i < executerCount; ++i)
    {
        g_Executer[i].Stop();
    }

    for (int i = 0; i < executerCount; ++i)
    {
        NNT_EXPECT_RESULT_SUCCESS(g_Executer[i].GetLastError());
    }

    // Finalize は TearDown に任せる
} // NOLINT(impl/function_size)

}}

extern "C" void nninitStartup()
{
    static NN_ALIGNAS(4096) char s_Buffer[1 * 1024 * 1024];
    nn::init::InitializeAllocator(s_Buffer, sizeof(s_Buffer));
}

extern "C" void nnMain()
{
#if defined(NNT_FS_STRESS_MULTIREADWRITELARGE_IS_APPLICATION)
    static const size_t MemorySize = 512 * 1024 * 1024;
#else
    static const size_t MemorySize = 96 * 1024 * 1024;
#endif
    static NN_ALIGNAS(4096) char s_Buffer[MemorySize];

    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    if (argc < 2)
    {
        NN_LOG("Usage: *.exe <source directory path>");
        nnt::Exit(EXIT_FAILURE);
    }
    nnt::fs::g_pRomVerifyPath = argv[1];
    NN_LOG("Verify Path is %s\n", nnt::fs::g_pRomVerifyPath);

    if (argc >= 3)
    {
        nnt::fs::g_MultiReadWriteTimeout = nn::TimeSpan::FromSeconds(atoi(argv[2]));
    }
    else
    {
        nnt::fs::g_MultiReadWriteTimeout = nn::TimeSpan(0);
    }

    nnt::fs::util::InitializeTestLibraryHeap(s_Buffer, sizeof(s_Buffer));
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

#if NNT_FS_STRESS_MULTIREADWRITELARGE_PRGRAMID == 0x005000C10000000 || defined(NN_BUILD_CONFIG_OS_WIN)
    nnt::fs::util::DeleteAllTestSaveData();
#endif

    nn::time::Initialize();

    auto testResult = RUN_ALL_TESTS();

    nn::time::Finalize();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(EXIT_FAILURE);
    }

    nnt::Exit(testResult);
}
