﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/time/time_PosixTime.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

size_t g_AllocatedMax = 0;
size_t g_Allocated = 0;

namespace {

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        g_Allocated += size;
        if (g_Allocated >= g_AllocatedMax)
        {
            g_AllocatedMax = g_Allocated;
        }
        return malloc(size);
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        g_Allocated -= size;
        free(p);
    }

    nn::Result FindSaveDataId(nn::fs::SaveDataId *outValue, nn::ncm::ApplicationId applicationId, nn::fs::UserId userId) NN_NOEXCEPT
    {
        const nn::fs::SaveDataSpaceId SpaceIds[] = {nn::fs::SaveDataSpaceId::User, nn::fs::SaveDataSpaceId::System};
        for(auto spaceId : SpaceIds)
        {
            std::unique_ptr<nn::fs::SaveDataIterator> iter;
            NNT_EXPECT_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, spaceId));

            while (NN_STATIC_CONDITION(true))
            {
                int64_t count;
                nn::fs::SaveDataInfo info;
                NNT_EXPECT_RESULT_SUCCESS(iter->ReadSaveDataInfo(&count, &info, 1));
                if (count == 0)
                {
                    break;
                }

                if (info.applicationId.value == applicationId.value && info.saveDataUserId == userId)
                {
                    *outValue = info.saveDataId;
                    NN_RESULT_SUCCESS;
                }
            }
        }
        return nn::fs::ResultTargetNotFound();
    }

    void ResetAllocated() NN_NOEXCEPT
    {
        g_Allocated = 0;
    }

    nn::fs::UserId GetUserIdByIndex(int index) NN_NOEXCEPT
    {
        nn::fs::UserId userId = {{ 0xFull, static_cast<uint32_t>(index) }};
        return userId;
    }

    nnt::fs::util::String GetMountNameByIndex(int index) NN_NOEXCEPT
    {
        nnt::fs::util::String mountName = "save" + nnt::fs::util::ToString(index);
        return mountName;
    }

    nn::fs::SystemSaveDataId GetSystemSaveDataIdByIndex(int index) NN_NOEXCEPT
    {
        const nn::fs::SystemSaveDataId BaseId = 0x8000000000004000;
        return BaseId + index;
    }
}

TEST(MountMultiSaveData, Basic)
{
    const size_t SaveDataSize = 512 * 1024;
    const size_t SaveDataJournalSize = 512 * 1024;
    const int SaveDataNum = 80;
    const auto ApplicationSaveDataMountCountLimit = 10;

    nn::ncm::ApplicationId appId = { 0x0005000C10000000 };
    nn::Bit64 ownerId = 0x0005000C10000000;

    // テストで使用するのと同 ID の既存セーブデータを削除
    for (int i = 1; i <= SaveDataNum; i++)
    {
        nn::fs::SaveDataId saveDataId = 0;
        auto result = FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i));
        if (result.IsSuccess())
        {
            nn::fs::DeleteSaveData(saveDataId);
        }
    }
    for (int i = 1; i <= SaveDataNum; i++)
    {
        const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
        nn::fs::DeleteSaveData(saveDataId);
    }

    // アプリケーションセーブデータを作成
    int applicationSaveDataCreateCount = 0;
    for (int i = 1; i <= ApplicationSaveDataMountCountLimit; i++)
    {
        auto result = nn::fs::CreateSaveData(
            appId, GetUserIdByIndex(i), ownerId, SaveDataSize, SaveDataJournalSize, 0);
        if (result.IsFailure())
        {
            NN_LOG("Failed to create save data #%d (%x)\n", i, result.GetInnerValueForDebug());
            break;
        }
        applicationSaveDataCreateCount = i;
    }
    EXPECT_EQ(applicationSaveDataCreateCount, ApplicationSaveDataMountCountLimit);

    // システムセーブデータを作成
    int systemSaveDataCreateCount = 0;
    for (int i = 1; i <= SaveDataNum - ApplicationSaveDataMountCountLimit; i++)
    {
        auto result = nn::fs::CreateSystemSaveData(
            GetSystemSaveDataIdByIndex(i), SaveDataSize, SaveDataJournalSize, 0);
        if (result.IsFailure())
        {
            NN_LOG("Failed to create system save data #%d (%x)\n",
                i, result.GetInnerValueForDebug());
            break;
        }
        systemSaveDataCreateCount = i;
    }
    EXPECT_EQ(systemSaveDataCreateCount, SaveDataNum - ApplicationSaveDataMountCountLimit);

    // 全アプリケーションセーブデータをマウント
    ResetAllocated();
    int applicationSaveDataMountCount = 0;
    for (int i = 1; i <= applicationSaveDataCreateCount; i++)
    {
        auto result = nn::fs::MountSaveData(
            GetMountNameByIndex(i).c_str(), appId, GetUserIdByIndex(i));
        if (result.IsFailure())
        {
            if (!nn::fs::ResultOpenCountLimit::Includes(result))
            {
                NN_LOG("Failed to mount save data #%d (%x)\n", i, result.GetInnerValueForDebug());
            }
            break;
        }
        applicationSaveDataMountCount = i;
    }
    EXPECT_EQ(applicationSaveDataMountCount, ApplicationSaveDataMountCountLimit);

    // 全システムセーブデータをマウント
    int systemSaveDataMountCount = 0;
    for (int i = 1; i <= systemSaveDataCreateCount; i++)
    {
        auto result = nn::fs::MountSystemSaveData(
            GetMountNameByIndex(i + applicationSaveDataMountCount).c_str(),
            GetSystemSaveDataIdByIndex(i));
        if (result.IsFailure())
        {
            if (!nn::fs::ResultOpenCountLimit::Includes(result))
            {
                NN_LOG("Failed to mount system save data #%d (%x)\n",
                    i, result.GetInnerValueForDebug());
            }
            break;
        }
        systemSaveDataMountCount = i;
    }
    EXPECT_EQ(systemSaveDataMountCount, systemSaveDataCreateCount);

    NN_LOG("Allocated: %zd bytes, (max %zd bytes)\n", g_Allocated, g_AllocatedMax);

    // 全システムセーブデータをアンマウントして削除
    for (int i = 1; i <= systemSaveDataCreateCount; i++)
    {
        if (i <= systemSaveDataMountCount)
        {
            nn::fs::Unmount(GetMountNameByIndex(i + applicationSaveDataMountCount).c_str());
        }
        const auto saveDataId = static_cast<nn::fs::SaveDataId>(GetSystemSaveDataIdByIndex(i));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataId));
    }

    // 全アプリケーションセーブデータをアンマウントして削除
    for (int i = 1; i <= applicationSaveDataCreateCount; i++)
    {
        if (i <= applicationSaveDataMountCount)
        {
            nn::fs::Unmount(GetMountNameByIndex(i).c_str());
        }
        nn::fs::SaveDataId saveDataId = 0;
        NNT_EXPECT_RESULT_SUCCESS(FindSaveDataId(&saveDataId, appId, GetUserIdByIndex(i)));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteSaveData(saveDataId));
    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);
    nn::fs::SetAllocator(Allocate, Deallocate);
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    nnt::Exit(testResult);
}
