﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_FormatString.h>

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;


namespace{


const char* FileSuffix = "";


class PrepareTestDir : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
public:
    nnt::fs::util::String testDirPath;

protected:
    PrepareTestDir() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        CreateWorkRootPath();
        testDirPath = GetWorkRootPath();
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteWorkRootPath();
    }
};


typedef PrepareTestDir RaceCondition;

void CreateAndDeleteFileTest(const char* fileName) NN_NOEXCEPT
{
    const int64_t FileSize = 1024;

    auto result = [&]() -> nn::Result
    {
        NN_RESULT_DO(CreateFile(fileName, FileSize));
        NN_RESULT_DO(DeleteFile(fileName));
        NN_RESULT_SUCCESS;
    }();

    if (result.IsFailure())
    {
        NN_LOG("CreateAndDeleteFileTest() failed: <%s>\n", fileName);
        nnt::fs::util::ListDirectoryRecursive(nnt::fs::util::GetParentPath(fileName).c_str());
    }
    NNT_EXPECT_RESULT_SUCCESS(result);
}


using namespace nnt::fs::util;

const int LoopCount  = 1000;
const int NameLength = 32;

struct Params
{
    int id;
    const char* hostPath;
};

struct AccessParams
{
    int id;
    const char* mountName;
};

void threadFunctionSd(void* arg) NN_NOEXCEPT
{
    NNT_FS_SCOPED_TRACE_SAFE("");

    auto pParams = (Params*)arg;

    int id = pParams->id;

    auto mountName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(mountName.get(), NameLength, "mount%d", id);

    auto fileName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(fileName.get(), NameLength, "%s:/file%s%d", mountName.get(), FileSuffix, id);

    NNT_EXPECT_RESULT_SUCCESS(MountSdCardForDebug(mountName.get()));
    DeleteFile(fileName.get());
    Unmount(mountName.get());

    for(int i = 0; i < LoopCount; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        if( (i % 100) == 0 )
        {
            NN_LOG("(SD     #%d) mount count : %d\n", id, i);
        }

        NNT_EXPECT_RESULT_SUCCESS(MountSdCardForDebug(mountName.get()));

        CreateAndDeleteFileTest(fileName.get());

        Unmount(mountName.get());
    }
}

void threadFunctionBis(void* arg) NN_NOEXCEPT
{
    NNT_FS_SCOPED_TRACE_SAFE("");

    auto pParams = (Params*)arg;

    int id = pParams->id;

    auto fileName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(fileName.get(), NameLength, "@System:/file%s%d", FileSuffix, id);

    NNT_EXPECT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
    DeleteFile(fileName.get());
    Unmount("@System");

    for(int i = 0; i < LoopCount; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        if( (i % 100) == 0 )
        {
            NN_LOG("(Bis    #%d) mount count : %d\n", id, i);
        }

        NNT_EXPECT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));

        CreateAndDeleteFileTest(fileName.get());

        Unmount("@System");
    }
}

void threadFunctionHost(void* arg) NN_NOEXCEPT
{
    NNT_FS_SCOPED_TRACE_SAFE("");

    auto pParams = (Params*)arg;

    int id = pParams->id;


    auto mountName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(mountName.get(), NameLength, "mount%d", id);

    auto fileName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(fileName.get(), NameLength, "%s:/file%s%d", mountName.get(), FileSuffix, id);

    NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.get(), pParams->hostPath));
    DeleteFile(fileName.get());
    Unmount(mountName.get());

    for(int i = 0; i < LoopCount; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        if( (i % 100) == 0 )
        {
            NN_LOG("(Host   #%d) mount count : %d\n", id, i);
        }

        NNT_EXPECT_RESULT_SUCCESS(MountHost(mountName.get(), pParams->hostPath));

        CreateAndDeleteFileTest(fileName.get());

        Unmount(mountName.get());
    }
}

void threadFunctionSystemSaveData(void* arg) NN_NOEXCEPT
{
    NNT_FS_SCOPED_TRACE_SAFE("");

    auto pParams = (Params*)arg;

    int id = pParams->id;

    auto mountName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(mountName.get(), NameLength, "mount%d", id);

    auto fileName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(fileName.get(), NameLength, "%s:/file%s%d", mountName.get(), FileSuffix, id);

    SystemSaveDataId systemSaveDataId = 0x8000000000004000 + id;

    for(int i = 0; i < LoopCount; i++)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));

        if( (i % 100) == 0 )
        {
            NN_LOG("(SysSav #%d) mount count : %d\n", id, i);
        }

        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData(mountName.get(), systemSaveDataId));

        DeleteFile(fileName.get()); // TODO: 削除
        CreateAndDeleteFileTest(fileName.get());

        Unmount(mountName.get());
    }

    DeleteSaveData(systemSaveDataId);
}

void threadFunctionAccess(void* arg) NN_NOEXCEPT
{
    NNT_FS_SCOPED_TRACE_SAFE("");

    auto pParams = (AccessParams*)arg;

    int id = pParams->id;
    const char* mountName = pParams->mountName;

    auto fileName = AllocateBuffer(NameLength);
    nn::util::SNPrintf(fileName.get(), NameLength, "%s:/file%s%d", mountName, FileSuffix, id);
    DeleteFile(fileName.get());

    for(int i = 0; i < LoopCount; i++)
    {
        CreateAndDeleteFileTest(fileName.get());
    }
}

}

//!< @brief 複数スレッドでのマウント並列実行
TEST_F(RaceCondition, Mount)
{
    nnt::fs::util::DeleteAllTestSaveData();

    const int ThreadCount = 10;
    const size_t StackSize = 64 * 1024;
    static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];

    nn::os::ThreadFunction threadFunctions[] = {
        threadFunctionSystemSaveData,
        threadFunctionSystemSaveData,
        threadFunctionSystemSaveData,
        threadFunctionSystemSaveData,

        threadFunctionHost,
        threadFunctionHost,

        threadFunctionBis,

        threadFunctionSd,
        threadFunctionSd,
        threadFunctionSd,
    };

    Params params[] = {
        {0,  testDirPath.c_str()},
        {1,  testDirPath.c_str()},
        {2,  testDirPath.c_str()},
        {3,  testDirPath.c_str()},
        {4,  testDirPath.c_str()},
        {5,  testDirPath.c_str()},
        {6,  testDirPath.c_str()},
        {7,  testDirPath.c_str()},
        {8,  testDirPath.c_str()},
        {9,  testDirPath.c_str()},
    };

    nn::os::ThreadType threads[ThreadCount];

    for (int i = 0; i < ThreadCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], threadFunctions[i], &params[i], stack[i], StackSize, nn::os::DefaultThreadPriority));
        nn::os::StartThread(&threads[i]);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&threads[i]);
        nn::os::DestroyThread(&threads[i]);
    }

    nnt::fs::util::DeleteAllTestSaveData();
}

TEST_F(RaceCondition, ParallelAccess)
{
    const char* SdMountName = "SdCard";
    AccessParams params[] = {
        {0, GetBisMountName(BisPartitionId::User)},
        {1, GetBisMountName(BisPartitionId::User)},
        {2, GetBisMountName(BisPartitionId::System)},
        {3, GetBisMountName(BisPartitionId::System)},
        {4, SdMountName},
        {5, SdMountName},
    };
    const int ThreadCount = sizeof(params) / sizeof(params[0]);
    const size_t StackSize = 64 * 1024;
    static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
    nn::os::ThreadType threads[ThreadCount];

    NNT_EXPECT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
    NNT_EXPECT_RESULT_SUCCESS(MountBis(BisPartitionId::System, nullptr));
    NNT_EXPECT_RESULT_SUCCESS(MountSdCardForDebug(SdMountName));

    for (int i = 0; i < ThreadCount; i++)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], threadFunctionAccess, &params[i], stack[i], StackSize, nn::os::DefaultThreadPriority));
    }

    TimeCount timeCount;
    timeCount.StartTime();
    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::StartThread(&threads[i]);
    }

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::WaitThread(&threads[i]);
    }
    timeCount.StopTime();
    timeCount.ViewTime(timeCount.GetTotalTime());

    for (int i = 0; i < ThreadCount; i++)
    {
        nn::os::DestroyThread(&threads[i]);
    }

    Unmount(SdMountName);
    Unmount(GetBisMountName(BisPartitionId::System));
    Unmount(GetBisMountName(BisPartitionId::User));
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    if( argc >= 2 )
    {
        FileSuffix = argv[1];
    }

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
