﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_DebugPrivate.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace {

    void CheckRomContents(int dataSrcNo)
    {
        NN_SDK_LOG("CheckRomContents: %d\n", dataSrcNo);

        nn::fs::FileHandle fileHandle;

        {
            // 共通のファイルを開く
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &fileHandle,
                    "rom:/test.txt",
                    OpenMode_Read
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };

            // ファイルの内容を読み込み
            const size_t bufferSize = 16;
            char buffer[bufferSize];
            size_t readSize;
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, bufferSize)
            );

            // ファイルの内容が正しいことを確認
            char expected_str[bufferSize];
            std::snprintf(expected_str, bufferSize, "dataSrc%d", dataSrcNo);
            NNT_FS_UTIL_EXPECT_MEMCMPEQ(expected_str, buffer, readSize);
        }

        {
            // 固有のファイルが開けることを確認
            char filename[32];
            std::snprintf(filename, sizeof(filename), "rom:/test%d.txt", dataSrcNo);
            NNT_EXPECT_RESULT_SUCCESS(
                nn::fs::OpenFile(
                    &fileHandle,
                    filename,
                    OpenMode_Read
                )
            );
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(fileHandle);
            };
        }
    }

}

TEST(Redirection, Rom)
{
    std::vector<char> cacheBuffer;

    // 最後に必ずリダイレクトなし設定に戻す
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::SetDataStorageRedirectTarget(nn::ncm::StorageId::None);
    };

#ifdef CHECK_SELF_ROM_FS
    // 通常通り自身のデータを読み込む（リダイレクトなし）
    nn::fs::SetDataStorageRedirectTarget(nn::ncm::StorageId::None);
    {
        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&cacheBufferSize));

        cacheBuffer.resize(cacheBufferSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom("rom", cacheBuffer.data(), cacheBufferSize)
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        // 内容が dataSrc2 のものであることを確認
        CheckRomContents(2);
    }
#endif

    // NAND へインストール済みの ROM のデータを読み込む（BuildInUser へリダイレクト）
    nn::fs::SetDataStorageRedirectTarget(nn::ncm::StorageId::BuildInUser);
    {
        size_t cacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&cacheBufferSize));

        cacheBuffer.resize(cacheBufferSize);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::fs::MountRom("rom", cacheBuffer.data(), cacheBufferSize)
        );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        // 内容が dataSrc1 のものであることを確認
        CheckRomContents(1);
    }
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nn::fs::SetEnabledAutoAbort(false);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
