﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/account.h>
#include <nn/fs.h>
#include <nn/account/account_ApiForSystemServices.h>
#include <nn/fs/fs_CacheStorageWithIndex.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SystemSaveDataPrivate.h>
#include <nn/fs/fs_TemporaryStorage.h>
#include <nn/fs/fs_Utility.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_Mutex.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace{
    enum class ResultType
    {
        Normal,
        Conflicting,
        Success,
        AlreadySuccess,
        FailureAndSuccess,
    };

    struct TestArgument
    {
        nn::account::Uid userId;
        nn::fs::SaveDataSpaceId saveDataSpaceId;
        nn::fs::SaveDataId saveDataId;
        int threadId;
        int64_t waitTime;

        nn::fs::UserId GetFsUid() NN_NOEXCEPT
        {
            return nn::fs::ConvertAccountUidToFsUserId(this->userId);
        }
    };

    const int SaveDataCount = 4;
    const int ThreadCount = SaveDataCount * 2;
    const size_t StackSize = 64 * 1024;
    const int DefaultTestProcessCount = 30;

    // 実行環境等に依存する値
    const int MinWaitTimeBeforeProcess = 1;             // ミリ秒
    const int MaxWaitTimeBeforeProcess = 10;            // ミリ秒
    const int WaitTimeBetweenMountAndUnmount = 1500;    // ミリ秒

    const int DefaultSaveDataSize = 16 * 1024 * 1024;
    const int DefaultJournalSize = 8 * 1024 * 1024;
    const int ExtendSaveDataSize = 4 * 1024 * 1024;
    const int ExtendJournalSize = 1 * 1024 * 1024;

    const int DefaultCacheStorageSize = 256 * 1024;
    const int DefaultCacheStorageJournalSize = 256 * 1024;

    int g_TestProcessCount = DefaultTestProcessCount;

    int g_SuccessCount1 = 0;
    int g_SuccessCount2 = 0;

    nn::os::Mutex g_SuccessCountMutex(false);

    nn::os::ThreadType g_Threads[ThreadCount];
    NN_OS_ALIGNAS_THREAD_STACK static char g_StackBuffer[StackSize * ThreadCount] = {};

    nn::account::Uid g_Uids[nn::account::UserCountMax];

    void IncrementSuccessCount(int threadId) NN_NOEXCEPT
    {
        std::lock_guard<nn::os::Mutex> scopedLock(g_SuccessCountMutex);
        if( threadId < SaveDataCount )
        {
            ++g_SuccessCount1;
        }
        else
        {
            ++g_SuccessCount2;
        }
    }

    void CheckSuccessCount(
             const char* functionName1,
             const char* functionName2,
             ResultType resultType
         ) NN_NOEXCEPT
    {
        const int AllProcessCount = g_TestProcessCount * SaveDataCount;
        NN_LOG(
            "Thread for %s: Success/Failure = %3d/%3d\n",
            functionName1,
            g_SuccessCount1,
            AllProcessCount - g_SuccessCount1
        );
        NN_LOG(
            "Thread for %s: Success/Failure = %3d/%3d\n",
            functionName2,
            g_SuccessCount2,
            AllProcessCount - g_SuccessCount2
        );

        switch( resultType )
        {
        case ResultType::Normal:
            {
                EXPECT_NE(0, g_SuccessCount1);
                EXPECT_NE(0, g_SuccessCount2);
                EXPECT_NE(AllProcessCount, g_SuccessCount1);
                EXPECT_NE(AllProcessCount, g_SuccessCount2);
            }
            break;
        case ResultType::Conflicting:
            {
                EXPECT_NE(0, g_SuccessCount1);
                EXPECT_NE(0, g_SuccessCount2);
                EXPECT_NE(AllProcessCount, g_SuccessCount1);
                EXPECT_NE(AllProcessCount, g_SuccessCount2);
                EXPECT_EQ(AllProcessCount, g_SuccessCount1 + g_SuccessCount2);
            }
            break;
        case ResultType::Success:
            {
                EXPECT_EQ(AllProcessCount, g_SuccessCount1);
                EXPECT_EQ(AllProcessCount, g_SuccessCount2);
            }
            break;
        case ResultType::AlreadySuccess:
            {
                EXPECT_NE(0, g_SuccessCount1);
                EXPECT_NE(AllProcessCount, g_SuccessCount1);
                EXPECT_EQ(AllProcessCount, g_SuccessCount2);
            }
            break;
        case ResultType::FailureAndSuccess:
            {
                EXPECT_EQ(0, g_SuccessCount1);
                EXPECT_EQ(AllProcessCount, g_SuccessCount2);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void MakeMountName(char* buffer, size_t bufferSize, int id) NN_NOEXCEPT
    {
        memset(buffer, 0, bufferSize);
        auto length = nn::util::SNPrintf(buffer, bufferSize, "save%d", id);
        ASSERT_LT(static_cast<size_t>(length), bufferSize);
    }

    template<typename ExpectResult>
    void CreateExistSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        nn::Result result = nn::fs::CreateSaveData(
                                nnt::fs::util::ApplicationId,
                                pTestArgument->GetFsUid(),
                                0,
                                DefaultSaveDataSize,
                                DefaultJournalSize,
                                0
                            );

        if( ExpectResult::Includes(result) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(result);
        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<typename ExpectResult>
    void MountSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        std::unique_ptr<char[]> mountName(new char[16]);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(MakeMountName(mountName.get(), 16, pTestArgument->threadId));

        nn::Result result = nn::fs::MountSaveData(mountName.get(), pTestArgument->GetFsUid());
        if( ExpectResult::Includes(result) )
        {
            return;
        }

        // ExpectResult 以外のエラーリザルトは想定外
        NNT_ASSERT_RESULT_SUCCESS(result);

        // 充分待ってからアンマウント
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitTimeBetweenMountAndUnmount));
        nn::fs::Unmount(mountName.get());
        IncrementSuccessCount(pTestArgument->threadId);
    }

    void UnmountSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        std::unique_ptr<char[]> mountName(new char[16]);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(MakeMountName(mountName.get(), 16, pTestArgument->threadId));
        nn::fs::Unmount(mountName.get());

        // 充分待ってから再マウント
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitTimeBetweenMountAndUnmount));
        NNT_ASSERT_RESULT_SUCCESS(
            nn::fs::MountSaveData(mountName.get(), pTestArgument->GetFsUid())
        );
        nn::fs::Unmount(mountName.get());
        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<typename ExpectResult>
    void DeleteSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        nn::Result result = nn::fs::DeleteSaveData(
                                nn::fs::SaveDataSpaceId::User,
                                pTestArgument->saveDataId
                            );

        if( ExpectResult::Includes(result) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(result);

        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<typename ExpectResult1, typename ExpectResult2 = nn::fs::ResultUnknown>
    void ExtendSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));
        nn::Result result
            = nn::fs::ExtendSaveData(
                  nn::fs::SaveDataSpaceId::User,
                  pTestArgument->saveDataId,
                  DefaultSaveDataSize + ExtendSaveDataSize,
                  DefaultJournalSize + ExtendJournalSize
              );

        if( ExpectResult1::Includes(result) || ExpectResult2::Includes(result) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(result);
        IncrementSuccessCount(pTestArgument->threadId);
    }

    void IsSaveDataExisting(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTestArgument);

        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        const auto isSaveDataExisting = nn::fs::IsSaveDataExisting(pTestArgument->GetFsUid());
        ASSERT_TRUE(isSaveDataExisting);

        IncrementSuccessCount(pTestArgument->threadId);
    }

    void CreateAndDeleteCacheStorage(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        const int cacheStorageIndex = pTestArgument->threadId;

        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(cacheStorageIndex, DefaultCacheStorageSize, DefaultCacheStorageJournalSize));

        {
            NN_UTIL_SCOPE_EXIT
            {
                // 同時実行されやすくするため、乱数時間停止する
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

                NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteCacheStorage(cacheStorageIndex));
            };

            // 同時実行されやすくするため、乱数時間停止する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

            int64_t cacheStorageDataSize;
            int64_t cacheStorageJournalSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetCacheStorageSize(&cacheStorageDataSize, &cacheStorageJournalSize, cacheStorageIndex));
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<typename ExpectResult>
    void AccessCacheStorageList(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        nn::fs::CacheStorageListHandle handle;
        auto resultOpen = nn::fs::OpenCacheStorageList(&handle);
        NNT_ASSERT_RESULT_SUCCESS(resultOpen);

        NN_UTIL_SCOPE_EXIT
        {
            if( resultOpen.IsSuccess() )
            {
                nn::fs::CloseCacheStorageList(handle);
            }
        };

        // この間に作成があればエラーになるので適当にスリープする
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(200));

        // 読み出し
        int index;
        nn::fs::CacheStorageInfo info;
        auto resultRead = nn::fs::ReadCacheStorageList(&index, &info, handle, 1);
        if( ExpectResult::Includes(resultRead) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(resultRead);

        IncrementSuccessCount(pTestArgument->threadId);
    }

    void EnsureSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(pTestArgument->userId));

        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<typename CreateFailureResult, int offsetUserIndex>
    void CreateAndDeleteSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        auto accountUserId = g_Uids[offsetUserIndex + pTestArgument->threadId];
        auto fsUserId = nn::fs::ConvertAccountUidToFsUserId(accountUserId);

        auto resultCreate = nn::fs::CreateSaveData(nnt::fs::util::ApplicationId, fsUserId, nnt::fs::util::ApplicationId.value, DefaultSaveDataSize, DefaultJournalSize, 0);
        if( CreateFailureResult::Includes(resultCreate) )
        {
            return;
        }
        NNT_ASSERT_RESULT_SUCCESS(resultCreate);

        {
            NN_UTIL_SCOPE_EXIT
            {
                // 同時実行されやすくするため、乱数時間停止する
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

                auto findImpl = [=](nn::util::optional<nn::fs::SaveDataInfo>* pOutValue) NN_NOEXCEPT -> nn::Result
                {
                    std::unique_ptr<nn::fs::SaveDataIterator> iter;
                    NN_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));

                    int64_t count;
                    nn::fs::SaveDataInfo info;
                    while( NN_STATIC_CONDITION(true) )
                    {
                        count = 0;
                        NN_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));
                        if( count == 0 )
                        {
                            *pOutValue = nn::util::nullopt;
                            NN_RESULT_SUCCESS;
                        }
                        if( info.saveDataUserId == fsUserId && info.applicationId == nnt::fs::util::ApplicationId )
                        {
                            *pOutValue = info;
                            NN_RESULT_SUCCESS;
                        }
                    }
                };

                for( ; ; )
                {
                    nn::util::optional<nn::fs::SaveDataInfo> info;
                    auto resultFind = findImpl(&info);
                    if( nn::fs::ResultInvalidHandle::Includes(resultFind) )
                    {
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
                        continue;
                    }
                    NNT_ASSERT_RESULT_SUCCESS(resultFind);
                    ASSERT_TRUE(info);
                    NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(nn::fs::SaveDataSpaceId::User, info->saveDataId));
                    break;
                }
            };

            // 同時実行されやすくするため、乱数時間停止する
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

            int64_t saveDataDataSize;
            int64_t saveDataJournalSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataSize(&saveDataDataSize, &saveDataJournalSize, accountUserId));
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    void IsSaveDataExistingAndGetSaveDataSize(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTestArgument);

        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        for( int i = 0; i < 4; ++i )
        {
            const auto isSaveDataExisting = nn::fs::IsSaveDataExisting(pTestArgument->userId);
            ASSERT_TRUE(isSaveDataExisting);

            int64_t saveDataSize;
            int64_t journalSize;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataSize(&saveDataSize, &journalSize, pTestArgument->userId));
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    void IsSaveDataExistingAndSetSaveDataFlags(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTestArgument);

        // 同時実行されやすくするため、乱数時間停止する
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(pTestArgument->waitTime));

        for( int i = 0; i < 4; ++i )
        {
            const auto isSaveDataExisting = nn::fs::IsSaveDataExisting(pTestArgument->userId);
            ASSERT_TRUE(isSaveDataExisting);

            NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSaveDataFlags(pTestArgument->saveDataId, nn::fs::SaveDataSpaceId::User, 0));
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<nn::fs::SaveDataSpaceId SpaceId>
    void MountAndUnmountSystemSaveData(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        std::unique_ptr<char[]> mountName(new char[16]);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(MakeMountName(mountName.get(), 16, pTestArgument->threadId));

        for( int i = 0; i < 4; ++i )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

            nn::Result result = nn::fs::MountSystemSaveData(mountName.get(), SpaceId, pTestArgument->saveDataId);
            NNT_ASSERT_RESULT_SUCCESS(result);

            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

            nn::fs::Unmount(mountName.get());
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    template<nn::fs::SaveDataSpaceId SpaceId>
    void GetSaveDataOwnerId(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pTestArgument);

        for( int i = 0; i < 4; ++i )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

            nn::Bit64 ownerId;
            NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSaveDataOwnerId(&ownerId, SpaceId, pTestArgument->saveDataId));
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }

    void MountAndUnmountTemporaryStorage(TestArgument* pTestArgument) NN_NOEXCEPT
    {
        std::unique_ptr<char[]> mountName(new char[16]);
        NNT_FS_ASSERT_NO_FATAL_FAILURE(MakeMountName(mountName.get(), 16, pTestArgument->threadId));

        for( int i = 0; i < 4; ++i )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

            nn::Result result = nn::fs::MountTemporaryStorage(mountName.get());
            if( !nn::fs::ResultTargetLocked::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            if( result.IsSuccess() )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(10));

                nn::fs::Unmount(mountName.get());
            }
        }

        IncrementSuccessCount(pTestArgument->threadId);
    }
}

class TestSaveDataRaceCondition : public ::testing::Test
{
public:
    typedef void (*TestFunction)(TestArgument* pTestArgument);

public:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::fs::util::DeleteAllTestSaveData();
        m_IsRealUserIdRequired = false;
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nnt::fs::util::DeleteAllTestSaveData();
    }

    // セーブデータを SaveDataCount 個作成し、
    // マルチスレッドで各々のセーブデータに対し function1 と function2 を同時実行します。
    // この処理を TestProcessCount 回繰り返します。
    void ExecuteRaceCondition(
        TestFunction function1,
        TestFunction function2
    ) NN_NOEXCEPT
    {
        ExecuteRaceCondition<>(
            [](TestArgument*) NN_NOEXCEPT{}, function1,
            [](TestArgument*) NN_NOEXCEPT{}, function2
        );
    }

    template<typename PreRun1, typename PreRun2>
    void ExecuteRaceCondition(
             PreRun1 preRun1, TestFunction function1,
             PreRun2 preRun2, TestFunction function2
         ) NN_NOEXCEPT
    {
        TestThreadArgument arguments[ThreadCount];

        for( int i = 0; i < ThreadCount; ++i )
        {
            arguments[i].SetTestFunction(i < SaveDataCount ? function1 : function2);
            arguments[i].GetTestArgument()->threadId = i;
        }
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        g_SuccessCount1 = 0;
        g_SuccessCount2 = 0;

        // マウントと拡張の同時実行を TestProcessCount 回行う
        for( int i = 0; i < g_TestProcessCount; ++i )
        {
            for( int j = 0; j < SaveDataCount; ++j )
            {
                int threadId1 = j;
                int threadId2 = j + SaveDataCount;

                if( m_IsRealUserIdRequired )
                {
                    arguments[threadId1].GetTestArgument()->userId = g_Uids[threadId1];
                    arguments[threadId2].GetTestArgument()->userId = g_Uids[threadId2];
                }
                else
                {
                    nn::account::Uid userId = { { 0, static_cast<nn::Bit64>(i + j) + 1 } };
                    arguments[threadId1].GetTestArgument()->userId = arguments[threadId2].GetTestArgument()->userId = userId;
                }

                arguments[threadId1].GetTestArgument()->saveDataSpaceId = nn::fs::SaveDataSpaceId::User;
                arguments[threadId1].GetTestArgument()->saveDataId = 0;
                arguments[threadId2].GetTestArgument()->saveDataSpaceId = nn::fs::SaveDataSpaceId::User;
                arguments[threadId2].GetTestArgument()->saveDataId = 0;

                NNT_FS_ASSERT_NO_FATAL_FAILURE(preRun1(arguments[threadId1].GetTestArgument()));
                NNT_FS_ASSERT_NO_FATAL_FAILURE(preRun2(arguments[threadId2].GetTestArgument()));

                arguments[threadId1].GetTestArgument()->waitTime
                    = std::uniform_int_distribution<int64_t>(
                          MinWaitTimeBeforeProcess,
                          MaxWaitTimeBeforeProcess
                      )(random);
                arguments[threadId2].GetTestArgument()->waitTime
                    = std::uniform_int_distribution<int64_t>(
                          MinWaitTimeBeforeProcess,
                          MaxWaitTimeBeforeProcess
                      )(random);

                NNT_ASSERT_RESULT_SUCCESS(
                    nn::os::CreateThread(
                        &g_Threads[threadId1],
                        OnThread,
                        &arguments[threadId1],
                        g_StackBuffer + StackSize * (threadId1),
                        StackSize,
                        nn::os::DefaultThreadPriority,
                        0
                    )
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    nn::os::CreateThread(
                        &g_Threads[threadId2],
                        OnThread,
                        &arguments[threadId2],
                        g_StackBuffer + StackSize * (threadId2),
                        StackSize,
                        nn::os::DefaultThreadPriority,
                        1
                    )
                );
            }

            for( auto& thread : g_Threads )
            {
                nn::os::StartThread(&thread);
            }
            for( auto& thread : g_Threads )
            {
                nn::os::WaitThread(&thread);
                nn::os::DestroyThread(&thread);
            }

            // 作成したセーブデータを削除する
            for( int j = 0; j < SaveDataCount; ++j )
            {
                (void)nn::fs::DeleteSaveData(
                    arguments[j].GetTestArgument()->saveDataSpaceId,
                    arguments[j].GetTestArgument()->saveDataId
                );
            }
            {
                int userCount = 0;
                NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&userCount, g_Uids, nn::account::UserCountMax));
                for( int userIndex = 0; userIndex < userCount; ++userIndex )
                {
                    nn::util::optional<nn::fs::SaveDataInfo> info;
                    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [=](nn::fs::SaveDataInfo& info_) NN_NOEXCEPT -> bool
                        {
                            return info_.applicationId == nnt::fs::util::ApplicationId
                                && info_.saveDataUserId == nn::fs::ConvertAccountUidToFsUserId(g_Uids[userIndex]);
                        });
                    if( info )
                    {
                        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteSaveData(info->saveDataId));
                    }
                }
            }
            nnt::fs::util::DeleteAllTestSaveData();

            // テストが予期せぬ失敗で終了していたら、継続しない
            if( ::testing::Test::HasFatalFailure() )
            {
                return;
            }
        }
    } // NOLINT(impl/function_size)

protected:
    void CheckAccount() NN_NOEXCEPT
    {
        int userCount = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::account::ListAllUsers(&userCount, g_Uids, nn::account::UserCountMax));
        ASSERT_EQ(nn::account::UserCountMax, userCount);

        // nmeta でユーザーアカウントセーブデータの設定を忘れていても EnsureSaveData は成功する
        // ので作れることを確認しておく
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(g_Uids[0]));
        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [=](nn::fs::SaveDataInfo& info_) NN_NOEXCEPT -> bool
            {
                return info_.applicationId == nnt::fs::util::ApplicationId
                    && info_.saveDataUserId == nn::fs::ConvertAccountUidToFsUserId(g_Uids[0]);
            });
        ASSERT_TRUE(info);

        m_IsRealUserIdRequired = true;
    }

    static void CreateBaseSaveData(TestArgument* pArgument) NN_NOEXCEPT
    {
        auto result = nn::fs::CreateSaveData(
            nnt::fs::util::ApplicationId,
            pArgument->GetFsUid(),
            0,
            DefaultSaveDataSize,
            DefaultJournalSize,
            0
        );
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }

        // UserId → SaveDataId のリストをここで作成しておく
        std::unique_ptr<nn::fs::SaveDataIterator> iter;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenSaveDataIterator(&iter, nn::fs::SaveDataSpaceId::User));
        std::unique_ptr<nn::fs::SaveDataInfo[]> infoList(new nn::fs::SaveDataInfo[SaveDataCount]);
        int64_t readCount = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(iter->ReadSaveDataInfo(&readCount, infoList.get(), SaveDataCount));

        for( int i = 0; i < readCount; ++i )
        {
            if( infoList[i].saveDataUserId == pArgument->GetFsUid() )
            {
                pArgument->saveDataSpaceId = nn::fs::SaveDataSpaceId::User;
                pArgument->saveDataId = infoList[i].saveDataId;
                break;
            }
        }
    }

private:
    class TestThreadArgument
    {
    public:
        TestThreadArgument() NN_NOEXCEPT
            : m_pTestFunction(nullptr)
        {
        }

        void SetTestFunction(TestFunction pFunction) NN_NOEXCEPT
        {
            m_pTestFunction = pFunction;
        }

        TestArgument* GetTestArgument() NN_NOEXCEPT
        {
            return &m_TestArgument;
        }

        const TestFunction GetTestFunction() const NN_NOEXCEPT
        {
            return m_pTestFunction;
        }

    private:
        TestFunction m_pTestFunction;
        TestArgument m_TestArgument;
    };

private:
    static void OnThread(void* pThreadParam) NN_NOEXCEPT
    {
        auto pTestThreadArgument = reinterpret_cast<TestThreadArgument*>(pThreadParam);
        nn::fs::SetPriorityRawOnCurrentThread(static_cast<nn::fs::PriorityRaw>(pTestThreadArgument->GetTestArgument()->threadId % 4));

        pTestThreadArgument->GetTestFunction()(pTestThreadArgument->GetTestArgument());
    }

private:
    bool m_IsRealUserIdRequired;
};

typedef TestSaveDataRaceCondition TestCacheStorageRaceCondition;

TEST_F(TestSaveDataRaceCondition, CreateVsDelete)
{
    // ResultSuccess::Includes は無いので、DeleteSaveData のテンプレート引数には適当なリザルトを入れておく
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        CreateExistSaveData<nn::fs::ResultPathAlreadyExists>,
        CreateBaseSaveData,
        DeleteSaveData<nn::fs::ResultUnknown>
    ));
    CheckSuccessCount("Create", "Delete", ResultType::AlreadySuccess);
}

TEST_F(TestSaveDataRaceCondition, MountVsMount)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        MountSaveData<nn::fs::ResultTargetLocked>,
        CreateBaseSaveData,
        MountSaveData<nn::fs::ResultTargetLocked>
    ));
    CheckSuccessCount("Mount", "Mount", ResultType::Conflicting);
}

TEST_F(TestSaveDataRaceCondition, MountVsExtend)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        MountSaveData<nn::fs::ResultSaveDataExtending>,
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetLocked>
    ));
    CheckSuccessCount("Mount ", "Extend", ResultType::Normal);
}

TEST_F(TestSaveDataRaceCondition, MountVsDelete)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        MountSaveData<nn::fs::ResultTargetNotFound>,
        CreateBaseSaveData,
        DeleteSaveData<nn::fs::ResultTargetLocked>
    ));
    CheckSuccessCount("Mount ", "Delete", ResultType::Conflicting);
}

TEST_F(TestSaveDataRaceCondition, ExtendVsUnmount)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetLocked>,
        [](TestArgument* pTestArgument) NN_NOEXCEPT
        {
            CreateBaseSaveData(pTestArgument);

            char mountName[16];
            NNT_FS_ASSERT_NO_FATAL_FAILURE(MakeMountName(mountName, 16, pTestArgument->threadId));
            NNT_ASSERT_RESULT_SUCCESS(
                nn::fs::MountSaveData(mountName, pTestArgument->GetFsUid())
            );
        },
        UnmountSaveData
    ));
    CheckSuccessCount("Extend ", "Unmount", ResultType::AlreadySuccess);
}

TEST_F(TestSaveDataRaceCondition, ExtendVsExtend)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetLocked>,
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetLocked>
    ));
    CheckSuccessCount("Extend", "Extend", ResultType::Normal);
}

TEST_F(TestSaveDataRaceCondition, ExtendVsDelete)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetNotFound, nn::fs::ResultPathNotFound>,
        CreateBaseSaveData,
        DeleteSaveData<nn::fs::ResultTargetLocked>
    ));
    CheckSuccessCount("Extend", "Delete", ResultType::Normal);
}

TEST_F(TestSaveDataRaceCondition, DeleteVsDelete)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        DeleteSaveData<nn::fs::ResultTargetNotFound>,
        CreateBaseSaveData,
        DeleteSaveData<nn::fs::ResultTargetNotFound>
    ));
    CheckSuccessCount("Delete", "Delete", ResultType::Conflicting);
}

TEST_F(TestSaveDataRaceCondition, ExtendVsIsExisting)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateBaseSaveData,
        ExtendSaveData<nn::fs::ResultTargetLocked>,
        CreateBaseSaveData,
        IsSaveDataExisting
    ));
    CheckSuccessCount("Extend", "IsExisting", ResultType::AlreadySuccess);
}

TEST_F(TestSaveDataRaceCondition, CreateAndDeleteVsEnsure)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckAccount());

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateAndDeleteSaveData<nn::fs::ResultPathAlreadyExists, 4>,
        EnsureSaveData
    ));
    CheckSuccessCount("CreateAndDeleteSaveData", "EnsureSaveData", ResultType::AlreadySuccess);
}

TEST_F(TestCacheStorageRaceCondition, CreateAndDeleteVsEnsureSaveData)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckAccount());

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        CreateAndDeleteCacheStorage,
        EnsureSaveData
    ));
    CheckSuccessCount("CreateAndDeleteCacheStorage", "EnsureSaveData", ResultType::Success);
}

TEST_F(TestCacheStorageRaceCondition, AccessCacheStorageListVsEnsureSaveData)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckAccount());

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        AccessCacheStorageList<nn::fs::ResultInvalidHandle>,
        EnsureSaveData
    ));
    CheckSuccessCount("AccessCacheStorageList", "EnsureSaveData", ResultType::Success);
}

TEST_F(TestSaveDataRaceCondition, IsSaveDataExistingAndGetSaveDataSizeAndSetSaveDataFlagsOnDifferentSaveData)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckAccount());

    auto createSaveData = [](TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(pTestArgument->userId));

        nn::util::optional<nn::fs::SaveDataInfo> saveDataInfo;
        nnt::fs::util::FindSaveData(&saveDataInfo, nn::fs::SaveDataSpaceId::User, [=](const nn::fs::SaveDataInfo& info_) NN_NOEXCEPT
            {
                return info_.saveDataUserId == pTestArgument->GetFsUid()
                    && info_.applicationId == nnt::fs::util::ApplicationId;
            });
        ASSERT_TRUE(saveDataInfo);

        pTestArgument->saveDataSpaceId = nn::fs::SaveDataSpaceId::User;
        pTestArgument->saveDataId = saveDataInfo->saveDataId;
    };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        createSaveData,
        IsSaveDataExistingAndGetSaveDataSize,
        createSaveData,
        IsSaveDataExistingAndSetSaveDataFlags
    ));
    CheckSuccessCount(
        "IsSaveDataExistingAndGetSaveDataSize",
        "IsSaveDataExistingAndSetSaveDataFlags",
        ResultType::Success
    );
}

TEST_F(TestSaveDataRaceCondition, GetSaveDataOwnerIdAndMountSdSystemSaveData)
{
    constexpr auto SpaceId = nn::fs::SaveDataSpaceId::SdSystem;

    auto createSystemSaveData = [](TestArgument* pTestArgument) NN_NOEXCEPT
    {
        pTestArgument->saveDataSpaceId = SpaceId;
        pTestArgument->saveDataId = 0x8000000000001000ULL + pTestArgument->threadId % SaveDataCount;
        auto result = nn::fs::CreateSystemSaveData(SpaceId, pTestArgument->saveDataId, nnt::fs::util::ApplicationId.value, DefaultSaveDataSize, DefaultJournalSize, 0);
        if( !nn::fs::ResultPathAlreadyExists::Includes(result) )
        {
            NNT_ASSERT_RESULT_SUCCESS(result);
        }
    };

    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        createSystemSaveData,
        GetSaveDataOwnerId<SpaceId>,
        createSystemSaveData,
        MountAndUnmountSystemSaveData<SpaceId>
    ));
    CheckSuccessCount(
        "GetSaveDataOwnerId",
        "MountAndUnmountSystemSaveData",
        ResultType::Success
    );
}

TEST_F(TestSaveDataRaceCondition, GetSaveDataOwnerIdAndMountTemporaryStorage)
{
    NNT_FS_ASSERT_NO_FATAL_FAILURE(CheckAccount());

    auto createTemporaryStorage = [](TestArgument* pTestArgument) NN_NOEXCEPT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::EnsureSaveData(pTestArgument->userId));

        nn::util::optional<nn::fs::SaveDataInfo> info;
        nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::Temporary, [](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
            {
                return i.applicationId == nnt::fs::util::ApplicationId;
            });
        ASSERT_TRUE(info);
        pTestArgument->saveDataSpaceId = nn::fs::SaveDataSpaceId::Temporary;
        pTestArgument->saveDataId = info->saveDataId;
    };


    NNT_FS_ASSERT_NO_FATAL_FAILURE(ExecuteRaceCondition(
        createTemporaryStorage,
        GetSaveDataOwnerId<nn::fs::SaveDataSpaceId::Temporary>,
        createTemporaryStorage,
        MountAndUnmountTemporaryStorage
    ));
    CheckSuccessCount(
        "GetSaveDataOwnerId",
        "MountAndUnmountTemporaryStorage",
        ResultType::Success
    );
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    for( int argumentIndex = 0; argumentIndex < argc; ++argumentIndex )
    {
        if( std::strcmp(argv[argumentIndex], "--test-process-count") == 0 )
        {
            ++argumentIndex;
            if( argumentIndex >= argc )
            {
                NN_LOG("test process count is not specified.\n");
                nnt::Exit(1);
            }

            g_TestProcessCount = std::atoi(argv[argumentIndex]);
        }
    }

    nn::fs::SetEnabledAutoAbort(false);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::account::InitializeForSystemService();

    auto result = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(result);
}
