﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataManagementPrivate.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

TEST(SaveDataDelayedDeletion, Stress)
{
    const int IdCount = 128;
    const int64_t IdBase = 0x8000000000003000ULL;

    SystemSaveDataId ids[IdCount];
    for(int i = 0; i < IdCount; i++)
    {
        ids[i] = IdBase + i;
    }

    // 前回試行の結果確認: 全存在か全削除である
    {
        Vector<SaveDataInfo> infos;

        FindSaveData(&infos, SaveDataSpaceId::System, [&](const SaveDataInfo& i) NN_NOEXCEPT
        {
            for(auto id : ids)
            {
                if(i.systemSaveDataId == id)
                {
                    NN_LOG("Found: %llx\n", id);
                    return true;
                }
            }
            return false;
        });

        EXPECT_TRUE(infos.empty() || infos.size() == IdCount);
    }


    const int64_t SaveDataSize = 32 * 1024;
    const int64_t SaveDataJournalSize = 32 * 1024;

    // 削除
    for(auto id : ids)
    {
        NN_LOG("Delete: %llx\n", id);
        DeleteSaveData(id);
    }

    // 作成
    for(auto id : ids)
    {
        NN_LOG("Create: %llx\n", id);
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(id, SaveDataSize, SaveDataJournalSize, 0));
    }

    // (一部)マウント中に遅延削除登録
    for(int i = 0; i < 10; i++)
    {
        char mountName[128];
        nn::util::SNPrintf(mountName, sizeof(mountName), "mount%d", i);
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData(mountName, IdBase + i));
    }

    // 削除登録
    NNT_EXPECT_RESULT_SUCCESS(RegisterSaveDataAtomicDeletion(ids, IdCount));
}

}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    nn::fs::SetEnabledAutoAbort(false);

    ::testing::InitGoogleTest(&argc, argv);

    auto ret = RUN_ALL_TESTS();

    nnt::Exit(ret);
}
