﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/fs/fs_CacheStoragePrivate.h>
#include <nn/fs/fs_QueryRange.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/ns/ns_AppletLauncherApi.h>
#include <nn/ns/ns_InitializationApi.h>

#if !defined( NNT_FS_INTEGRATION_QUERYRANGE_SYSTEM_PROGRAM )
#include <nn/fs/fs_AddOnContent.h>
#endif

#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

enum class BootType
{
    SystemProgram,
    Host,
    Nspd,
    Nand,
    SdCard,
    GameCard,
    NandPatch,
    SdCardPatch,
    GameCardPatch,

    Count
};

BootType g_BootType;

int32_t GetExpectAesCtrKeyTypeFlag(BootType bootType, nn::fs::SpeedEmulationMode speedEmulationMode) NN_NOEXCEPT
{
    static const int SpeedEmulationModeCount = 4; // SpeedEmulationMode のモードの総数
    NN_ASSERT_RANGE(static_cast<int>(speedEmulationMode), 0, SpeedEmulationModeCount);
    NN_ASSERT_RANGE(static_cast<int>(bootType), 0, static_cast<int>(BootType::Count));

    static const int32_t InternalSoftware = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);
    static const int32_t InternalHardware = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes);
    static const int32_t ExternalHardware = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::ExternalKeyForHardwareAes);

    static const int32_t KeyTypeTable[SpeedEmulationModeCount][static_cast<int>(BootType::Count)] =
    {
    //            SystemProgram     Host              Nspd  Nand              SdCard            GameCard          NandPatch         SdCardPatch       GameCardPatch
    /* None   */ {InternalSoftware, InternalHardware, 0,    InternalHardware, InternalHardware, InternalSoftware, ExternalHardware, ExternalHardware, ExternalHardware},
    /* Faster */ {InternalSoftware, InternalSoftware, 0,    InternalSoftware, InternalSoftware, InternalSoftware, ExternalHardware, ExternalHardware, ExternalHardware},
    /* Slower */ {InternalSoftware, InternalHardware, 0,    InternalHardware, InternalHardware, InternalSoftware, ExternalHardware, ExternalHardware, ExternalHardware},
    /* Random */ {InternalSoftware, InternalSoftware, 0,    InternalSoftware, InternalSoftware, InternalSoftware, ExternalHardware, ExternalHardware, ExternalHardware},
    };

    return KeyTypeTable[static_cast<int>(speedEmulationMode)][static_cast<int>(bootType)];
}

}

class QueryRangeTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetSpeedEmulationMode(&m_DefaultMode));
    }

    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(m_DefaultMode));
    }

private:
    nn::fs::SpeedEmulationMode m_DefaultMode;
};

TEST_F(QueryRangeTest, Rom)
{
    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&size));

    auto buffer = nnt::fs::util::AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountRom("rom", buffer.get(), size));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("rom");
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "rom:/file0.dat", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(file);
    };

    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, file));

    nn::fs::QueryRangeInfo info;

    static const nn::fs::SpeedEmulationMode SpeedEmulationModes[] =
    {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Random
    };
    for( auto mode : SpeedEmulationModes )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));
        NNT_ASSERT_RESULT_SUCCESS(QueryRange(&info, file, 0, fileSize));

        EXPECT_EQ(GetExpectAesCtrKeyTypeFlag(g_BootType, mode), info.aesCtrKeyTypeFlag);

        int32_t expectSpeedEmulationTypeFlag = 0;
        if( g_BootType == BootType::Nand
            || g_BootType == BootType::NandPatch
            || g_BootType == BootType::SdCard
            || g_BootType == BootType::SdCardPatch )
        {
            expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);
        }
        else if( g_BootType == BootType::SystemProgram )
        {
            expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);
        }

        if( expectSpeedEmulationTypeFlag != 0 && mode != nn::fs::SpeedEmulationMode::None )
        {
            expectSpeedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }

        EXPECT_EQ(expectSpeedEmulationTypeFlag, info.speedEmulationTypeFlag);
    }
}

TEST_F(QueryRangeTest, SaveData)
{
    nnt::fs::util::CreateAndMountSystemSaveData("save", 0x8000000000004000);
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("save");
        nnt::fs::util::DeleteAllTestSaveData();
    };

    static const auto FileSize = 256;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("save:/file0.dat", FileSize));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile("save:/file0.dat"));
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "save:/file0.dat", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    nn::fs::SpeedEmulationMode speedEmulationModes[] =
    {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Random
    };

    for( auto mode : speedEmulationModes )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));

        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryRange(&info, file, 0, FileSize));

        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);

        int32_t expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled);
        if( mode != nn::fs::SpeedEmulationMode::None )
        {
            expectSpeedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }
        EXPECT_EQ(expectSpeedEmulationTypeFlag, info.speedEmulationTypeFlag);
    }
}

TEST_F(QueryRangeTest, SdCard)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountSdCardForDebug("sd"));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("sd");
    };

    static const auto FileSize = 256;
    char filePath[16];

    for( int i = 0; i < 1000; ++i )
    {
        nn::util::SNPrintf(filePath, sizeof(filePath), "sd:/test%03d", i);

        auto result = nn::fs::CreateFile(filePath, FileSize);
        if( result.IsSuccess() )
        {
            break;
        }
        if( nn::fs::ResultPathAlreadyExists::Includes(result) && i < 999 )
        {
            continue;
        }

        NNT_ASSERT_RESULT_SUCCESS(result);
    }
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile(filePath));
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    nn::fs::SpeedEmulationMode speedEmulationModes[] =
    {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Random
    };

    for( auto mode : speedEmulationModes )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));

        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryRange(&info, file, 0, FileSize));
        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);

        int32_t expectSpeedEmulationFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled);
        if( mode != nn::fs::SpeedEmulationMode::None )
        {
            expectSpeedEmulationFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }
        EXPECT_EQ(expectSpeedEmulationFlag, info.speedEmulationTypeFlag);
    }
}

TEST_F(QueryRangeTest, CacheStorage)
{
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateCacheStorage(nnt::fs::util::ApplicationId, 0, 1024 * 1024, 1024 * 1024, 0));
    NN_UTIL_SCOPE_EXIT
    {
        nnt::fs::util::DeleteAllTestSaveData();
    };

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCacheStorage("cache", nnt::fs::util::ApplicationId));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("cache");
    };

    static const auto FileSize = 256;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile("cache:/file0.dat", FileSize));
    NN_UTIL_SCOPE_EXIT
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::DeleteFile("cache:/file0.dat"));
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "cache:/file0.dat", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    nn::fs::SpeedEmulationMode speedEmulationModes[] =
    {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Random
    };

    for( auto mode : speedEmulationModes )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));

        nn::fs::QueryRangeInfo info;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryRange(&info, file, 0, FileSize));

        EXPECT_EQ(0, info.aesCtrKeyTypeFlag);

        int32_t expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                             | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                             | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);
        if( mode != nn::fs::SpeedEmulationMode::None )
        {
            expectSpeedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
        }
        EXPECT_EQ(expectSpeedEmulationTypeFlag, info.speedEmulationTypeFlag);
    }
}

#if !defined( NNT_FS_INTEGRATION_QUERYRANGE_SYSTEM_PROGRAM )
TEST_F(QueryRangeTest, Aoc)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(g_BootType != BootType::Host && g_BootType != BootType::Nspd);

    const nn::aoc::AddOnContentIndex AocIndex = 1;

    size_t size;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::QueryMountAddOnContentCacheSize(&size, AocIndex));

    auto buffer = nnt::fs::util::AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountAddOnContent("aoc", AocIndex, buffer.get(), size));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount("aoc");
    };

    nn::fs::FileHandle file;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&file, "aoc:/file0.dat", nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        CloseFile(file);
    };

    int64_t fileSize = 0;
    NNT_ASSERT_RESULT_SUCCESS(GetFileSize(&fileSize, file));

    nn::fs::QueryRangeInfo info;

    static const nn::fs::SpeedEmulationMode SpeedEmulationModes[] =
    {
        nn::fs::SpeedEmulationMode::None,
        nn::fs::SpeedEmulationMode::Faster,
        nn::fs::SpeedEmulationMode::Slower,
        nn::fs::SpeedEmulationMode::Random
    };
    for( auto mode : SpeedEmulationModes )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::SetSpeedEmulationMode(mode));
        NNT_ASSERT_RESULT_SUCCESS(QueryRange(&info, file, 0, fileSize));

        int32_t expectAesCtrKeyFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes);
        if( g_BootType == BootType::GameCard
            || g_BootType == BootType::GameCardPatch
            || mode == nn::fs::SpeedEmulationMode::Faster
            || mode == nn::fs::SpeedEmulationMode::Random )
        {
            expectAesCtrKeyFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForSoftwareAes);
        }
        EXPECT_EQ(expectAesCtrKeyFlag, info.aesCtrKeyTypeFlag);

        int32_t expectSpeedEmulationTypeFlag = 0;
        if( g_BootType == BootType::Nand
            || g_BootType == BootType::NandPatch
            || g_BootType == BootType::SdCard
            || g_BootType == BootType::SdCardPatch )
        {
            expectSpeedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

            if( mode != nn::fs::SpeedEmulationMode::None )
            {
                expectSpeedEmulationTypeFlag |= static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::StorageSpeedEmulationEnabled);
            }
        }

        EXPECT_EQ(expectSpeedEmulationTypeFlag, info.speedEmulationTypeFlag);
    }
}

TEST_F(QueryRangeTest, LaunchSystemProgram)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(g_BootType == BootType::SystemProgram);

    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::ns::Finalize();
    };

    nn::os::ProcessId processId;
    nn::ncm::SystemProgramId sysId = {0x0100000000001FFF};
    NNT_ASSERT_RESULT_SUCCESS(nn::ns::LaunchLibraryApplet(&processId, sysId));

    nn::os::SystemEvent event;
    nn::ns::GetApplicationShellEvent(&event);
    event.Wait();

    const auto eventInfoArrayLength = 256;
    std::unique_ptr<nn::ns::ApplicationShellEventInfo[]> eventInfoArray(new nn::ns::ApplicationShellEventInfo[eventInfoArrayLength]);
    ASSERT_TRUE(eventInfoArray);
    {
        const auto infoCount = nn::ns::PopApplicationShellEventInfo(eventInfoArray.get(), eventInfoArrayLength);
        for( auto i = 0; i < infoCount; ++i )
        {
            if( eventInfoArray[i].processId == processId )
            {
                ASSERT_EQ(nn::ns::ApplicationShellEvent::Exit, eventInfoArray[i].event);
            }
        }
    }
}
#endif

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

#if defined( NNT_FS_INTEGRATION_QUERYRANGE_SYSTEM_PROGRAM )
    g_BootType = BootType::SystemProgram;
#else
    // google test の引数が argv から取り除かれて null になっているため
    // argc を実際の argv の個数に合わせる
    int argcRemoved = 0;
    for( int i = 0; i < argc && argv[i] != nullptr; ++i )
    {
        ++argcRemoved;
    }

    if( argcRemoved < 2 )
    {
        NN_LOG("Usage: *.nsp <Host or Nspd, SystemProgram, Nand, SdCard, GameCard, NandPatch, SdCardPatch, GameCardPatch>\n");
        return;
    }

    if( strncmp(argv[1], "Host", 4) == 0 )
    {
        g_BootType = BootType::Host;
    }
    else if( strncmp(argv[1], "Nspd", 4) == 0 )
    {
        g_BootType = BootType::Nspd;
    }
    else if( strncmp(argv[1], "SystemProgram", 13) == 0 )
    {
        g_BootType = BootType::SystemProgram;
    }
    else if( strncmp(argv[1], "NandPatch", 9) == 0 )
    {
        g_BootType = BootType::NandPatch;
    }
    else if( strncmp(argv[1], "Nand", 4) == 0 )
    {
        g_BootType = BootType::Nand;
    }
    else if( strncmp(argv[1], "SdCardPatch", 11) == 0 )
    {
        g_BootType = BootType::SdCardPatch;
    }
    else if( strncmp(argv[1], "SdCard", 6) == 0 )
    {
        g_BootType = BootType::SdCard;
    }
    else if( strncmp(argv[1], "GameCardPatch", 13) == 0 )
    {
        g_BootType = BootType::GameCardPatch;
    }
    else if( strncmp(argv[1], "GameCard", 8) == 0 )
    {
        g_BootType = BootType::GameCard;
    }
    else
    {
        NN_ABORT("Invalid argument: %s\n", argv[1]);
    }

#endif

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
