﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace{

const char MountName[]          = "perftest";
const char TestFileName[]       = "0";
const int  LoopCount = 10;

class PerformanceTestWithMount : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
protected:
    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        CreateWorkRootPath();
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteWorkRootPath();
    }
};

struct ParamsStructure{
    size_t saveSize;
    size_t fileSize;
};

class PerformanceTestWithMountSave : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture,
    public ::testing::WithParamInterface<ParamsStructure>
{
};
}

//!< @brief Mount, Unmount の実行時間計測
TEST_F(PerformanceTestWithMount, MountHost)
{
    // Mount、Unmount の実行時間計測
    NN_LOG("\nMount Performance Test\n");

    TimeCount timeCountMount;
    TimeCount timeCountUnmount;

    for(int i = 0; i < LoopCount; i++)
    {
        NNT_FS_SCOPED_TRACE("loop count: %d\n", i);
        timeCountMount.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(MountHost(MountName, GetWorkRootPath().c_str()));
        timeCountMount.StopTime();

        timeCountUnmount.StartTime();
        Unmount(MountName);
        timeCountUnmount.StopTime();
     }
     NN_LOG("\nMount (%d counts)\n", LoopCount);
     timeCountMount.ViewTime(timeCountMount.GetTotalTime());
     NN_LOG("\nUnmount (%d counts)\n", LoopCount);
     timeCountUnmount.ViewTime(timeCountUnmount.GetTotalTime());
}

//!< @brief MountHostRoot, UnmountHostRoot の実行時間計測
TEST_F(PerformanceTestWithMount, MountHostRoot)
{
    // MountHostRoot、UnmountHostRoot の実行時間計測
    NN_LOG("\nMountHostRoot Performance Test\n");

    TimeCount timeCountMountHostRoot;
    TimeCount timeCountUnmountHostRoot;

    for(int i = 0; i < LoopCount; i++)
    {
        NNT_FS_SCOPED_TRACE("loop count: %d\n", i);
        timeCountMountHostRoot.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(MountHostRoot());
        timeCountMountHostRoot.StopTime();

        timeCountUnmountHostRoot.StartTime();
        UnmountHostRoot();
        timeCountUnmountHostRoot.StopTime();
    }
    NN_LOG("\nMountHostRoot (%d counts)\n", LoopCount);
    timeCountMountHostRoot.ViewTime(timeCountMountHostRoot.GetTotalTime());
    NN_LOG("\nUnmountHostRoot (%d counts)\n", LoopCount);
    timeCountUnmountHostRoot.ViewTime(timeCountUnmountHostRoot.GetTotalTime());
}

TEST_P(PerformanceTestWithMountSave, MountSystemSaveData)
{
    SystemSaveDataId id = 0x8000000000000003;
    String filePath = String(MountName) + ":/" + TestFileName;
    ParamsStructure saveParam = GetParam();

    nnt::fs::util::DeleteAllTestSaveData();

    // MountSystemSaveData、CommitSaveData、Unmount の実行時間計測
    NN_LOG("\nMountSystemSaveData Performance Test (saveSize=%d, fileSize=%d)\n", saveParam.saveSize, saveParam.fileSize);

    TimeCount timeCountCreateSave;
    TimeCount timeCountMount;
    TimeCount timeCountWriteFile;
    TimeCount timeCountCommit;
    TimeCount timeCountUnmount;
    TimeCount timeCountDeleteSave;

    for(int i = 0; i < LoopCount; i++)
    {
        NNT_FS_SCOPED_TRACE("loop count: %d\n", i);

        timeCountCreateSave.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(CreateSystemSaveData(id, saveParam.saveSize, saveParam.saveSize, 0));
        timeCountCreateSave.StopTime();

        timeCountMount.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(MountSystemSaveData(MountName, id));
        timeCountMount.StopTime();

        NNT_EXPECT_RESULT_SUCCESS(CreateFile(filePath.c_str(), saveParam.fileSize));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(MountName));

        timeCountWriteFile.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::WriteFileWith32BitCount(filePath.c_str(), saveParam.fileSize, 0));
        timeCountWriteFile.StopTime();

        timeCountCommit.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(MountName));
        timeCountCommit.StopTime();

        NNT_EXPECT_RESULT_SUCCESS(DeleteFile(filePath.c_str()));
        NNT_EXPECT_RESULT_SUCCESS(CommitSaveData(MountName));

        timeCountUnmount.StartTime();
        Unmount(MountName);
        timeCountUnmount.StopTime();

        timeCountDeleteSave.StartTime();
        nnt::fs::util::DeleteAllTestSaveData();
        timeCountDeleteSave.StopTime();
    }
    NN_LOG("\nCreateSystemSaveData    (%d counts)\n", LoopCount);
    timeCountCreateSave.ViewTime(timeCountCreateSave.GetTotalTime());
    NN_LOG("\nMountSystemSaveData     (%d counts)\n", LoopCount);
    timeCountMount.ViewTime(timeCountMount.GetTotalTime());
    NN_LOG("\nWriteFileWith32BitCount(%d counts)\n", LoopCount);
    timeCountWriteFile.ViewTime(timeCountWriteFile.GetTotalTime());
    NN_LOG("\nCommitSaveData          (%d counts)\n", LoopCount);
    timeCountCommit.ViewTime(timeCountCommit.GetTotalTime());
    NN_LOG("\nUnmount                 (%d counts)\n", LoopCount);
    timeCountUnmount.ViewTime(timeCountUnmount.GetTotalTime());
    NN_LOG("\nDeleteSaveData          (%d counts)\n", LoopCount);
    timeCountDeleteSave.ViewTime(timeCountDeleteSave.GetTotalTime());
}

ParamsStructure saveSizeParam[] = {
    /* saveSize,  fileSize */
    { 0x00100000, 0x00008000 }, //  1MB,  32kB
    { 0x00100000, 0x00080000 }, //  1MB, 512kB
    { 0x02100000, 0x00008000 }, // 33MB,  32kB
    { 0x02100000, 0x00080000 }, // 33MB, 512kB
    { 0x02100000, 0x02000000 }  // 33MB,  32MB
};

INSTANTIATE_TEST_CASE_P(WithVariousSize,
                        PerformanceTestWithMountSave,
                        ::testing::ValuesIn(saveSizeParam));

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    if (CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
