﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_ContentStorage.h>
#include <nn/fs/fs_ImageDirectory.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fs_MmcPrivate.h>

#include <nn/util/util_FormatString.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fs;
using namespace nnt::fs::util;

namespace{

const int  MaxNameLength = 64;

const int  FileCount  = 4096;
const int  DirCount   = 256;
#if defined(NN_BUILD_CONFIG_OS_WIN32)
const int  NestCount  = 50;
#else /* !NN_BUILD_CONFIG_OS_WIN32 */
const int  NestCount  = 100;
#endif /* !NN_BUILD_CONFIG_OS_WIN32 */
const int64_t BigFileSize = 2LL * 1024 * 1024 * 1024;
const int  BigFileWorkSize = 512 * 1024;

class PerformanceTest : public ::testing::Test
{
protected:
    static char m_FsName[MaxNameLength];
    static char m_MountName[MaxNameLength];
    static char m_FlatTestPath[MaxNameLength];
    static char m_NestTestPath[MaxNameLength];
    static char m_BigFileTestPath[MaxNameLength];
    static int m_FileCount;
    static int m_DirCount;
    static int m_NestCount;
    static int64_t m_BigFileSize;

    char m_pEntryPath[FileCount][NameMaxLength];

    virtual void TestReadDirectoryFlat() NN_NOEXCEPT            { NN_LOG("<SKIP>\n"); };
    virtual void TestReadDirectoryNest() NN_NOEXCEPT            { NN_LOG("<SKIP>\n"); };
    virtual void TestGetEntryTypeFile() NN_NOEXCEPT             { NN_LOG("<SKIP>\n"); };
    virtual void TestGetEntryTypeDirectory() NN_NOEXCEPT        { NN_LOG("<SKIP>\n"); };
    virtual void TestGetDirectoryEntryCount() NN_NOEXCEPT       { NN_LOG("<SKIP>\n"); };
    virtual void TestOpenCloseFile() NN_NOEXCEPT                { NN_LOG("<SKIP>\n"); };
    virtual void TestCreateRenameDeleteFile() NN_NOEXCEPT       { NN_LOG("<SKIP>\n"); };
    virtual void TestCreateRenameDeleteDirectory() NN_NOEXCEPT  { NN_LOG("<SKIP>\n"); };
    virtual void TestReadBigFile() NN_NOEXCEPT                  { NN_LOG("<SKIP>\n"); };
    virtual void TestWriteBigFile() NN_NOEXCEPT                 { NN_LOG("<SKIP>\n"); };

    void PerfReadDirectoryFlat(const char* basePath) NN_NOEXCEPT;
    void PerfReadDirectoryNest(const char* basePath, const int nestCount) NN_NOEXCEPT;
    void PerfGetEntryType(char (*pPath)[NameMaxLength], const char* basePath, const int count, const bool isRandom, const bool isFile) NN_NOEXCEPT;
    void PerfGetDirectoryEntryCount(const char* basePath, const int fileCount, const int dirCount) NN_NOEXCEPT;
    void PerfCreateRenameDeleteFile(char (*pPath)[NameMaxLength], const int fileCount) NN_NOEXCEPT;
    void PerfCreateRenameDeleteDirectory(char (*pPath)[NameMaxLength], const int dirCount) NN_NOEXCEPT;
    void PerfOpenCloseFile(const char (*pPath)[NameMaxLength], const int fileCount, const int diffCount) NN_NOEXCEPT;
    void PerfReadFile(const char* pPath, const int64_t fileSize, const int readSize, const int adjust) NN_NOEXCEPT;
    void PerfWriteFile(const char* pPath, const int64_t fileSize, const int writeSize, const int adjust) NN_NOEXCEPT;

};
class PerformanceTestCommon : public PerformanceTest
{
protected:
    static void Initialize() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FlatTestPath, sizeof(m_FlatTestPath), "%s:/flat/", m_MountName);
        nn::util::SNPrintf(m_NestTestPath, sizeof(m_NestTestPath), "%s:/nest/", m_MountName);
        nn::util::SNPrintf(m_BigFileTestPath, sizeof(m_BigFileTestPath), "%s:/big/", m_MountName);

        m_FileCount   = FileCount;
        m_DirCount    = DirCount;
        m_NestCount   = NestCount;
        m_BigFileSize = BigFileSize;
    }

    virtual void TestReadDirectoryFlat() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestReadDirectoryNest() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestGetEntryTypeFile() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestGetEntryTypeDirectory() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestGetDirectoryEntryCount() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestOpenCloseFile() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestReadBigFile() NN_NOEXCEPT NN_OVERRIDE;
};
class PerformanceTestWritable : public PerformanceTestCommon
{
public:
    static void TearDownTestCase() NN_NOEXCEPT
    {
        DeletePerfTestDirectory();
        Unmount(m_MountName);
    }

protected:
    static char m_WritableTestPath[MaxNameLength];

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteDirectoryRecursively(m_WritableTestPath);
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(m_WritableTestPath));
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        DeleteDirectoryRecursively(m_WritableTestPath);
    }

    static void Initialize() NN_NOEXCEPT
    {
        PerformanceTestCommon::Initialize();
        nn::util::SNPrintf(m_WritableTestPath, sizeof(m_WritableTestPath), "%s:/write/", m_MountName);
    }
    static void CreatePerfTestDirectory() NN_NOEXCEPT
    {
        DeleteDirectoryRecursively(m_FlatTestPath);
        DeleteDirectoryRecursively(m_NestTestPath);
        DeleteDirectoryRecursively(m_BigFileTestPath);
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(m_FlatTestPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(m_NestTestPath));
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(m_BigFileTestPath));

        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, nullptr, m_FlatTestPath, m_FileCount, 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, nullptr, m_FlatTestPath, m_DirCount, DIRTYPE_FLAT, false, 0));
        // Host の時は作らない
        if(strcmp(m_FsName, "Host") != 0)
        {
            NNT_ASSERT_RESULT_SUCCESS(CreateTestDirectory(nullptr, nullptr, m_NestTestPath, m_NestCount, DIRTYPE_NEST, true, 0));
        }
        NNT_ASSERT_RESULT_SUCCESS(CreateTestFile(nullptr, nullptr, m_BigFileTestPath, 1, m_BigFileSize));
    }
    static void DeletePerfTestDirectory() NN_NOEXCEPT
    {
        DeleteDirectoryRecursively(m_FlatTestPath);
        DeleteDirectoryRecursively(m_NestTestPath);
        DeleteDirectoryRecursively(m_BigFileTestPath);
        DeleteDirectoryRecursively(m_WritableTestPath);
    }

    virtual void TestCreateRenameDeleteFile() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestCreateRenameDeleteDirectory() NN_NOEXCEPT NN_OVERRIDE;
    virtual void TestWriteBigFile() NN_NOEXCEPT NN_OVERRIDE;
};

class PerformanceTestHost : public PerformanceTestWritable
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "Host");
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "host");
        PerformanceTestWritable::Initialize();

        m_HostDirectory.Create();
        std::strncpy(m_HostRootPath, m_HostDirectory.GetPath().c_str(), sizeof(m_HostRootPath));
        NNT_ASSERT_RESULT_SUCCESS(MountHost(m_MountName, m_HostRootPath));
        PerformanceTestWritable::CreatePerfTestDirectory();
    }
    static void TearDownTestCase() NN_NOEXCEPT
    {
        PerformanceTestWritable::DeletePerfTestDirectory();
        Unmount(m_MountName);
        m_HostDirectory.Delete();
    }

protected:
    static nnt::fs::util::TemporaryHostDirectory m_HostDirectory;
    static char m_HostRootPath[260];
};

class PerformanceTestBis : public PerformanceTestWritable
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "Bis");
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "%s", GetBisMountName(BisPartitionId::User));
        PerformanceTestWritable::Initialize();

        NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));
        PerformanceTestWritable::CreatePerfTestDirectory();
    }
};
class PerformanceTestSd : public PerformanceTestWritable
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "SdCard");
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "sd");
        PerformanceTestWritable::Initialize();

        NNT_ASSERT_RESULT_SUCCESS(MountSdCard(m_MountName));
        PerformanceTestWritable::CreatePerfTestDirectory();
    }
};

class PerformanceTestContentStorage : public PerformanceTestWritable
{
protected:
    static void Initialize(ContentStorageId contentStorageId) NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "%s", GetContentStorageMountName(contentStorageId));
        PerformanceTestWritable::Initialize();

        NNT_ASSERT_RESULT_SUCCESS(MountContentStorage(contentStorageId));
        PerformanceTestWritable::CreatePerfTestDirectory();
    }
};
class PerformanceTestContentStorageBis : public PerformanceTestContentStorage
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "ContentStorage(User)");
        PerformanceTestContentStorage::Initialize(ContentStorageId::User);
    }
};
class PerformanceTestContentStorageSd : public PerformanceTestContentStorage
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "ContentStorage(SdCard)");
        PerformanceTestContentStorage::Initialize(ContentStorageId::SdCard);
    }
};

class PerformanceTestImageDirectory : public PerformanceTestWritable
{
protected:
    static void Initialize(ImageDirectoryId imageDirectoryId) NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "image");
        PerformanceTestWritable::Initialize();

        NNT_ASSERT_RESULT_SUCCESS(MountImageDirectory(m_MountName, imageDirectoryId));
        PerformanceTestWritable::CreatePerfTestDirectory();
    }
};
class PerformanceTestImageDirectoryNand : public PerformanceTestImageDirectory
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "ImageDirectory(Nand)");
        PerformanceTestImageDirectory::Initialize(ImageDirectoryId::Nand);
    }
};
class PerformanceTestImageDirectorySd : public PerformanceTestImageDirectory
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "ImageDirectory(SdCard)");
        PerformanceTestImageDirectory::Initialize(ImageDirectoryId::SdCard);
    }
};

class PerformanceTestSystemSave : public PerformanceTestWritable
{
public:
    static void TearDownTestCase() NN_NOEXCEPT
    {
        Unmount(m_MountName);
        Unmount(m_FlatMountName);
        Unmount(m_NestMountName);
        Unmount(m_BigFileMountName);
        DeleteAllTestSaveData();
    }

protected:
    static char m_FlatMountName[MaxNameLength];
    static char m_NestMountName[MaxNameLength];
    static char m_BigFileMountName[MaxNameLength];

    static void Initialize(SaveDataSpaceId saveDataSpaceId) NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "save");
        nn::util::SNPrintf(m_FlatMountName, sizeof(m_FlatMountName), "saveflat");
        nn::util::SNPrintf(m_NestMountName, sizeof(m_NestMountName), "savenest");
        nn::util::SNPrintf(m_BigFileMountName, sizeof(m_BigFileMountName), "savebig");
        PerformanceTestWritable::Initialize();
        m_BigFileSize = 32 * 1024 * 1024;
        nn::util::SNPrintf(m_FlatTestPath, sizeof(m_FlatTestPath), "%s:/flat/", m_FlatMountName);
        nn::util::SNPrintf(m_NestTestPath, sizeof(m_NestTestPath), "%s:/nest/", m_NestMountName);
        nn::util::SNPrintf(m_BigFileTestPath, sizeof(m_BigFileTestPath), "%s:/big/", m_BigFileMountName);

        const SystemSaveDataId id        = 0x8000000000004000;
        const SystemSaveDataId flatId    = 0x8000000000004001;
        const SystemSaveDataId nestId    = 0x8000000000004002;
        const SystemSaveDataId bigFileId = 0x8000000000004003;
        const size_t saveSize = 64 * 1024 * 1024;
        DeleteAllTestSaveData();
        NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(saveDataSpaceId, id, 0, saveSize, saveSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(saveDataSpaceId, flatId, 0, saveSize, saveSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(saveDataSpaceId, nestId, 0, saveSize, saveSize, 0));
        NNT_ASSERT_RESULT_SUCCESS(CreateSystemSaveData(saveDataSpaceId, bigFileId, 0, saveSize, saveSize, 0));

        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData(m_MountName, saveDataSpaceId, id));
        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData(m_FlatMountName, saveDataSpaceId, flatId));
        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData(m_NestMountName, saveDataSpaceId, nestId));
        NNT_ASSERT_RESULT_SUCCESS(MountSystemSaveData(m_BigFileMountName, saveDataSpaceId, bigFileId));
        PerformanceTestWritable::CreatePerfTestDirectory();
        NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(m_MountName));
        NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(m_FlatMountName));
        NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(m_NestMountName));
        NNT_ASSERT_RESULT_SUCCESS(CommitSaveData(m_BigFileMountName));
    }
};
class PerformanceTestSystemSaveNand : public PerformanceTestSystemSave
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "SystemSaveData(System)");
        PerformanceTestSystemSave::Initialize(SaveDataSpaceId::System);
    }
};
class PerformanceTestSystemSaveSd : public PerformanceTestSystemSave
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "SystemSaveData(SdSystem)");
        PerformanceTestSystemSave::Initialize(SaveDataSpaceId::SdSystem);
    }
};

class PerformanceTestRom : public PerformanceTestCommon
{
public:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::util::SNPrintf(m_FsName, sizeof(m_FsName), "Rom");
        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "rom");
        PerformanceTestCommon::Initialize();
        m_NestCount = 40;

        size_t mountRomCacheBufferSize = 0;
        NNT_ASSERT_RESULT_SUCCESS(QueryMountRomCacheSize(&mountRomCacheBufferSize));
        m_MountRomCacheBuffer.reset(new char[mountRomCacheBufferSize]);

        NNT_ASSERT_RESULT_SUCCESS(MountRom(m_MountName, m_MountRomCacheBuffer.get(), mountRomCacheBufferSize));
    }
    static void TearDownTestCase() NN_NOEXCEPT
    {
        Unmount(m_MountName);
        m_MountRomCacheBuffer.reset(nullptr);
    }

protected:
    static std::unique_ptr<char[]> m_MountRomCacheBuffer;

#if 1
    virtual void TestReadBigFile() NN_NOEXCEPT NN_OVERRIDE { NN_LOG("<SKIP>\n"); };
#endif
};
}

char PerformanceTest::m_FsName[MaxNameLength];
char PerformanceTest::m_MountName[MaxNameLength];
char PerformanceTest::m_FlatTestPath[MaxNameLength];
char PerformanceTest::m_NestTestPath[MaxNameLength];
char PerformanceTest::m_BigFileTestPath[MaxNameLength];
int PerformanceTest::m_FileCount;
int PerformanceTest::m_DirCount;
int PerformanceTest::m_NestCount;
int64_t PerformanceTest::m_BigFileSize;

char PerformanceTestWritable::m_WritableTestPath[MaxNameLength];
nnt::fs::util::TemporaryHostDirectory PerformanceTestHost::m_HostDirectory;
char PerformanceTestHost::m_HostRootPath[260];
char PerformanceTestSystemSave::m_FlatMountName[MaxNameLength];
char PerformanceTestSystemSave::m_NestMountName[MaxNameLength];
char PerformanceTestSystemSave::m_BigFileMountName[MaxNameLength];
std::unique_ptr<char[]> PerformanceTestRom::m_MountRomCacheBuffer;

/**
* @brief テスト対象のパスを配列に設定します。
* @param[in/out] pPath     テスト対象となるパスの配列
* @param[in]     basePath  テストディレクトリのベースパス
* @param[in]     count     テスト対象となるパスの配列の数
* @param[in]     isRandom  ランダムなパス番号を設定するか
* @param[in]     isFile    ファイルかディレクトリか
*/
void SetTestPath(char (*pPath)[NameMaxLength], const char* basePath, const int count, const bool isRandom, const bool isFile) NN_NOEXCEPT
{
    const char *addPath = (isFile) ? ".file" : "";

    if(isRandom == true)
    {
        std::unique_ptr<int[]> pathCount(new int[count]);
        CreateRandomArray(pathCount.get(), count, 1);
        for(int i = 0; i < count; i++)
        {
            nn::util::SNPrintf(pPath[i], NameMaxLength, "%s%04d%s", basePath, pathCount[i], addPath);
        }
    }
    else
    {
        for(int i = 0; i < count; i++)
        {
            nn::util::SNPrintf(pPath[i], NameMaxLength, "%s%04d%s", basePath, i, addPath);
        }
    }
}

//!< @brief 複数ファイルを持ったディレクトリを読み込む時間計測
void PerformanceTest::PerfReadDirectoryFlat(const char* basePath) NN_NOEXCEPT
{
    TimeCount timeCount;

    for(int i = 1; i <= 128; i *= 2)
    {
        DirectoryHandle dirHandle;
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, basePath, OpenDirectoryMode_File));

        while(NN_STATIC_CONDITION(true))
        {
            int64_t readNum = 0;
            std::unique_ptr<DirectoryEntry[]> dirEntry(new DirectoryEntry[i]);

            timeCount.StartTime();
            NNT_EXPECT_RESULT_SUCCESS(ReadDirectory(&readNum, dirEntry.get(), dirHandle, i));
            timeCount.StopTime();

            if( readNum == 0 )
            {
                break;
            }
        }
        CloseDirectory(dirHandle);

        NN_LOG("<RESULT>: %s / ReadDirectoryFlat (EntryBuffer=%3d) : ", m_FsName, i);
        timeCount.ViewSimpleTime(timeCount.GetTotalTime());
        timeCount.ResetTime();
    }
}

//!< @brief 複数階層のディレクトリを読み込む時間計測（各階層に1ファイルずつ）
void PerformanceTest::PerfReadDirectoryNest(const char* basePath, const int nestCount) NN_NOEXCEPT
{
    // Host の時はスキップ
    NNT_FS_UTIL_SKIP_TEST_UNLESS(strcmp(m_FsName, "Host") != 0);

    TimeCount timeCount;
    DirectoryHandle dirHandle;
    String dirPath = basePath;

    // その各階層に 1 つずつあるファイルのエントリを読み込む時間を計測
    for(int i = 0; i < nestCount; i++)
    {
        char dirName[4];
        nn::util::SNPrintf(dirName, sizeof(dirName), "/%d", i % 10);
        dirPath += dirName;
        NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, directory path: %s\n", i, dirPath.c_str());
        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, dirPath.c_str(), OpenDirectoryMode_All));

        const int entryBufferCount = 2;
        DirectoryEntry dirEntry[entryBufferCount];
        int64_t readNum = 0;

        timeCount.StartTime();
        NNT_EXPECT_RESULT_SUCCESS(ReadDirectory(&readNum, dirEntry, dirHandle, entryBufferCount));
        timeCount.StopTime();

        CloseDirectory(dirHandle);

        if(i == nestCount - 1)
        {
            ASSERT_EQ(entryBufferCount - 1, readNum);
        }
        else
        {
            ASSERT_EQ(entryBufferCount, readNum);
        }
    }

    NN_LOG("<RESULT>: %s / ReadDirectoryNest (NestCount=%4d) : ", m_FsName, nestCount);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());
}

//!< @brief エントリタイプを読み込む時間計測
void PerformanceTest::PerfGetEntryType(char (*pPath)[NameMaxLength], const char* basePath, const int count, const bool isRandom, const bool isFile) NN_NOEXCEPT
{
    TimeCount timeCount;

    SetTestPath(pPath, basePath, count, isRandom, isFile);

    for(int i = 0; i < count; i++)
    {
        NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, path: %s\n", i, pPath[i]);
        DirectoryEntryType entryType;
        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(GetEntryType(&entryType, pPath[i]));
        timeCount.StopTime();
    }

    NN_LOG("<RESULT>: %s / GetEntryType%s%s (Count=%4d) : ", m_FsName, (isFile) ? "File" : "Directory", (isRandom) ? "Random    " : "Sequencial", count);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());
}

//!< @brief ファイルとディレクトリのエントリの個数を取得する時間計測
void PerformanceTest::PerfGetDirectoryEntryCount(const char* basePath, const int fileMaxCount, const int dirMaxCount) NN_NOEXCEPT
{
    struct TestPattern {
        int fileCount;
        int dirCount;
    } pattern[] = {{fileMaxCount, 0}, {0, dirMaxCount}, {fileMaxCount, dirMaxCount}};

    for(int i = 0; i < (sizeof(pattern) / sizeof(pattern[0])); i++)
    {
        TimeCount timeCount;
        DirectoryHandle dirHandle;
        OpenDirectoryMode mode;
        int fileCount = pattern[i].fileCount;
        int dirCount = pattern[i].dirCount;

        if (dirCount == 0)
        {
            mode = OpenDirectoryMode_File;
        }
        else if (fileCount == 0)
        {
            mode = OpenDirectoryMode_Directory;
        }
        else
        {
            mode = OpenDirectoryMode_All;
        }

        NNT_ASSERT_RESULT_SUCCESS(OpenDirectory(&dirHandle, basePath, mode));
        int64_t entryNum = 0;

        timeCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(GetDirectoryEntryCount(&entryNum, dirHandle));
        timeCount.StopTime();

        ASSERT_EQ(fileCount + dirCount, entryNum);
        CloseDirectory(dirHandle);

        NN_LOG("<RESULT>: %s / GetDirectoryEntryCount (File=%4d, Directory=%4d) : ", m_FsName, fileCount, dirCount);
        timeCount.ViewSimpleTime(timeCount.GetTotalTime());
    }
}

//!< @brief ファイルの作成、リネーム、削除の時間計測
void PerformanceTest::PerfCreateRenameDeleteFile(char (*pPath)[NameMaxLength], const int fileCount) NN_NOEXCEPT
{
    TimeCount timeCreateCount;
    TimeCount timeRenameCount;
    TimeCount timeDeleteCount;

    for(int i = 0; i < fileCount; i++)
    {
        timeCreateCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateFile(pPath[i], 0));
        timeCreateCount.StopTime();
    }
    for(int i = 0; i < fileCount; i++)
    {
        char newPath[NameMaxLength];
        nn::util::SNPrintf(newPath, NameMaxLength, "%s_", pPath[i]);

        timeRenameCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameFile(pPath[i], newPath));
        timeRenameCount.StopTime();

        std::strncpy(pPath[i], newPath, NameMaxLength);
    }
    for(int i = 0; i < fileCount; i++)
    {
        timeDeleteCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteFile(pPath[i]));
        timeDeleteCount.StopTime();
    }

    NN_LOG("<RESULT>: %s / CreateFile (File=%4d) : ", m_FsName, fileCount);
    timeCreateCount.ViewSimpleTime(timeCreateCount.GetTotalTime());
    NN_LOG("<RESULT>: %s / RenameFile (File=%4d) : ", m_FsName, fileCount);
    timeRenameCount.ViewSimpleTime(timeRenameCount.GetTotalTime());
    NN_LOG("<RESULT>: %s / DeleteFile (File=%4d) : ", m_FsName, fileCount);
    timeDeleteCount.ViewSimpleTime(timeDeleteCount.GetTotalTime());
}

//!< @brief ディレクトリの作成、リネーム、削除の時間計測
void PerformanceTest::PerfCreateRenameDeleteDirectory(char (*pPath)[NameMaxLength], const int dirCount) NN_NOEXCEPT
{
    TimeCount timeCreateCount;
    TimeCount timeRenameCount;
    TimeCount timeDeleteCount;

    for(int i = 0; i < dirCount; i++)
    {
        timeCreateCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(CreateDirectory(pPath[i]));
        timeCreateCount.StopTime();
    }
    for(int i = 0; i < dirCount; i++)
    {
        char newPath[NameMaxLength];
        nn::util::SNPrintf(newPath, NameMaxLength, "%s_", pPath[i]);

        timeRenameCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(RenameDirectory(pPath[i], newPath));
        timeRenameCount.StopTime();

        strncpy(pPath[i], newPath, NameMaxLength);
    }
    for(int i = 0; i < dirCount; i++)
    {
        timeDeleteCount.StartTime();
        NNT_ASSERT_RESULT_SUCCESS(DeleteDirectory(pPath[i]));
        timeDeleteCount.StopTime();
    }

    NN_LOG("<RESULT>: %s / CreateDirectory (Directory=%4d) : ", m_FsName, dirCount);
    timeCreateCount.ViewSimpleTime(timeCreateCount.GetTotalTime());
    NN_LOG("<RESULT>: %s / RenameDirectory (Directory=%4d) : ", m_FsName, dirCount);
    timeRenameCount.ViewSimpleTime(timeRenameCount.GetTotalTime());
    NN_LOG("<RESULT>: %s / DeleteDirectory (Directory=%4d) : ", m_FsName, dirCount);
    timeDeleteCount.ViewSimpleTime(timeDeleteCount.GetTotalTime());
}

//!< @brief ファイルのオープンの時間計測
void PerformanceTest::PerfOpenCloseFile(const char (*pPath)[NameMaxLength], const int fileCount, const int diffCount) NN_NOEXCEPT
{
    TimeCount timeOpenCount;
    TimeCount timeOpenCountFirst;
    TimeCount timeOpenCountLast;
    TimeCount timeCloseCount;
    FileHandle handle;

    for(int i = 0; i < fileCount; i++)
    {
        timeOpenCount.StartTime();
        if (i < diffCount)
        {
            timeOpenCountFirst.StartTime();
        }
        if (i >= (fileCount - diffCount))
        {
            timeOpenCountLast.StartTime();
        }
        NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, pPath[i], OpenMode_Read));
        if (i < diffCount)
        {
            timeOpenCountFirst.StopTime();
        }
        if (i >= (fileCount - diffCount))
        {
            timeOpenCountLast.StopTime();
        }
        timeOpenCount.StopTime();

        timeCloseCount.StartTime();
        CloseFile(handle);
        timeCloseCount.StopTime();
    }

    NN_LOG("<RESULT>: %s / OpenFile       (File =%4d) : ", m_FsName, fileCount);
    timeOpenCount.ViewSimpleTime(timeOpenCount.GetTotalTime());
    NN_LOG("<RESULT>: %s / OpenFile First (Count=%4d) : ", m_FsName, diffCount);
    timeOpenCountFirst.ViewSimpleTime(timeOpenCountFirst.GetTotalTime());
    NN_LOG("<RESULT>: %s / OpenFile Last  (Count=%4d) : ", m_FsName, diffCount);
    timeOpenCountLast.ViewSimpleTime(timeOpenCountLast.GetTotalTime());
    NN_LOG("<RESULT>: %s / CloseFile      (File =%4d) : ", m_FsName, fileCount);
    timeCloseCount.ViewSimpleTime(timeCloseCount.GetTotalTime());
}

//!< @brief ファイルの読み込み時間計測
void PerformanceTest::PerfReadFile(const char* pPath, const int64_t fileSize, const int readSize, const int adjust) NN_NOEXCEPT
{
    TimeCount timeCount;
    FileHandle handle;
    int64_t offset;
    auto buffer = AllocateBuffer(readSize);

    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, pPath, OpenMode_Read));

    offset = adjust;
    timeCount.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, buffer.get(), readSize));
    timeCount.StopTime();

    NN_LOG("<RESULT>: %s / ReadFile First (Size=%10lld, Adjust=%d) : ", m_FsName, fileSize, adjust);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());
    timeCount.ResetTime();

    offset = fileSize - readSize - adjust;
    timeCount.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(ReadFile(handle, offset, buffer.get(), readSize));
    timeCount.StopTime();

    NN_LOG("<RESULT>: %s / ReadFile Last  (Size=%10lld, Adjust=%d) : ", m_FsName, fileSize, adjust);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());

    CloseFile(handle);
}

//!< @brief ファイルの書き込み時間計測
void PerformanceTest::PerfWriteFile(const char* pPath, const int64_t fileSize, const int writeSize, const int adjust) NN_NOEXCEPT
{
    TimeCount timeCount;
    FileHandle handle;
    int64_t offset;
    auto buffer = AllocateBuffer(writeSize);

    NNT_ASSERT_RESULT_SUCCESS(OpenFile(&handle, pPath, OpenMode_Write));

    offset = adjust;
    timeCount.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(WriteFile(handle, offset, buffer.get(), writeSize, WriteOption::MakeValue(0)));
    NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
    timeCount.StopTime();

    NN_LOG("<RESULT>: %s / WriteFile First (Size=%10lld, Adjust=%d) : ", m_FsName, fileSize, adjust);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());
    timeCount.ResetTime();

    offset = fileSize - writeSize - adjust;
    timeCount.StartTime();
    NNT_ASSERT_RESULT_SUCCESS(WriteFile(handle, offset, buffer.get(), writeSize, WriteOption::MakeValue(0)));
    NNT_ASSERT_RESULT_SUCCESS(FlushFile(handle));
    timeCount.StopTime();

    NN_LOG("<RESULT>: %s / WriteFile Last  (Size=%10lld, Adjust=%d) : ", m_FsName, fileSize, adjust);
    timeCount.ViewSimpleTime(timeCount.GetTotalTime());

    CloseFile(handle);
}

//!< @brief 4096 個のファイルを持ったディレクトリを読み込む時間計測
void PerformanceTestCommon::TestReadDirectoryFlat() NN_NOEXCEPT
{
    PerfReadDirectoryFlat(m_FlatTestPath);
}
//!< @brief 100階層のディレクトリを読み込む時間計測（各階層に1ファイルずつ）
void PerformanceTestCommon::TestReadDirectoryNest() NN_NOEXCEPT
{
    PerfReadDirectoryNest(m_NestTestPath, m_NestCount);
}
//!< @brief 4096 個のファイルのエントリタイプを取得する時間計測
void PerformanceTestCommon::TestGetEntryTypeFile() NN_NOEXCEPT
{
    PerfGetEntryType(m_pEntryPath, m_FlatTestPath, m_FileCount, false, true);
    PerfGetEntryType(m_pEntryPath, m_FlatTestPath, m_FileCount, true, true);
}
//!< @brief 256 個のディレクトリのエントリタイプを取得する時間計測
void PerformanceTestCommon::TestGetEntryTypeDirectory() NN_NOEXCEPT
{
    PerfGetEntryType(m_pEntryPath, m_FlatTestPath, m_DirCount, false, false);
    PerfGetEntryType(m_pEntryPath, m_FlatTestPath, m_DirCount, true, false);
}
//!< @brief ディレクトリとファイルのエントリの個数を取得する時間計測
void PerformanceTestCommon::TestGetDirectoryEntryCount() NN_NOEXCEPT
{
    PerfGetDirectoryEntryCount(m_FlatTestPath, m_FileCount, m_DirCount);
}
//!< @brief ファイルのオープンの時間計測
void PerformanceTestCommon::TestOpenCloseFile() NN_NOEXCEPT
{
    SetTestPath(m_pEntryPath, m_FlatTestPath, m_FileCount, false, true);
    PerfOpenCloseFile(m_pEntryPath, m_FileCount, 100);
}
//!< @brief ファイルの作成、リネーム、削除の時間計測
void PerformanceTestWritable::TestCreateRenameDeleteFile() NN_NOEXCEPT
{
    SetTestPath(m_pEntryPath, m_WritableTestPath, m_FileCount, false, true);
    PerfCreateRenameDeleteFile(m_pEntryPath, m_FileCount);
}
//!< @brief ディレクトリの作成、リネーム、削除の時間計測
void PerformanceTestWritable::TestCreateRenameDeleteDirectory() NN_NOEXCEPT
{
    SetTestPath(m_pEntryPath, m_WritableTestPath, m_DirCount, false, false);
    PerfCreateRenameDeleteDirectory(m_pEntryPath, m_DirCount);
}
//!< @brief Bigファイルの読み込み時間計測
void PerformanceTestCommon::TestReadBigFile() NN_NOEXCEPT
{
    SetTestPath(m_pEntryPath, m_BigFileTestPath, 1, false, true);
    PerfReadFile(m_pEntryPath[0], m_BigFileSize, BigFileWorkSize, 0);
    PerfReadFile(m_pEntryPath[0], m_BigFileSize, BigFileWorkSize, 1);
}
//!< @brief Bigファイルの書き込み時間計測
void PerformanceTestWritable::TestWriteBigFile() NN_NOEXCEPT
{
    SetTestPath(m_pEntryPath, m_BigFileTestPath, 1, false, true);
    PerfWriteFile(m_pEntryPath[0], m_BigFileSize, BigFileWorkSize, 0);
    PerfWriteFile(m_pEntryPath[0], m_BigFileSize, BigFileWorkSize, 1);
}


#define DEFTEST(classname) \
TEST_F(classname, ReadDirectoryFlat)            { TestReadDirectoryFlat(); }            \
TEST_F(classname, ReadDirectoryNest)            { TestReadDirectoryNest(); }            \
TEST_F(classname, GetEntryTypeFile)             { TestGetEntryTypeFile(); }             \
TEST_F(classname, GetEntryTypeDirectory)        { TestGetEntryTypeDirectory(); }        \
TEST_F(classname, GetDirectoryEntryCount)       { TestGetDirectoryEntryCount(); }       \
TEST_F(classname, OpenCloseFile)                { TestOpenCloseFile(); }                \
TEST_F(classname, CreateRenameDeleteFile)       { TestCreateRenameDeleteFile(); }       \
TEST_F(classname, CreateRenameDeleteDirectory)  { TestCreateRenameDeleteDirectory(); }  \
TEST_F(classname, ReadBigFile)                  { TestReadBigFile(); }                  \
TEST_F(classname, WriteBigFile)                 { TestWriteBigFile(); }

DEFTEST(PerformanceTestHost)
DEFTEST(PerformanceTestBis)
DEFTEST(PerformanceTestSd)
DEFTEST(PerformanceTestContentStorageBis)
DEFTEST(PerformanceTestContentStorageSd)
DEFTEST(PerformanceTestImageDirectoryNand)
DEFTEST(PerformanceTestImageDirectorySd)
DEFTEST(PerformanceTestSystemSaveNand)
DEFTEST(PerformanceTestSystemSaveSd)
DEFTEST(PerformanceTestRom)


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    nnt::fs::util::SetFsTestPerformanceConfiguration();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    // MmcPatrol 休止
    nn::fs::SuspendMmcPatrol();

    auto testResult = RUN_ALL_TESTS();

    // MmcPatrol 再開
    nn::fs::ResumeMmcPatrol();

    if (CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
