﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_MemoryReportInfo.h>
#include <nn/fssrv/fssrv_MemoryResourceFromExpHeap.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace
{

const int ExpHeapSize = 4 * 1024;
NN_ALIGNAS(4096) char g_ExpHeapStack[ExpHeapSize];

}

class PeakCheckableMemoryResourceFromExpHeapTest : public testing::TestWithParam<size_t>
{
};

INSTANTIATE_TEST_CASE_P(Default, PeakCheckableMemoryResourceFromExpHeapTest, testing::Values(1, 4, 7, 8, 32, 1023, 1024, 1025));

TEST_P(PeakCheckableMemoryResourceFromExpHeapTest, BulkAllocate)
{
    nn::lmem::HeapHandle heapHandle = nn::lmem::CreateExpHeap(g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill | nn::lmem::CreationOption_ThreadSafe);
    ASSERT_NE(nullptr, heapHandle);

    size_t allocateSize = GetParam();
    nnt::fs::util::Vector<void*> memorys;
    const auto firstFreeSize = nn::lmem::GetExpHeapTotalFreeSize(heapHandle);
    nn::fssrv::PeakCheckableMemoryResourceFromExpHeap allocator(firstFreeSize);
    allocator.SetHeapHandle(heapHandle);

    NN_UTIL_SCOPE_EXIT
    {
        for( auto p : memorys )
        {
            allocator.deallocate(p, allocateSize);
        }
        nn::lmem::DestroyExpHeap(heapHandle);
        // すべて解放すると、最初に戻るはず
        ASSERT_EQ(firstFreeSize, allocator.GetFreeSizeCurrent());
    };

    do
    {
        void* p = allocator.allocate(allocateSize);
        if( p == nullptr )
        {
            break;
        }
        memorys.push_back(p);
        ASSERT_LE(nn::lmem::GetExpHeapTotalFreeSize(heapHandle), allocator.GetFreeSizePeak());
    } while( NN_STATIC_CONDITION(true) );
}

TEST_P(PeakCheckableMemoryResourceFromExpHeapTest, RandomFree)
{
    nn::lmem::HeapHandle heapHandle = nn::lmem::CreateExpHeap(g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill | nn::lmem::CreationOption_ThreadSafe);
    ASSERT_NE(nullptr, heapHandle);

    size_t allocateSize = GetParam();
    nnt::fs::util::Vector<void*> memorys;
    const auto firstFreeSize = nn::lmem::GetExpHeapTotalFreeSize(heapHandle);
    nn::fssrv::PeakCheckableMemoryResourceFromExpHeap allocator(firstFreeSize);
    allocator.SetHeapHandle(heapHandle);

    std::mt19937 mt(nnt::fs::util::GetRandomSeed());

    NN_UTIL_SCOPE_EXIT
    {
        for( auto p : memorys )
        {
            allocator.deallocate(p, allocateSize);
        }
        nn::lmem::DestroyExpHeap(heapHandle);
        // すべて解放すると、最初に戻るはず
        ASSERT_EQ(firstFreeSize, allocator.GetFreeSizeCurrent());
    };

    do
    {
        void* p = allocator.allocate(allocateSize);
        if( p == nullptr )
        {
            ASSERT_LE(nn::lmem::GetExpHeapTotalFreeSize(heapHandle), allocator.GetFreeSizePeak());
            break;
        }
        if( std::uniform_int_distribution<size_t>(0, 1)(mt) )
        {
            allocator.deallocate(p, allocateSize);
        }
        else
        {
            memorys.push_back(p);
        }
    } while( NN_STATIC_CONDITION(true) );
}

TEST(MemoryReport, GetMemoryReportInfo)
{
    nn::fs::MemoryReportInfo memoryReportInfoBefore;
    nn::fs::MemoryReportInfo memoryReportInfoAfter;

    {
        nn::fs::MountHostRoot();
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::GetAndClearMemoryReportInfo(&memoryReportInfoBefore));
        nn::fs::UnmountHostRoot();
    }

    NN_LOG("pooledBufferPeakFreeSize : %X\n", memoryReportInfoBefore.pooledBufferPeakFreeSize);
    NN_LOG("bufferManagerPeakFreeSize: %X\n", memoryReportInfoBefore.bufferManagerPeakFreeSize);
    NN_LOG("bufferManagerPeakTotalAllocatableSize: %X\n", memoryReportInfoBefore.bufferManagerPeakTotalAllocatableSize);
    NN_LOG("expHeapPeakFreeSize      : %X\n", memoryReportInfoBefore.expHeapPeakFreeSize);
    NN_LOG("bufferPoolPeakFreeSize   : %X\n", memoryReportInfoBefore.bufferPoolPeakFreeSize);
    NN_LOG("pooledBufferRetriedCount : %u\n", memoryReportInfoBefore.pooledBufferRetriedCount);
    NN_LOG("bufferManagerRetriedCount: %u\n", memoryReportInfoBefore.bufferManagerRetriedCount);
    NN_LOG("pooledBufferReduceAllocationCount : %u\n", memoryReportInfoBefore.pooledBufferReduceAllocationCount);

    NNT_ASSERT_RESULT_SUCCESS( nn::fs::GetAndClearMemoryReportInfo(&memoryReportInfoAfter) );

    NN_LOG("pooledBufferPeakFreeSize : %X\n", memoryReportInfoAfter.pooledBufferPeakFreeSize);
    NN_LOG("bufferManagerPeakFreeSize: %X\n", memoryReportInfoAfter.bufferManagerPeakFreeSize);
    NN_LOG("bufferManagerPeakTotalAllocatableSize: %X\n", memoryReportInfoAfter.bufferManagerPeakTotalAllocatableSize);
    NN_LOG("expHeapPeakFreeSize      : %X\n", memoryReportInfoAfter.expHeapPeakFreeSize);
    NN_LOG("bufferPoolPeakFreeSize   : %X\n", memoryReportInfoAfter.bufferPoolPeakFreeSize);
    NN_LOG("pooledBufferRetriedCount : %u\n", memoryReportInfoAfter.pooledBufferRetriedCount);
    NN_LOG("bufferManagerRetriedCount: %u\n", memoryReportInfoAfter.bufferManagerRetriedCount);
    NN_LOG("pooledBufferReduceAllocationCount : %u\n", memoryReportInfoAfter.pooledBufferReduceAllocationCount);

    // 少なくともクリア前より大きいはず
    EXPECT_GE(memoryReportInfoAfter.pooledBufferPeakFreeSize,  memoryReportInfoBefore.pooledBufferPeakFreeSize);
    EXPECT_GE(memoryReportInfoAfter.bufferManagerPeakFreeSize, memoryReportInfoBefore.bufferManagerPeakFreeSize);
    EXPECT_GE(memoryReportInfoAfter.bufferManagerPeakTotalAllocatableSize, memoryReportInfoBefore.bufferManagerPeakTotalAllocatableSize);
    EXPECT_GE(memoryReportInfoAfter.expHeapPeakFreeSize,       memoryReportInfoBefore.expHeapPeakFreeSize);
    EXPECT_GE(memoryReportInfoAfter.bufferPoolPeakFreeSize,    memoryReportInfoBefore.bufferPoolPeakFreeSize);
    // 少なくともクリア前より小さいはず
    EXPECT_LE(memoryReportInfoAfter.pooledBufferRetriedCount,  memoryReportInfoBefore.pooledBufferRetriedCount);
    EXPECT_LE(memoryReportInfoAfter.bufferManagerRetriedCount, memoryReportInfoBefore.bufferManagerRetriedCount);

    EXPECT_LE(memoryReportInfoAfter.pooledBufferReduceAllocationCount, memoryReportInfoBefore.pooledBufferReduceAllocationCount);
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(testResult);
}
