﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/util/util_FormatString.h>
#include <nn/os.h>

#include <nnt/nntest.h>
#include <nnt/nnt_Argument.h.>
#include <nnt/base/testBase_Exit.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/fsUtil/testFs_util.h>

namespace {

const int NameLength = 256;

class FsLargeRomFs : public ::testing::Test, public nnt::fs::util::PrepareWorkDirFixture
{
protected:
    char m_MountName[NameLength];

    virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
    {
        size_t cacheBufferSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheBufferSize));
        m_CacheBuffer.reset(new char[cacheBufferSize]);
        NN_ABORT_UNLESS_NOT_NULL(m_CacheBuffer);

        nn::util::SNPrintf(m_MountName, sizeof(m_MountName), "rom");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(m_MountName, m_CacheBuffer.get(), cacheBufferSize));
    }
    virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
    {
        nn::fs::Unmount(m_MountName);
    }

private:
    std::unique_ptr<char[]> m_CacheBuffer;
};

void DumpDir(const char* srcDir, int* outDirCount, int* outFileCount)
{
    nn::fs::DirectoryHandle srcHandle;
    char srcPath[NameLength];
    nn::fs::DirectoryEntry entryBuffer;

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcDir, nn::fs::OpenDirectoryMode_Directory));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s/%s", srcDir, entryBuffer.name);
        NN_LOG("Directory(%s)\n", srcPath);
        (*outDirCount)++;

        DumpDir(srcPath, outDirCount, outFileCount);
    }
    nn::fs::CloseDirectory(srcHandle);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&srcHandle, srcDir, nn::fs::OpenDirectoryMode_File));
    while (1)
    {
        int64_t outValue = 0;
        NNT_ASSERT_RESULT_SUCCESS(nn::fs::ReadDirectory(&outValue, &entryBuffer, srcHandle, 1));
        if (outValue == 0)
        {
            break;
        }

        nn::util::SNPrintf(srcPath, sizeof(srcPath), "%s/%s", srcDir, entryBuffer.name);
        NN_LOG("File(%s)\n", srcPath);
        (*outFileCount)++;
    }
    nn::fs::CloseDirectory(srcHandle);
}

}

TEST_F(FsLargeRomFs, DumpDirectoryRecursive)
{
    char rootPath[NameLength];
    nn::util::SNPrintf(rootPath, NameLength, "%s:/", m_MountName);

#if 1
    int outDirCount = 0;
    int outFileCount = 0;
    DumpDir(rootPath, &outDirCount, &outFileCount);
    ASSERT_EQ(4, outDirCount);
    ASSERT_EQ(143, outFileCount);
#else
    NNT_ASSERT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive(rootPath));
#endif
}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto ret = RUN_ALL_TESTS();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
