﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/fs.h>

#include <nn/os/os_Tick.h>
#include <nn/util/util_ScopeExit.h>

#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Integration_FileDataCache_Util.h"


const size_t DefaultFileDataCacheSize = 32 * 1024 * 1024;

void RandomReadTest(void* theArg)
{
    RandomReadTestArg* arg = static_cast<RandomReadTestArg*>(theArg);
    arg->success = RandomReadTest(arg->mountName, arg->seed, arg->trialCount, arg->trialTime, arg->trialSize);
}

bool RandomReadTest(const char* mountName, int64_t seed, int trialCount, nn::TimeSpan trialTime)
{
    return RandomReadTest(mountName, seed, trialCount, trialTime, 0);
}

bool RandomReadTest(const char* mountName, int64_t seed, int trialCount, nn::TimeSpan trialTime, int64_t trialSize)
{
    int64_t fileCount;
    nn::fs::DirectoryEntry* fileEntries;
    {
        char path[16];
        std::snprintf(path, 16, "%s:/", mountName);

        nn::fs::DirectoryHandle directoryHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, path, nn::fs::OpenDirectoryMode_File));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&fileCount, directoryHandle));

        fileEntries = static_cast<nn::fs::DirectoryEntry*>(std::malloc(sizeof(nn::fs::DirectoryEntry) * static_cast<size_t>(fileCount)));

        int64_t readFileCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&readFileCount, fileEntries, directoryHandle, fileCount));
        NN_ABORT_UNLESS(readFileCount == fileCount);

        nn::fs::CloseDirectory(directoryHandle);
    }
    NN_UTIL_SCOPE_EXIT
    {
        std::free(fileEntries);
    };

    std::mt19937 mt(static_cast<unsigned int>(seed));
    std::uniform_int_distribution<int> fileSelector(0, static_cast<int>(fileCount) - 1);

    FILE_DATA_CACHE_TEST_INFO_LOG("random read test: seed = %lld\n", seed);

    int64_t readSize = 0;
    nn::os::Tick startTick = nn::os::GetSystemTick();
    for (int trial = 0; trial < trialCount || trialCount == 0; trial++)
    {
        const nn::fs::DirectoryEntry& fileEntry = fileEntries[fileSelector(mt)];

        std::uniform_int_distribution<int64_t> dist(0, fileEntry.fileSize);

        int64_t offset = dist(mt);
        size_t size = static_cast<size_t>(dist(mt));
        while (size > static_cast<size_t>(fileEntry.fileSize - offset))
        {
            size = static_cast<size_t>(dist(mt));
        }

        nn::Bit8* pFileBuffer = new nn::Bit8[size + 16];
        NN_UTIL_SCOPE_EXIT
        {
            delete[] pFileBuffer;
        };
        std::memset(pFileBuffer, 0xcd, 8);
        std::memset(pFileBuffer + 8 + size, 0xcd, 8);

        char path[16];
        std::snprintf(path, 16, "%s:/%s", mountName, fileEntry.name);

        nn::fs::FileHandle fileHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, offset, pFileBuffer + 8, size));

        if (!nnt::fs::util::IsFilledWithValue(pFileBuffer, 8, 0xcd))
        {
            FILE_DATA_CACHE_TEST_ERROR_LOG("memory corruption (head), seed = %lld, trial = %d\n", seed, trial);
            return false;
        }
        if (!nnt::fs::util::IsFilledWith8BitCount(pFileBuffer + 8, size, offset))
        {
            FILE_DATA_CACHE_TEST_ERROR_LOG("compare failed, seed = %lld, trial = %d\n", seed, trial);
            return false;
        }
        if (!nnt::fs::util::IsFilledWithValue(pFileBuffer + 8 + size, 8, 0xcd))
        {
            FILE_DATA_CACHE_TEST_ERROR_LOG("memory corruption (tail), seed = %lld, trial = %d\n", seed, trial);
            return false;
        }

        nn::os::Tick currentTick = nn::os::GetSystemTick();
        if (trialTime != nn::TimeSpan(0) && nn::os::ConvertToTimeSpan(currentTick - startTick) > trialTime)
        {
            break;
        }

        readSize += static_cast<int64_t>(size);
        if (trialSize > 0 && readSize > trialSize)
        {
            break;
        }
    }

    return true;
}

void ExpectNoMemoryLeak()
{
    bool memoryLeaked = nnt::fs::util::CheckMemoryLeak();
    EXPECT_FALSE(memoryLeaked);
    if (memoryLeaked)
    {
        nnt::fs::util::ResetAllocateCount();
    }
}

const size_t MinAcceptableCacheMemorySize = 1 * 1024 * 1024;

size_t GetMaxAllocatableCacheMemorySize() NN_NOEXCEPT
{
    const size_t memorySizeUsedByTest = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_WIN)

    // TORIAEZU: 適当なサイズを設定
    const size_t maxAllocatableMemorySize = 1 * 1024 * 1024 * 1024 - memorySizeUsedByTest;

#elif defined(NN_BUILD_TARGET_PLATFORM_NX)

    nn::os::MemoryInfo memoryInfo;
    nn::os::QueryMemoryInfo(&memoryInfo);
    // 利用可能なメモリからテストで使われる分を引く
    NN_ABORT_UNLESS(memoryInfo.allocatedMemoryHeapSize > memorySizeUsedByTest);
    const size_t maxAllocatableMemorySize = memoryInfo.allocatedMemoryHeapSize - memorySizeUsedByTest;

#else
#error "unknown platform"
#endif

    // 二分探索で実際に割り当てられそうなメモリサイズを探す
    size_t upperLimit = maxAllocatableMemorySize;
    size_t lowerLimit = 0;

    while (upperLimit - lowerLimit > 1 * 1024 * 1024)  // 厳密な値は要らないので適当なところで切り上げる
    {
        size_t testSize = lowerLimit + (upperLimit - lowerLimit) / 2;
        void* p = std::malloc(testSize);

        if (p)
        {
            std::free(p);
            lowerLimit = testSize;
        }
        else
        {
            upperLimit = testSize;
        }
    }

    const size_t availableMemorySize = lowerLimit;
    NN_ABORT_UNLESS(availableMemorySize > 0);

    return availableMemorySize;
}

void ReadFileTestFixture::SetUpFileSystem() NN_NOEXCEPT
{
    size_t fileSystemCacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&fileSystemCacheSize));

    m_pCache = std::malloc(fileSystemCacheSize);
    NN_ABORT_UNLESS_NOT_NULL(m_pCache);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom(RomMountName, m_pCache, fileSystemCacheSize));
}

void ReadFileTestFixture::SetUpFileHandle() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&m_FileHandle, TestTargetFilePath, nn::fs::OpenMode_Read));

    int64_t fileSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, m_FileHandle));
    m_FileSizeBufferSize = static_cast<size_t>(fileSize);

    m_pFileSizeBuffer = std::malloc(m_FileSizeBufferSize);
    NN_ABORT_UNLESS_NOT_NULL(m_pFileSizeBuffer);
}

void ReadFileTestFixture::TearDownFileHandle() NN_NOEXCEPT
{
    std::free(m_pFileSizeBuffer);

    nn::fs::CloseFile(m_FileHandle);
}

void ReadFileTestFixture::TearDownFileSystem() NN_NOEXCEPT
{
    nn::fs::Unmount("rom");

    std::free(m_pCache);
}

void ReadFileTestFixture::SetUp() NN_NOEXCEPT
{
    SetUpFileSystem();
    SetUpFileHandle();
    AutoAbortDisabledTestFixture::SetUp();
}

void ReadFileTestFixture::TearDown() NN_NOEXCEPT
{
    AutoAbortDisabledTestFixture::TearDown();
    TearDownFileHandle();
    TearDownFileSystem();
}

void TestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size)
{
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::ReadFile(fileHandle, offset, buffer, size);
    }
    {
        cacheDisabledResult = nn::fs::ReadFile(fileHandle, offset, buffer, size);
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());
}

void TestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option)
{
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::ReadFile(fileHandle, offset, buffer, size, option);
    }
    {
        cacheDisabledResult = nn::fs::ReadFile(fileHandle, offset, buffer, size, option);
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());
}

void TestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size)
{
    size_t cacheEnabledReadSize = SIZE_MAX;
    size_t cacheDisabledReadSize = SIZE_MAX;
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size);
        if (readSize)
        {
            cacheEnabledReadSize = *readSize;
        }
    }
    {
        cacheDisabledResult = nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size);
        if (readSize)
        {
            cacheDisabledReadSize = *readSize;
        }
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());

    if (cacheEnabledResult.GetInnerValueForDebug() == cacheDisabledResult.GetInnerValueForDebug())
    {
        EXPECT_EQ(cacheEnabledReadSize, cacheDisabledReadSize);
    }
}

void TestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option)
{
    size_t cacheEnabledReadSize = SIZE_MAX;
    size_t cacheDisabledReadSize = SIZE_MAX;
    nn::Result cacheEnabledResult;
    nn::Result cacheDisabledResult;
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        cacheEnabledResult = nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size, option);
        if (readSize)
        {
            cacheEnabledReadSize = *readSize;
        }
    }
    {
        cacheDisabledResult = nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size, option);
        if (readSize)
        {
            cacheDisabledReadSize = *readSize;
        }
    }
    EXPECT_EQ(cacheEnabledResult.GetInnerValueForDebug(), cacheDisabledResult.GetInnerValueForDebug());

    if (cacheEnabledResult.GetInnerValueForDebug() == cacheDisabledResult.GetInnerValueForDebug())
    {
        EXPECT_EQ(cacheEnabledReadSize, cacheDisabledReadSize);
    }
}

void DeathTestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(fileHandle, offset, buffer, size), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(fileHandle, offset, buffer, size), "");
    }
}

void DeathTestReadFile(nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(fileHandle, offset, buffer, size, option), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(fileHandle, offset, buffer, size, option), "");
    }
}

void DeathTestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size), "");
    }
}

void DeathTestReadFile(size_t* readSize, nn::fs::FileHandle fileHandle, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size, option), "");
    }
    {
        EXPECT_DEATH_IF_SUPPORTED(
            nn::fs::ReadFile(readSize, fileHandle, offset, buffer, size, option), "");
    }
}
