﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <random>

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>

#include <nn/fs.h>
#include <nn/os/os_Tick.h>
#include <nn/os/os_Thread.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>

#include "testFs_Integration_FileDataCache_Util.h"


namespace {

const char* RomMountName = "rom";

const nn::TimeSpan ZeroTimeSpan(0);

const int MaxThreadInfoCount = 2;
ThreadInfo g_ThreadInfo[MaxThreadInfoCount];

}

// ----------------------------------------------------------------------------

TEST(OperationOrder, EnableBeforeMountDisableAfterUnmount)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        {
            ScopedRomFsMounter romFs(RomMountName);

            EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, EnableAfterMountDisableAfterUnmount)
{
    {
        ScopedRomFsMounter romFs(RomMountName);
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));

        romFs.Unmount();
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, EnableBeforeMountDisableBeforeUnmount)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
        ScopedRomFsMounter romFs(RomMountName);

        EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));

        fileDataCache.Disable();
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, EnableAfterMountDisableBeforeUnmount)
{
    {
        ScopedRomFsMounter romFs(RomMountName);
        {
            ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

            EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, RepeatCacheSwitchingWhileKeepingMounted)
{
    {
        ScopedRomFsMounter romFs(RomMountName);

        for (int i = 0; i < 5; i++)
        {
            EXPECT_TRUE(RandomReadTest(RomMountName, i * 2, 16, ZeroTimeSpan));
            {
                ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

                EXPECT_TRUE(RandomReadTest(RomMountName, i * 2 + 1, 16, ZeroTimeSpan));
            }
        }
    }
    ExpectNoMemoryLeak();
}

TEST(OperationOrder, RepeatMountSwitchingWhileKeepingCacheEnabled)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        for (int i = 0; i < 5; i++)
        {
            ScopedRomFsMounter romFs(RomMountName);

            EXPECT_TRUE(RandomReadTest(RomMountName, i, 32, ZeroTimeSpan));
        }
    }
    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

TEST(CacheSize, SmallCacheSize)
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(MinAcceptableCacheMemorySize);
        ScopedRomFsMounter romFs(RomMountName);

        EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

#if defined(NN_BUILD_CONFIG_OS_WIN32) && defined(NN_BUILD_CONFIG_ADDRESS_32)
// Win32 環境では安定した最大サイズのメモリ確保ができないため、無効化しておく (SIGLO-82973)
TEST(CacheSize, DISABLED_LargeCacheSize)
#else
TEST(CacheSize, LargeCacheSize)
#endif
{
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(GetMaxAllocatableCacheMemorySize());
        ScopedRomFsMounter romFs(RomMountName);

        EXPECT_TRUE(RandomReadTest(RomMountName, 0, 128, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

#if defined(NN_BUILD_CONFIG_OS_WIN32) && defined(NN_BUILD_CONFIG_ADDRESS_32)
// Win32 環境では安定した最大サイズのメモリ確保ができないため、無効化しておく (SIGLO-82973)
TEST(CacheSize, DISABLED_RandomCacheSize)
#else
TEST(CacheSize, RandomCacheSize)
#endif
{
    int64_t seed = nn::os::GetSystemTick().GetInt64Value();
    std::mt19937 mt(static_cast<unsigned int>(seed));
    std::uniform_int_distribution<size_t> dist(MinAcceptableCacheMemorySize, GetMaxAllocatableCacheMemorySize());

    for (int i = 0; i < 16; i++)
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(dist(mt));
        ScopedRomFsMounter romFs(RomMountName);

        EXPECT_TRUE(RandomReadTest(RomMountName, 0, 16, ZeroTimeSpan));
    }
    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

namespace {

struct RomFsReadThreadArg
{
    int64_t seed;
    nn::TimeSpan trialTime;
};

void RomFsReadThreadFunc(void* theArg)
{
    RomFsReadThreadArg* arg = static_cast<RomFsReadThreadArg*>(theArg);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick) < arg->trialTime)
    {
        ScopedRomFsMounter romFs(RomMountName);
        int64_t seed = nn::os::GetSystemTick().GetInt64Value();
        EXPECT_TRUE(RandomReadTest(RomMountName, seed, 16, ZeroTimeSpan));
    }
}

struct CacheSwitchingThreadArg
{
    int64_t seed;
    nn::TimeSpan trialTime;
};
void CacheSwitchingThreadFunc(void* theArg)
{
    CacheSwitchingThreadArg* arg = static_cast<CacheSwitchingThreadArg*>(theArg);

    std::mt19937 mt(static_cast<unsigned int>(arg->seed));
    std::uniform_int_distribution<int64_t> msSelector(0, 20);

    nn::os::Tick startTick = nn::os::GetSystemTick();

    while (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - startTick) < arg->trialTime)
    {
        ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(msSelector(mt)));
    }
}

}  // namespace unnamed

TEST(ThreadSafety, RomFsReadAndCacheSwitching)
{
    CacheSwitchingThreadArg cacheSwitchingThreadArg;
    cacheSwitchingThreadArg.seed = nn::os::GetSystemTick().GetInt64Value();
    cacheSwitchingThreadArg.trialTime = nn::TimeSpan::FromSeconds(10);

    RomFsReadThreadArg romFsReadThreadArg;
    romFsReadThreadArg.seed = nn::os::GetSystemTick().GetInt64Value();
    romFsReadThreadArg.trialTime = nn::TimeSpan::FromSeconds(10);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[0].threadType,
            CacheSwitchingThreadFunc,
            &cacheSwitchingThreadArg,
            g_ThreadInfo[0].threadStack,
            g_ThreadInfo[0].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            0));
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &g_ThreadInfo[1].threadType,
            RomFsReadThreadFunc,
            &romFsReadThreadArg,
            g_ThreadInfo[1].threadStack,
            g_ThreadInfo[1].ThreadStackSize,
            nn::os::DefaultThreadPriority,
            1));

    for (int i = 0; i < 2; i++)
    {
        nn::os::StartThread(&g_ThreadInfo[i].threadType);
    }
    for (int i = 0; i < 2; i++)
    {
        nn::os::WaitThread(&g_ThreadInfo[i].threadType);
        nn::os::DestroyThread(&g_ThreadInfo[i].threadType);
    }

    ExpectNoMemoryLeak();
}

// ----------------------------------------------------------------------------

TEST(Reading, ReadFileBeyondItsSizeLimit)
{
    ScopedGlobalFileDataCacheEnabler fileDataCache(DefaultFileDataCacheSize);
    ScopedRomFsMounter romFs(RomMountName);

    int64_t fileCount;
    nn::fs::DirectoryEntry* fileEntries;
    {
        char path[16];
        std::snprintf(path, 16, "%s:/", RomMountName);

        nn::fs::DirectoryHandle directoryHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenDirectory(&directoryHandle, path, nn::fs::OpenDirectoryMode_File));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetDirectoryEntryCount(&fileCount, directoryHandle));

        fileEntries = static_cast<nn::fs::DirectoryEntry*>(std::malloc(sizeof(nn::fs::DirectoryEntry) * static_cast<size_t>(fileCount)));

        int64_t readFileCount;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadDirectory(&readFileCount, fileEntries, directoryHandle, fileCount));
        NN_ABORT_UNLESS(readFileCount == fileCount);

        nn::fs::CloseDirectory(directoryHandle);
    }
    NN_UTIL_SCOPE_EXIT
    {
        std::free(fileEntries);
    };

    for (int64_t i = 0; i < fileCount; i++)
    {
        char path[16];
        std::snprintf(path, 16, "%s:/%s", RomMountName, fileEntries[i].name);

        nn::fs::FileHandle fileHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(fileHandle);
        };

        // SIGLO-80384
        // 32MB ファイル末尾を超えて読もうとしてみる (ある程度読み込みサイズを超過させないと発覚しない不具合だった)
        size_t bufferSize = static_cast<size_t>(fileEntries[i].fileSize + 32 * 1024 * 1024);
        void* buffer = std::malloc(bufferSize);
        NN_UTIL_SCOPE_EXIT
        {
            std::free(buffer);
        };

        size_t readSize;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&readSize, fileHandle, 0, buffer, bufferSize));

        EXPECT_EQ(readSize, fileEntries[i].fileSize);
        EXPECT_TRUE(nnt::fs::util::IsFilledWith8BitCount(buffer, readSize, 0));
    }
}
