﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_DbmRomPathTool.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsApiRo/testFs_ApiRo.h>
#include <nnt/result/testResult_Assert.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace nnt { namespace fs { namespace api {
    class FsApiHtmlDocumentFileSystem : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiHtmlDocumentFileSystem() NN_NOEXCEPT
            : FsApiIntegrationTestFileSystem("htmlDoc")
        {
            nn::ncm::ApplicationId id = { 0x0005000c10000000 };

            // 0x0005000c10000000 の HtmlDocument ありのアプリとしてインストール済みで実行
            nn::ns::Initialize();
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::MountApplicationHtmlDocument("htmlDoc", id));

#if !defined(FS_TEST_NS_API)
            nn::fs::Unmount("htmlDoc");
            nn::ns::Finalize();

            // 0x0005000c10000000 の HtmlDocument ありのアプリとしてインストール済み、パッチあり / 無しで実行
            // 前回の MountApplicationHtmlDocument() による RegisterHtmlDocumentPath() に引きずられる点に注意
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountContent("htmlDoc", id, nn::fs::ContentType_Manual));
#endif
        }

        virtual ~FsApiHtmlDocumentFileSystem() NN_NOEXCEPT NN_OVERRIDE
        {
            nn::fs::Unmount("htmlDoc");
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<nnt::fs::api::TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiHtmlDocumentFileSystem> romFs(new FsApiHtmlDocumentFileSystem());

        std::unique_ptr<TestFileSystemInfo> info(new nnt::fs::api::TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(romFs);
        info->type = FileSystemType_HtmlDocumentFileSystem;
        info->rootDirPath = "htmlDoc:/html-document";

        auto& attribute = info->attribute;

        static const int NameLengthMax  = nn::fs::RomPathTool::MaxPathLength;
        static const int PathLengthMax  = nn::fs::RomPathTool::MaxPathLength;
        attribute.directoryNameLengthMax = NameLengthMax;
        attribute.fileNameLengthMax      = NameLengthMax;
        attribute.directoryPathLengthMax = PathLengthMax;
        attribute.filePathLengthMax      = PathLengthMax;

        attribute.fileSizeMax = 0x0000100000000000;
        attribute.storageSize = 0x0000100000000000;

        attribute.fileOpenMax      = 1024;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;

        attribute.aesCtrKeyTypeFlag = static_cast<int32_t>(nn::fs::AesCtrKeyTypeFlag::InternalKeyForHardwareAes);
        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::GameCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nn::os::GetHostArgc();
    char**  argv = nn::os::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllRoTests();

    const int HeapSize = 1024 * 1024;
    std::unique_ptr<char[]> heapStack(new char[HeapSize]);

    nnt::fs::util::InitializeTestLibraryHeap(heapStack.get(), HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto ret = 0;
    {
        nnt::fs::util::GlobalFileDataCache dataCache(argc, argv);

        ret = RUN_ALL_TESTS();
    }

    if( nnt::fs::util::CheckMemoryLeak() )
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
