﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/fs/fs_CloudBackupWorkStorage.h>
#include <nn/os.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Integration_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const char MountName[] = "CbWork";
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
}

namespace nnt { namespace fs { namespace api {
    class FsApiCloudBackupWorkStorage : public FsApiIntegrationTestFileSystem
    {
    public:
        FsApiCloudBackupWorkStorage() NN_NOEXCEPT
            : FsApiIntegrationTestFileSystem(MountName)
        {
            NN_ABORT_UNLESS(nn::fs::MountCloudBackupWorkStorage(MountName, nn::fs::CloudBackupWorkStorageId::Nand).IsSuccess());
        }
        ~FsApiCloudBackupWorkStorage() NN_NOEXCEPT
        {
            nn::fs::Unmount(MountName);
        }
    };

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<FsApiCloudBackupWorkStorage> workStorage(new FsApiCloudBackupWorkStorage());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Integration));
        info->fileSystem = std::move(workStorage);
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int FatNameLengthMax   = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax   = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int MountNameLength    = (sizeof(MountName) - 1) + 1;  // マウント名に ':' を足した文字数
        static const int CloudBackupWorkStoragePathLength = 9;   // CloudBackupWorkStorage 内部で使用されるルートディレクトリ名の文字数 ("A:\cbwork")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax = FatPathLengthMax + MountNameLength - CloudBackupWorkStoragePathLength;
        attribute.filePathLengthMax      = FatPathLengthMax + MountNameLength - CloudBackupWorkStoragePathLength;

        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = 12 * 1024 * 1024 * 1024LL;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::MmcStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled)
                                         | static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::UsbStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isSupportedGetFileTimeStamp = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isReadOverloadsSupported = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCloudBackupWorkStorage(MountName, nn::fs::CloudBackupWorkStorageId::Nand));
    nn::util::SNPrintf(g_RootDirPath, sizeof(g_RootDirPath), "%s:/API_TEST", MountName);
    nn::fs::DeleteDirectoryRecursively(g_RootDirPath);
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_RootDirPath));
    nn::fs::Unmount(MountName);

    auto ret = RUN_ALL_TESTS();

    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountCloudBackupWorkStorage(MountName, nn::fs::CloudBackupWorkStorageId::Nand));
    nn::fs::DeleteDirectory(g_RootDirPath);
    nn::fs::Unmount(MountName);

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
