﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <memory>
#include <mutex>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_Optional.h>

#include <nnt/nntest.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

struct TestStat
{
public:
    static const int MessageLegnthMax = 512;

public:
    TestStat()
    {
        memset(this, 0, sizeof(*this));
    }

public:
    int64_t loopCount;  // 何らかのループ回数
    int64_t entryCount; // 何らかの処理単位数
    int64_t byteCount;  // 何らかの処理バイト数
    char message[MessageLegnthMax]; // 自由記述メッセージ
};


class TestStatHolder
{

public:
    TestStatHolder()
        : m_Mutex(false)
        , m_Priority(nn::fs::PriorityRaw_Normal)
        , m_IsExitRequired(false)
        , m_PreviousLoopCount(0)
    {
    }

    void SetName(const char* name)
    {
        m_Name = name;
    }

    nnt::fs::util::String GetName()
    {
        return m_Name;
    }

    void SetTestStat(const TestStat& testStat)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        memcpy(&m_TestStsat, &testStat, sizeof(TestStat));
    }


    void GetTestStat(TestStat* pOutValue)
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        memcpy(pOutValue, &m_TestStsat, sizeof(TestStat));
    }

    void SetPriority(nn::fs::PriorityRaw priority)
    {
        m_Priority = priority;
    }

    nn::fs::PriorityRaw GetPriority()
    {
        return m_Priority;
    }

    void RequireExit()
    {
        m_IsExitRequired = true;
    }

    bool IsExitRequired()
    {
        return m_IsExitRequired;
    }

    bool IsLoopCountUpdated()
    {
        bool isLoopCountUpdated = false;

        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        if(m_TestStsat.loopCount > m_PreviousLoopCount)
        {
            isLoopCountUpdated = true;
        }

        m_PreviousLoopCount = m_TestStsat.loopCount;
        return isLoopCountUpdated;
    }

private:
    TestStat m_TestStsat;
    nn::os::Mutex m_Mutex;
    nnt::fs::util::String m_Name;
    nn::fs::PriorityRaw m_Priority;
    bool m_IsExitRequired;
    int64_t m_PreviousLoopCount;
};


class TestStatManager
{
public:
    TestStatManager()
        : m_Mutex(false)
        , m_IsExitRequired(false)
    {
    }

    TestStatHolder* Register()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        std::unique_ptr<TestStatHolder> pHolder(new TestStatHolder);
        if(m_IsExitRequired)
        {
            pHolder->RequireExit();
        }
        m_pHolderList.push_back(std::move(pHolder));
        return m_pHolderList.back().get();
    }

    void ShowAllStat()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        NN_SDK_LOG("****** stats:               name  priority    loops  entries        bytes log\n");
        for (int i = 0; i < static_cast<int>(m_pHolderList.size()); i++)
        {
            TestStatHolder* pHolder = m_pHolderList[i].get();
            TestStat stat;
            pHolder->GetTestStat(&stat);
            NN_SDK_LOG("%32s: %8d %8lld %8lld %12lld %s\n",
                pHolder->GetName().c_str(),
                pHolder->GetPriority(),
                stat.loopCount,
                stat.entryCount,
                stat.byteCount,
                stat.message
            );
        }
        NN_SDK_LOG("\n\n");
    }

    void CheckAllLoopCount()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);

        for (int i = 0; i < static_cast<int>(m_pHolderList.size()); i++)
        {
            NN_ABORT_UNLESS(
                m_pHolderList[i]->IsLoopCountUpdated(),
                "Error: LoopCount has not been updated. (%s)\n",
                m_pHolderList[i]->GetName().c_str()
            );
        }
    }

    void RequireExitAll()
    {
        std::lock_guard<nn::os::Mutex> scopedLock(m_Mutex);
        m_IsExitRequired = true;
        for (int i = 0; i < static_cast<int>(m_pHolderList.size()); i++)
        {
            m_pHolderList[i]->RequireExit();
        }
    }

private:
    nn::os::Mutex m_Mutex;
    nnt::fs::util::Vector<std::unique_ptr<TestStatHolder>> m_pHolderList;
    bool m_IsExitRequired;


};
