﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/util/util_ScopeExit.h>

#include "testFs_Integration_AgingRw_TestCase.h"
#include "testFs_Integration_AgingRw_TestStat.h"
#include "testFs_Integration_AgingRw_IFileSystemProvider.h"

using namespace nn;
using namespace nn::fs;
using namespace nn::fs::detail;
using namespace nnt::fs::util;

namespace {

os::TlsSlot g_TestParameterTlsSlot;

struct TestParameter
{
    int64_t minRandomAccessSize;
};

TestParameter DefaultTestParameter = { 0 };

TestParameter* GetThreadLocalTestParameter()
{
    auto* pParam = reinterpret_cast<TestParameter*>(os::GetTlsValue(g_TestParameterTlsSlot));
    if (pParam == nullptr)
    {
        return &DefaultTestParameter;
    }
    else
    {
        return pParam;
    }
}

}

void AllocateTestParameterTlsSlot()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(os::AllocateTlsSlot(&g_TestParameterTlsSlot, nullptr));
}


namespace {

// データ生成の際の最大サイズ
int64_t g_maxDirectoryTreeSize  = static_cast<int64_t>(1024) * 1024 * 1024 * 6;

int64_t GetMaxDirectoryTreeSize()
{
    return g_maxDirectoryTreeSize;
}

}

void SetMaxDirectoryTreeSize(int64_t size)
{
    if (g_maxDirectoryTreeSize > size)
    {
        g_maxDirectoryTreeSize = size;
    }
}

// 個別ファイルデータキャッシュに使えるバッファの最大サイズ
size_t g_maxBufferSizeForIndividualFileDataCache = 0;

size_t GetBufferSizeForIndividualFileDataCache() NN_NOEXCEPT
{
    return g_maxBufferSizeForIndividualFileDataCache;
}

void SetBufferSizeForIndividualFileDataCache(size_t size) NN_NOEXCEPT
{
    g_maxBufferSizeForIndividualFileDataCache = size;
}

namespace {

// maxSize 以下のランダムなコーナーアドレスを得る
int64_t GetNearestAddressCornerCondition(int64_t maxSize)
{
    // コーナー条件のため様々な定数を定義
    const int64_t DeviceBufferSize  = 8   * 1024 * 1024;
    const int64_t WorkBufferSize    = 512 * 1024;
    const int64_t SdmmcAlignment    = 16  * 1024;
    const int64_t StorageSectorSize = 512;

    std::mt19937 random(nnt::fs::util::GetRandomSeed());

    // どれも 1 つも入らない場合は適当
    if (maxSize < StorageSectorSize)
    {
        return (random() % maxSize);
    }

    // ここは絶対降順
    int64_t baseSizeArray[] =
    {
        DeviceBufferSize,
        WorkBufferSize,
        SdmmcAlignment,
        StorageSectorSize
    };


    int64_t totalOffset = 0;
    for (int i = 0; i < sizeof(baseSizeArray) / sizeof(baseSizeArray[0]); i++)
    {
        std::uniform_real_distribution<> dist(0.0, 1.0);

        // 正方向
        if (totalOffset < baseSizeArray[i] || dist(random) > 0.5) // 負に行けない時は正方向にしか行かない
        {
            int64_t restSize = maxSize - totalOffset;
            // ベースサイズよりもサイズが小さい場合は何もしない
            if (restSize < (baseSizeArray[i]))
            {
                continue;
            }
            int64_t restBlock = restSize / baseSizeArray[i];
            int64_t blockNum = random() % restBlock;
            totalOffset += (baseSizeArray[i]) * blockNum;
        }
        // 負方向
        else
        {
            int64_t restBlock = totalOffset / baseSizeArray[i];
            int64_t blockNum = random() % restBlock;
            if (totalOffset < (baseSizeArray[i]) * blockNum)
            {
                NN_LOG("max Size   : 0x%llx ", maxSize);
                NN_LOG("totalOffset : 0x%llx ", totalOffset);
                NN_ABORT("=== Generate Offset Rogic Error!!!=== \n");
            }
            totalOffset -= (baseSizeArray[i]) * blockNum;
        }
    }
    // ここでは maxSize - totalOffset > 512 のはずなので大丈夫なはず
    if (maxSize - 512 < totalOffset || totalOffset < 0)
    {
        NN_LOG("max Size   : 0x%llx\n", maxSize);
        NN_LOG("totalOffset : 0x%llx\n", totalOffset);
        NN_ABORT("=== Generate Offset Rogic Error!!!=== \n");
    }
    // 10 未満なら終わり
    if (totalOffset < 10)
    {
        return totalOffset;
    }
    // 最後に 1 ケタずらす
    std::uniform_real_distribution<> dist(0.0, 1.0);
    int plusminus = (dist(random) > 0.5) ? -1 : 1;
    int randomDigis = random() % 10;
    totalOffset += randomDigis * plusminus;

    return totalOffset;
}

// ランダムなオフセット、サイズを得る
void GetRandomOffsetAndSize(int64_t* pOutOffset, size_t* pOutSize, int64_t fileSize, int64_t maxSize, bool isCorner)
{
    // isCorner の場合、 CornerRaito の割合でコーナーにする。残りはランダムにする。
    // TODO : 完全なコーナーオンリーの需要が出たら bool isCorner ではなく struct にしてここは switch にする
    const float CornerRaito = 0.5;
    std::mt19937 random(nnt::fs::util::GetRandomSeed());
    std::uniform_real_distribution<> dist(0.0, 1.0);

    int64_t offset;
    int64_t size;

    if (isCorner && (dist(random) > CornerRaito))
    {
        // とりあえずコーナー的なオフセットを決める
        offset = GetNearestAddressCornerCondition(fileSize);
        // サイズ上限はバッファサイズかファイルの残りサイズの小さい方
        auto sizeLimit = std::min((fileSize - offset), maxSize);
        size = GetNearestAddressCornerCondition(sizeLimit);
    }
    else
    {
        offset = random() % fileSize;
        auto minSize = GetThreadLocalTestParameter()->minRandomAccessSize;
        size   = std::min(static_cast<int64_t>(minSize + random()), maxSize);
    }
    if (offset + size > fileSize)
    {
        size = fileSize - offset;
    }

    *pOutOffset = offset;
    *pOutSize   = static_cast<size_t>(size);
}

/**
* @brief ファイルをランダムなオフセット・サイズで Read します。
* @param[in]  path            ファイルのパス
* @param[in]  isCorner        AtCorner テストかどうか : オフセット・アドレスの選び方に影響
* @param[in]  isCheck         Read 後 verify するかどうか
*                             verify は 32BitCount で行う
*/
Result RandomRead(const char* path, bool isCorner, bool isCheck, bool isFileNameOffsetEnable)
{
    FileHandle handle;
    NN_FS_TEST_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{
        CloseFile(handle);
    };

    int64_t fileSize = 0;
    NN_FS_TEST_RESULT_DO(GetFileSize(&fileSize, handle));
    if (fileSize == 0)
    {
        NN_RESULT_SUCCESS;
    }

    //　とりあえず 16MB に設定する
    const size_t BufferSize = 16 * 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);
    if (buffer.get() == nullptr)
    {
        NN_LOG("BufferAllocate Error %s , l.%d\n", __FUNCTION__, __LINE__);
        MyAbort(nn::fs::ResultUnknown());
    }
    memset(buffer.get(), 0, BufferSize);

    int64_t offset = 0;
    size_t readSize = 0;

    auto fileDataCacheBuffer = AllocateBuffer(16);
    fileDataCacheBuffer.reset();

    // ファイルサイズが設定値以下の場合に、ランダムに個別ファイルデータキャッシュを有効化する。
    // バッファが取れなかったときはやめる
    if (fileSize < GetBufferSizeForIndividualFileDataCache())
    {
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        std::uniform_real_distribution<> dist(0.0, 1.0);
        if (dist(random) > 0.5)
        {
            fileDataCacheBuffer = AllocateBuffer(fileSize);
            if (fileDataCacheBuffer != nullptr)
            {
                NN_RESULT_TRY(EnableIndividualFileDataCache(path, fileDataCacheBuffer.get(), fileSize))
                    NN_RESULT_CATCH(ResultIndividualFileDataCacheAlreadyEnabled)
                    {
                        // fail through
                    }
                    NN_RESULT_CATCH_ALL
                    {
                        NN_FS_TEST_RESULT_DO(NN_RESULT_CURRENT_RESULT);
                    }
                NN_RESULT_END_TRY;
            }
        }
    }

    GetRandomOffsetAndSize(&offset, &readSize, fileSize, BufferSize, isCorner);

    auto result = (ReadFile(handle, offset, buffer.get(), readSize));
    if (result.IsFailure())
    {
        NN_LOG("offset = 0x%llx, size = 0x%llx, fileSize = %llx\n", offset, readSize, fileSize);
        NN_FS_TEST_RESULT_DO(result);
    }
    if (isCheck)
    {
        if(isFileNameOffsetEnable)
        {
            offset += GetOffsetByFileName(path);
        }
        if (!IsFilledWith32BitCount(buffer.get(), readSize, offset))
        {
            NN_LOG("read data mismatch \n");
            NN_LOG("path = %s fileSize = %lld\n", path, fileSize);
            NN_LOG("offset = %lld size= %d nameOffset1 = %lld nameOffset2 = %lld\n", offset, readSize, GetOffsetByFileName("(KAZKwTB6hQ@Qn#QKsS(aJBK5'Q", strnlen("(KAZKwTB6hQ@Qn#QKsS(aJBK5'Q", 32)), GetOffsetByFileName(path));
            return nn::fs::ResultUnknown();
        }
    }

    DisableIndividualFileDataCache(path);

    NN_RESULT_SUCCESS;
}

/**
* @brief ファイルをランダムなオフセット・サイズで Write します。
* @param[in]  path                ファイルのパス
* @param[in]  mountNameForCommit  nullptr 以外が入っていた場合、そのパスに向けて CommitSaveData を行う
* @param[in]  isCorner            AtCorner テストかどうか : オフセット・アドレスの選び方に影響
* @param[in]  isCheck             Write 後 Read -> verify するかどうか
* @param[in]  isRandomValue       Write するデータをランダムにするか 32BitCount にするか
*/
Result RandomWrite(const char* path, const char* mountNameForCommit, bool isCorner, bool isCheck, bool isRandomValue)
{
    //　とりあえず 16MB に設定する
    const size_t BufferSize = 16 * 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);
    if (buffer.get() == nullptr)
    {
        NN_LOG("BufferAllocate Error %s , l.%d\n", __FUNCTION__, __LINE__);
        MyAbort(nn::fs::ResultUnknown());
    }
    auto tempBuffer = AllocateBuffer(BufferSize);
    if (tempBuffer.get() == nullptr)
    {
        NN_LOG("BufferAllocate Error %s , l.%d\n", __FUNCTION__, __LINE__);
        MyAbort(nn::fs::ResultUnknown());
    }

    int64_t fileSize = 0;
    int64_t offset = 0;
    size_t size = 0;

    auto fileDataCacheBuffer = AllocateBuffer(16);
    fileDataCacheBuffer.reset();

    {
        FileHandle handle;
        NN_FS_TEST_RESULT_DO(OpenFile(&handle, path, OpenMode_Write));
        NN_UTIL_SCOPE_EXIT{
            CloseFile(handle);
            // mountName が指定されていたら CommitSaveData を行う
            if(mountNameForCommit != nullptr)
            {
                // ランダムに CommitSaveData を呼ぶ
                std::mt19937 random(nnt::fs::util::GetRandomSeed());
                std::uniform_real_distribution<> dist(0.0, 1.0);
                if(dist(random) > 0.5)
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS(CommitSaveData(mountNameForCommit));
                }
            }
        };

        NN_FS_TEST_RESULT_DO(GetFileSize(&fileSize, handle));
        if (fileSize == 0)
        {
            NN_RESULT_SUCCESS;
        }

        GetRandomOffsetAndSize(&offset, &size, fileSize, BufferSize, isCorner);

        // ファイルサイズが設定値以下の場合に、ランダムに個別ファイルデータキャッシュを有効化する。
        // バッファが取れなかったときはやめる
        if (fileSize < GetBufferSizeForIndividualFileDataCache())
        {
            std::mt19937 random(nnt::fs::util::GetRandomSeed());
            std::uniform_real_distribution<> dist(0.0, 1.0);
            if (dist(random) > 0.5)
            {
                fileDataCacheBuffer = AllocateBuffer(fileSize);
                if (fileDataCacheBuffer != nullptr)
                {
                    NN_RESULT_TRY(EnableIndividualFileDataCache(path, fileDataCacheBuffer.get(), fileSize))
                        NN_RESULT_CATCH(ResultIndividualFileDataCacheAlreadyEnabled)
                        {
                            // fail through
                        }
                        NN_RESULT_CATCH_ALL
                        {
                            NN_FS_TEST_RESULT_DO(NN_RESULT_CURRENT_RESULT);
                        }
                    NN_RESULT_END_TRY;
                }
            }
        }

        // isRandomValue かどうかで埋める値をスイッチする
        if (isRandomValue) FillBufferWithRandomValue(buffer.get(), BufferSize);
        else FillBufferWith32BitCount(buffer.get(), size, offset);

        memcpy(tempBuffer.get(), buffer.get(), BufferSize);

        auto result = (WriteFile(handle, offset, buffer.get(), size, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush)));
        if (result.IsFailure())
        {
            NN_LOG("offset = 0x%llx, size = 0x%llx, fileSize = %llx\n", offset, size, fileSize);
            NN_FS_TEST_RESULT_DO(result);
        }
    }

    if(isCheck)
    {
        FileHandle handle;
        NN_FS_TEST_RESULT_DO(OpenFile(&handle, path, OpenMode_Read));
        NN_UTIL_SCOPE_EXIT{
            CloseFile(handle);
        };

        auto result = (ReadFile(handle, offset, buffer.get(), size));
        if (result.IsFailure())
        {
            NN_LOG("offset = 0x%llx, size = 0x%llx, fileSize = %llx\n", offset, size, fileSize);
            NN_FS_TEST_RESULT_DO(result);
        }

        if (memcmp(tempBuffer.get(), buffer.get(), size) != 0)
        {
            NN_LOG("[error] %s %d path : %s\n", __FUNCTION__, __LINE__, path);
            NN_LOG("offset = 0x%llx, size = 0x%llx, fileSize = %llx\n", offset, size, fileSize);
            size_t counter = 0;
            for (size_t i = 0; i < size; i++)
            {
                if (*(tempBuffer.get() + i) != *(buffer.get() + i))
                {
                    NN_LOG("%d exp = 0x%02x act = 0x%02x\n", *(tempBuffer.get() + i), *(buffer.get() + i));
                    counter++;
                    if (counter > 100) break;
                }
            }
            return nn::fs::ResultUnknown();
        }
    }

    DisableIndividualFileDataCache(path);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result RandomRead(const char* path, bool isCorner)
{
    auto result = RandomRead(path, isCorner, false, false);
    if (result.IsFailure())
    {
        NN_LOG("error occrued %s : %s\n", __FUNCTION__, path);
    }
    return result;
}

Result RandomReadWithCheck(const char* path, bool isCorner, bool isFileNameOffsetEnable)
{
    auto result = RandomRead(path, isCorner, true, isFileNameOffsetEnable);
    if (result.IsFailure())
    {
        NN_LOG("error occrued %s : %s\n", __FUNCTION__, path);
    }
    return result;
}
Result RandomReadWithCheck(const char* path, bool isCorner)
{
    return RandomRead(path, isCorner, true, false);
}


Result WriteReadVerifyRandomDataByRandomOffsetAndSize(const char* path, const char* mountNameForCommit, bool isCorner)
{
    auto result = RandomWrite(path, mountNameForCommit, isCorner, true, true);
    if (result.IsFailure())
    {
        NN_LOG("error occrued %s : %s\n", __FUNCTION__, path);
    }
    return result;
}

Result WriteWith32BitCount(const char* path, const char* mountNameForCommit, bool isCorner)
{
    auto result = RandomWrite(path, mountNameForCommit, isCorner, false, false);
    if (result.IsFailure())
    {
        NN_LOG("error occrued %s : %s\n", __FUNCTION__, path);
    }
    return result;
}

}

// テストケースの実装

// fs 内を DumpDirectoryRecursive する
void TestCaseReadOnlyDumpDirectoryRecursive(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    auto mountName = pFsProvider->GetMountName();

    const size_t BufferSize = 1024 * 1024;
    auto buffer = AllocateBuffer(BufferSize);
    if (buffer.get() == nullptr)
    {
        NN_LOG("BufferAllocate Error %s , l.%d\n", __FUNCTION__, __LINE__);
        MyAbort(nn::fs::ResultUnknown());
    }

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DumpDirectoryRecursive((mountName + (":/")).c_str(), false, buffer.get(), BufferSize, true));

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        pTestStatHolder->SetTestStat(stat);
    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

namespace {
void TestCaseReadOnlyRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder, bool isCorner)
{
    auto mountName = pFsProvider->GetMountName();

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, (mountName + (":/")).c_str()));

    // ファイル数が 0 なら以下の処理は行わない
    if (fileList.size() == 0)
    {
        return;
    }

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        // fs 内のファイルをランダムに選びランダムなレンジで read する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        int fileIndex = random() % fileList.size();
        NN_ABORT_UNLESS_RESULT_SUCCESS(RandomRead(fileList[fileIndex].c_str(), isCorner));

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "read %s", fileList[fileIndex].c_str());
        pTestStatHolder->SetTestStat(stat);
    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

// fs 内のファイルをランダムに選びランダムなレンジで read + verify / write する
void TestCaseReadWriteRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder,bool isCommit, bool isCorner)
{
    auto mountName = pFsProvider->GetMountName();

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, (mountName + (":/")).c_str()));

    // ファイル数が 0 なら以下の処理は行わない
    if (fileList.size() == 0)
    {
        return;
    }

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        // fs 内のファイルをランダムに選びランダムなレンジで read or write with verify する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        int fileIndex = random() % fileList.size();
        std::uniform_real_distribution<> dist(0.0, 1.0);

        bool isRead = dist(random) > 0.5;

        if (isRead)
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(RandomRead(fileList[fileIndex].c_str(), isCorner));
        }
        else
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(WriteReadVerifyRandomDataByRandomOffsetAndSize(fileList[fileIndex].c_str(), isCommit ? mountName.c_str() : nullptr, isCorner));
        }

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "%s %s", isRead ? "read " : "write", fileList[fileIndex].c_str());
        pTestStatHolder->SetTestStat(stat);

    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

// fs 内のファイルにランダムに 32bit カウントデータを書き込みんだ後ランダムに Read して verify する
void TestCaseReadWriteWith32BitCount(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder, bool isCorner)
{
    auto mountName = pFsProvider->GetMountName();

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, (mountName + (":/")).c_str()));

    // ファイル数が 0 なら以下の処理は行わない
    if (fileList.size() == 0)
    {
        return;
    }

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        // fs 内のファイルをランダムに選びランダムなレンジで read して verify する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        int fileIndex = random() % fileList.size();
        NN_ABORT_UNLESS_RESULT_SUCCESS(WriteWith32BitCount(fileList[fileIndex].c_str(), nullptr, isCorner));
        NN_ABORT_UNLESS_RESULT_SUCCESS(RandomReadWithCheck(fileList[fileIndex].c_str(), isCorner));
        fileIndex = random() % fileList.size();
        NN_ABORT_UNLESS_RESULT_SUCCESS(RandomReadWithCheck(fileList[fileIndex].c_str(), isCorner));

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "read %s", fileList[fileIndex].c_str());
        pTestStatHolder->SetTestStat(stat);
    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

// fs 内のファイルをランダムに選びランダムなレンジで read したあと 32Bit カウントデータで verify する
// TODO : TestCaseReadOnlyRandom のコピペを何とかする
void TestCaseReadOnlyRandomWith32BitCountCheck(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder, bool isCorner, bool isFileNameOffsetEnable)
{
    auto mountName = pFsProvider->GetMountName();

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, (mountName + (":/")).c_str()));

    // ファイル数が 0 なら以下の処理は行わない
    if (fileList.size() == 0)
    {
        return;
    }

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        // fs 内のファイルをランダムに選びランダムなレンジで read して verify する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        int fileIndex = random() % fileList.size();
        NN_ABORT_UNLESS_RESULT_SUCCESS(RandomReadWithCheck(fileList[fileIndex].c_str(), isCorner, isFileNameOffsetEnable));

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "read %s", fileList[fileIndex].c_str());
        pTestStatHolder->SetTestStat(stat);
    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

} // nemaspace

void TestCaseReadOnlyRandomAtCorner(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandom(pFsProvider, pTestStatHolder, true);
}


// fs 内のファイルをランダムに選びランダムなレンジで read する
void TestCaseReadOnlyRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandom(pFsProvider, pTestStatHolder, false);
}

void TestCaseReadOnlyRandomLargeAccessSize(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    GetThreadLocalTestParameter()->minRandomAccessSize = 2 * 1024 * 1024;
    TestCaseReadOnlyRandom(pFsProvider, pTestStatHolder, false);
}


void TestCaseReadWriteRandomAtCorner(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadWriteRandom(pFsProvider, pTestStatHolder, false, true);
}

void TestCaseReadWriteCommitRandomAtCorner(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadWriteRandom(pFsProvider, pTestStatHolder, true, true);
}

void TestCaseReadWriteRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadWriteRandom(pFsProvider, pTestStatHolder, false, false);
}

void TestCaseReadWriteWith32BitCountAtCorner(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadWriteWith32BitCount(pFsProvider, pTestStatHolder, true);
}

void TestCaseReadWriteWith32BitCount(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadWriteWith32BitCount(pFsProvider, pTestStatHolder, false);
}

void TestCaseReadOnlyRandomWith32BitCountCheckAtCorner(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandomWith32BitCountCheck(pFsProvider, pTestStatHolder, true, false);
}

void TestCaseReadOnlyRandomWith32BitCountCheck(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandomWith32BitCountCheck(pFsProvider, pTestStatHolder, false, false);
}

void TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffset(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandomWith32BitCountCheck(pFsProvider, pTestStatHolder, true, true);
}

void TestCaseReadOnlyRandomWith32BitCountCheckAtCornerByFileNameOffsetLargeAccessSize(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    GetThreadLocalTestParameter()->minRandomAccessSize = 2 * 1024 * 1024;
    TestCaseReadOnlyRandomWith32BitCountCheck(pFsProvider, pTestStatHolder, true, true);
}

void TestCaseReadOnlyRandomWith32BitCountCheckByFileNameOffset(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseReadOnlyRandomWith32BitCountCheck(pFsProvider, pTestStatHolder, false, true);
}

// C:/Windows/temp/dataSrc にランダムなディレクトリツリーを生成する
void TestCreateRandomDirectoryTree(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    NN_LOG("====== Start Create Directory\n");

    auto testDirectoryPath = pFsProvider->GetMountName() + ":/dataSrc/";
    DeleteDirectoryRecursively(testDirectoryPath.c_str());

    TestStat stat;
    const int64_t TotalSizeForLargeFile  = 10;
    const int64_t TotalSizeForMiddleFile = 100;
    const int64_t TotalSizeForSmallFile  = 1000;
    MY_ABORT_UNLESS_RESULT_SUCCESS(CreateDirectory(testDirectoryPath.c_str()), "<%s>\n", pFsProvider->GetMountName().c_str());

    // DirectoryTree 全体で maxDirectoryTreeSize 程度のサイズになるように調整
    const int64_t maxDirectoryTreeSize = GetMaxDirectoryTreeSize();
    NN_SDK_LOG("DataSize: %lld\n", maxDirectoryTreeSize);
    const int64_t totalSize = static_cast<int64_t>(maxDirectoryTreeSize / 3);

    NN_LOG("Create Large\n");
    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Create Large");
    stat.loopCount++;
    pTestStatHolder->SetTestStat(stat);
    MY_ABORT_UNLESS_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectoryTreeRandomlyByFileNameOffset(testDirectoryPath.c_str(), totalSize, TotalSizeForLargeFile), pFsProvider->GetMountName().c_str());

    NN_LOG("Create Middle\n");
    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Create Middle");
    stat.loopCount++;
    pTestStatHolder->SetTestStat(stat);
    MY_ABORT_UNLESS_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectoryTreeRandomlyByFileNameOffset(testDirectoryPath.c_str(), totalSize, TotalSizeForMiddleFile), pFsProvider->GetMountName().c_str());

    NN_LOG("Create Small\n");
    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Create Small");
    stat.loopCount++;
    pTestStatHolder->SetTestStat(stat);
    MY_ABORT_UNLESS_RESULT_SUCCESS(nnt::fs::util::CreateTestDirectoryTreeRandomlyByFileNameOffset(testDirectoryPath.c_str(), totalSize, TotalSizeForSmallFile), pFsProvider->GetMountName().c_str());

    NN_LOG("====== End Create Directory\n");
    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    stat.loopCount++;
    pTestStatHolder->SetTestStat(stat);
}

void TestDisturbDirectoryTreeForPatchTest(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    NN_LOG("====== Start Disturb Directory\n");

    auto testDirectoryPath = pFsProvider->GetMountName() + ":/dataSrc/";

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, testDirectoryPath.c_str()));

    // ファイル数が 0 なら以下の処理は行わない
    if (fileList.size() == 0)
    {
        return;
    }

    TestStat stat;
    stat.loopCount = 0;
    while (stat.loopCount < 5000)
    {
        // fs 内のファイルをランダムに選びランダムなレンジで read して verify する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        int fileIndex = random() % fileList.size();
        NN_ABORT_UNLESS_RESULT_SUCCESS(RandomWrite(fileList[fileIndex].c_str(), nullptr, false, false, true));
        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "read %s", fileList[fileIndex].c_str());
        pTestStatHolder->SetTestStat(stat);
    }

    NN_LOG("====== End Disturb Directory\n");
    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

// fs 内のファイル、ディレクトリをランダムに選び削除、作成する
void TestCaseCreateDeleteRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder, int64_t totalSize, int64_t totalEntryCount)
{
    auto mountName = pFsProvider->GetMountName();
    int64_t restSize = totalSize;

    Vector<String> fileList;
    Vector<String> dirList;

    NN_ABORT_UNLESS_RESULT_SUCCESS(ListDirectoryRecursive(&fileList, &dirList, (mountName + (":/")).c_str()));

    TestStat stat;

    while(!pTestStatHolder->IsExitRequired())
    {
        // fs 内のファイル/ディレクトリをランダムに選び削除、作成する
        std::mt19937 random(nnt::fs::util::GetRandomSeed());
        std::uniform_real_distribution<> dist(0.0, 1.0);

        bool isFile = dist(random) > 0.5;
        auto listSize = isFile ? fileList.size() : dirList.size();
        bool isCreate;
        if (listSize <= (isFile ? 0 : 1))
        {
            isCreate = true;
        }
        else if (listSize > totalEntryCount / 2)
        {
            isCreate = false;
        }
        else
        {
            isCreate = dist(random) > 0.5;
        }

        String path;
        if (isCreate)
        {
            auto index = random() % dirList.size();
            String parentDirectory = dirList[index];

            auto name = GenerateRandomLengthEntryName(32);
            path = parentDirectory + "/" + name;
            if (path.length() > 90)
            {
                continue;
            }

            if (isFile)
            {
                // 適当にサイズ決める
                int64_t size = static_cast<int64_t>(dist(random) * totalSize / totalEntryCount * 2);
                size = std::min(size, restSize);

                auto result = CreateFileWith32BitCount(path.c_str(), size, 0);
                if (ResultTooLongPath::Includes(result) ||
                    ResultPathAlreadyExists::Includes(result))
                {
                    continue; // skip
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                restSize -= size;
                fileList.push_back(path);
            }
            else
            {
                auto result = CreateDirectory(path.c_str());
                if (ResultPathAlreadyExists::Includes(result))
                {
                    continue; // skip
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                dirList.push_back(path);
            }
        }
        else
        {
            auto index = random() % listSize;
            if (isFile)
            {
                path = fileList[index];

                FileHandle handle;
                int64_t size;
                NN_ABORT_UNLESS_RESULT_SUCCESS(OpenFile(&handle, path.c_str(), OpenMode_Read));
                NN_ABORT_UNLESS_RESULT_SUCCESS(GetFileSize(&size, handle));
                CloseFile(handle);

                NN_ABORT_UNLESS_RESULT_SUCCESS(DeleteFile(path.c_str()));

                restSize += size;
                fileList.erase(fileList.begin() + index);
            }
            else
            {
                path = dirList[index];

                auto result = DeleteDirectory(path.c_str());
                if (ResultDirectoryNotEmpty::Includes(result))
                {
                    continue; // skip
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                dirList.erase(dirList.begin() + index);
            }
        }

        nn::os::YieldThread();

        // テスト状況のセット(toriaezu)
        stat.loopCount++;
        stat.entryCount = fileList.size() + dirList.size();
        util::SNPrintf(stat.message, stat.MessageLegnthMax, "%s(%s) %s", isCreate ? "create" : "delete", isFile ? "File" : "Dir", path.c_str());
        pTestStatHolder->SetTestStat(stat);

    }

    util::SNPrintf(stat.message, stat.MessageLegnthMax, "Finalized");
    pTestStatHolder->SetTestStat(stat);
}

void TestCaseCreateDeleteRandom(IFileSystemProvider* pFsProvider, TestStatHolder* pTestStatHolder)
{
    TestCaseCreateDeleteRandom(pFsProvider, pTestStatHolder, 1024 * 1024 * 10, 30);
}
