﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <nn/util/util_FormatString.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/base/testBase_Exit.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nnt/nnt_Argument.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_SpeedEmulation.h>

using namespace nn::fs;
using namespace nnt::fs::util;


namespace
{
    nn::os::ThreadType g_Thread;
    static const size_t ThreadStackSize = 64 * 1024;
    char g_ThreadName[] = "FsAging";
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];
}

void* operator new(size_t size) NN_NOEXCEPT
{
    return malloc(size);
}

void operator delete(void* p) NN_NOEXCEPT
{
    free(p);
}

void RunThreadFunction(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);
    const size_t WorkBufferSize = 64 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    NN_LOG("start thread\n");
    // 幅優先探索
    NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("rom:/", true, workBuffer.get(), WorkBufferSize, true));
    NN_LOG("done thread\n");
}

//!< ROM をマウント後 ROM 内の特定ファイルを各バッファサイズで読み込む時間を計測する
TEST(AgingTestReadRom, ReadRom)
{
    size_t size;
    NNT_EXPECT_RESULT_SUCCESS(QueryMountRomCacheSize(&size));

    auto buffer = AllocateBuffer(size);
    ASSERT_NE(nullptr, buffer);

    const size_t WorkBufferSize = 64 * 1024 * 1024;
    auto workBuffer = AllocateBuffer(WorkBufferSize);

    NNT_ASSERT_RESULT_SUCCESS(MountRom("rom", buffer.get(), size));

    // 複数スレッドでアクセス
    int priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread());
    nn::Result result = nn::os::CreateThread( &g_Thread, RunThreadFunction, nullptr, g_ThreadStack, ThreadStackSize, priority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::SetThreadNamePointer(&g_Thread, g_ThreadName);
    nn::os::StartThread( &g_Thread );

    NN_LOG("start main thread\n");
    NNT_EXPECT_RESULT_SUCCESS(DumpDirectoryRecursive("rom:/", true, workBuffer.get(), WorkBufferSize));
    NN_LOG("done main thread\n");

    nn::os::WaitThread( &g_Thread );
    nn::os::DestroyThread( &g_Thread );
    Unmount("rom");
}


extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    SetAllocator(Allocate, Deallocate);
    ResetAllocateCount();
    nn::fs::SetEnabledAutoAbort(false);

    auto testResult = RUN_ALL_TESTS();
    EXPECT_FALSE(CheckMemoryLeak());

    nnt::Exit(testResult);
}
