﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SaveDataForDebug.h>

#include "testFs_FsLib_Mount.h"
#include "testFs_FsLib_Mount_Bis.h"

// SetSaveDataRootPath を一度呼ぶと元の状態に戻せないので最後に実行する

namespace
{
    nn::Result ConfirmFileTest(const char* fileNameBuffer)
    {
        NN_RESULT_DO(nn::fs::MountHostRoot());
        // Confirm File Existance
        nn::fs::FileHandle saveHandle;
        NN_RESULT_DO(nn::fs::OpenFile(&saveHandle, fileNameBuffer, nn::fs::OpenMode_Read));
        nn::fs::CloseFile(saveHandle);
        nn::fs::UnmountHostRoot();
        NN_RESULT_SUCCESS;
    }

    nn::Result CreateDirectoryTest(const char* createDirBuffer)
    {
        NN_RESULT_DO(nn::fs::MountHostRoot());
        // Create Directory
        NN_RESULT_DO(nn::fs::CreateDirectory(createDirBuffer));
        nn::fs::UnmountHostRoot();
        NN_RESULT_SUCCESS;
    }

    nn::Result DeleteDirectoryTest(const char* deleteDirBuffer)
    {
        NN_RESULT_DO(nn::fs::MountHostRoot());
        // Delete Directory
        NN_RESULT_DO(nn::fs::DeleteDirectoryRecursively(deleteDirBuffer));
        nn::fs::UnmountHostRoot();
        NN_RESULT_SUCCESS;
    }

    struct SaveDataStructure
    {
        int value;
    };

    nn::Result MountTest()
    {
        // マウント名 "save" としてセーブデータをマウント
        NN_RESULT_DO(nn::fs::MountSaveDataForDebug("save"));
        const size_t FileSize = sizeof(SaveDataStructure);

        // ファイルが存在しない場合は作成
        {
            nn::fs::DirectoryEntryType directoryEntryType;
            if( nn::fs::GetEntryType(&directoryEntryType, "save:/file").IsFailure() )
            {
                NN_RESULT_DO(nn::fs::CreateFile("save:/file", FileSize));

                // 初期データを書き込み
                nn::fs::FileHandle fileHandle;
                NN_RESULT_DO(nn::fs::OpenFile(&fileHandle, "save:/file", nn::fs::OpenMode_Write));

                SaveDataStructure data = { 0 };
                NN_RESULT_DO(nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));

                nn::fs::CloseFile(fileHandle);
            }
        }
        // セーブデータの更新内容をコミット
        NN_RESULT_DO(nn::fs::CommitSaveData("save"));

        // アンマウントする
        nn::fs::Unmount("save");
        NN_RESULT_SUCCESS;
    }
}

#if 0
TEST(MountSaveBasic, MountSaveDefault)
{
    NNT_EXPECT_RESULT_SUCCESS(MountTest());
}
#endif

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
TEST(MountSaveBasic, MountSaveRoot)
{
    nnt::fs::util::DeleteAllTestSaveData();

    nn::fs::SetSaveDataRootPath(g_TestDirPath.GetPath().c_str());
    NNT_EXPECT_RESULT_SUCCESS(MountTest());

    nn::util::optional<nn::fs::SaveDataInfo> info;
    nnt::fs::util::FindSaveData(&info, nn::fs::SaveDataSpaceId::User, [&](const nn::fs::SaveDataInfo& i) NN_NOEXCEPT
        {
            return i.applicationId.value == nnt::fs::util::UserSaveDataApplicationId && i.saveDataType == nn::fs::SaveDataType::Account && i.saveDataUserId == nn::fs::InvalidUserId;
        });
    EXPECT_NE(info, nn::util::nullopt);

    char path[256];
    nn::util::SNPrintf(path, sizeof(path), "%s/%016llx", g_TestDirPath.GetPath().c_str(), info->saveDataId);

    NNT_EXPECT_RESULT_SUCCESS(ConfirmFileTest((nnt::fs::util::String(path) + "/0/file").c_str()));
    // NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(path));

    // システムセーブはリダイレクトされないこと
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::CreateAndMountSystemSaveData("save", 0x8000000000000003));
    nn::fs::DirectoryEntryType type;
    NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, GetEntryType(&type, g_TestDirPath.GetPath().append("/8000000000000003").c_str()));
    nn::fs::Unmount("save");
    nn::fs::UnmountHostRoot();
}
#endif

// 以下のテストは fssrv 側の Indexer の上書きができないため、どれか 1 のテストしか回すことができません。
// TORIAEZU で if 0 と if 1 で切り替えます。
#if defined(NN_BUILD_CONFIG_OS_WIN32)

#if 0
TEST(MountSaveBasic, MountSaveAndBisRoot)
{
    // TODO: SetBisRootForHostHelper の削除

    nn::fs::SetSaveDataRootPath(nnt::fs::util::GetHostTemporaryPath());
    SetBisRootForHostHelper setBisRoot(nn::fs::BisPartitionId::System, nnt::fs::util::GetHostTemporaryPath());
    NNT_EXPECT_RESULT_SUCCESS(MountTest());
    NNT_EXPECT_RESULT_SUCCESS(ConfirmFileTest(GetTestPath("/0000000000000001/0/file").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(GetTestPath("/0000000000000001").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(GetTestPath("/save").c_str()));
}
#endif

#if 0
TEST(MountSaveBasic, MountBisRoot)
{
    // TODO: SetBisRootForHostHelper の削除

    SetBisRootForHostHelper setBisRootUser(nn::fs::BisPartitionId::User, nnt::fs::util::GetHostTemporaryPath());
    SetBisRootForHostHelper setBisRootSystem(nn::fs::BisPartitionId::System, nnt::fs::util::GetHostTemporaryPath());
    NNT_EXPECT_RESULT_SUCCESS(MountTest());
    NNT_EXPECT_RESULT_SUCCESS(ConfirmFileTest(GetTestPath("/save/0000000000000001/0/file").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(GetTestPath("/save").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(GetTestPath("/saveMeta").c_str()));
}

#endif

#endif // WIN 版のみ

// MountSaveDataForDebug は kvdb::AutoBuffer で new を使うため New/Delete のチェックを行わない
TEST(Mount, VariousSaveDataRootPath)
{
    char testDir[256];
    char path[256];

    // テスト用のディレクトリ
    nn::util::SNPrintf(testDir, sizeof(testDir), "%s\\VariousSaveDataRootPath", g_TestDirPath.GetPath().c_str());
    NNT_EXPECT_RESULT_SUCCESS(CreateDirectoryTest(testDir));

    // SetSaveDataRootPath で存在するパスを指定する
    nn::fs::SetSaveDataRootPath(testDir);
    NNT_EXPECT_RESULT_SUCCESS(MountTest());

    // SetSaveDataRootPath でセパレータで終わるパスを指定する
    nn::util::SNPrintf(path, sizeof(path), "%s\\", testDir);
    nn::fs::SetSaveDataRootPath(path);
    NNT_EXPECT_RESULT_SUCCESS(MountTest());

    // SetSaveDataRootPath で存在しないドライブを指定する
    nn::fs::SetSaveDataRootPath("X:\\");
    NNT_EXPECT_RESULT_FAILURE(
        nn::fs::ResultTargetNotFound,
        nn::fs::MountSaveDataForDebug("mount"));

    // テスト用のディレクトリを削除
    NNT_EXPECT_RESULT_SUCCESS(DeleteDirectoryTest(testDir));


#if defined(NN_BUILD_CONFIG_OS_WIN32)
    // SaveDataRootPath の設定を解除
    nn::fs::SetSaveDataRootPath("");
#else
    // SaveDataRootPath を存在するパスに設定しておく
    nn::fs::SetSaveDataRootPath(g_TestDirPath.GetPath().c_str());
#endif
}
