﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_Application.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_Content.h>
#include <nn/fs/fs_Logo.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
    #include <detail/fssrv_RuntimeConfiguration.h>
#endif

#include "testFs_FsLib_Mount.h"

using namespace nn::fs;

TEST(MountContent, ParseDirOnIllegalStorage)
{
    NNT_ASSERT_RESULT_SUCCESS(MountBis(BisPartitionId::User, nullptr));

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory("@User:/data/"));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile("@User:/data/test.bin", 1024));

    {
        const nn::ncm::ProgramId dummyId = {};
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountContent("content", "@User:/", ContentType::ContentType_Meta));
        NNT_EXPECT_RESULT_FAILURE(ResultPermissionDenied, MountContent("content", "@User:/", dummyId, ContentType::ContentType_Manual));
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively("@User:/data/"));
    Unmount("@User");
}

TEST(MountContent, ParseDirOnLegalStorage)
{
    NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_TestDirPath.GetPath().append("/data/").c_str()));
    NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(g_TestDirPath.GetPath().append("/data/test.bin").c_str(), 1024));

    {
        const nn::ncm::ProgramId dummyId = {};
        NNT_EXPECT_RESULT_SUCCESS(MountContent("content", ("@Host:/" + g_TestDirPath.GetPath() + "/").c_str(), ContentType::ContentType_Meta));
        Unmount("content");
        NNT_EXPECT_RESULT_SUCCESS(MountContent("content", ("@Host:/" + g_TestDirPath.GetPath() + "/").c_str(), dummyId, ContentType::ContentType_Manual));
        Unmount("content");
    }

    NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(g_TestDirPath.GetPath().append("/data/").c_str()));
    UnmountHostRoot();
}

#if 0
TEST(Mount, MountApplicationPackage)
{
    auto path = String("@Host:/D:/tmp/test.nsp");

    const char* mountAppName = "Appli";
    NNT_ASSERT_RESULT_SUCCESS(SearchContentMeta(mountAppName, path));

    const char* mountMetaName = "AppliMeta";
    std::unique_ptr<char> buffer;
    size_t metaSize;
    NNT_EXPECT_RESULT_SUCCESS(ReadContentMeta(&metaSize, mountMetaName, &buffer, path));
}

TEST(Mount, MountApplicationPackageDirectory)
{
    auto path = String("@Host:/D:/tmp/test.nspd/");

    const char* mountAppName = "Appli";
    NNT_ASSERT_RESULT_SUCCESS(SearchContentMeta(mountAppName, path));

    const char* mountMetaName = "AppliMeta";
    std::unique_ptr<char> buffer;
    size_t metaSize;
    NNT_EXPECT_RESULT_SUCCESS(ReadContentMeta(&metaSize, mountMetaName, &buffer, path));

    NNT_EXPECT_RESULT_SUCCESS(MountContent("test", "@Host:/D:/tmp/test.nspd/control0.ncd/", ContentType_Control));
}
#endif

#if 0 // TODO: 自動化
TEST(Mount, MountLogo)
{
    NNT_EXPECT_RESULT_SUCCESS(MountLogo("logo", ("@Host:/" + GetTestPath("/test.nca")).c_str()));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("logo:/"));
    Unmount("logo");
}

TEST(Mount, MountLogoNotFound)
{
    NNT_EXPECT_RESULT_FAILURE(ResultPartitionNotFound, MountLogo("logo", ("@Host:/" + GetTestPath("/test2.nca")).c_str()));
}
#endif

#if 0
TEST_F(MountBasic, MountCode) // Todo: 自動化
{
    NNT_ASSERT_RESULT_SUCCESS(MountCode("code", ("@Host:/" + GetTestPath("/gcc3.nca")).c_str()));
    DirectoryHandle dirHandle;
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenDirectory(&dirHandle, "code:/", OpenDirectoryMode_File));
    while( NN_STATIC_CONDITION(true) )
    {
        int64_t readNum = 0;
        const int entryBufferNum = 3;
        DirectoryEntry dirEntry[entryBufferNum];
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::ReadDirectory(&readNum, dirEntry, dirHandle, entryBufferNum));
        for( int i = 0; i<readNum; i++ )
        {
            // Check results
            NN_LOG("entry: <%s> %d %lld\n", dirEntry[i].name, dirEntry[i].directoryEntryType, dirEntry[i].fileSize);
        }
        if( readNum == 0 )
        {
            break;
        }
    }
    nn::fs::CloseDirectory(dirHandle);
    Unmount("code");
}
#endif

#if 0
TEST_F(MountBasic, MountContent) // Todo: 自動化
{
    NNT_ASSERT_RESULT_SUCCESS(MountContent("control", ("@Host:/" + GetTestPath("/ncm/content/contents/06272ccccf0a84ea6cca6e64eec95a93.nca")).c_str(), ContentType_Control));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("control:/"));
    Unmount("control");

    NNT_ASSERT_RESULT_SUCCESS(MountContent("manual", ("@Host:/" + GetTestPath("/ncm/content/contents/67a960ddd1a4f723cf1cb11ca2ce7fa4.nca")).c_str(), ContentType_Manual));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("manual:/"));
    Unmount("manual");

    NNT_ASSERT_RESULT_SUCCESS(MountContent("meta", ("@Host:/" + GetTestPath("/ncm/content/contents/12d5094a84abc9c4b15ef8ed2746fccd.nca")).c_str(), ContentType_Meta));
    NNT_EXPECT_RESULT_SUCCESS(nnt::fs::util::DumpDirectoryRecursive("meta:/"));
    Unmount("meta");
}
#endif

namespace
{
    const nn::ncm::ProgramId ProgramId = { 0x0100000000002064 };
    const nn::ncm::DataId DataId = { 0 };

    // 適当にビルドした nca に対して作られた main.npdm データ
    // (実機で検証できればよい)
    const uint8_t Data[] = {
        0x4D, 0x45, 0x54, 0x41, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x2C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00,
        0x46, 0x73, 0x4C, 0x69, 0x62, 0x4D, 0x6F, 0x75, 0x6E, 0x74, 0x54, 0x65, 0x73, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x03, 0x00, 0x00, 0x40, 0x01, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xE0, 0x02, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x41, 0x43, 0x49, 0x44, 0xE0, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0x40, 0x02, 0x00, 0x00, 0x2C, 0x00, 0x00, 0x00, 0x70, 0x02, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x80, 0x02, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2A, 0x00, 0x2A, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xF7, 0x43, 0x00, 0x03, 0xCF, 0xFF, 0xFF, 0x1F, 0xEF, 0xFF, 0x7F, 0x20, 0x6F, 0x06, 0xE0, 0x47, 0x0F, 0xE0, 0xFF, 0x7F, 0xEF, 0xFF, 0xE7, 0x9F, 0xEF, 0x13, 0x00, 0xA0, 0x3F, 0x00, 0x00, 0x00,
        0xBF, 0xFF, 0xFF, 0x07, 0x3F, 0x00, 0x00, 0x80, 0xBF, 0xFF, 0xFF, 0x07, 0x3F, 0x71, 0x80, 0x03, 0x3F, 0x01, 0x00, 0x00, 0x7F, 0x06, 0x00, 0x06, 0x7F, 0x06, 0x00, 0x06, 0x7F, 0x0D, 0x00, 0x06,
        0x7F, 0x9F, 0x00, 0x07, 0x7F, 0x10, 0x01, 0x07, 0x7F, 0xEF, 0x02, 0x07, 0xFF, 0xF7, 0xFF, 0xFF, 0xFF, 0x5F, 0x00, 0x00, 0xFF, 0xBF, 0x48, 0x00, 0xFF, 0x7F, 0xFF, 0x03, 0xFF, 0xFF, 0x02, 0x00,
        0x41, 0x43, 0x49, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x0C, 0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x40, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x1F, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0xC0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x06, 0x66, 0x73, 0x70, 0x2D, 0x73, 0x72, 0x76, 0x01, 0x6C, 0x6D, 0x05, 0x6E, 0x73, 0x3A, 0x61, 0x6D, 0x32, 0x02, 0x68, 0x74, 0x63, 0x07, 0x68, 0x74, 0x63, 0x3A, 0x74, 0x65, 0x6E, 0x76, 0x00,
        0xB7, 0x43, 0x00, 0x03, 0xCF, 0xFF, 0xFF, 0x1F, 0xEF, 0xFF, 0x7F, 0x20, 0x6F, 0x06, 0xE0, 0x47, 0x0F, 0xE0, 0xFF, 0x7F, 0xEF, 0xFF, 0xE7, 0x9F, 0xEF, 0x13, 0x00, 0xA0, 0x3F, 0x00, 0xA0, 0x02,
        0x3F, 0x00, 0x08, 0x00, 0x3F, 0x00, 0xB8, 0x02, 0x3F, 0x00, 0x10, 0x00, 0x3F, 0x00, 0xC0, 0x02, 0x3F, 0x00, 0x08, 0x00, 0xBF, 0x02, 0x00, 0x03, 0x3F, 0x01, 0x00, 0x00, 0x3F, 0x05, 0x00, 0x03,
        0x3F, 0x01, 0x00, 0x00, 0x3F, 0x00, 0x80, 0x03, 0x3F, 0x02, 0x00, 0x00, 0x3F, 0x00, 0x80, 0x03, 0x3F, 0x02, 0x00, 0x00, 0x3F, 0x00, 0x80, 0x03, 0x3F, 0x02, 0x00, 0x00, 0x3F, 0x06, 0x80, 0x03,
        0x3F, 0x01, 0x00, 0x00, 0x3F, 0x18, 0x80, 0x03, 0x3F, 0x08, 0x00, 0x00, 0x3F, 0x40, 0x80, 0x03, 0x3F, 0x01, 0x00, 0x00, 0x3F, 0x71, 0x80, 0x03, 0x3F, 0x01, 0x00, 0x00, 0x3F, 0x60, 0x81, 0x03,
        0x3F, 0x20, 0x00, 0x00, 0x7F, 0x06, 0x00, 0x06, 0x7F, 0x06, 0x00, 0x06, 0x7F, 0x0D, 0x00, 0x06, 0x7F, 0x9F, 0x00, 0x07, 0x7F, 0x10, 0x01, 0x07, 0x7F, 0xEF, 0x02, 0x07, 0xFF, 0xB7, 0xC2, 0xFF,
        0xFF, 0x17, 0xC6, 0x18, 0xFF, 0x97, 0x86, 0x1A, 0xFF, 0x17, 0x87, 0x21, 0xFF, 0xD7, 0x8B, 0x2F, 0xFF, 0x5F, 0x00, 0x00, 0xFF, 0xBF, 0x48, 0x00, 0xFF, 0x7F, 0x00, 0x02, 0xFF, 0xFF, 0x02, 0x00
    };

    void SetUpMountContent() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::fssrv::detail::SetEnabledProgramVerification(false);
#endif

        NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_TestDirPath.GetPath().append("/data/").c_str()));

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_TestDirPath.GetPath().append("/code/").c_str()));

        auto filePath = g_TestDirPath.GetPath().append("/code/main.npdm");
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateFile(filePath.c_str(), sizeof(Data)));

        nn::fs::FileHandle file;
        {
            auto result = nn::fs::OpenFile(&file, filePath.c_str(), nn::fs::OpenMode_Write);
            NNT_EXPECT_RESULT_SUCCESS(result);
            if( result.IsSuccess() )
            {
                NNT_EXPECT_RESULT_SUCCESS(nn::fs::WriteFile(file, 0, Data, sizeof(Data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
                nn::fs::CloseFile(file);
            }
        }

        NNT_EXPECT_RESULT_SUCCESS(nn::fs::CreateDirectory(g_TestDirPath.GetPath().append("/logo/").c_str()));

        UnmountHostRoot();
    }

    void TearDownMountContent() NN_NOEXCEPT
    {
#if defined(NN_BUILD_CONFIG_OS_WIN)
        nn::fssrv::detail::SetEnabledProgramVerification(true);
#endif

        NNT_ASSERT_RESULT_SUCCESS(MountHostRoot());
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(g_TestDirPath.GetPath().append("/data/").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(g_TestDirPath.GetPath().append("/code/").c_str()));
        NNT_EXPECT_RESULT_SUCCESS(nn::fs::DeleteDirectoryRecursively(g_TestDirPath.GetPath().append("/logo/").c_str()));
        UnmountHostRoot();
    }

    nnt::fs::util::String GetPath() NN_NOEXCEPT
    {
        return ("@Host:/" + g_TestDirPath.GetPath() + "/");
    }

    template<bool IsReservedMountNameSupported>
    const nnt::fs::util::MountTestAttribute GetAttribute() NN_NOEXCEPT
    {
        nnt::fs::util::MountTestAttribute attribute = {};
        attribute.setUp = SetUpMountContent;
        attribute.tearDowm = TearDownMountContent;
        attribute.isReservedMountNameSupported = IsReservedMountNameSupported;
        return attribute;
    }

    // MountApplicationPackage
    nn::Result MountApplicationPackageForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountApplicationPackage(mountName, GetPath().c_str()));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountApplicationPackageForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountApplicationPackage(mountName, path));
        NN_RESULT_SUCCESS;
    }

    // MountContent(const char* name, const char* path, ncm::ProgramId id, ContentType contentType)
    nn::Result MountContentType1ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, GetPath().c_str(), ProgramId, nn::fs::ContentType_Manual));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountContentType1ForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, path, ProgramId, nn::fs::ContentType_Manual));
        NN_RESULT_SUCCESS;
    }

    // MountContent(const char* name, const char* path, ncm::DataId id, ContentType contentType)
    nn::Result MountContentType2ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, GetPath().c_str(), DataId, nn::fs::ContentType_Manual));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountContentType2ForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, path, DataId, nn::fs::ContentType_Manual));
        NN_RESULT_SUCCESS;
    }

    // MountContent(const char* name, const char* path, ContentType contentType)
    nn::Result MountContentType3ForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, GetPath().c_str(), nn::fs::ContentType_Meta));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountContentType3ForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountContent(mountName, path, nn::fs::ContentType_Meta));
        NN_RESULT_SUCCESS;
    }

    // MountContent(const char* name, ncm::ProgramId id, ContentType contentType)
    // → testFs_FsLib_Mount_Content_HtmlDocument.cpp でテストする

#if defined(NN_BUILD_CONFIG_OS_WIN)
    // MountCode
    nn::Result MountCodeForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountCode(mountName, GetPath().c_str(), ProgramId));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountCodeForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountCode(mountName, path, ProgramId));
        NN_RESULT_SUCCESS;
    }
#endif // defined(NN_BUILD_CONFIG_OS_WIN)

    // MountLogo
    nn::Result MountLogoForMountNameTest(const char* mountName) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountLogo(mountName, GetPath().c_str(), ProgramId));
        NN_RESULT_SUCCESS;
    }

    nn::Result MountLogoForMountPathTest(const char* mountName, const char* path) NN_NOEXCEPT
    {
        NN_RESULT_DO(nn::fs::MountLogo(mountName, path, ProgramId));
        NN_RESULT_SUCCESS;
    }

    const nnt::fs::util::MountTestParameter MountTestParameters[] = {
        { "MountApplicationPackage", MountApplicationPackageForMountNameTest, MountApplicationPackageForMountPathTest, GetAttribute<false> },
        { "MountContentType1", MountContentType1ForMountNameTest, MountContentType1ForMountPathTest, GetAttribute<true> },
        { "MountContentType2", MountContentType2ForMountNameTest, MountContentType2ForMountPathTest, GetAttribute<true> },
        { "MountContentType3", MountContentType3ForMountNameTest, MountContentType3ForMountPathTest, GetAttribute<true> },
#if defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountCode", MountCodeForMountNameTest, MountCodeForMountPathTest, GetAttribute<false> },
#endif // defined(NN_BUILD_CONFIG_OS_WIN)
        { "MountLogo", MountLogoForMountNameTest, MountLogoForMountPathTest, GetAttribute<false> },
    };
}

NNT_FS_INSTANTIATE_TEST_CASE_MOUNT(WithMountContent1, ::testing::ValuesIn(MountTestParameters));
