﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/album.h>
#include <nn/nifm.h>

#include <nn/album/album_ShareId.h>
#include <nn/album/album_SaveScreenshot.h>
#include <nn/album/album_SaveAndShareScreenshot.h>

#include "testFs_FsLib_AccessLog.h"

#if defined(NN_BUILD_CONFIG_HARDWARE_NX)

namespace
{
    // 画面写真として保存する画像データを生成します。
    // 画像データの色フォーマットは R8G8B8A8 ですが、アルファチャンネルは無視されます。
    uint8_t* CreateImageData(size_t* pOutSize, int width, int height) NN_NOEXCEPT
    {
        const int PixelSize = 4;
        size_t size = PixelSize * width * height;
        uint8_t* pImage = reinterpret_cast<uint8_t*>(malloc(size));
        NN_ASSERT_NOT_NULL(pImage);

        for(int y = 0; y < height; y++)
        {
            uint8_t* pLine = pImage + PixelSize * width* y;
            for(int x = 0; x < width; x++)
            {
                uint8_t* pPixel = pLine + PixelSize * x;
                int v = ((x / 32) + (y / 32)) % 2;
                if(v == 0)
                {
                    pPixel[0] = 255; // R
                    pPixel[1] = 255; // G
                    pPixel[2] = 255; // B
                    pPixel[3] = 255; // A （無視される）
                }
                else
                {
                    // +---------+---------+
                    // |   Red   |  Green  |
                    // +---------+---------+
                    // |   Blue  |  Black  |
                    // +---------+---------+
                    if(y < height / 2)
                    {
                        if(x < width / 2)
                        {
                            pPixel[0] = 255; // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                        else
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 255; // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                    }
                    else
                    {
                        if(x < width / 2)
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 255; // B
                            pPixel[3] = 255; // A （無視される）
                        }
                        else
                        {
                            pPixel[0] = 0;   // R
                            pPixel[1] = 0;   // G
                            pPixel[2] = 0;   // B
                            pPixel[3] = 255; // A （無視される）
                        }
                    }
                }

            }
        }

        *pOutSize = size;
        return pImage;
    }

    void DestroyImageData(uint8_t* pImage) NN_NOEXCEPT
    {
        free(pImage);
    }

    const nn::album::ImageSize ScreenShotSize = nn::album::ImageSize_1280x720;
    const nn::album::AlbumReportOption ReportOption = nn::album::AlbumReportOption_NoReport;
    const nn::album::ImageOrientation ImageOrientaiton = nn::album::ImageOrientation_Rotate180;

    const int Width = nn::album::GetImageWidth(ScreenShotSize);
    const int Height = nn::album::GetImageHeight(ScreenShotSize);
}

class ExecuteAlbum : public testing::Test
{
public:
    static void SetUpTestCase()
    {
        nn::nifm::Initialize();
        nn::album::Initialize();

        g_pImageData = CreateImageData(&g_ImageDataSize, Width, Height);
    }
    static void TearDownTestCase()
    {
        DestroyImageData(g_pImageData);
        nn::album::Finalize();
    }
public:
    static size_t g_ImageDataSize;
    static uint8_t* g_pImageData;
};

size_t ExecuteAlbum::g_ImageDataSize = 0;
uint8_t* ExecuteAlbum::g_pImageData = nullptr;

TEST_F(ExecuteAlbum, SaveScreenshot_Success)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveScreenshot", image_data_size: 3686400, width: 1280, height: 720, album_report_option: 0x0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveScreenshot", image_data_size: 3686400, width: 1280, height: 720, album_report_option: 0x0 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0 }
*/
{
    NNT_ASSERT_RESULT_SUCCESS(
        nn::album::SaveScreenshot(g_pImageData, g_ImageDataSize,
        ScreenShotSize, ReportOption));
    NNT_ASSERT_RESULT_SUCCESS(
        nn::album::SaveScreenshot(g_pImageData, g_ImageDataSize,
        ScreenShotSize, ImageOrientaiton, ReportOption));
}

TEST_F(ExecuteAlbum, DISABLED_SaveScreenshot_Failure)
{}

TEST_F(ExecuteAlbum, DISABLED_SaveAndShareScreenshot_Success)
{}

TEST_F(ExecuteAlbum, SaveAndShareScreenshot_Failure)
/*
AccessLogTest: NX
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 0, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 0, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
AccessLogTest: NX-system
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 0, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 0, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
FS_ACCESS: { start: 0, end: 0, result: 0x00000000, handle: 0x0000000000000000, priority: Normal, function: "SaveAndShareScreenshot", image_data_size: 3686400, width: 1280, height: 720, image_orientation: 2, album_report_option: 0x0, userid: 0x10000ABCDEF0123456789ABCDEF01234 }
*/
{
    const nn::account::Uid uid = nnt::fs::GetDefaultUid();
    ASSERT_TRUE(
        nn::album::SaveAndShareScreenshot(g_pImageData, g_ImageDataSize,
        ScreenShotSize, ReportOption, uid).IsFailure());
    ASSERT_TRUE(
        nn::album::SaveAndShareScreenshot(g_pImageData, g_ImageDataSize,
        ScreenShotSize, ImageOrientaiton, ReportOption, uid).IsFailure());
    nn::album::ShareId shareId = {};
    ASSERT_TRUE(
        nn::album::SaveAndShareScreenshot(&shareId, g_pImageData, g_ImageDataSize,
        ScreenShotSize, ReportOption, uid).IsFailure());
    ASSERT_TRUE(
        nn::album::SaveAndShareScreenshot(&shareId, g_pImageData, g_ImageDataSize,
        ScreenShotSize, ImageOrientaiton, ReportOption, uid).IsFailure());
}

#endif
