﻿# coding: UTF-8

# Copyright (C)Nintendo All rights reserved.
#
# These coded instructions, statements, and computer programs contain proprietary
# information of Nintendo and/or its licensed developers and are protected by
# national and international copyright laws. They may not be disclosed to third
# parties or copied or duplicated in any form, in whole or in part, without the
# prior written consent of Nintendo.
#
# The content herein is highly confidential and should be handled accordingly.

import difflib
import os
import re
import shutil
import subprocess
import sys

BUILD_TYPE = "Develop"

g_isTestEnd = False
g_isTestPdm = False
g_isTestIndexer = False
g_isTestAccount = False
g_sdk_root_path = ""
g_expect_access_log_dict = {}

def check_exists(path):
    if not os.path.exists(path):
        print("Not Found : " + path)
        sys.exit(1)

def delete_directory_recursive(dir_path):
    if os.path.exists(dir_path):
        shutil.rmtree(dir_path)

def update_expect_access_log_dict(filepath):
    # ソースコードから、想定するアクセスログを抽出する
    global g_expect_access_log_dict
    test_name = ""
    with open(filepath, encoding="utf-8_sig") as f:
        for line in f:
            if line[:15] == "AccessLogTest: ":
                test_name = line[15:].strip()
            if line[:11] == "FS_ACCESS: ":
                if not test_name in g_expect_access_log_dict:
                    g_expect_access_log_dict[test_name] = ""
                g_expect_access_log_dict[test_name] += line

def create_expect_access_log_dict():
    # ソースコードから、想定するアクセスログを抽出する
    source_dir_path = os.path.join(g_sdk_root_path, "Tests", "Fs", "Sources", "Tests", "FsLib", "AccessLog")
    # ソースファイルの抽出順序を理想的な順序に変える
    source_file_name_list = sorted(os.listdir(source_dir_path), key=lambda k: tuple(reversed(k.split("."))))
    for name in source_file_name_list:
        if name[-4:] == ".cpp" or name[-2:] == ".h":
            source_file_path = os.path.join(source_dir_path, name)
            update_expect_access_log_dict(source_file_path)

def get_access_log_Impl(specs, platform, target=""):
    # testFs_FsLib_AccessLog.testlist.yml を実行し、アクセスログを取得する
    print("Output AccessLog (" + specs + "): wait few minutes...")
    os.putenv("NINTENDO_SDK_ROOT", os.path.normpath(g_sdk_root_path))
    os.putenv("NNTEST_RUNNER_TOOLS_PATH", os.path.join(g_sdk_root_path, "Tools", "CommandLineTools"))
    os.putenv("NNTEST_BUILD_SYSTEM_ROOT", os.path.normpath(g_sdk_root_path))

    test_runner_path = os.path.join(g_sdk_root_path, "Integrate", "CommandLineTools", "NNTest.cmd")
    yaml_file_path = os.path.join(g_sdk_root_path, "Tests", "Fs", "Sources", "Tests", "FsLib", "AccessLog", "testFs_FsLib_AccessLog", "testFs_FsLib_AccessLog.testlist.yml")

    check_exists(test_runner_path)
    check_exists(yaml_file_path)

    command = [
        test_runner_path,
        "-File", yaml_file_path,
        "-BuildType", BUILD_TYPE,
        "-Platform", platform,
        "-Verbose"
    ]
    if target:
      command.extend(["-TargetAddressPattern", target])

    prcs = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
    std_out, std_err = prcs.communicate()
    base_text = std_out.decode("shift_jis")
    if len(std_err) > 0 or re.search("- ERROR\\s*:\\s*[0-9]" , base_text) or re.search("- TIMEOUT\\s*:\\s*[0-9]" , base_text) or re.search("- NO_FILE\\s*:\\s*[0-9]" , base_text):
        print(std_err)
        print("  NG.")
        print(base_text)
        sys.exit(1)
    print("  OK.")
    return base_text

def delete_save_data_Generic():
    artifact_dir_path = os.path.join(g_sdk_root_path, "Tests", "Outputs", "x64-v140", "Tests", "testFs_FsLib_AccessLog", BUILD_TYPE)
    delete_directory_recursive(os.path.join(artifact_dir_path, "Album_NAND"))
    delete_directory_recursive(os.path.join(artifact_dir_path, "Album_SD"))
    delete_directory_recursive(os.path.join(artifact_dir_path, "bis"))
    delete_directory_recursive(os.path.join(artifact_dir_path, "sdcard"))

def get_access_log_Nx(target):
    return get_access_log_Impl("NX", "NXFP2-a64", target)

def get_access_log_Generic():
    delete_save_data_Generic()
    return get_access_log_Impl("Generic", "Win64_VS2015")

def normalize_access_log_line(line):
    global g_isTestEnd
    if "sdk_version" in line:
        g_isTestEnd = False
    elif "testFs_FsLib_AccessLog-spec.NX.autogen.vcxproj.test_detail.xml" in line:
        g_isTestEnd = True
    if g_isTestEnd:
        return ""
    
    global g_isTestPdm
    if ("OpenFile" in line) and ("result: 0x00000000" in line) and ("pdm:/" in line):
        g_isTestPdm = True
    elif g_isTestPdm and ("CloseFile" in line):
        g_isTestPdm = False
    
    global g_isTestIndexer
    if ("MountSystemSaveData" in line) and ("saveDataIxrDb" in line):
        g_isTestIndexer = True
        return ""
    elif ("Unmount" in line) and ("saveDataIxrDb" in line):
        g_isTestIndexer = False
        return ""
    if g_isTestIndexer:
        return ""
    if ("MountSystemSaveData" in line) and ("nsdsave" in line):
        return ""
    
    global g_isTestAccount
    if ("name: \"account\"" in line) or ("path: \"account:" in line):
        if ("OpenFile" in line) and ("result: 0x00000000" in line):
            g_isTestAccount = True
        return ""
    if g_isTestAccount and ("CloseFile" in line):
        g_isTestAccount = False
        return ""
    if g_isTestAccount:
        return ""
    
    # FS_ACCESS より前の文字列を排除
    line = re.sub(".*FS_ACCESS",                            "FS_ACCESS", line)
    # バージョンを統一
    line = re.sub("sdk_version: *[0-9\.]+",                  "sdk_version: 0.0.0", line)
    # 開始時間、終了時間を 0 にする
    line = re.sub("start: *[0-9]+",                         "start: 0", line)
    line = re.sub("end: *[0-9]+",                           "end: 0", line)
    # 有効なハンドルを適当な文字列にする
    line = re.sub(" handle: 0x0000000000000000",            " handle: tmp", line)
    line = re.sub(" handle: 0x[0-9a-f]+",                   " handle: 0xFEDCBA0987654321", line)
    line = re.sub(" handle: tmp",                           " handle: 0x0000000000000000", line)
    # 他のハンドルを置換
    line = re.sub("_handle: 0x[0-9a-fA-F]+",                "_handle: 0xFEDCBA09", line)
    # userid と savedataid を適当な文字列にする
    line = re.sub("userid: 0x[0-9A-F]+",                    "userid: 0x10000ABCDEF0123456789ABCDEF01234", line)
    line = re.sub("savedataid: 0x[0-9A-F]+",                "savedataid: 0x1234", line)
    # パスを統一する
    # 絶対パスはアルファベット 1 文字のドライブ名から始まっているという前提
    line = re.sub("\"[a-zA-Z]:.*romfs\.bin\"",              "\"romfs.bin\"", line)
    line = re.sub("\"[a-zA-Z]:.*testFs_FsLib_AccessLog",    "\"testFs_FsLib_AccessLog", line)
    line = re.sub("\"[a-zA-Z]:.*\"",                        "\"C:\\Windows\\Temp\"", line)
    # 不定なサイズを置換
    if g_isTestPdm or ('"GetSaveDataOwnerId"' in line):
        line = re.sub("size: *[0-9]+", "size: 1", line)
    return line

def normalize_access_log(raw_access_log):
    # start/end/handle/userid などの不定値を定数に変換
    last_line = ""
    for line in raw_access_log.splitlines():
        if not "FS_ACCESS" in line:
            continue
        normalized_line = normalize_access_log_line(line)
        if normalized_line == "":
            continue
        # ReadSaveDataInfo が 2 回連続する場合は、2 回目以降を無視する
        if (normalized_line == last_line) and ('"ReadSaveDataInfo"' in normalized_line):
            continue
        last_line = normalized_line
        yield normalized_line

def read_expect_file(test_name):
    if not test_name in g_expect_access_log_dict:
        return ""
    return g_expect_access_log_dict[test_name]

def read_expect_file_Nx():
    expect_file_path = os.path.join(g_sdk_root_path, "Tests", "Fs", "Sources", "Tests", "FsLib", "AccessLog", "testFs_FsLib_AccessLog", "expect_accesslog_outputtest_Nx.txt")
    return read_expect_file("NX") + read_expect_file("NX-system")

def read_expect_file_Generic():
    expect_file_path = os.path.join(g_sdk_root_path, "Tests", "Fs", "Sources", "Tests", "FsLib", "AccessLog", "testFs_FsLib_AccessLog", "expect_accesslog_outputtest_Generic.txt")
    return read_expect_file("Generic") + read_expect_file("Generic-system")

def get_diff(text1, text2):
    return os.linesep.join(difflib.unified_diff(text1.splitlines(), text2.splitlines()))

def check_access_log_expected_Nx(access_log_Nx):
    print("diff (Nx)")
    expect_access_log_Nx = read_expect_file_Nx()
    diff_Nx = get_diff(access_log_Nx, expect_access_log_Nx)
    if len(diff_Nx):
        print(diff_Nx)
        print("  NG.")
        sys.exit(1)
    else:
        print("  OK.")

def check_access_log_expected_Generic(access_log_Generic):
    print("diff (Generic)")
    expect_access_log_Generic = read_expect_file_Generic()
    diff_Generic = get_diff(access_log_Generic, expect_access_log_Generic)
    if len(diff_Generic):
        print(diff_Generic)
        print("  NG.")
        sys.exit(1)
    else:
        print("  OK.")

def set_sdk_root_path(sdk_root_path):
    global g_sdk_root_path
    g_sdk_root_path = os.path.normpath(sdk_root_path)

def main():
    args = sys.argv
    if len(args) <= 3:
        print("usage: <sdk_root_path> <target> <platform>")
        sys.exit(1)
    set_sdk_root_path(args[1])
    target = args[2]
    platform = args[3]

    if platform == "NX":
        raw_access_log_Nx = get_access_log_Nx(target)
        access_log_Nx = os.linesep.join(normalize_access_log(raw_access_log_Nx))

        # 取得したアクセスログと期待するアクセスログを比較し、差異が無いことを確認する。
        create_expect_access_log_dict()
        check_access_log_expected_Nx(access_log_Nx)
    else:
        raw_access_log_Generic = get_access_log_Generic()
        access_log_Generic = os.linesep.join(normalize_access_log(raw_access_log_Generic))

        # 取得したアクセスログと期待するアクセスログを比較し、差異が無いことを確認する。
        create_expect_access_log_dict()
        check_access_log_expected_Generic(access_log_Generic)

if __name__ == '__main__':
  main()
