﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

#include <nn/fssystem/fs_TmFileSystem.h>
#include <nn/fs/fs_ResultHandler.h>

using nn::fssystem::TmFileSystem;

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>
#endif

namespace {
    const int HeapSize = 128 * 1024;
    char g_HeapStack[HeapSize];
    char g_RootDirPath[256];
    nnt::fs::util::TemporaryHostDirectory g_HostDirectory;
}

namespace nnt { namespace fs { namespace api {
    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index)
    {
        NN_UNUSED(index);
        std::unique_ptr<TmFileSystem> hostFs(new TmFileSystem());
        NNT_ASSERT_RESULT_SUCCESS(hostFs->Initialize());

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));
        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(hostFs)));
        info->type = FileSystemType_HostFileSystem;
        info->rootDirPath = g_RootDirPath;

        auto& attribute = info->attribute;

        static const int HostFsPathLengthMax      = 260 - 1; // HostFileSystem::WindowsPathLengthMax からヌル文字分を引いた値
        static const int DirectoryPathLengthMax   = 248 - 1; // Windows の CreateDirectory に渡せるパスの最大長（'\0' を含まない）
        attribute.directoryNameLengthMax = DirectoryPathLengthMax;
        attribute.fileNameLengthMax      = HostFsPathLengthMax;
        attribute.directoryPathLengthMax = DirectoryPathLengthMax;
        attribute.filePathLengthMax      = HostFsPathLengthMax;

        attribute.fileSizeMax = 0x0000000220000000LL;
        attribute.storageSize = 0x0000000240000000LL;

        attribute.fileOpenMax      = 128;
        attribute.directoryOpenMax = 64;

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedGetFileTimeStamp = true;
        attribute.isSupportedQueryRange = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

    g_HostDirectory.Create();
    strncpy(g_RootDirPath, g_HostDirectory.GetPath().c_str(), sizeof(g_RootDirPath));

    auto ret = RUN_ALL_TESTS();

    g_HostDirectory.Delete();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
