﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nn/result/result_HandlingUtility.h>
#include <nnt/nnt_Argument.h>

#include <nn/util/util_TFormatString.h>
#include <nn/fs/fs_ResultHandler.h>

#include <nn/fssystem/fs_TmFileSystem.h>
#include <nn/fssystem/fs_ProxyFileSystem.h>

#include "shim/fs_FileSystemProxyServiceObject.h"
#include "shim/fs_FileSystemServiceObjectAdapter.h"

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>
#endif

namespace {
const int HeapSize = 512 * 1024;
char g_HeapStack[HeapSize];
char g_RootDirPath[256] = "";
nnt::fs::util::TemporaryHostDirectory g_HostDirectory;


using namespace nn::fs;

#if defined(NN_BUILD_CONFIG_OS_WIN)

// HostFileSystem on HostPc
Result OpenHostFs(std::unique_ptr<nn::fs::fsa::IFileSystem>* pOutValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    nn::fssrv::sf::FspPath sfPath;
    sfPath.str[0] = '\0';
    NN_RESULT_DO(fileSystemProxy->OpenHostFileSystem(&fileSystem, sfPath));

    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
    fileSystemAbstract.reset(new nn::fs::detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailed());

    *pOutValue = std::move(fileSystemAbstract);
    NN_RESULT_SUCCESS;
}

#else // defined(NN_BUILD_CONFIG_OS_WIN)

class File : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
public:
    explicit File(std::unique_ptr<nn::fs::fsa::IFile>&& pBaseFile) NN_NOEXCEPT
        : m_pBaseFile(std::move(pBaseFile))
    {
    }

private:
    virtual nn::Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual nn::Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationID,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::unique_ptr<nn::fs::fsa::IFile> m_pBaseFile;
};

nn::Result File::DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pBaseFile->Read(outValue, offset, buffer, size, option));
    NN_RESULT_SUCCESS;
}

nn::Result File::DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pBaseFile->Write(offset, buffer, size, option));
    NN_RESULT_SUCCESS;
}

nn::Result File::DoFlush() NN_NOEXCEPT
{
    NN_RESULT_DO(m_pBaseFile->Flush());
    NN_RESULT_SUCCESS;
}

nn::Result File::DoSetSize(int64_t size) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pBaseFile->SetSize(size));
    NN_RESULT_SUCCESS;
}

nn::Result File::DoGetSize(int64_t* outValue) NN_NOEXCEPT
{
    NN_RESULT_DO(m_pBaseFile->GetSize(outValue));
    NN_RESULT_SUCCESS;
}

nn::Result File::DoOperateRange(
    void* outBuffer, size_t outBufferSize, nn::fs::OperationId operationId, int64_t offset, int64_t size, const void* inBuffer, size_t inBufferSize) NN_NOEXCEPT
{
    if( operationId == nn::fs::OperationId::QueryRange )
    {
        // FIXME : FileServiceObjectAdapter::DoOperateRange でチェックしてほしい
        NN_FSP_REQUIRES(outBuffer != nullptr, nn::fs::ResultNullptrArgument());
    }

    NN_RESULT_DO(
        m_pBaseFile->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            offset,
            size,
            inBuffer,
            inBufferSize
        )
    );
    NN_RESULT_SUCCESS;
}

class FileSystem : public nn::fssystem::ProxyFileSystem
{
public:
    explicit FileSystem(std::unique_ptr<fsa::IFileSystem>&& pBaseFileSystem)
        : ProxyFileSystem(pBaseFileSystem.get(), nn::fs::ResultUnknown()),
          m_pBaseFileSystem(std::move(pBaseFileSystem))
    {
    }

public:
    virtual Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE
    {
        std::unique_ptr<nn::fs::fsa::IFile> pFile;
        NN_RESULT_DO(nn::fssystem::ProxyFileSystem::DoOpenFile(&pFile, path, mode));
        outValue->reset(new File(std::move(pFile)));
        NN_RESULT_SUCCESS;
    }

private:
    std::unique_ptr<nn::fs::fsa::IFileSystem> m_pBaseFileSystem;
};

// FatFileSystem on SD card
Result OpenFatFs(std::unique_ptr<nn::fs::fsa::IFileSystem>* pOutValue) NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystemProxy> fileSystemProxy = nn::fs::detail::GetFileSystemProxyServiceObject();
    nn::sf::SharedPointer<nn::fssrv::sf::IFileSystem> fileSystem;
    NN_RESULT_DO(fileSystemProxy->OpenSdCardFileSystem(&fileSystem));

    std::unique_ptr<fsa::IFileSystem> fileSystemAbstract;
    fileSystemAbstract.reset(new nn::fs::detail::FileSystemServiceObjectAdapter(std::move(fileSystem)));
    NN_RESULT_THROW_UNLESS(fileSystemAbstract, ResultAllocationMemoryFailed());

    std::unique_ptr<fsa::IFileSystem> proxyFileSystem;
    proxyFileSystem.reset(new FileSystem(std::move(fileSystemAbstract)));
    NN_RESULT_THROW_UNLESS(proxyFileSystem, ResultAllocationMemoryFailed());

    *pOutValue = std::move(proxyFileSystem);
    NN_RESULT_SUCCESS;
}

#endif // defined(NN_BUILD_CONFIG_OS_WIN)

}



namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, const int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<nn::fs::fsa::IFileSystem> fs;
        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));
        auto& attribute = info->attribute;

        // TODO: 統一する
#if defined(NN_BUILD_CONFIG_OS_WIN32)
        NNT_ASSERT_RESULT_SUCCESS(OpenHostFs(&fs));
        info->type = FileSystemType_HostFileSystem;
        info->rootDirPath = g_RootDirPath;

        static const int HostFsPathLengthMax      = 260 - 1; // HostFileSystem::WindowsPathLengthMax からヌル文字分を引いた値
        static const int DirectoryPathLengthMax   = 248 - 1; // Windows の CreateDirectory に渡せるパスの最大長（'\0' を含まない）
        attribute.directoryNameLengthMax = DirectoryPathLengthMax;
        attribute.fileNameLengthMax      = HostFsPathLengthMax;
        attribute.directoryPathLengthMax = DirectoryPathLengthMax;
        attribute.filePathLengthMax      = HostFsPathLengthMax;

        attribute.fileSizeMax = 0x0000100000000000LL;
        attribute.storageSize = 0x0000100000000000LL;

        attribute.isSupportedMultiBytePath = true;
#else
        NNT_ASSERT_RESULT_SUCCESS(OpenFatFs(&fs));
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = g_RootDirPath;

        static const int FatNameLengthMax  = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax  = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int DriveLetterLength = 2;       // FatFileSystem で付与されるドライブレターの文字数 ("A:")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax = FatPathLengthMax - DriveLetterLength;
        attribute.filePathLengthMax      = FatPathLengthMax - DriveLetterLength;

        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = 0x400000000ULL; // 16GB sdcard

        attribute.speedEmulationTypeFlag = static_cast<int32_t>(nn::fs::SpeedEmulationTypeFlag::SdCardStorageContextEnabled);

        attribute.isSupportedMultiBytePath = false;
#endif
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.fileSizeAlignment = 1;

        attribute.fileOpenMax      = 256 - 1; // ncm がSDのコンテンツメタデータベース保存用に一つオープンしている場合がある
        attribute.directoryOpenMax = 31;

        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(fs)));

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    g_HostDirectory.Create();
    strncpy(g_RootDirPath, g_HostDirectory.GetPath().c_str(), sizeof(g_RootDirPath));
#else
    {
        std::unique_ptr<nn::fs::fsa::IFileSystem> fs;
        NNT_ASSERT_RESULT_SUCCESS(OpenFatFs(&fs));
        std::strncpy(g_RootDirPath, "/API_TEST", sizeof(g_RootDirPath));
        fs->DeleteDirectoryRecursively(g_RootDirPath);
        NNT_ASSERT_RESULT_SUCCESS(fs->CreateDirectory(g_RootDirPath));
    }
#endif
    auto ret = RUN_ALL_TESTS();

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    g_HostDirectory.Delete();
#else
    {
        std::unique_ptr<nn::fs::fsa::IFileSystem> fs;
        NNT_ASSERT_RESULT_SUCCESS(OpenFatFs(&fs));
        NNT_ASSERT_RESULT_SUCCESS(fs->DeleteDirectory(g_RootDirPath));
    }
#endif

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
