﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api.h>
#include <nnt/nnt_Argument.h>

#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_FileStorage.h>

//#define USE_FILE

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 512 * 1024;
    char g_HeapStack[HeapSize];

#ifdef USE_FILE
    // File Storage version
    const auto StorageSize = 8ULL * 1024 * 1024 * 1024;
    nn::fs::FileHandle baseFileHandle;
#else
    // Memory Storage version
    const auto StorageSize = 8 * 1024 * 1024;
    std::unique_ptr<char[]> storageBuffer;
#endif

    std::unique_ptr<char[]> cacheBuffer;
    std::unique_ptr<nn::fs::IStorage> baseStorage;


    void SetUpFatFs(std::unique_ptr<nn::fat::FatFileSystem>* pOutValue) NN_NOEXCEPT
    {
        // FatFs を生成
        std::unique_ptr<nn::fat::FatFileSystem> fatFs(new nn::fat::FatFileSystem());
        ASSERT_NE(fatFs, nullptr);

        // キャッシュバッファ・IStorage を割り当て
        ASSERT_NE(cacheBuffer, nullptr);
        size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Initialize(baseStorage.get(), cacheBuffer.get(), cacheBufferSize));

        static bool s_IsFormated = false;
        if( !s_IsFormated )
        {
            // FAT にフォーマット
            NNT_ASSERT_RESULT_SUCCESS(fatFs->Format());
            s_IsFormated = true;
        }

        // fatfs 内部のマウント処理
        NNT_ASSERT_RESULT_SUCCESS(fatFs->Mount());

        *pOutValue = std::move(fatFs);
    }

}

namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, const int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<nn::fat::FatFileSystem> fatFs;
        SetUpFatFs(&fatFs);

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));

        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestFileSystem(std::move(fatFs)));
        info->type = FileSystemType_FatFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        static const int FatNameLengthMax  = 255;     // prfile2 の LONG_NAME_CHARS
        static const int FatPathLengthMax  = 260 - 1; // prfile2 の LONG_NAME_PATH_CHARS
        static const int DriveLetterLength = 2;       // FatFileSystem で付与されるドライブレターの文字数 ("A:")
        attribute.directoryNameLengthMax = FatNameLengthMax;
        attribute.fileNameLengthMax      = FatNameLengthMax;
        attribute.directoryPathLengthMax = FatPathLengthMax - DriveLetterLength;
        attribute.filePathLengthMax      = FatPathLengthMax - DriveLetterLength;

        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = StorageSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.isSupportedMultiBytePath = false;
        attribute.isSupportedGetFreeSpaceSize = false;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isRootExcessFailureResultPathNotFound = true;

#if !defined(USE_FILE)
        attribute.isMemoryStorage = true;
#endif // !defined(USE_FILE)

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }
}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::api::LoadAllTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();

    nn::fs::SetEnabledAutoAbort(false);

#ifdef USE_FILE
    nnt::fs::util::TemporaryHostDirectory tempRootDirPath;
    nnt::fs::util::String baseFilePath;
    tempRootDirPath.Create();
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::MountHostRoot());
    baseFilePath = tempRootDirPath.GetPath() + "/FatBaseFile.bin";
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::CreateFile(baseFilePath.c_str(), StorageSize));
    NNT_ASSERT_RESULT_SUCCESS(nn::fs::OpenFile(&baseFileHandle, baseFilePath.c_str(), nn::fs::OpenMode_Read | nn::fs::OpenMode_Write));
    baseStorage.reset(new nn::fs::FileHandleStorage(baseFileHandle));
#else
    storageBuffer.reset(new char[StorageSize]);
    ASSERT_NE(storageBuffer, nullptr);
    memset(storageBuffer.get(), 0xCD, StorageSize);
    baseStorage.reset(new nn::fs::MemoryStorage(storageBuffer.get(), StorageSize));
#endif

    size_t cacheBufferSize = nn::fat::FatFileSystem::GetCacheBufferSize();
    cacheBuffer.reset(new char[cacheBufferSize]);


    auto ret = RUN_ALL_TESTS();

    baseStorage.reset(nullptr);

#ifdef USE_FILE
    nn::fs::FlushFile(baseFileHandle);
    nn::fs::CloseFile(baseFileHandle);
    nn::fs::DeleteFile(baseFilePath.c_str());
    nn::fs::UnmountHostRoot();
    tempRootDirPath.Delete();
#endif

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
