﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/crypto.h>

#include <nn/fssystem/fs_ApplicationTemporaryFileSystem.h>
#include <nn/fat/fat_FatFileSystem.h>
#include <nn/fs/fs_MemoryStorage.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

#include <nnt/nntest.h>
#include <nnt/base/testBase_Exit.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsApi/testFs_Unit_Api_SaveData.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/nnt_Argument.h>

using namespace nn::fssystem;

#include <nnt/fsUtil/testFs_util_GlobalNewDeleteChecker.Impl.h>

namespace {
    const int HeapSize = 10 * 1024 * 1024;
    char g_HeapStack[HeapSize];

    // SaveData Format Parameters
    const int64_t AvailableSize = 32 * 1024 * 1024;
    const size_t BlockSize = 16 * 1024;
    const uint32_t CountBlock = AvailableSize / BlockSize;
    int64_t g_TotalSize = 0;
    save::HierarchicalIntegrityVerificationStorageControlArea::InputParam g_ParamIntegrity;

    std::unique_ptr<nnt::fs::util::AccessCountedMemoryStorage> g_pBaseStorage;

    const auto MaxCacheCount = 1024;
    const auto SizeBlock = 16 * 1024;
    const auto BufferManagerHeapSize = 8 * 1024 * 1024;
    const size_t BufferManagerWorkBufferSize = 128 * 1024;
    NN_ALIGNAS(4096) char g_BufferManagerHeap[BufferManagerHeapSize];
    char g_BufferManagerWorkBuffer[BufferManagerWorkBufferSize];
    nn::fssystem::FileSystemBufferManager g_BufferManager;

    void SetUpBaseStorage() NN_NOEXCEPT
    {
        {
            g_pBaseStorage.reset(new nnt::fs::util::AccessCountedMemoryStorage());
            g_pBaseStorage->Initialize(static_cast<size_t>(g_TotalSize));
            g_pBaseStorage->FillBuffer(0, static_cast<size_t>(g_TotalSize), 0xCD);
        }

        g_BufferManager.Initialize(
            MaxCacheCount,
            reinterpret_cast<uintptr_t>(g_BufferManagerHeap),
            sizeof(g_BufferManagerHeap),
            SizeBlock,
            g_BufferManagerWorkBuffer,
            BufferManagerWorkBufferSize
        );

        {
            // フォーマット（実装は fssrv_FileSystemProxyImpl.cpp を参照）
            nn::fs::SubStorage storage(g_pBaseStorage.get(), 0, g_TotalSize);
            ASSERT_TRUE(save::IntegritySaveDataFileSystemDriver::Format(storage, BlockSize, g_ParamIntegrity, CountBlock, &g_BufferManager, nnt::fs::util::GetMacGenerator()).IsSuccess());
        }
    }

    void TearDownBaseStorage() NN_NOEXCEPT
    {
        g_pBaseStorage.reset();
        g_BufferManager.Finalize();
    }

    void SetUpApplicationTemporaryFs(std::unique_ptr<nn::fssystem::ApplicationTemporaryFileSystem>* pOutValue) NN_NOEXCEPT
    {
        // SaveDataFs を生成
        std::unique_ptr<nn::fssystem::ApplicationTemporaryFileSystem> saveDataFs(new nn::fssystem::ApplicationTemporaryFileSystem());
        ASSERT_NE(saveDataFs, nullptr);

        // 初期化・マウント
        ASSERT_TRUE(saveDataFs->Initialize(g_pBaseStorage.get(), &g_BufferManager, nnt::fs::util::GetMacGenerator()).IsSuccess());

        *pOutValue = std::move(saveDataFs);
    }
}

namespace nnt { namespace fs { namespace api {

    void GetTestFileSystemInfo(std::unique_ptr<TestFileSystemInfo>* outValue, int index) NN_NOEXCEPT
    {
        NN_UNUSED(index);

        std::unique_ptr<nn::fssystem::ApplicationTemporaryFileSystem> applicationTemporaryFs;
        SetUpApplicationTemporaryFs(&applicationTemporaryFs);

        std::unique_ptr<TestFileSystemInfo> info(new TestFileSystemInfo(FsApiTestType::Unit));

        nn::fssystem::ISaveDataExtraDataAccessor* pSaveDataExtraDataAccessor = applicationTemporaryFs.get();
        info->fileSystem = std::unique_ptr<ITestFileSystem>(new FsApiUnitTestSaveDataFileSystem(std::move(applicationTemporaryFs), pSaveDataExtraDataAccessor));
        info->type = FileSystemType_SaveDataFileSystem;
        info->rootDirPath = "";

        auto& attribute = info->attribute;

        attribute.directoryNameLengthMax = dbm::MaxDirectoryNameLength;
        attribute.fileNameLengthMax      = dbm::MaxFileNameLength;
        // EntryNameLengthMax 以上のパスが fs_SavePath で弾かれるので設定できるのは EntryNameLengthMax - 1 まで
        attribute.directoryPathLengthMax = nn::fs::EntryNameLengthMax - 1;
        attribute.filePathLengthMax      = nn::fs::EntryNameLengthMax - 1;
        attribute.fileSizeMax = 0xFFFFFFFF;
        attribute.storageSize = g_TotalSize;

        attribute.fileOpenMax      = 30;
        attribute.directoryOpenMax = 20;

        attribute.isSupportedMultiBytePath = true;
        attribute.isSupportedGetFreeSpaceSize = true;
        attribute.isConcatenationFileSystem = false;
        attribute.isSupportedQueryRange = true;
        attribute.isMemoryStorage = true;

        attribute.fileSizeAlignment = 1;

        *outValue = std::move(info);
    }

    void GetTestSaveDataExtraDataInfo(std::unique_ptr<TestSaveDataExtraDataInfo>* outValue, ITestFileSystem* pFileSystem) NN_NOEXCEPT
    {
        std::unique_ptr<TestSaveDataExtraDataInfo> info(new TestSaveDataExtraDataInfo);
        info->saveDataExtraDataAccessor = reinterpret_cast<FsApiUnitTestSaveDataFileSystem*>(pFileSystem)->GetSaveDataExtraDataAccessor();

        info->staticSaveDataExtraDataReader = [](void* pExtraData, size_t sizeExtraData) NN_NOEXCEPT -> nn::Result
        {
            NN_ABORT_UNLESS_LESS_EQUAL(sizeExtraData, sizeof(nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData));

            int64_t storageSize;
            NN_RESULT_DO(g_pBaseStorage->GetSize(&storageSize));

            nn::fs::SubStorage subStorage(g_pBaseStorage.get(), 0, storageSize);
            nn::fssystem::save::IntegritySaveDataFileSystem::ExtraData extraData;
            NN_RESULT_DO(nn::fssystem::save::IntegritySaveDataFileSystem::ReadExtraData(
                &extraData,
                subStorage,
                &g_BufferManager,
                nnt::fs::util::GetMacGenerator()
            ));

            std::memcpy(pExtraData, &extraData, sizeExtraData);

            NN_RESULT_SUCCESS;
        };

        *outValue = std::move(info);
    }

}}}

extern "C" void nnMain()
{
    int     argc = nnt::GetHostArgc();
    char**  argv = nnt::GetHostArgv();

    ::testing::InitGoogleTest(&argc, argv);

    nnt::fs::util::InitializeSaveDataTestHelper();

    nnt::fs::api::LoadAllTests();
    nnt::fs::api::LoadAllSaveDataExtraDataTests();

    nnt::fs::util::InitializeTestLibraryHeap(g_HeapStack, HeapSize);
    nn::fs::SetAllocator(nnt::fs::util::Allocate, nnt::fs::util::Deallocate);
    nnt::fs::util::ResetAllocateCount();
    save::HierarchicalIntegrityVerificationStorage::SetGenerateRandomFunction(nnt::fs::util::GenerateRandomForSaveDataTest);

    nn::fs::SetEnabledAutoAbort(false);

    nnt::fs::util::String tempRootDirPath;
    nnt::fs::util::String baseFilePath;

    // サイズ計算
    {
        g_ParamIntegrity.sizeBlockLevel[0] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[1] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[2] = BlockSize;
        g_ParamIntegrity.sizeBlockLevel[3] = BlockSize;

        NNT_ASSERT_RESULT_SUCCESS(save::IntegritySaveDataFileSystemDriver::QueryTotalSize(&g_TotalSize, BlockSize, g_ParamIntegrity, CountBlock));
    }

    static const size_t BufferPoolSize = 1024 * 1024;
    static NN_ALIGNAS(4096) char g_BufferPool[BufferPoolSize];
    static const size_t WorkBufferSize = nn::fssystem::BufferPoolWorkSize;
    static NN_ALIGNAS(8) char g_WorkBuffer[WorkBufferSize];
    nn::fssystem::InitializeBufferPool(g_BufferPool, BufferPoolSize, g_WorkBuffer, WorkBufferSize);

    SetUpBaseStorage();

    auto ret = RUN_ALL_TESTS();

    TearDownBaseStorage();

    nnt::fs::util::FinalizeSaveDataTestHelper();

    if (nnt::fs::util::CheckMemoryLeak())
    {
        nnt::Exit(1);
    }

    nnt::Exit(ret);
}
