﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const OpenDirectoryMode g_OpenDirectoryModes[] =
    {
        OpenDirectoryMode_Directory,
        OpenDirectoryMode_File,
        OpenDirectoryMode_All
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadTargetExistenceRoTests()
    {
        return;
    }

    class TargetExistenceRoFile : public GetFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_ExistFileName = GetTestRootPath().append("/TargetExistenceRo/test.file");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }
    protected:
        String m_ExistFileName;
    };

    class TargetExistenceRoDirectory : public GetFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_ExistDirectoryName = GetTestRootPath().append("/TargetExistenceRo/testDir");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }
    protected:
        String m_ExistDirectoryName;
    };

    class TargetExistenceRoNonExistent : public GetFileSystemTestFixture
    {
    };

    class TargetExistenceRoInvalidNotDir : public TargetExistenceRoFile
    {
    };

    class TargetExistenceRoInvalidNotFile : public TargetExistenceRoDirectory
    {
    };

    class TargetExistenceRoFileAsDirectory : public GetFileSystemTestFixture
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            m_FileAsDirectoryName = GetTestRootPath().append("/TargetExistenceRo/test.file/test");
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::TearDown();
        }
    protected:
        String m_FileAsDirectoryName;
    };

    //! @brief GetEntryType に存在しないパスを渡すと失敗します。
    TEST_F(TargetExistenceRoNonExistent, GetEntryType)
    {
        String noExistantEntryName = GetTestRootPath().append("/TargetExistenceRo/noexist");
        DirectoryEntryType deType;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            GetFs().GetEntryType(&deType, noExistantEntryName.c_str())
        );
    }

    //! @brief OpenFile に存在しないパスを渡すと失敗します。
    TEST_F(TargetExistenceRoNonExistent, OpenFile)
    {
        String noExistantFileName = GetTestRootPath().append("/TargetExistenceRo/noexist.file");
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            GetFs().OpenFile(&file, noExistantFileName.c_str(), OpenMode_Read)
        );
        // 読み込み専用のためこのテストは通りません。
        //NNT_EXPECT_RESULT_FAILURE(
        //    ResultPathNotFound,
        //    GetFs().OpenFile(&file, noExistantFileName.c_str(), OpenMode_Write)
        //);

        // 存在するファイル名と存在しないディレクトリ名の組み合わせ
        String badParentFileName = GetTestRootPath().append("/TargetExistenceRo/noexist/test.file");
        NNT_EXPECT_RESULT_FAILURE(
            nn::fs::ResultPathNotFound,
            GetFs().OpenFile(&file, badParentFileName.c_str(), OpenMode_Read));
    }

    //! @brief OpenDirectory に存在しないパスを渡すと失敗します。
    TEST_F(TargetExistenceRoNonExistent, OpenDirectory)
    {
        String noExistantDirectoryName = GetTestRootPath().append("/noexist");
        String badParentDirectoryName
            = GetTestRootPath().append("/TargetExistenceRo/noexist/testDir");
        for( auto openDirectoryMode : g_OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultPathNotFound,
                GetFs().OpenDirectory(
                    &directory, noExistantDirectoryName.c_str(), openDirectoryMode));

            // 存在するディレクトリ名と存在しないディレクトリ名の組み合わせ
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultPathNotFound,
                GetFs().OpenDirectory(
                    &directory, badParentDirectoryName.c_str(), openDirectoryMode));
        }
    }

    //! @brief GetEntryType に存在するファイルをディレクトリとして使ったパスを渡すと失敗します
    TEST_F(TargetExistenceRoFileAsDirectory, GetEntryType)
    {
        DirectoryEntryType deType;
        auto result = GetFs().GetEntryType(&deType, m_FileAsDirectoryName.c_str());
        if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDbmInvalidOperation, result);
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
        }
    }

    //! @brief OpenFile に存在するファイルをディレクトリとして使ったパスを渡すと失敗します
    TEST_F(TargetExistenceRoFileAsDirectory, OpenFile)
    {
        std::unique_ptr<ITestFile> file;
        auto result = GetFs().OpenFile(&file, m_FileAsDirectoryName.c_str(), OpenMode_Read);
        if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDbmInvalidOperation, result);
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
        }
    }

    //! @brief OpenDirectory に存在するファイルをディレクトリとして使ったパスを渡すと失敗します
    TEST_F(TargetExistenceRoFileAsDirectory, OpenDirectory)
    {
        for( auto openDirectoryMode : g_OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            auto result = GetFs().OpenDirectory(
                &directory,
                m_FileAsDirectoryName.c_str(),
                openDirectoryMode
            );
            if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
            {
                NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDbmInvalidOperation, result);
            }
            else
            {
                NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
            }
        }
    }

    //! @brief OpenFile に存在するディレクトリを渡すと失敗します。
    TEST_F(TargetExistenceRoInvalidNotFile, OpenFile)
    {
        std::unique_ptr<ITestFile> file;
        auto result = GetFs().OpenFile(&file, m_ExistDirectoryName.c_str(), OpenMode_Read);
        if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDbmInvalidOperation, result);
        }
        else
        {
            NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
        }

        // 読み込み専用のためこのテストは通りません。
        //NNT_EXPECT_RESULT_FAILURE(
        //    ResultPathNotFound,
        //    GetFs().OpenFile(&file, m_ExistDirectoryName.c_str(), OpenMode_Write)
        //);
    }

    //! @brief OpenDirectory に存在するファイルを渡すと失敗します。
    TEST_F(TargetExistenceRoInvalidNotDir, OpenDirectory)
    {
        for( auto openDirectoryMode : g_OpenDirectoryModes )
        {
            std::unique_ptr<ITestDirectory> directory;
            auto result = GetFs().OpenDirectory(&directory, m_ExistFileName.c_str(), openDirectoryMode);
            if( GetFsType() == nnt::fs::api::FileSystemType_HtmlDocumentFileSystem )
            {
                NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultDbmInvalidOperation, result);
            }
            else
            {
                NNT_EXPECT_RESULT_FAILURE(nn::fs::ResultPathNotFound, result);
            }
        }
    }
}}}
