﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <numeric>
#include <nn/fs.h>
#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nn/util/util_FormatString.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、並列同時実行観点を定義します。
*/
namespace {

    // テストファイル定義
    const char TestFileExt[]                = ".file";

    // 実行スレッド数定義
    const int ThreadCount                   = 2;
    const int ThreadCount2                  = 8;

    // スレッドスタックサイズ定義
    const size_t StackSize                  = 64 * 1024;

    // テスト対象ファイル数／ディレクトリ数定義
    const int TestFileCount                 = 1;
    const int TestFilesCount                = ThreadCount;
    const int TestDirectoriesCount          = ThreadCount;

    // バッファサイズ定義
    const int BufferSize                    = 1024;
    // エントリサイズ定義
    const int EntrySize                     = 2;
    // エントリ数テストエントリファイル数定義
    const int TestEntryCount                = 10;

    // テストを繰り返す回数定義
    const int TestMaxLoopCount              = 1000;
    const int TestOpenMaxLoopCount          = 10000;

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };

}

namespace nnt { namespace fs { namespace api {
    void LoadRaceConditionRoTests()
    {
        return;
    }

    // パラメータ用構造体
    struct ParamsStructure
    {
        ITestFileSystem*    fileSystem;
        ITestFile*          file;
        ITestDirectory*     directory;
        String*             path;
        int                 idx;
        int64_t             fileSize;
        char*               buffer;
        int                 bufferSize;
        DirectoryEntry*     entryBuffer;
        int                 entrySize;
        char*               randomString;
    };

    /**
     * @brief テストスレッドを待機する関数です
     *   @param[in]  pThreads               スレッドオブジェクト構造体配列ポインタ
     *   @param[in]  threadCount            待機するスレッド数
     */
    void TestThreadWait(const nn::os::ThreadType *pThreads, const int32_t threadCount)
    {
        NN_ASSERT_NOT_NULL(pThreads);

        for (int i = 0; i < threadCount; i++)
        {
            nn::os::WaitThread(const_cast<nn::os::ThreadType*>(&pThreads[i]));
            nn::os::DestroyThread(const_cast<nn::os::ThreadType*>(&pThreads[i]));
        }
    }

    /**
        * @brief 指定エントリタイプが存在するか確認する関数です
        *   @param[in]  fileSystem             ファイルシステム
        *   @param[in]  name                   確認するパスを指定します
        *   @param[in]  entryType              確認するファイル／ディレクトリのエントリタイプを指定します
        *
        *   @return     指定エントリタイプの結果が返ります。
        *   @retval     true                   指定のエントリタイプで存在した場合に返ります。
        *   @retval     false                  指定のエントリタイプで存在しない場合に返ります。
        */
    bool IsSpecifiedEntryTypeExist(
        ITestFileSystem* fileSystem, const String name, const DirectoryEntryType entryType) NN_NOEXCEPT
    {
        DirectoryEntryType entry;

        // 指定パスのエントリタイプを取得します
        if (fileSystem->GetEntryType(&entry, name.c_str()).IsSuccess() != true)
        {
            return false;
        }

        // 取得したエントリタイプの識別を判定します
        return ((entry & entryType) == entryType);
    }

    /**
     * @brief   テストファイル作成を事前準備するテストフィクスチャです。
     */
    class RaceConditionRoTestFilePreparationFixture : public GetFileSystemTestFixture
    {
    public:

        /**
         * @brief コンストラクタ
         *  テストファイル作成時のテストファイル数、ファイルサイズを設定します
         */
        RaceConditionRoTestFilePreparationFixture(
            const int32_t testFileCount, const int64_t fileSize) NN_NOEXCEPT
        {
            this->m_FileSize  = fileSize;
            this->m_FileCount = testFileCount;
        }

        /**
         * @brief テストファイルをオープンする関数です。
         *   @param[in]  mode               ファイルオープンモード
         */
        void OpenTestFile(const int32_t mode) NN_NOEXCEPT
        {
            for (int i = 0; i < this->m_FileCount; ++i)
            {
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenFile(
                        &this->m_FileArray[i],
                        this->m_FileNameArray[i].c_str(),
                        static_cast<OpenMode>(mode)));
            }
        }

    protected:
        void MakeExpectImage(char* pBuffer, const int64_t bufferSize) NN_NOEXCEPT
        {
            for (int64_t cnt = 0; cnt < bufferSize; cnt++)
            {
                pBuffer[cnt] = cnt & 0xffU;
            }
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            NN_ASSERT(m_FileSize < 10000);

            MakeExpectImage(this->m_RandomString, this->m_FileSize);

            for (int i = 0; i < this->m_FileCount; ++i)
            {
                char num[5];
                nn::util::SNPrintf(num, sizeof(num), "%04d", i);
                this->m_FileNameArray[i]
                    = GetTestRootPath().append("/RaceConditionRo/").append(num).append(TestFileExt);
            }
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < this->m_FileCount; ++i)
            {
                this->m_FileArray[i].reset(nullptr);
            }
            GetFileSystemTestFixture::TearDown();
        }

        String* GetFileName(int pos) NN_NOEXCEPT
        {
            NN_ASSERT_LESS_EQUAL(pos, TestFilesCount);
            return &m_FileNameArray[pos];
        }

        ITestFile* GetFile(int pos) NN_NOEXCEPT
        {
            NN_ASSERT_LESS_EQUAL(pos, TestFilesCount);
            return m_FileArray[pos].get();
        }

        char* GetRandomString() NN_NOEXCEPT
        {
            return m_RandomString;
        }

        int64_t GetFileSize() NN_NOEXCEPT
        {
            return m_FileSize;
        }

    private:
        std::unique_ptr<ITestFile> m_FileArray[TestFilesCount];
        String m_FileNameArray[TestFilesCount];
        char m_RandomString[4096];
        int64_t m_FileSize;
        int32_t m_FileCount;
    };

    /**
     * @brief   テストディレクトリ作成、オープンまでを事前準備するテストフィクスチャです。
     */
    class RaceConditionRoTestDirectoryPreparationFixture : public GetFileSystemTestFixture
    {
    public:

        /**
         * @brief コンストラクタ
         *  テストディレクトリを作成するディレクトリ数を設定します
         */
        explicit RaceConditionRoTestDirectoryPreparationFixture(
            const int32_t testDirCount) NN_NOEXCEPT
        {
            this->m_DirectoryCount  = testDirCount;
        }

        /**
         * @brief テストディレクトリをオープンする関数です。
         *   @param[in]  mode               ディレクトリオープンモード
         */
        void OpenTestDirectory(const int32_t mode) NN_NOEXCEPT
        {
            for (int i = 0; i < this->m_DirectoryCount; ++i)
            {
                NNT_FS_SCOPED_TRACE_SAFE(
                    "loop count: %d, directory path: %s\n", i, this->m_DirectoryName.c_str());
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(
                        &this->m_DirectoryArray[i],
                        this->m_DirectoryName.c_str(),
                        static_cast<OpenDirectoryMode>(mode)));
            }
        }

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            GetFileSystemTestFixture::SetUp();
            nnt::fs::util::String rootPath = GetTestRootPath().append("/");
            this->m_DirectoryName = rootPath.append("RaceConditionRo/Test/");
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < this->m_DirectoryCount; ++i)
            {
                this->m_DirectoryArray[i].reset(nullptr);
            }
            GetFileSystemTestFixture::TearDown();
        }

        ITestDirectory* GetDirectory(int pos) NN_NOEXCEPT
        {
            NN_ASSERT_LESS_EQUAL(pos, TestDirectoriesCount);
            return m_DirectoryArray[pos].get();
        }

        String* GetDirectoryName() NN_NOEXCEPT
        {
            return &m_DirectoryName;
        }

    private:
        std::unique_ptr<ITestDirectory> m_DirectoryArray[TestDirectoriesCount];
        String m_DirectoryName;
        int32_t m_DirectoryCount;

    };

    /**
     * @brief   RaceConditionRoOpenFile で利用するテストフィクスチャです。
     */
    class RaceConditionRoOpenFile : public RaceConditionRoTestFilePreparationFixture
    {
    public:

        RaceConditionRoOpenFile() NN_NOEXCEPT
              : RaceConditionRoTestFilePreparationFixture(TestFilesCount, BufferSize * ThreadCount)
        {
        }

        /**
         * @brief ファイルオープンを同時実行するスレッド関数です
         *   @param[in]  pArg                    ParamsStructure構造体ポインタ
         */
        static void OpenFileThreadFunction(void* pArg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);
            for (int i = 0; i < TestOpenMaxLoopCount; ++i)
            {
                std::unique_ptr<ITestFile> file;
                NNT_ASSERT_RESULT_SUCCESS(
                    params->fileSystem->OpenFile(&file, params->path->c_str(), OpenMode_Read));
            }
        }
    };

    /**
    * @brief   RaceConditionRoReadFile で利用するテストフィクスチャです。
    */
    class RaceConditionRoReadFile : public RaceConditionRoTestFilePreparationFixture
    {
    public:
        char m_Buffer[ThreadCount][BufferSize];

        RaceConditionRoReadFile() NN_NOEXCEPT
            : RaceConditionRoTestFilePreparationFixture(TestFileCount, BufferSize * ThreadCount)
        {
        }

        /**
        * @brief ファイル読み込みを同時実行するスレッド関数です
        *   @param[in]  pArg                    ParamsStructure構造体ポインタ
        */
        static void ReadFileThreadFunction(void* pArg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);
            const int64_t offset = params->idx * params->bufferSize;

            for( int i = 0; i < TestMaxLoopCount; ++i )
            {
                auto read = ReadFunctions[i % NN_ARRAY_SIZE(ReadFunctions)];

                NNT_EXPECT_RESULT_SUCCESS(
                    read(
                        params->file, offset, params->buffer, params->bufferSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(
                    params->randomString + offset, params->buffer, params->bufferSize);
                std::memset(params->buffer, i, params->bufferSize);
            }
        }
    };

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
    /**
     * @brief   RaceConditionRoReadFile2 で利用するテストフィクスチャです。
     */
    class RaceConditionRoReadFile2 : public RaceConditionRoTestFilePreparationFixture
    {
    public:
        struct ThreadInfo
        {
            ITestFile* pFile;
            size_t bufferSize;
            int64_t offset;
            int64_t readSize;
        };

    public:
        RaceConditionRoReadFile2() NN_NOEXCEPT
            : RaceConditionRoTestFilePreparationFixture(0, 0)
        {
        }

        // バッファの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        static void FillBuffer(char* buffer, size_t size, int64_t offset) NN_NOEXCEPT
        {
            std::iota(reinterpret_cast<uint8_t*>(buffer), reinterpret_cast<uint8_t*>(buffer + size), static_cast<uint8_t>(offset));
        }

        // バッファが先頭から dataOffset + 0x00, 0x01, 0x02, ... で埋められていることを確認する
        static void VerifyFilledData(const char* buffer, size_t size, int64_t dataOffset, char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
        {
            // workBufferSize が 0x100 の倍数ならループごとに FillBuffer() する必要がない
            // workBufferSize >= size の場合はループしないので 0x100 の倍数でなくて良い
            ASSERT_TRUE( workBufferSize >= size || workBufferSize % 0x100 == 0);
            ASSERT_NE(nullptr, workBuffer);

            const size_t bufferSize = std::min(size, workBufferSize);
            FillBuffer(workBuffer, bufferSize, dataOffset);

            for( size_t compareOffset = 0; compareOffset < size; compareOffset += bufferSize )
            {
                const size_t compareSize = std::min(bufferSize, static_cast<size_t>(size - compareOffset));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(workBuffer, buffer + compareOffset, compareSize);
            }
        }

        // ファイル読み込みを行うスレッド関数
        static void ReadFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            ASSERT_NE(nullptr, arg);

            const size_t BufferSizeMax = 1 * 1024 * 1024;

            ThreadInfo* pInfo = static_cast<ThreadInfo*>(arg);
            auto buffer = nnt::fs::util::AllocateBuffer(pInfo->bufferSize);
            ASSERT_NE(nullptr, buffer);

            const auto workBufferSize = std::min(pInfo->bufferSize, BufferSizeMax);
            auto workBuffer = nnt::fs::util::AllocateBuffer(workBufferSize);
            ASSERT_NE(nullptr, workBuffer);

            int index = 0;
            for( int64_t offset = pInfo->offset;
                offset < pInfo->offset + pInfo->readSize; offset += pInfo->bufferSize )
            {
                auto read = ReadFunctions[index % NN_ARRAY_SIZE(ReadFunctions)];
                ++index;

                const size_t readSize = std::min(
                    pInfo->bufferSize, static_cast<size_t>(pInfo->offset + pInfo->readSize - offset));
                NNT_EXPECT_RESULT_SUCCESS(read(
                    pInfo->pFile,
                    offset,
                    buffer.get(),
                    readSize));

                // 読み込み内容の確認
                VerifyFilledData(buffer.get(), readSize, offset, workBuffer.get(), workBufferSize);
            }
        }
    };
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

    /**
     * @brief   RaceConditionRoReadDirectory で利用するテストフィクスチャです。
     */
    class RaceConditionRoReadDirectory : public RaceConditionRoTestDirectoryPreparationFixture
    {
    public:
        DirectoryEntry entryReadBuffer[ThreadCount][EntrySize];

        RaceConditionRoReadDirectory() NN_NOEXCEPT
            : RaceConditionRoTestDirectoryPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief ディレクトリ読み込みを同時実行するスレッド関数です
         *   @param[in]  pArg                    ParamsStructure構造体ポインタ
         */
        static void ReadDirectoryThreadFunction(void* pArg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);

            for (int i = 0; i < TestMaxLoopCount; ++i)
            {
                int64_t resultEntry = 0;
                // 同一ディレクトリハンドルの場合、Readするとスレッド間でそれぞれFindNextされエントリの終端が予測不可となるため、実行のみの確認です
                NNT_EXPECT_RESULT_SUCCESS(
                    params->directory->Read(&resultEntry, params->entryBuffer, params->entrySize));
            }
        }
    };

    /**
     * @brief   RaceConditionRoGetDirectoryEntryCount で利用するテストフィクスチャです。
     */
    class RaceConditionRoGetDirectoryEntryCount
        : public RaceConditionRoTestDirectoryPreparationFixture
    {
    public:

        RaceConditionRoGetDirectoryEntryCount() NN_NOEXCEPT
            : RaceConditionRoTestDirectoryPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief エントリ数取得を同時実行するスレッド関数です
         *   @param[in]  pArg                    ParamsStructure構造体ポインタ
         */
        static void GetEntryCountThreadFunction(void* pArg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);

            for (int i = 0; i < TestMaxLoopCount; ++i)
            {
                int64_t resultEntry = 0;
                NNT_EXPECT_RESULT_SUCCESS(params->directory->GetEntryCount(&resultEntry));
                EXPECT_EQ(TestEntryCount, resultEntry);
            }
        }
    };

    /**
     * @brief   RaceConditionRoOpenDirectory で利用するテストフィクスチャです。
     */
    class RaceConditionRoOpenDirectory : public RaceConditionRoTestDirectoryPreparationFixture
    {
    public:

        RaceConditionRoOpenDirectory() NN_NOEXCEPT
            : RaceConditionRoTestDirectoryPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief オープンディレクトリを同時実行するスレッド関数です
         *   @param[in]  pArg                    ParamsStructure構造体ポインタ
         */
        static void OpenDirectoryThreadFunction(void* pArg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);

            for (int i = 0; i < TestOpenMaxLoopCount; ++i)
            {
                std::unique_ptr<ITestDirectory> directory;
                NNT_ASSERT_RESULT_SUCCESS(
                    params->fileSystem->OpenDirectory(
                        &directory, params->path->c_str(), OpenDirectoryMode_File));
            }
        }
    };

    /**
     * @brief   RaceConditionRoSetGetFileSize で利用するテストフィクスチャです。
     */
    class RaceConditionRoSetGetFileSize : public RaceConditionRoTestFilePreparationFixture
    {
    public:

        /**
         * @brief コンストラクタ
         */
        RaceConditionRoSetGetFileSize() NN_NOEXCEPT
              : RaceConditionRoTestFilePreparationFixture(TestFilesCount, BufferSize * ThreadCount)
        {
        }

        /**
         * @brief ファイルサイズ取得を同時実行するスレッド関数です
         *   @param[in]  pArg                    ParamsStructure構造体ポインタ
         */
        static void GetFileSizeThreadFunction(void* pArg) NN_NOEXCEPT
        {
            int64_t size;
            ParamsStructure* params = static_cast<ParamsStructure*>(pArg);
            for (int i = 0; i < TestMaxLoopCount; ++i)
            {
                NNT_EXPECT_RESULT_SUCCESS(params->file->GetSize(&size));
                EXPECT_TRUE(params->fileSize == size);
            }
        }
    };

    //!< @brief １ファイルをマルチスレッドで繰り返し同一ファイルのオープンが成功すること
    TEST_F(RaceConditionRoOpenFile, OpenFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].path          = this->GetFileName(i);
            params[i].fileSystem    = &GetFs();

            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    OpenFileThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    nn::os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief １ファイルを開き、マルチスレッドで繰り返し同一ファイルの読み込みが成功すること
    TEST_F(RaceConditionRoReadFile, ReadFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].file          = this->GetFile(0);
            params[i].idx           = i;
            params[i].buffer        = m_Buffer[i];
            params[i].bufferSize    = BufferSize;
            params[i].randomString  = this->GetRandomString();

            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    ReadFileThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
    // 複数スレッドでファイルの同じ位置を読み込む
    TEST_F(RaceConditionRoReadFile2, ReadFile)
    {
        char path[128];
        nn::util::SNPrintf(path, sizeof(path), "%s/RaceConditionRo/file.dat", GetTestRootPath().c_str());

        std::unique_ptr<ITestFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, path, nn::fs::OpenMode_Read));

        static const auto AllocatedStackBufferSize = StackSize + nn::os::ThreadStackAlignment;

        auto pThreadStackBuffer = nnt::fs::util::AllocateBuffer(ThreadCount2 * AllocatedStackBufferSize);

        {
            int64_t fileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&fileSize));

            ThreadInfo infoArray[ThreadCount2];
            nn::os::ThreadType threadArray[ThreadCount2];
            for( auto i = 0; i < ThreadCount2; ++i )
            {
                infoArray[i].pFile = pFile.get();
                switch( i )
                {
                case 0:
                case 1:
                    // ファイル全体を一度に読み込み
                    infoArray[i].bufferSize = static_cast<size_t>(fileSize);
                    infoArray[i].offset = 0;
                    infoArray[i].readSize = fileSize;
                    break;
                case 2:
                case 3:
                    // ファイル全体を分割して読み込み
                    infoArray[i].bufferSize
                        = static_cast<size_t>(std::min<int64_t>(256 * 1024, fileSize));
                    infoArray[i].offset = 0;
                    infoArray[i].readSize = fileSize;
                    break;
                default:
                    // ファイルの一部を分割して読み込み
                    infoArray[i].bufferSize
                        = static_cast<size_t>(std::min<int64_t>(256 * 1024, fileSize));
                    infoArray[i].offset = std::min<int64_t>(i * 13, fileSize);
                    infoArray[i].readSize = std::min(
                        static_cast<int64_t>(infoArray[i].bufferSize * i),
                        static_cast<int64_t>(fileSize - infoArray[i].offset));
                }
                NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &threadArray[i],
                    ReadFileThreadFunction,
                    &infoArray[i],
                    reinterpret_cast<char*>(nn::util::align_up(
                        reinterpret_cast<uintptr_t>(pThreadStackBuffer.get() + AllocatedStackBufferSize * i),
                        nn::os::ThreadStackAlignment)),
                    StackSize,
                    nn::os::DefaultThreadPriority));
            }

            for( auto& thread : threadArray )
            {
                nn::os::StartThread(&thread);
            }

            for( auto& thread : threadArray )
            {
                nn::os::WaitThread(&thread);
                nn::os::DestroyThread(&thread);
            }
        }
    }
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

    //!< @brief 同一ディレクトリに対して複数のハンドルで開き、マルチスレッドで繰り返しエントリ読み込みが成功すること
    TEST_F(RaceConditionRoReadDirectory, ReadDirectory)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestDirectory(OpenDirectoryMode_File);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].fileSystem    = &GetFs();
            params[i].directory     = this->GetDirectory(i);
            params[i].idx           = i;
            params[i].entryBuffer   = entryReadBuffer[i];
            params[i].entrySize     = EntrySize;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].directory);
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    ReadDirectoryThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief 同一ディレクトリに対して複数のハンドルで開き、マルチスレッドで繰り返しエントリ数の取得が成功すること
    TEST_F(RaceConditionRoGetDirectoryEntryCount, GetDirectoryEntryCount)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestDirectory(OpenDirectoryMode_File);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].fileSystem    = &GetFs();
            params[i].directory     = this->GetDirectory(i);
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].directory);
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    GetEntryCountThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief 同一ディレクトリに対して、マルチスレッドで繰り返し複数のハンドルでディレクトリのオープンが成功すること
    TEST_F(RaceConditionRoOpenDirectory, OpenDirectory)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].fileSystem    = &GetFs();
            params[i].path          = this->GetDirectoryName();
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].path);
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    OpenDirectoryThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief １つのファイルを開き、マルチスレッドで繰り返し同一ファイルのサイズ取得が成功すること
    TEST_F(RaceConditionRoSetGetFileSize, GetFileSize)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; ++i)
        {
            params[i].fileSystem    = &GetFs();
            params[i].file          = this->GetFile(0);
            params[i].fileSize      = nn::util::align_up(
                                        this->GetFileSize(), GetFsAttribute()->fileSizeAlignment); // ファイルサイズのアラインメントを考慮
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(
                nn::os::CreateThread(
                    &threads[i],
                    GetFileSizeThreadFunction,
                    &params[i],
                    stack[i],
                    StackSize,
                    os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }
}}}
