﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/fsApi/testFs_Api.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace {
    const int FileSize = 32;
}

namespace nnt { namespace fs { namespace api {
    void LoadTargetInUseTests() NN_NOEXCEPT
    {
        return;
    }

    class TargetInUseFile
        : public CleanupFileSystemTestFixture, public ::testing::WithParamInterface<OpenMode>
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_LockedFileName = GetTestRootPath().append("/locked.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_LockedFileName.c_str(), FileSize));
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(&m_File, m_LockedFileName.c_str(), GetParam()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_File.reset(nullptr);
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_LockedFileName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetLockedFileName() NN_NOEXCEPT
        {
            return m_LockedFileName;
        }
    private:
        std::unique_ptr<ITestFile> m_File;
        String m_LockedFileName;
    };

    class TargetInUseDirectory
        : public CleanupFileSystemTestFixture, public ::testing::WithParamInterface<OpenDirectoryMode>
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_LockedDirectoryName = GetTestRootPath().append("/lockedDir");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_LockedDirectoryName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenDirectory(&m_Directory, m_LockedDirectoryName.c_str(), GetParam()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_Directory.reset(nullptr);
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_LockedDirectoryName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetLockedDirectoryName() NN_NOEXCEPT
        {
            return m_LockedDirectoryName;
        }

    private:
        std::unique_ptr<ITestDirectory> m_Directory;
        String m_LockedDirectoryName;
    };

    class TargetInUseFileInDirectory
        : public CleanupFileSystemTestFixture, public ::testing::WithParamInterface<OpenMode>
    {
    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_TestDirectoryName = GetTestRootPath().append("/testDir");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateDirectory(m_TestDirectoryName.c_str()));
            m_LockedFileName = GetTestRootPath().append("/testDir/locked.file");
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(m_LockedFileName.c_str(), FileSize));
            NNT_ASSERT_RESULT_SUCCESS(
                GetFs().OpenFile(&m_File, m_LockedFileName.c_str(), GetParam()));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_File.reset(nullptr);
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(m_LockedFileName.c_str()));
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteDirectory(m_TestDirectoryName.c_str()));
            CleanupFileSystemTestFixture::TearDown();
        }

        const String& GetTestDirectoryName() NN_NOEXCEPT
        {
            return m_TestDirectoryName;
        }
    private:
        std::unique_ptr<ITestFile> m_File;
        String m_LockedFileName;
        String m_TestDirectoryName;
    };

    class TargetInUseFileWithOpenModeRead : public TargetInUseFile
    {
    };

    class TargetInUseFileWithOpenModeWrite : public TargetInUseFile
    {
    };


    String OpenModeInfo(const int openMode)
    {
        auto directoryName = AllocateBuffer(NameMaxLength);
        String mode;

        nn::util::SNPrintf(
            directoryName.get(),
            NameMaxLength,
            "%s",
            openMode == OpenMode_Read ? " Read" :openMode & OpenMode_Read ? " Read |" :"");
        mode.append(directoryName.get());
        nn::util::SNPrintf(
            directoryName.get(),
            NameMaxLength,
            "%s",
            openMode & OpenMode_Write ? openMode & OpenMode_AllowAppend ? " Write |" :" Write" :"");
        mode.append(directoryName.get());
        nn::util::SNPrintf(
            directoryName.get(),
            NameMaxLength,
            "%s",
            openMode & OpenMode_AllowAppend ? " AllowAppend" :"");
        mode.append(directoryName.get());

        return mode;
    }

//!< 全てのファイルオープンモードの組み合わせでオープンされているファイルを対象とする

    //!< @brief DeleteFile を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseFile, DeleteFile)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        NNT_EXPECT_RESULT_FAILURE(ResultTargetLocked, GetFs().DeleteFile(GetLockedFileName().c_str()));
    }

    //!< @brief RenameFile を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseFile, RenameFile)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        auto fileName = GetTestRootPath().append("/rename.file");
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().RenameFile(GetLockedFileName().c_str(), fileName.c_str()));
    }

    //!< @brief 取得できた EntryType が DirectoryEntryType_File であることを確認する
    TEST_P(TargetInUseFile, GetEntryType)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        DirectoryEntryType directoryEntryType;
        NNT_EXPECT_RESULT_SUCCESS(
            GetFs().GetEntryType(&directoryEntryType, GetLockedFileName().c_str()));
        EXPECT_EQ(DirectoryEntryType_File, directoryEntryType);
    }

//!< 二重にオープンする際のオープンモードに制限があることを確認する

    //!< @brief OpenMode_Read の際にオープンできることを確認する
    TEST_P(TargetInUseFileWithOpenModeRead, OpenFile)
    {
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, GetLockedFileName().c_str(), OpenMode_Read));
        file.reset(nullptr);

        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().OpenFile(&file, GetLockedFileName().c_str(), OpenMode_Write));
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked,
            GetFs().OpenFile(
                &file,
                GetLockedFileName().c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
    }

    //!< @brief OpenFile を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseFileWithOpenModeWrite, OpenFile)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        std::unique_ptr<ITestFile> file;
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().OpenFile(&file, GetLockedFileName().c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().OpenFile(&file, GetLockedFileName().c_str(), OpenMode_Write));
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked,
            GetFs().OpenFile(
                &file,
                GetLockedFileName().c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
    }

    const OpenMode AllOpenMode[] = {
        OpenMode_Read,
        OpenMode_Write,
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write),
        static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)
    };

    const OpenMode WriteMode[] = {
        OpenMode_Write,
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write),
        static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)
    };

    INSTANTIATE_TEST_CASE_P(WithOpenMode,
                            TargetInUseFile,
                            ::testing::ValuesIn(AllOpenMode));

    INSTANTIATE_TEST_CASE_P(WithOpenMode,
                            TargetInUseFileWithOpenModeRead,
                            ::testing::Values(OpenMode_Read));

    INSTANTIATE_TEST_CASE_P(WithOpenMode_WriteAllowAppend,
                            TargetInUseFileWithOpenModeWrite,
                            ::testing::ValuesIn(WriteMode));

//!< 全てのディレクトリオープンモードの組み合わせでオープンされているディレクトリを対象とする

    //!< @brief DeleteDirectory を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseDirectory, DeleteDirectory)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().DeleteDirectory(GetLockedDirectoryName().c_str()));
    }

    //!< @brief DeleteDirectoryRecursively を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseDirectory, DeleteDirectoryRecursively)
    {
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().DeleteDirectoryRecursively(GetLockedDirectoryName().c_str()));
    }

    //!< @brief RenameDirectory を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseDirectory, RenameDirectory)
    {
        auto directoryName = GetTestRootPath().append("/rename");
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked,
            GetFs().RenameDirectory(GetLockedDirectoryName().c_str(), directoryName.c_str()));
    }

    //!< @brief 取得できた EntryType が DirectoryEntryType_Directory であることを確認する
    TEST_P(TargetInUseDirectory, GetEntryType)
    {
        DirectoryEntryType directoryEntryType;
        NNT_EXPECT_RESULT_SUCCESS(
            GetFs().GetEntryType(&directoryEntryType, GetLockedDirectoryName().c_str()));
        EXPECT_EQ(DirectoryEntryType_Directory, directoryEntryType);
    }

    //!< @brief 二重にオープンできることを確認する
    TEST_P(TargetInUseDirectory, OpenDirectory)
    {
        std::unique_ptr<ITestDirectory> dir;
        NNT_EXPECT_RESULT_SUCCESS(
            GetFs().OpenDirectory(&dir, GetLockedDirectoryName().c_str(), OpenDirectoryMode_File));
        dir.reset(nullptr);
    }

    const OpenDirectoryMode AllOpenDirectoryMode[] = {
        OpenDirectoryMode_File,
        OpenDirectoryMode_Directory,
        OpenDirectoryMode_All
    };

    INSTANTIATE_TEST_CASE_P(WithOpenDirectoryMode,
                            TargetInUseDirectory,
                            ::testing::ValuesIn(AllOpenDirectoryMode));

//!< 全てのファイルオープンモードの組み合わせでオープンされているファイルを対象とする

    //!< @brief DeleteDirectory を実行した際に ResultDirectoryNotEmpty が返却されることを確認する
    TEST_P(TargetInUseFileInDirectory, DeleteDirectory)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        NNT_EXPECT_RESULT_FAILURE(
            ResultDirectoryNotEmpty, GetFs().DeleteDirectory(GetTestDirectoryName().c_str()));
    }

    //!< @brief DeleteDirectoryRecursively を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseFileInDirectory, DeleteDirectoryRecursively)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked, GetFs().DeleteDirectoryRecursively(GetTestDirectoryName().c_str()));
    }

    //!< @brief RenameDirectory を実行した際に ResultTargetLocked が返却されることを確認する
    TEST_P(TargetInUseFileInDirectory, RenameDirectory)
    {
        NNT_FS_SCOPED_TRACE("open mode: %s\n", OpenModeInfo(GetParam()).c_str());
        auto directoryName = GetTestRootPath().append("/rename");
        NNT_EXPECT_RESULT_FAILURE(
            ResultTargetLocked,
            GetFs().RenameDirectory(GetTestDirectoryName().c_str(), directoryName.c_str()));
    }

    INSTANTIATE_TEST_CASE_P(WithOpenMode,
                            TargetInUseFileInDirectory,
                            ::testing::ValuesIn(AllOpenMode));
}}}
