﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <numeric>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nnt/fsUtil/testFs_util.h>
#include <nnt/result/testResult_Assert.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt { namespace fs { namespace api {
    void LoadSizeArgumentTests() NN_NOEXCEPT
    {
        return;
    }

namespace{

    struct ParamsStructure{
        int64_t fileSize;
        int64_t offset;
        int64_t size;
    };

    const OpenMode OpenFileModes[] =
    {
        OpenMode_Read,
        OpenMode_Write,
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend),
        static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)
    };

    bool IsInvalidAccess(nn::Result result, bool isWriteMode)
    {
        NN_LOG("Result (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return ResultOutOfRange::Includes(result) || ResultInvalidArgument::Includes(result) ||
            (isWriteMode && ResultFileExtensionWithoutOpenModeAllowAppend::Includes(result));
    }

    enum class AccessRange
    {
        InRange,
        OutOfRange
    };

    typedef nn::Result(*ReadFunction)(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize);

    nn::Result ReadFunction0(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        if( (accessRange == AccessRange::InRange) && (size != expectedSize) )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                pTestFile->Read(offset, buffer, size, nn::fs::ReadOption())
            );
        }

        auto accessSize = ((accessRange == AccessRange::OutOfRange && size > 0) ? size : expectedSize);
        NN_RESULT_DO(pTestFile->Read(offset, buffer, accessSize, nn::fs::ReadOption()));

        *outValue = expectedSize;

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(size);

        if( (accessRange == AccessRange::InRange) && (size != expectedSize) )
        {
            NNT_EXPECT_RESULT_FAILURE(
                nn::fs::ResultOutOfRange,
                pTestFile->Read(offset, buffer, size)
            );
        }

        auto accessSize = ((accessRange == AccessRange::OutOfRange && size > 0) ? size : expectedSize);
        NN_RESULT_DO(pTestFile->Read(offset, buffer, accessSize));

        *outValue = expectedSize;

        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(accessRange);

        util::InvalidateVariable(outValue);
        NN_RESULT_DO(pTestFile->Read(outValue, offset, buffer, size, nn::fs::ReadOption()));
        EXPECT_EQ(expectedSize, *outValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(size_t* outValue, nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size, AccessRange accessRange, size_t expectedSize) NN_NOEXCEPT
    {
        NN_UNUSED(accessRange);

        util::InvalidateVariable(outValue);
        NN_RESULT_DO(pTestFile->Read(outValue, offset, buffer, size));
        EXPECT_EQ(expectedSize, *outValue);
        NN_RESULT_SUCCESS;
    }

    bool IsReadOverload(ReadFunction read) NN_NOEXCEPT
    {
        return read != ReadFunction2;
    }

    const ReadFunction ReadFunctions[] = {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

    class SizeArgumentFileSize : public nnt::fs::api::CleanupFileSystemTestFixture,
        public ::testing::WithParamInterface<int64_t>
    {
    };

    class SizeArgumentFileReadWrite : public nnt::fs::api::CleanupFileSystemTestFixture,
        public ::testing::WithParamInterface<ParamsStructure>
    {
        enum ReadWriteType {
            ReadWriteType_Read,
            ReadWriteType_WriteRead
        };

    protected:
        // Read・Writeの実行順の選択
        void ReadWriteTest(
            const int64_t fileSize, const int64_t offset, const size_t size, ReadFunction read) NN_NOEXCEPT
        {
            if(fileSize >= static_cast<int64_t>(offset + size))
            {
                // Write先行（fileSize に (offset + size) が収まる場合、write()先行で実行する。）
                // （ Write → Read の順で実行 ）
                ReadWriteTestProcess(fileSize, offset, size, ReadWriteType_WriteRead, read);
            }
            else
            {
                // Read先行（事前の fileSize に (offset + size) が fileSize を超える場合、
                //  Write() でファイルが生成される前に確認するためにread()先行で実行する。）
                // （ Read → Write → Read の順で実行 ）
                ReadWriteTestProcess(fileSize, offset, size, ReadWriteType_Read, read);
                ReadWriteTestProcess(fileSize, offset, size, ReadWriteType_WriteRead, read);
            }
            FsApiTestDebugPrint("\n");
        }

        // ファイル生成・オープン・Read/Write・クローズの実施及び結果の照合
        void ReadWriteTestProcess(
            const int64_t fileSize, int64_t offset, const size_t size, const int type, ReadFunction read) NN_NOEXCEPT
        {
            String fileName = GetTestRootPath().append("/test.file");
            NNT_FS_UTIL_SKIP_TEST_UNLESS( fileSize < GetFsAttribute()->storageSize &&
                offset < GetFsAttribute()->storageSize &&
                static_cast<int64_t>(size) < GetFsAttribute()->storageSize);

            // 大容量バッファ動的確保時のサイズ調整
            size_t       workSize = size;                           // バッファサイズ
            int64_t      remainSize = size;                         // 残数値
#if defined(NN_BUILD_CONFIG_OS_WIN32)
            const size_t ThresholdValue = 64 * 1024 * 1024;   // 大容量確認の閾値
#else
            const size_t ThresholdValue = 256 * 1024 * 1024;   // 大容量確認の閾値
#endif
            if(size > ThresholdValue){
                workSize = ThresholdValue;
                FsApiTestDebugPrint("  size changed : 0x%x to 0x%x \n",
                    static_cast<uint32_t>(size),
                    static_cast<uint32_t>(workSize));
            }

            const auto allocateSize = (workSize == 0 ? 1 : workSize);
            auto writeBuffer(nnt::fs::util::AllocateBuffer(allocateSize));
            auto readBuffer(nnt::fs::util::AllocateBuffer(allocateSize));

            if(writeBuffer.get() == nullptr || readBuffer.get() == nullptr)
            {
                ADD_FAILURE();
                NN_LOG("Buffer Area Out Of Memory Allocate Size 0x%x\n",
                    static_cast<uint32_t>(workSize));
                return;
            }

            // ファイル作成・オープン
            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileSize));
            if (GetFsAttribute()->isSaveFileSystem)
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());
            }
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

            FillBufferWithRandomValue(writeBuffer.get(), workSize);

            // 大容量バッファ対応（バッファサイズを丸めた際、繰り返すためのループ処理）
            do
            {
                size_t readSize = 0;
                util::InvalidateVariable(readBuffer.get(), static_cast<int>(workSize));
                // write確認（fileSize >= (offset + size)
                if(type == ReadWriteType_WriteRead)
                {
                    FsApiTestDebugPrint(
                        "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x (write) %s \n",
                        fileSize,
                        offset,
                        static_cast<uint32_t>(workSize),
                        size == workSize ? "" : "(loop)");
                    NNT_EXPECT_RESULT_SUCCESS(
                        file->Write(offset, writeBuffer.get(), workSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
                    // 結果照合（write後のファイルサイズ確認とread確認）
                    FsApiTestDebugPrint(
                        "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x ",
                        fileSize, offset, static_cast<uint32_t>(workSize));

                    int64_t currentFileSize = 0;
                    NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&currentFileSize));
                    if (size == 0 || static_cast<int64_t>(offset + workSize) <= fileSize)
                    {
                        // 0 バイト書き込み・ファイルサイズを超えない書き込みの場合はファイルサイズは更新されない
                        EXPECT_EQ(
                            nn::util::align_up(fileSize, GetFsAttribute()->fileSizeAlignment),
                            currentFileSize);
                    }
                    else
                    {
                        EXPECT_EQ(
                            nn::util::align_up(offset + workSize,
                                GetFsAttribute()->fileSizeAlignment),
                            currentFileSize);
                    }

                    if (size == 0 && fileSize < offset)
                    {
                        // 0 バイト書き込みの場合はファイルサイズが更新されないので ResultOutOfRange が返る
                        readSize = 0;
                        const auto result = read(
                            &readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::OutOfRange, workSize);
                        if( 0 < workSize )
                        {
                            NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, result);
                        }
                        else
                        {
                            NNT_EXPECT_RESULT_SUCCESS(result);
                        }
                    }
                    else
                    {
                        readSize = 0;
                        NNT_EXPECT_RESULT_SUCCESS(
                            read(
                                &readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::InRange, workSize));
                    }
                    FsApiTestDebugPrint(
                        "(read(0x%x)) %s \n",
                        static_cast<uint32_t>(readSize),
                        size == workSize ? "" : "(loop)");
                    NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), workSize);
                }
                // read確認（fileSize < (offset + size)
                if(type == ReadWriteType_Read)
                {
                    if(fileSize < offset)
                    {
                        // （無効なファイルポジション時）fileSize < offset の場合は、1 バイト以上の読み込みならば常に ResultOutOfRange が返ること。
                        FsApiTestDebugPrint(
                            "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x ",
                            fileSize, offset, static_cast<uint32_t>(workSize));
                        readSize = 0;
                        const auto result = read(
                            &readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::OutOfRange, 0);
                        if( 0 < workSize)
                        {
                            NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, result);
                        }
                        else
                        {
                            NNT_EXPECT_RESULT_SUCCESS(result);
                        }
                        FsApiTestDebugPrint(
                            "(read(0x%x)) %s \n",
                            static_cast<uint32_t>(readSize),
                            size == workSize ? "" : "(loop)");
                    }
                    else
                    {   // （正常なファイルポジション）fileSize >= offset の場合
                        FsApiTestDebugPrint(
                            "  fileSize = 0x%llx \toffset = 0x%llx \tsize = 0x%x ",
                            fileSize, offset, workSize);
                        size_t expectedSize = 0;
                        if( fileSize >= static_cast<int64_t>(offset + workSize) )
                        {
                            // （実行後ファイルサイズに収まるケース）fileSize >= (offset + workSize) の場合は、workSize と readSize が一致すること。
                            expectedSize = workSize;
                        }
                        else
                        {
                            // （実行後ファイルサイズを超えるケース）fileSize < (offset + workSize) の場合で、
                            //   offset > fileSize の場合は、0、
                            //   offset <= fileSize の場合は、(fileSize - offset) 分が read されること。
                            expectedSize = fileSize <= offset ? 0 : static_cast<size_t>(fileSize - offset);
                        }
                        readSize = 0;
                        NNT_EXPECT_RESULT_SUCCESS(
                            read(
                                &readSize, file.get(), offset, readBuffer.get(), workSize, AccessRange::InRange, expectedSize));
                        FsApiTestDebugPrint(
                            "(read(0x%x)) %s \n",
                            static_cast<uint32_t>(readSize),
                            size == workSize ? "" : "(loop)");
                    }
                }
                // 残数値更新
                remainSize = remainSize - workSize;
                offset = offset + workSize;
                workSize = static_cast<size_t>(remainSize) < workSize ?
                    static_cast<size_t>(remainSize) : workSize;

            } while (remainSize > 0);  // end of while

            file.reset(nullptr);
            if (GetFsAttribute()->isSaveFileSystem)
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());
            }
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
            if (GetFsAttribute()->isSaveFileSystem)
            {
                NNT_EXPECT_RESULT_SUCCESS(GetFs().Commit());
            }
        } // NOLINT(impl/function_size)
    };

    class SizeArgumentVariousAccess : public nnt::fs::api::CleanupFileSystemTestFixture, public ::testing::WithParamInterface<ReadFunction>
    {
    public:
        SizeArgumentVariousAccess() NN_NOEXCEPT
            : m_File()
            , m_FilePath()
            , m_FileSize(0)
            , m_WorkBuffer(nullptr, nnt::fs::util::DeleterBuffer)
            , m_WorkBufferSize(0)
        {
        }

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();

            m_FilePath = GetTestRootPath().append("/test.file");

            // 既に同名ファイルが存在していたらそれを削除
            nn::Result result = GetFs().DeleteFile(m_FilePath.c_str());
            if( !nn::fs::ResultPathNotFound::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            // 空き容量に合わせてファイルサイズを調整
            static const int FileSizeMax = 256 * 1024 * 1024;
            int64_t fileSize = FileSizeMax;
            int64_t freeSpaceSize = 0;
            result = GetFs().GetFreeSpaceSize(&freeSpaceSize, GetTestRootPath().c_str());
            if( result.IsSuccess() )
            {
                if( freeSpaceSize < fileSize )
                {
                    fileSize = freeSpaceSize;
                }
            }
            else if( !nn::fs::ResultNotImplemented::Includes(result) )
            {
                NNT_ASSERT_RESULT_SUCCESS(result);
            }

            // ファイル作成
            // 空き容量取得ができないファイルシステムのために容量不足はサイズを減らしてリトライする
            static const int FileSizeMin = 32;
            for( ; fileSize >= FileSizeMin; fileSize /= 2 )
            {
                result = GetFs().CreateFile(m_FilePath.c_str(), fileSize);
                if( !nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
                {
                    break;
                }

                // TmFileSystem では容量不足で CreateFile に失敗してもファイルが作成される場合があるのでリトライのために削除する
                const auto deleteResult = GetFs().DeleteFile(m_FilePath.c_str());
                if( !nn::fs::ResultPathNotFound::Includes(deleteResult) )
                {
                    NNT_ASSERT_RESULT_SUCCESS(deleteResult);
                }
            }
            NNT_ASSERT_RESULT_SUCCESS(result);
            m_FileSize = fileSize;

            // VerifyFilledData で使用するバッファを確保
            m_WorkBufferSize = static_cast<size_t>(std::min(static_cast<int64_t>(BufferSizeMax), fileSize));
            m_WorkBuffer = nnt::fs::util::AllocateBuffer(m_WorkBufferSize);

            // テストで使うためにファイルを開き、初期状態の内容にする
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(
                &m_File,
                m_FilePath.c_str(),
                static_cast<OpenMode>(nn::fs::OpenMode_Read | nn::fs::OpenMode_Write)));
            FillFile();

            NN_LOG("FileSize: %lld\n", m_FileSize);
        }

        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            m_File.reset();
            NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteFile(m_FilePath.c_str()));
            m_FilePath = "";
            m_FileSize = 0;

            CleanupFileSystemTestFixture::TearDown();
        }

        ITestFile* GetFile() NN_NOEXCEPT
        {
            return m_File.get();
        }

        int64_t GetFileSize() const NN_NOEXCEPT
        {
            return m_FileSize;
        }

        // バッファが先頭から dataOffset + 0x00, 0x01, 0x02, ... で埋められていることを確認する
        void VerifyFilledData(const char* buffer, size_t size, int64_t dataOffset) NN_NOEXCEPT
        {
            // workBufferSize が 0x100 の倍数ならループごとに FillBuffer() する必要がない
            // workBufferSize >= size の場合はループしないので 0x100 の倍数でなくて良い
            ASSERT_TRUE(m_WorkBufferSize >= size || m_WorkBufferSize % 0x100 == 0);
            ASSERT_NE(nullptr, m_WorkBuffer);

            const size_t bufferSize = std::min(size, m_WorkBufferSize);
            FillBuffer(m_WorkBuffer.get(), bufferSize, dataOffset);

            for( size_t compareOffset = 0; compareOffset < size; compareOffset += bufferSize )
            {
                const size_t compareSize = std::min(bufferSize, static_cast<size_t>(size - compareOffset));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(m_WorkBuffer.get(), buffer + compareOffset, compareSize);
            }
        }

    private:
        const size_t BufferSizeMax = 1 * 1024 * 1024;

    private:
        // バッファの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        void FillBuffer(char* buffer, size_t size, int64_t offset) NN_NOEXCEPT
        {
            std::iota(reinterpret_cast<uint8_t*>(buffer), reinterpret_cast<uint8_t*>(buffer + size), static_cast<uint8_t>(offset));
        }

        // ファイルの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        void FillFile() NN_NOEXCEPT
        {
            const size_t bufferSize = std::min(BufferSizeMax, static_cast<size_t>(m_FileSize));
            auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
            ASSERT_NE(nullptr, writeBuffer);
            FillBuffer(writeBuffer.get(), bufferSize, 0);

            NN_UTIL_SCOPE_EXIT
            {
                NNT_ASSERT_RESULT_SUCCESS(GetFile()->Flush());
            };

            for( int64_t offset = 0; offset < m_FileSize; offset += bufferSize )
            {
                const size_t writeSize = std::min(bufferSize, static_cast<size_t>(m_FileSize - offset));
                auto result = GetFile()->Write(
                    offset,
                    writeBuffer.get(),
                    writeSize,
                    nn::fs::WriteOption::MakeValue(0)
                );

                // Generic 版の HostFileSystem では CreateFile で Sparse なファイルが作成されるので
                // 書き込み時に容量不足で失敗する場合がある
                if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
                {
                    // 書き込めた分までをファイルサイズとして再設定する
                    m_FileSize = offset;
                    NNT_ASSERT_RESULT_SUCCESS(GetFile()->SetSize(m_FileSize));
                    return;
                }

                NNT_ASSERT_RESULT_SUCCESS(result);
            }
        }

    private:
        std::unique_ptr<ITestFile> m_File;
        nnt::fs::util::String m_FilePath;
        int64_t m_FileSize;
        std::unique_ptr<char, decltype(&nnt::fs::util::DeleterBuffer)> m_WorkBuffer;
        size_t m_WorkBufferSize;
    };

    class SizeArgumentFile : public CleanupFileSystemTestFixture
    {
    protected:
        static const int64_t FileSize = 1024;

    protected:
        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            m_FileName = GetTestRootPath().append("/test.file");

            NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(GetFileName().c_str(), FileSize));
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            // ファイルシステムによっては失敗を記憶しているために DeleteFile に失敗する
            (void) GetFs().DeleteFile(GetFileName().c_str());
            CleanupFileSystemTestFixture::TearDown();
        }

        const nnt::fs::util::String& GetFileName() NN_NOEXCEPT
        {
            return m_FileName;
        }

    private:
        nnt::fs::util::String m_FileName;
    };

    typedef SizeArgumentFileSize SizeArgumentFileSizeIllegal;
    typedef SizeArgumentFileReadWrite SizeArgumentSetSizeIllegal;
    typedef SizeArgumentFileReadWrite SizeArgumentOffsetAndSize;
    typedef SizeArgumentFileReadWrite SizeArgumentFileSizeSet;
    typedef SizeArgumentFileReadWrite SizeArgumentOffsetAndSizeIllegal;
    typedef SizeArgumentFileSize SizeArgumentOverFileSize;
    typedef SizeArgumentFileSize SizeArgumentOffsetAndSizeFull;
    typedef SizeArgumentFileSize SizeArgumentEmpty;
    typedef SizeArgumentFileSize SizeArgumentOffsetFileSize;

    // TODO: ファイル作成部分の共通化

    //! @brief GetParam() サイズのファイルを作成出来、GetSize() でサイズを取得できること
    TEST_P(SizeArgumentFileSize, CreateFile_GetSize)
    {
        String fileName = GetTestRootPath().append("/test.file");
        int64_t fileSize = GetParam();

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileSize));
        std::unique_ptr<ITestFile> file;
        int64_t actualSize;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        FsApiTestDebugPrint("  fileSize = 0x%llx actualSize = 0x%llx \n", fileSize, actualSize);
        EXPECT_EQ(fileSize, actualSize);
        file.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief GetParam() で サイズのファイルを作成し、SetSize() でサイズを設定（および GetSize() でサイズを取得）できること
    TEST_P(SizeArgumentFileSizeSet, SetSize_GetSize)
    {
        String fileName = GetTestRootPath().append("/test.file");
        ParamsStructure fileParam = GetParam();

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.size <= GetFsAttribute()->fileSizeMax);
        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.fileSize <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileParam.fileSize));
        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
        int64_t actualSize;
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        FsApiTestDebugPrint("  fileSize = 0x%llx to 0x%llx \n", actualSize, fileParam.size);
        NNT_EXPECT_RESULT_SUCCESS(file->SetSize(fileParam.size));
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        EXPECT_EQ(fileParam.size, actualSize);
        file.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief 負のサイズ指定でファイルを作成（CreateFile()）を実行すると OutOfRange が返ること
    TEST_P(SizeArgumentFileSizeIllegal, CreateFile)
    {
        String fileName = GetTestRootPath().append("/test.file");
        int64_t fileSize = GetParam();
        NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, GetFs().CreateFile(fileName.c_str(), fileSize));
        NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief GetParam() サイズのファイルを作成後、負のサイズ指定で SetSize() を実行すると OutOfRange が返ること
    TEST_P(SizeArgumentSetSizeIllegal, SetSize)
    {
        String fileName = GetTestRootPath().append("/test.file");
        ParamsStructure fileParam = GetParam();

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.fileSize <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileParam.fileSize));

        for( auto openMode : OpenFileModes )
        {
            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), openMode));
            NNT_EXPECT_RESULT_FAILURE(ResultOutOfRange, file->SetSize(fileParam.size));
        }

        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief GetParam() サイズのファイル作成後、ファイルサイズ + 1 のサイズで SetSize() 出来、GetSize() で取得できること
    TEST_P(SizeArgumentOverFileSize, SetSize_GetSize)
    {
        String fileName = GetTestRootPath().append("/test.file");
        int64_t fileSize = GetParam();

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize + 1 <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileSize));
        std::unique_ptr<ITestFile> file;
        int64_t actualSize;
        int64_t expectSize = fileSize + 1;

        NNT_FS_UTIL_SKIP_TEST_UNLESS( expectSize < GetFsAttribute()->storageSize);
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
        FsApiTestDebugPrint("  fileSize = 0x%llx to 0x%llx \n", fileSize, fileSize + 1);
        NNT_EXPECT_RESULT_SUCCESS(file->SetSize(expectSize));
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&actualSize));
        EXPECT_EQ(expectSize, actualSize);
        file.reset(nullptr);
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    //! @brief GetParam() サイズのファイルを作成し、offset、size の組合せでRead/Writeが正しく動作すること
    TEST_P(SizeArgumentOffsetAndSize, WriteFile_ReadFile)
    {
        ParamsStructure fileParam = GetParam();

        const auto fileSizeMax = GetFsAttribute()->fileSizeMax;
        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.fileSize <= fileSizeMax);
        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.offset + fileParam.size <= fileSizeMax);

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            ReadWriteTest(fileParam.fileSize, fileParam.offset, static_cast<size_t>(fileParam.size), read);
        }
    }

    //! @brief GetParam() サイズのファイルを作成し、ファイルサイズに等しいオフセットで Read が正しく動作すること
    TEST_P(SizeArgumentOffsetFileSize, ReadFile)
    {
        const String fileName = GetTestRootPath().append("/test.file");
        const int64_t fileSize = static_cast<size_t>(GetParam());

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
        };

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            std::unique_ptr<ITestFile> file;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Read));

            const size_t BufferSize = 32;
            auto readBuffer = AllocateBuffer(BufferSize);
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(
                read(&readSize, file.get(), fileSize, readBuffer.get(), BufferSize, AccessRange::InRange, static_cast<size_t>(0)));
        }
    }

    //! @brief GetParam() サイズのファイルを作成し、1 バイト単位でファイル末尾まで正しく書き込めること
    TEST_P(SizeArgumentOffsetAndSizeFull, WriteFile_ReadFile_Heavy)
    {
        String fileName = GetTestRootPath().append("/test.file");
        int64_t fileSize = GetParam();
        const int BufferSize = 16 * 1024;
        ASSERT_LE(fileSize, BufferSize);

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileSize + BufferSize <= GetFsAttribute()->fileSizeMax);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
        };

        static char writeBuffer[BufferSize];
        FillBufferWithRandomValue(writeBuffer, BufferSize);

        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                fileName.c_str(),
                static_cast<OpenMode>(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend)));

        size_t accessSizeArray[] = { 1 };
        for( auto accessSize : accessSizeArray )
        {
            size_t wrote = 0;
            while( wrote != static_cast<size_t>(fileSize) )
            {
                NNT_EXPECT_RESULT_SUCCESS(
                    file->Write(wrote, writeBuffer + wrote, accessSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
                wrote += accessSize;
            }

            static char readBuffer[BufferSize] = { 0 };
            size_t read = 0;
            int readFunctionIndex = 0;
            while( read != static_cast<size_t>(fileSize) )
            {
                auto readFunction = ReadFunctions[readFunctionIndex % NN_ARRAY_SIZE(ReadFunctions)];
                size_t readSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(
                    readFunction(&readSize, file.get(), read, readBuffer + read, accessSize, AccessRange::InRange, accessSize));
                read += readSize;
                ++readFunctionIndex;
            }

            NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer, readBuffer, static_cast<size_t>(fileSize));
        }
    }

    //! @brief GetParam() のサイズでファイルを作成し、GetParam() のオフセット・サイズ指定で Write() または、Read() を実行すると所定のエラー Result が帰ること
    TEST_P(SizeArgumentOffsetAndSizeIllegal, WriteFile_ReadFile)
    {
        String fileName = GetTestRootPath().append("/test.file");
        ParamsStructure fileParam = GetParam();
        const int BufferSize = 32;

        NNT_FS_UTIL_SKIP_TEST_UNLESS(fileParam.fileSize <= GetFsAttribute()->fileSizeMax && fileParam.fileSize <= GetFsAttribute()->storageSize);

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), fileParam.fileSize));
        NN_UTIL_SCOPE_EXIT
        {
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
        };

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            size_t accessSize = std::min(static_cast<int>(fileParam.size), BufferSize);
            {
                std::unique_ptr<ITestFile> file;
                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Write)));
                auto writeBuffer = AllocateBuffer(BufferSize);
                EXPECT_TRUE(IsInvalidAccess(file->Write(fileParam.offset, writeBuffer.get(), accessSize, WriteOption()), true));
            }
            {
                std::unique_ptr<ITestFile> file;
                NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read)));
                auto readBuffer = AllocateBuffer(BufferSize);
                size_t readSize = 0;
                EXPECT_TRUE(IsInvalidAccess(read(&readSize, file.get(), fileParam.offset, readBuffer.get(), accessSize, AccessRange::OutOfRange, static_cast<size_t>(0)), false));
            }
        }
    }

    //! サイズ 0 のファイルを作成し、サイズ 0 を指定した Write(), Read() および Flush() が成功すること
    TEST_F(SizeArgumentEmpty, WriteFile_ReadFile_FlushFile)
    {
        const auto fileName = GetTestRootPath().append("/test.file");
        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), 0));

        std::unique_ptr<ITestFile> file;
        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read | OpenMode_Write)));

        int64_t fileSize = -1;
        NNT_EXPECT_RESULT_SUCCESS(file->GetSize(&fileSize));
        EXPECT_EQ(0, fileSize);

        char buffer[1] = {};
        size_t readSize = std::numeric_limits<size_t>::max();
        NNT_EXPECT_RESULT_SUCCESS(file->Write(0, buffer, 0, WriteOption()));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(0, buffer, 0, ReadOption()));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(0, buffer, 0));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, 0, buffer, 0, ReadOption()));
        EXPECT_EQ(0, readSize);
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, 0, buffer, 0));
        EXPECT_EQ(0, readSize);

        // サイズ 0 の場合はバッファが nullptr でも成功する
        NNT_EXPECT_RESULT_SUCCESS(file->Write(0, nullptr, 0, WriteOption()));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(0, nullptr, 0, ReadOption()));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(0, nullptr, 0));
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, 0, nullptr, 0, ReadOption()));
        EXPECT_EQ(0, readSize);
        NNT_EXPECT_RESULT_SUCCESS(file->Read(&readSize, 0, nullptr, 0));
        EXPECT_EQ(0, readSize);

        NNT_EXPECT_RESULT_SUCCESS(file->Flush());

        file.reset();
        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }

    int64_t param1[] = { 0LL, 32LL, 4096LL, 16384LL };
    int64_t param2[] = { 0xFFFFFFFFLL };
    int64_t param3[] = { -1LL };
    int64_t param4[] = { 16384LL };

    ParamsStructure setGetSizeParam[] = {
      // | fileSize  |   offset      |   size    |
      //  -----------------------------------------
        { 0x0,        0x0,          0x0         },
        { 0x0,        0x0,          0x20        },
        { 0x20,       0x0,          0x0         },
        { 0x20,       0x0,          0x10        },
        { 0x20,       0x0,          0x20        },
        { 0x20,       0x0,          0x1000      },
        { 0x20,       0x0,          0xFFFFFFFF  },
        { 0x20,       0x0,          0x100000000 }
    };

    ParamsStructure setSizeIllegalParam[] = {
      // | fileSize |   offset    |   size    |
      //  -----------------------------------------
        { 0x0,        0x0,          -1 },
        { 0x20,       0x0,          -1 }
    };

    ParamsStructure offsetAndSizeParam[] = {
      // | fileSize |   offset    |   size    |
      //  -----------------------------------------
        { 0x0,        0x0,          0x0     },
        { 0x0,        0x0,          0x20    },
        { 0x0,        0x20,         0x0     },
        { 0x0,        0x20,         0x20    },
        { 0x20,       0x0,          0x0     },
        { 0x20,       0x0,          0x20    },
        { 0x20,       0x0,          0x10    },
        { 0x20,       0x0,          0x21    },
        { 0x20,       0x20,         0x0     },
        { 0x20,       0x20,         0x10    },
        { 0x20,       0x10,         0x0     },
        { 0x20,       0x10,         0x20    },
        { 0x20,       0x10,         0x10    },
        { 0x20,       0x21,         0x0     },
        { 0x20,       0x21,         0x20    },
        { 0x20,       0x40,         0x0     },
        { 0x20,       0x40,         0x20    },
        { 0x1000,     0x0,          0x0     },
        { 0x1000,     0x0,          0x1000  },
        { 0x1000,     0x0,          0x800   },
        { 0x1000,     0x0,          0x1001  },
        { 0x1000,     0x1000,       0x0     },
        { 0x1000,     0x1000,       0x800   },
        { 0x1000,     0x800,        0x0     },
        { 0x1000,     0x800,        0x20    },
        { 0x1000,     0x800,        0x1000  },
        { 0x1000,     0x800,        0x800   },
        { 0x1000,     0x1001,       0x0     },
        { 0x1000,     0x1001,       0x20    },
        { 0x4000,     0x0,          0x0     },
        { 0x4000,     0x0,          0x4000  },
        { 0x4000,     0x0,          0x2000  },
        { 0x4000,     0x0,          0x4001  },
        { 0x4000,     0x4000,       0x0     },
        { 0x4000,     0x4000,       0x2000  },
        { 0x4000,     0x2000,       0x0     },
        { 0x4000,     0x2000,       0x20    },
        { 0x4000,     0x2000,       0x4000  },
        { 0x4000,     0x2000,       0x2000  },
        { 0x4000,     0x4001,       0x0     },
        { 0x4000,     0x4001,       0x20    }
    };

    ParamsStructure offsetOrSize32bitIllegalParam[] = {
        // | fileSize    |   offset       |   size    |
        { 0xFFFFFFFFLL + 1,  0xFFFFFFFFLL + 3,  0x1 },
        { 0xFFFFFFFFLL + 2,  0xFFFFFFFFLL + 3,  0x1 },
        { 0x10,              0xFFFFFFFFLL + 1,  0x1 },
        { 0x10,              0xFFFFFFFFLL + 2,  0x1 },
        { 0x10,              -1,                0x1 },
    };

    ParamsStructure offsetAndSizeHeavyParam[] = {
      // | fileSize  |   offset    |   size    |
      //  -----------------------------------------
        { 0xFFFFFFFF, 0x0,          0x0        },
        { 0xFFFFFFFF, 0x0,          0xFFFFFFFF },
        { 0xFFFFFFFF, 0x0,          0x7FFFFFFF },
        { 0xFFFFFFFF, 0xFFFFFFFF,   0x0        },
        { 0xFFFFFFFF, 0xFFFFFFFF,   0x20       },
        { 0xFFFFFFFF, 0x100000000,  0x0        },
        { 0xFFFFFFFF, 0x100000000,  0x20       },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0x0        },
        { 0xFFFFFFFF, 0x7FFFFFFF,   0x20       },
        { 0x7CFF0000, 0x0,          0x7CFF0000 },
        //{ 0xFFFFFFFF, 0x7FFFFFFF,   0xFFFFFFFF }, // MountHost で失敗する
        //{ 0xFFFFFFFF, 0x7FFFFFFF,   0x7FFFFFFF }, // MountHost で失敗する
        { 0x0,        0x0,          0xFFFFFFFF },
        { 0x0200000000, 0x0100000000, 0x20 },
        { 0x0400000000, 0x0200000000, 0x20 },
        { 0x0800000000, 0x0400000000, 0x20 },
        { 0x1000000000, 0x0800000000, 0x20 },
    };

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentFileSize,
                            ::testing::ValuesIn(param1));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentFileSizeSet,
                            ::testing::ValuesIn(setGetSizeParam));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentFileSizeIllegal,
                            ::testing::ValuesIn(param3));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentSetSizeIllegal,
                            ::testing::ValuesIn(setSizeIllegalParam));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentOffsetAndSize,
                            ::testing::ValuesIn(offsetAndSizeParam));

    INSTANTIATE_TEST_CASE_P(WithVariousSizeHeavy,
                            SizeArgumentOffsetAndSize,
                            ::testing::ValuesIn(offsetAndSizeHeavyParam));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentOffsetAndSizeFull,
                            ::testing::ValuesIn(param4));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentOffsetAndSizeIllegal,
                            ::testing::ValuesIn(offsetOrSize32bitIllegalParam));

    INSTANTIATE_TEST_CASE_P(WithVariousSizeHeavy,
                            SizeArgumentOverFileSize,
                            ::testing::ValuesIn(param2));

    INSTANTIATE_TEST_CASE_P(WithVariousSizeHeavy,
                            SizeArgumentFileSize,
                            ::testing::ValuesIn(param2));

    INSTANTIATE_TEST_CASE_P(WithVariousSize,
                            SizeArgumentOffsetFileSize,
                            ::testing::ValuesIn(param1));



    typedef nnt::fs::api::CleanupFileSystemTestFixture SizeArgumentTransferSize;

#if 0 // SIGLO-30380

    //! @brief 0 ～ FileSizeMax のサイズでファイルの書き込み・読み込みが出来ること
    TEST_F(SizeArgumentTransferSize, DISABLED_WriteFile_ReadFile_Heavy) // 非常に重いので基本無効
    {
        String fileName = GetTestRootPath().append("/test.file");

        const int FileSizeMax = 256 * 1024;
        const int Step = 1;

        NNT_ASSERT_RESULT_SUCCESS(GetFs().CreateFile(fileName.c_str(), 0));

        for( auto read : ReadFunctions )
        {
            if( IsReadOverload(read) && !GetFsAttribute()->isReadOverloadsSupported )
            {
                continue;
            }

            for(int fileSize = 0; fileSize <= FileSizeMax; fileSize += Step)
            {
                NN_LOG("size: %d\n", fileSize);

                std::unique_ptr<ITestFile> file;
                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
                NNT_EXPECT_RESULT_SUCCESS(file->SetSize(fileSize));

                auto writeBuffer(nnt::fs::util::AllocateBuffer(fileSize));
                FillBufferWithRandomValue(writeBuffer.get(), fileSize);
                NNT_EXPECT_RESULT_SUCCESS(file->Write(0, writeBuffer.get(), fileSize, WriteOption::MakeValue(WriteOptionFlag_Flush)));
                file.reset(nullptr);

                NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), static_cast<OpenMode>(OpenMode_Read)));
                auto readBuffer(nnt::fs::util::AllocateBuffer(fileSize));
                size_t readSize = 0;

                NNT_EXPECT_RESULT_SUCCESS(read(&readSize, file.get(), 0, readBuffer.get(), fileSize, fileSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(writeBuffer.get(), readBuffer.get(), fileSize);

                file.reset(nullptr);
            }
        }

        NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
    }
#endif

    //! @brief 様々なパターンのオフセットとサイズで読み込む
    TEST_P(SizeArgumentVariousAccess, ReadFileWithPattern)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        // 1 分以内にテストが終わるような大きさのパターンを用意
        // パターンはオフセット、サイズの領域を 4 等分したものと、それを + 1 したもの
        static const int OffsetPatternLength = 8;
        NN_STATIC_ASSERT(OffsetPatternLength % 2 == 0);
        int64_t offsetPatterns[OffsetPatternLength];
        for( int i = 0; i < OffsetPatternLength - 1; i += 2 )
        {
            offsetPatterns[i] = GetFileSize() / OffsetPatternLength * i;
            offsetPatterns[i + 1] = offsetPatterns[i] + 1;
        }

        const size_t bufferSize = static_cast<size_t>(std::min<int64_t>(2 * 1024 * 1024, GetFileSize()));
        auto buffer = nnt::fs::util::AllocateBuffer(bufferSize);

        static const int SizePatternLength = 8;
        NN_STATIC_ASSERT(SizePatternLength % 2 == 0);
        size_t sizePatterns[SizePatternLength];
        for( int i = 0; i < SizePatternLength - 1; i += 2 )
        {
            sizePatterns[i] = bufferSize / SizePatternLength * i;
            sizePatterns[i + 1] = sizePatterns[i] + 1;
        }

        for( auto size : sizePatterns )
        {
            for( auto offset : offsetPatterns )
            {
                const size_t accessSize = static_cast<size_t>(std::min(static_cast<int64_t>(size), GetFileSize() - offset));
                size_t readSize = 0;
                NNT_EXPECT_RESULT_SUCCESS(read(&readSize, GetFile(), offset, buffer.get(), accessSize, AccessRange::InRange, accessSize));
                VerifyFilledData(buffer.get(), readSize, offset);
            }
        }
    }

    //! @brief ランダムなオフセットとサイズで読み込む
    TEST_P(SizeArgumentVariousAccess, ReadFileRandom)
    {
        auto read = GetParam();
        NNT_FS_UTIL_SKIP_TEST_UNLESS(!IsReadOverload(read) || GetFsAttribute()->isReadOverloadsSupported);

        const int64_t sizeMax = std::min<int64_t>(2 * 1024 * 1024, GetFileSize());
        auto buffer = nnt::fs::util::AllocateBuffer(static_cast<size_t>(sizeMax));

        std::mt19937 mt(nnt::fs::util::GetRandomSeed());
        static const int LoopCount = 500;
        for( int i = 0; i < LoopCount; ++i )
        {
            const size_t size = static_cast<size_t>(std::uniform_int_distribution<int64_t>(1, sizeMax)(mt));
            const int64_t offset = std::uniform_int_distribution<int64_t>(0, sizeMax - static_cast<int64_t>(size))(mt);
            NNT_FS_SCOPED_TRACE("offset: %llx size: %llx\n", offset, static_cast<uint64_t>(size));
            size_t readSize = 0;
            NNT_EXPECT_RESULT_SUCCESS(read(&readSize, GetFile(), offset, buffer.get(), size, AccessRange::InRange, size));
            VerifyFilledData(buffer.get(), size, offset);
        }
    }

    INSTANTIATE_TEST_CASE_P(WithReadOverloads,
        SizeArgumentVariousAccess,
        ::testing::ValuesIn(ReadFunctions));

    //!< @brief OperateRange の QueryRange で不正なサイズを渡すと失敗することを確認する
    TEST_F(SizeArgumentFile, InvalidSize_OperateRange_QueryRange)
    {
        // ファイルオープン
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(
            GetFs().OpenFile(
                &file,
                GetFileName().c_str(),
                static_cast<nn::fs::OpenMode>(
                    nn::fs::OpenMode::OpenMode_Read |
                    nn::fs::OpenMode::OpenMode_Write |
                    nn::fs::OpenMode::OpenMode_AllowAppend)
            )
        );

        nn::fs::QueryRangeInfo info;

        if( GetFsAttribute()->isSupportedQueryRange )
        {
            NNT_ASSERT_RESULT_FAILURE(nn::fs::ResultInvalidSize,
                file->OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, FileSize, nullptr, 0));
        }
        else
        {
            auto result = file->OperateRange(&info, 0, nn::fs::OperationId::QueryRange, 0, FileSize, nullptr, 0);
            if( !nn::fs::ResultUnsupportedOperation::Includes(result)
                && !nn::fs::ResultInvalidSize::Includes(result) )
            {
                NN_LOG("Unexpected result: 0x%08X\n", result.GetInnerValueForDebug());
                FAIL();
            }
        }
    }
}}}
