﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <numeric>
#include <nnt/fsApi/testFs_Api.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_BitUtil.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

/**
* @file
* @brief    ファイルシステム関数、並列同時実行観点を定義します。
*/
namespace {

    // テストファイル定義
    const char TestFileExt[]                = ".file";
    const char TestEntryFileFormat[]        = "f%02d.file";

    // 実行スレッド数定義
    const int ThreadCount                   = 2;
    const int ThreadCount2                  = 8;

    // スレッドスタックサイズ定義
    const size_t StackSize                  = 64 * 1024;

    // テスト対象ファイル数／ディレクトリ数定義
    const int TestFileCount                 = 1;
    const int TestFilesCount                = ThreadCount;
    const int TestDirectoriesCount          = ThreadCount;

    // バッファサイズ定義
    const int BufferSize                    = 1024;
    // エントリサイズ定義
    const int EntrySize                     = 2;
    // エントリ数テストエントリファイル数定義
    const int TestEntryCount                = 100;

    // テストを繰り返す回数定義
    const int TestMaxLoopCount              = 1000;
    const int TestOpenMaxLoopCount          = 10000;

    // ランダムなファイルサイズの最大値定義
    const int TestRandomMaxFileSize         = 1024;

    //! parentDirectoryNameLength は スラッシュも含みます
    int64_t GetDirectoryNameMaxLength(
            const nnt::fs::api::FileSystemAttribute* attribute,
            size_t parentDirectoryNameLength
        ) NN_NOEXCEPT
    {
        if( attribute->directoryPathLengthMax <= static_cast<int>(parentDirectoryNameLength) )
        {
            return 0;
        }
        return static_cast<int64_t>(
            std::min(
                attribute->directoryNameLengthMax,
                attribute->directoryPathLengthMax - static_cast<int>(parentDirectoryNameLength)
            )
        );
    }

    //! parentDirectoryNameLength は スラッシュも含みます
    int64_t GetFileNameMaxLength(
            const nnt::fs::api::FileSystemAttribute* attribute,
            size_t parentDirectoryNameLength
        ) NN_NOEXCEPT
    {
        if( attribute->filePathLengthMax <= static_cast<int>(parentDirectoryNameLength) )
        {
            return 0;
        }
        return static_cast<int64_t>(
            std::min(
                attribute->fileNameLengthMax,
                attribute->filePathLengthMax - static_cast<int>(parentDirectoryNameLength)
            )
        );
    }

    typedef nn::Result(*ReadFunction)(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size);

    nn::Result ReadFunction0(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction1(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        NN_RESULT_DO(pTestFile->Read(offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction2(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size, nn::fs::ReadOption()));
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadFunction3(nnt::fs::api::ITestFile* pTestFile, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
    {
        size_t sizeRead;
        NN_RESULT_DO(pTestFile->Read(&sizeRead, offset, buffer, size));
        NN_RESULT_SUCCESS;
    }

    const ReadFunction ReadFunctions[] =
    {
        ReadFunction0,
        ReadFunction1,
        ReadFunction2,
        ReadFunction3
    };
}

namespace nnt { namespace fs { namespace api {
    void LoadRaceConditionTests() NN_NOEXCEPT
    {
        return;
    }

    // パラメータ用構造体
    struct ParamsStructure
    {
        ITestFileSystem*    fileSystem;
        ITestFile*          file;
        ITestDirectory*     dir;
        String*             path;
        int                 idx;
        int64_t             fileSize;
        char*               buffer;
        int                 bufferSize;
        DirectoryEntry*     entryBuffer;
        int                 entrySize;
        String*             randomString;
    };

    /**
     * @brief 内部使用のテストファイル生成する関数です
     *   @param[in]  fileSystem             ファイルシステム
     *   @param[in]  path                   パス + ファイル名
     *   @param[in]  pStringBuffer          書き込むバッファポインタ
     */
    void CreateTestFileInternal(ITestFileSystem* fileSystem, const String* path, const String* pStringBuffer) NN_NOEXCEPT
    {
        std::unique_ptr<ITestFile> file;

        NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateFile(path->c_str(), 0));
        if  (pStringBuffer->length() > 0)
        {
            NNT_ASSERT_RESULT_SUCCESS(fileSystem->OpenFile(&file, path->c_str(), static_cast<OpenMode>(OpenMode_Write | OpenMode_AllowAppend)));
            NNT_EXPECT_RESULT_SUCCESS(file->Write(0, pStringBuffer->c_str(), pStringBuffer->length(), WriteOption()));
            NNT_EXPECT_RESULT_SUCCESS(file->Flush());
            file.reset(nullptr);
        }
    }

    /**
     * @brief 内部使用のテストディレクトリ生成する関数です
     *   @param[in]  fileSystem             ファイルシステム
     *   @param[in]  path                   パス(ルートパス/作成するディレクトリ名)
     *   @param[in]  fileCount              指定パス内に作成するファイル数
     */
    void CreateTestDirectoryInternal(ITestFileSystem* fileSystem, const String* path, const int32_t fileCount) NN_NOEXCEPT
    {
        auto fileName = AllocateBuffer(NameMaxLength);
        String entryFile;

        NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateDirectory(path->c_str()));

        for (int32_t i = 0; i < fileCount; i++)
        {
            nn::util::SNPrintf(fileName.get(), NameMaxLength, TestEntryFileFormat, i);
            entryFile = *path;
            entryFile.append("/").append(fileName.get());
            NNT_FS_SCOPED_TRACE_SAFE("loop count: %d, file path: %s\n", i, entryFile.c_str());
            NNT_ASSERT_RESULT_SUCCESS(fileSystem->CreateFile(entryFile.c_str(), 0));
        }
    }

    /**
     * @brief テストスレッドを待機する関数です
     *   @param[in]  pThreads               スレッドオブジェクト構造体配列ポインタ
     *   @param[in]  threadCount            待機するスレッド数
     */
    void TestThreadWait(const nn::os::ThreadType *pThreads, const int32_t threadCount) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pThreads);

        for (int i = 0; i < threadCount; i++)
        {
            nn::os::WaitThread(const_cast<nn::os::ThreadType*>(&pThreads[i]));
            nn::os::DestroyThread(const_cast<nn::os::ThreadType*>(&pThreads[i]));
        }
    }

    /**
     * @brief   テストファイル作成、オープンまでを事前準備するテストフィクスチャです。
     */
    class RaceConditionTestFilePreparationFixture : public CleanupFileSystemTestFixture
    {
    public:

        /**
         * @brief コンストラクタ
         *  テストファイル作成時のテストファイル数、ファイルサイズを設定します
         */
        RaceConditionTestFilePreparationFixture(const int32_t testFileCount, const int64_t fileSize) NN_NOEXCEPT
        {
            this->fileSize  = fileSize;
            this->fileCount = testFileCount;
        }

        /**
         * @brief テストファイルをオープンする関数です。
         *   @param[in]  mode               ファイルオープンモード
         */
        void OpenTestFile(const int32_t mode) NN_NOEXCEPT
        {
            for (int i = 0; i < this->fileCount; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE(
                    "loop count: %d, file path: %s\n", i, this->fileNameArray[i].c_str()
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenFile(
                        &this->fileArray[i],
                        this->fileNameArray[i].c_str(),
                        static_cast<OpenMode>(mode)
                    )
                );
            }
        }

    protected:
        std::unique_ptr<ITestFile> fileArray[TestFilesCount];
        String fileNameArray[TestFilesCount];
        String randomString;
        int64_t fileSize;
        int32_t fileCount;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();

            this->randomString = String(GenerateRandomString(this->fileSize));

            for (int i = 0; i < this->fileCount; i++)
            {
                do
                {
                    nnt::fs::util::String rootPath = GetTestRootPath().append("/");
                    auto maxFileNameLength
                        = GetFileNameMaxLength(GetFsAttribute(), rootPath.length());
                    auto prefixLength = static_cast<int64_t>(strlen(TestFileExt));
                    this->fileNameArray[i]
                        = rootPath.append(
                            GenerateRandomLengthEntryName(
                                std::min(maxFileNameLength - prefixLength, static_cast<int64_t>(128))
                            )
                        ).append(TestFileExt);
                } while (IsEntryExist(&GetFs(), this->fileNameArray[i].c_str(), DirectoryEntryType_File) == true);

                NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, this->fileNameArray[i].c_str());
                CreateTestFileInternal(&GetFs(), &this->fileNameArray[i], &this->randomString);
            }
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < this->fileCount; i++)
            {
                this->fileArray[i].reset(nullptr);
                NNT_FS_SCOPED_TRACE("loop count: %d, file path: %s\n", i, this->fileNameArray[i].c_str());
                NNT_ASSERT_RESULT_SUCCESS(GetFs().DeleteFile(this->fileNameArray[i].c_str()));
            }
            CleanupFileSystemTestFixture::TearDown();
        }

    };

    /**
     * @brief   テストディレクトリ作成、オープンまでを事前準備するテストフィクスチャです。
     */
    class RaceConditionTestDirPreparationFixture : public CleanupFileSystemTestFixture
    {
    public:

        /**
         * @brief コンストラクタ
         *  テストディレクトリを作成するディレクトリ数を設定します
         */
        NN_IMPLICIT RaceConditionTestDirPreparationFixture(const int32_t testDirCount) NN_NOEXCEPT    // NN_IMPLICIT は、checkCPPでの単一引数コンストラクタのエラー回避です
        {
            this->dirCount  = testDirCount;
        }

        /**
         * @brief テストディレクトリをオープンする関数です。
         *   @param[in]  mode               ディレクトリオープンモード
         */
        void OpenTestDirectory(const int32_t mode) NN_NOEXCEPT
        {
            for (int i = 0; i < this->dirCount; i++)
            {
                NNT_FS_SCOPED_TRACE_SAFE(
                    "loop count: %d, directory path: %s\n", i, this->dirName.c_str()
                );
                NNT_ASSERT_RESULT_SUCCESS(
                    GetFs().OpenDirectory(
                        &this->dirArray[i],
                        this->dirName.c_str(),
                        static_cast<OpenDirectoryMode>(mode)
                    )
                );
            }
        }

    protected:
        std::unique_ptr<ITestDirectory>dirArray[TestDirectoriesCount];
        String dirName;
        int32_t dirCount;

        virtual void SetUp() NN_NOEXCEPT NN_OVERRIDE
        {
            CleanupFileSystemTestFixture::SetUp();
            nnt::fs::util::String rootPath = GetTestRootPath().append("/");
            auto maxDirectoryNameLength
                = GetDirectoryNameMaxLength(GetFsAttribute(), rootPath.length());
            this->dirName
                = rootPath.append(
                    GenerateRandomLengthEntryName(
                        std::min(maxDirectoryNameLength, static_cast<int64_t>(128))
                    )
                );
            CreateTestDirectoryInternal(&GetFs(), &this->dirName, TestEntryCount);
        }
        virtual void TearDown() NN_NOEXCEPT NN_OVERRIDE
        {
            for (int i = 0; i < this->dirCount; i++)
            {
                this->dirArray[i].reset(nullptr);
            }
            NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectoryRecursively(this->dirName.c_str()));

            CleanupFileSystemTestFixture::TearDown();
        }
    };

    /**
     * @brief   RaceConditionOpenFile で利用するテストフィクスチャです。
     */
    class RaceConditionOpenFile : public RaceConditionTestFilePreparationFixture
    {
    public:

        RaceConditionOpenFile() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(TestFilesCount, 0)
        {
        }

        /**
         * @brief ファイルオープンを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void OpenFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            for (int i = 0; i < TestOpenMaxLoopCount; i++)
            {
                std::unique_ptr<ITestFile> file;
                NNT_ASSERT_RESULT_SUCCESS(params->fileSystem->OpenFile(&file, params->path->c_str(), OpenMode_Read));
            }
        }
    };

    /**
     * @brief   RaceConditionReadFile で利用するテストフィクスチャです。
     */
    class RaceConditionReadFile : public RaceConditionTestFilePreparationFixture
    {
    public:
        char readBuffer[ThreadCount][BufferSize];

        RaceConditionReadFile() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(TestFileCount, BufferSize * ThreadCount)
        {
        }

        /**
         * @brief ファイル読み込みを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void ReadFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            const int64_t offset = params->idx * params->bufferSize;

            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                auto read = ReadFunctions[i % NN_ARRAY_SIZE(ReadFunctions)];

                NNT_EXPECT_RESULT_SUCCESS(read(params->file, offset, params->buffer, params->bufferSize));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(params->randomString->c_str() + offset, params->buffer, params->bufferSize);
                std::memset(params->buffer, i, params->bufferSize);
            }
        }
    };

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
    /**
     * @brief   RaceConditionReadFile2 で利用するテストフィクスチャです。
     */
    class RaceConditionReadFile2 : public RaceConditionTestFilePreparationFixture
    {
    public:
        struct ThreadInfo
        {
            ITestFile* pFile;
            size_t bufferSize;
            int64_t offset;
            int64_t readSize;
        };

    public:
        RaceConditionReadFile2() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(0, 0)
        {
        }

        // バッファの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        static void FillBuffer(char* buffer, size_t size, int64_t offset) NN_NOEXCEPT
        {
            std::iota(reinterpret_cast<uint8_t*>(buffer), reinterpret_cast<uint8_t*>(buffer + size), static_cast<uint8_t>(offset));
        }

        // バッファが先頭から dataOffset + 0x00, 0x01, 0x02, ... で埋められていることを確認する
        static void VerifyFilledData(const char* buffer, size_t size, int64_t dataOffset, char* workBuffer, size_t workBufferSize) NN_NOEXCEPT
        {
            // workBufferSize が 0x100 の倍数ならループごとに FillBuffer() する必要がない
            // workBufferSize >= size の場合はループしないので 0x100 の倍数でなくて良い
            ASSERT_TRUE( workBufferSize >= size || workBufferSize % 0x100 == 0);
            ASSERT_NE(nullptr, workBuffer);

            const size_t bufferSize = std::min(size, workBufferSize);
            FillBuffer(workBuffer, bufferSize, dataOffset);

            for( size_t compareOffset = 0; compareOffset < size; compareOffset += bufferSize )
            {
                const size_t compareSize = std::min(bufferSize, static_cast<size_t>(size - compareOffset));
                NNT_FS_UTIL_EXPECT_MEMCMPEQ(workBuffer, buffer + compareOffset, compareSize);
            }
        }

        // ファイルの内容を先頭から 0x00, 0x01, 0x02, ... で埋める
        void FillFile(const char* filePath, int64_t testFileSize) NN_NOEXCEPT
        {
            const size_t BufferSizeMax = 1 * 1024 * 1024;

            std::unique_ptr<ITestFile> pFile;
            NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, filePath, nn::fs::OpenMode_Write));

            const size_t bufferSize = std::min(BufferSizeMax, static_cast<size_t>(testFileSize));
            auto writeBuffer = nnt::fs::util::AllocateBuffer(bufferSize);
            ASSERT_NE(nullptr, writeBuffer);
            FillBuffer(writeBuffer.get(), bufferSize, 0);

            for( int64_t offset = 0; offset < testFileSize; offset += bufferSize )
            {
                const size_t writeSize
                    = std::min(bufferSize, static_cast<size_t>(testFileSize - offset));
                NNT_ASSERT_RESULT_SUCCESS(pFile->Write(
                    offset,
                    writeBuffer.get(),
                    writeSize,
                    nn::fs::WriteOption::MakeValue(0)));
            }
            NNT_ASSERT_RESULT_SUCCESS(pFile->Flush());
        }

        // ファイル読み込みを行うスレッド関数
        static void ReadFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            ASSERT_NE(nullptr, arg);

            const size_t BufferSizeMax = 1 * 1024 * 1024;

            ThreadInfo* pInfo = static_cast<ThreadInfo*>(arg);
            auto buffer = nnt::fs::util::AllocateBuffer(pInfo->bufferSize);
            ASSERT_NE(nullptr, buffer);

            const auto workBufferSize = std::min(pInfo->bufferSize, BufferSizeMax);
            auto workBuffer = nnt::fs::util::AllocateBuffer(workBufferSize);
            ASSERT_NE(nullptr, workBuffer);

            int index = 0;
            for( int64_t offset = pInfo->offset;
                offset < pInfo->offset + pInfo->readSize; offset += pInfo->bufferSize )
            {
                auto read = ReadFunctions[index % NN_ARRAY_SIZE(ReadFunctions)];
                ++index;

                const size_t readSize = std::min(
                    pInfo->bufferSize, static_cast<size_t>(pInfo->offset + pInfo->readSize - offset));
                NNT_EXPECT_RESULT_SUCCESS(read(
                    pInfo->pFile,
                    offset,
                    buffer.get(),
                    readSize));

                // 読み込み内容の確認
                VerifyFilledData(buffer.get(), readSize, offset, workBuffer.get(), workBufferSize);
            }
        }
    };
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

    /**
     * @brief   RaceConditionReadDirectory で利用するテストフィクスチャです。
     */
    class RaceConditionReadDirectory : public RaceConditionTestDirPreparationFixture
    {
    public:
        DirectoryEntry entryReadBuffer[ThreadCount][EntrySize];

        RaceConditionReadDirectory() NN_NOEXCEPT
            : RaceConditionTestDirPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief ディレクトリ読み込みを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void ReadDirectoryThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);

            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                int64_t resultEntry = 0;
                // 同一ディレクトリハンドルの場合、Readするとスレッド間でそれぞれFindNextされエントリの終端が予測不可となるため、実行のみの確認です
                NNT_EXPECT_RESULT_SUCCESS(params->dir->Read(&resultEntry, params->entryBuffer, params->entrySize));
            }
        }
    };

    /**
     * @brief   RaceConditionGetDirectoryEntryCount で利用するテストフィクスチャです。
     */
    class RaceConditionGetDirectoryEntryCount : public RaceConditionTestDirPreparationFixture
    {
    public:

        RaceConditionGetDirectoryEntryCount() NN_NOEXCEPT
            : RaceConditionTestDirPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief エントリ数取得を同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void GetEntryCountThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);

            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                int64_t resultEntry = 0;
                NNT_EXPECT_RESULT_SUCCESS(params->dir->GetEntryCount(&resultEntry));
                EXPECT_EQ(TestEntryCount, resultEntry);
            }
        }
    };

    /**
     * @brief   RaceConditionOpenDirectory で利用するテストフィクスチャです。
     */
    class RaceConditionOpenDirectory : public RaceConditionTestDirPreparationFixture
    {
    public:

        RaceConditionOpenDirectory() NN_NOEXCEPT
            : RaceConditionTestDirPreparationFixture(TestDirectoriesCount)
        {
        }

        /**
         * @brief オープンディレクトリを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void OpenDirectoryThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);

            for (int i = 0; i < TestOpenMaxLoopCount; i++)
            {
                std::unique_ptr<ITestDirectory> dir;
                NNT_ASSERT_RESULT_SUCCESS(params->fileSystem->OpenDirectory(&dir, params->path->c_str(), OpenDirectoryMode_File));
            }
        }
    };

    /**
     * @brief   RaceConditionWriteFile(排他対象) で利用するテストフィクスチャです。
     */
    class RaceConditionWriteFile : public RaceConditionTestFilePreparationFixture
    {
    public:

        RaceConditionWriteFile() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(TestFilesCount, 0)
        {
        }

        /**
         * @brief ファイル書き込みを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void WriteFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            String randomString;
            int64_t size = BufferSize;
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            randomString = GenerateRandomString(size);

            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(params->file->Write(0, randomString.c_str(), randomString.length(), WriteOption()));
            }

            // FSA の上に構築する fs 向けの flush
            NNT_EXPECT_RESULT_SUCCESS(params->file->Flush());
        }
    };

    /**
     * @brief   RaceConditionFlushFile(排他対象) で利用するテストフィクスチャです。
     */
    class RaceConditionFlushFile : public RaceConditionTestFilePreparationFixture
    {
    public:

        RaceConditionFlushFile() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(TestFilesCount, 0)
        {
        }

        /**
         * @brief キャッシュフラッシュを同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void FlushFileThreadFunction(void* arg) NN_NOEXCEPT
        {
            String randomString;
            int64_t size = BufferSize * ThreadCount;
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            randomString = String(GenerateRandomString(size).c_str());

            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(params->file->Write(0, randomString.c_str(), randomString.length(), WriteOption()));
                NNT_EXPECT_RESULT_SUCCESS(params->file->Flush());
            }
        }
    };

    /**
     * @brief   RaceConditionSetGetFileSize(SetFileSizeのみ排他対象) で利用するテストフィクスチャです。
     */
    class RaceConditionSetGetFileSize : public RaceConditionTestFilePreparationFixture
    {
    public:

        /**
         * @brief コンストラクタ
         */
        RaceConditionSetGetFileSize() NN_NOEXCEPT
            : RaceConditionTestFilePreparationFixture(TestFilesCount, GetRandomFileSize(TestRandomMaxFileSize))
        {
        }

        /**
         * @brief ファイルサイズ設定を同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void SetFileSizeThreadFunction(void* arg) NN_NOEXCEPT
        {
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(params->file->SetSize(params->fileSize));
            }
        }

        /**
         * @brief ファイルサイズ取得を同時実行するスレッド関数です
         *   @param[in]  arg                    ParamsStructure構造体ポインタ
         */
        static void GetFileSizeThreadFunction(void* arg) NN_NOEXCEPT
        {
            int64_t size;
            ParamsStructure* params = static_cast<ParamsStructure*>(arg);
            for (int i = 0; i < TestMaxLoopCount; i++)
            {
                NNT_EXPECT_RESULT_SUCCESS(params->file->GetSize(&size));
                EXPECT_EQ(params->fileSize, size);
            }
        }

        /**
         * @brief 指定のサイズ範囲でランダムなファイルサイズを返す関数です
         */
        int64_t GetRandomFileSize(const int maxFileSize) NN_NOEXCEPT
        {
            static std::mt19937 mt(nnt::fs::util::GetRandomSeed());
            return std::uniform_int_distribution<>(1, maxFileSize)(mt);
        }
    };

    //!< @brief １ファイルをマルチスレッドで繰り返し同一ファイルのオープンが成功すること
    TEST_F(RaceConditionOpenFile, OpenFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].path          = &this->fileNameArray[i];
            params[i].fileSystem    = &GetFs();

            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], OpenFileThreadFunction, &params[i], stack[i], StackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief １ファイルを開き、マルチスレッドで繰り返し同一ファイルの読み込みが成功すること
    TEST_F(RaceConditionReadFile, ReadFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].file          = this->fileArray[0].get();
            params[i].idx           = i;
            params[i].buffer        = readBuffer[i];
            params[i].bufferSize    = BufferSize;
            params[i].randomString  = &this->randomString;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], ReadFileThreadFunction, &params[i], stack[i], StackSize, nn::os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

#if !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)
    // 複数スレッドでファイルの同じ位置を読み込む
    TEST_F(RaceConditionReadFile2, ReadFile)
    {
        static const int64_t FileSizeMax = 256 * 1024 * 1024;

        char path[128];
        nn::util::SNPrintf(path, sizeof(path), "%s/file.dat", GetTestRootPath().c_str());

        // ファイルを作成する
        {
            int64_t testFileSize = std::min(FileSizeMax, GetFsAttribute()->fileSizeMax);

            {
                auto result = GetFs().DeleteFile(path);
                if( !nn::fs::ResultPathNotFound::Includes(result) )
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }
            }

            {
                int64_t freeSpaceSize = 0;
                auto result = GetFs().GetFreeSpaceSize(&freeSpaceSize, GetTestRootPath().c_str());
                if( result.IsSuccess() )
                {
                    if( freeSpaceSize < testFileSize )
                    {
                        testFileSize = freeSpaceSize;
                    }
                }
                else if( !nn::fs::ResultNotImplemented::Includes(result) )
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                }
            }

            for ( ; ; )
            {
                auto result = GetFs().CreateFile(path, testFileSize);
                if (!nn::fs::ResultUsableSpaceNotEnough::Includes(result))
                {
                    NNT_ASSERT_RESULT_SUCCESS(result);
                    break;
                }
                testFileSize -= 16 * 1024;
                ASSERT_LT(0, testFileSize);
            }

            FillFile(path, testFileSize);
        }
        NN_UTIL_SCOPE_EXIT
        {
            GetFs().DeleteFile(path);
        };

        std::unique_ptr<ITestFile> pFile;
        NNT_ASSERT_RESULT_SUCCESS(GetFs().OpenFile(&pFile, path, nn::fs::OpenMode_Read));

        static const auto AllocatedStackBufferSize = StackSize + nn::os::ThreadStackAlignment;

        auto pThreadStackBuffer = nnt::fs::util::AllocateBuffer(ThreadCount2 * AllocatedStackBufferSize);

        {
            int64_t testFileSize = 0;
            NNT_ASSERT_RESULT_SUCCESS(pFile->GetSize(&testFileSize));

            ThreadInfo infoArray[ThreadCount2];
            nn::os::ThreadType threadArray[ThreadCount2];
            for( auto i = 0; i < ThreadCount2; ++i )
            {
                infoArray[i].pFile = pFile.get();
                switch( i )
                {
                case 0:
                case 1:
                    // ファイル全体を一度に読み込み
                    infoArray[i].bufferSize = static_cast<size_t>(testFileSize);
                    infoArray[i].offset = 0;
                    infoArray[i].readSize = testFileSize;
                    break;
                case 2:
                case 3:
                    // ファイル全体を分割して読み込み
                    infoArray[i].bufferSize
                        = static_cast<size_t>(std::min<int64_t>(256 * 1024, testFileSize));
                    infoArray[i].offset = 0;
                    infoArray[i].readSize = testFileSize;
                    break;
                default:
                    // ファイルの一部を分割して読み込み
                    infoArray[i].bufferSize
                        = static_cast<size_t>(std::min<int64_t>(256 * 1024, testFileSize));
                    infoArray[i].offset = std::min<int64_t>(i * 13, testFileSize);
                    infoArray[i].readSize = std::min(
                        static_cast<int64_t>(infoArray[i].bufferSize * i),
                        static_cast<int64_t>(testFileSize - infoArray[i].offset));
                }
                NNT_ASSERT_RESULT_SUCCESS(nn::os::CreateThread(
                    &threadArray[i],
                    ReadFileThreadFunction,
                    &infoArray[i],
                    reinterpret_cast<char*>(nn::util::align_up(
                        reinterpret_cast<uintptr_t>(pThreadStackBuffer.get() + AllocatedStackBufferSize * i),
                        nn::os::ThreadStackAlignment)),
                    StackSize,
                    nn::os::DefaultThreadPriority));
            }

            for( auto& thread : threadArray )
            {
                nn::os::StartThread(&thread);
            }

            for( auto& thread : threadArray )
            {
                nn::os::WaitThread(&thread);
                nn::os::DestroyThread(&thread);
            }
        }
    }
#endif // !defined(NN_BUILD_CONFIG_ADDRESS_32) || !defined(NN_BUILD_CONFIG_OS_WIN)

    //!< @brief 同一ディレクトリに対して複数のハンドルで開き、マルチスレッドで繰り返しエントリ読み込みが成功すること
    TEST_F(RaceConditionReadDirectory, ReadDirectory)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestDirectory(OpenDirectoryMode_File);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].dir           = this->dirArray[i].get();
            params[i].idx           = i;
            params[i].entryBuffer   = entryReadBuffer[i];
            params[i].entrySize     = EntrySize;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].dir);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], ReadDirectoryThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief 同一ディレクトリに対して複数のハンドルで開き、マルチスレッドで繰り返しエントリ数の取得が成功すること
    TEST_F(RaceConditionGetDirectoryEntryCount, GetDirectoryEntryCount)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestDirectory(OpenDirectoryMode_File);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].dir           = this->dirArray[i].get();
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].dir);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], GetEntryCountThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief 同一ディレクトリに対して、マルチスレッドで繰り返し複数のハンドルでディレクトリのオープンが成功すること
    TEST_F(RaceConditionOpenDirectory, OpenDirectory)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].path          = &this->dirName;
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, directory path: %s\n", i, params[i].path);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], OpenDirectoryThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief ２つのファイルをマルチスレッドで繰り返しそれぞれのファイル書き込みが成功すること
    TEST_F(RaceConditionWriteFile, WriteFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].file          = this->fileArray[i].get();
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], WriteFileThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief ２つのファイルをマルチスレッドで繰り返しそれぞれのファイル書き込みに対してキャッシュフラッシュが成功すること
    TEST_F(RaceConditionFlushFile, FlushFile)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].file          = this->fileArray[i].get();
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], FlushFileThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief ２つのファイルをマルチスレッドで繰り返しそれぞれのファイルサイズ設定が成功すること
    TEST_F(RaceConditionSetGetFileSize, SetFileSize)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].file          = this->fileArray[i].get();
            params[i].fileSize      = GetRandomFileSize(TestRandomMaxFileSize);
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], SetFileSizeThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }

    //!< @brief １つのファイルを開き、マルチスレッドで繰り返し同一ファイルのサイズ取得が成功すること
    TEST_F(RaceConditionSetGetFileSize, GetFileSize)
    {
        static NN_ALIGNAS(4096) uint8_t stack[ThreadCount][StackSize];
        nn::os::ThreadType threads[ThreadCount];
        ParamsStructure params[ThreadCount];

        OpenTestFile(OpenMode_Read | OpenMode_Write | OpenMode_AllowAppend);

        // ２つのスレッドにひき渡す各パラメータ設定、スレッドを生成／起動します。
        for (int i = 0; i < ThreadCount; i++)
        {
            params[i].fileSystem    = &GetFs();
            params[i].file          = this->fileArray[0].get();
            params[i].fileSize      = nn::util::align_up(this->fileSize, GetFsAttribute()->fileSizeAlignment); // ファイルサイズのアラインメントを考慮
            params[i].idx           = i;

            NNT_FS_SCOPED_TRACE("thread count: %d, file path: %s\n", i, params[i].file);
            NNT_EXPECT_RESULT_SUCCESS(nn::os::CreateThread(&threads[i], GetFileSizeThreadFunction, &params[i], stack[i], StackSize, os::DefaultThreadPriority));
            nn::os::StartThread(&threads[i]);
        }

        // ２つのスレッドが終了するまで待機します。
        TestThreadWait(threads, ThreadCount);
    }


}}}
