﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/time.h>
#include <nnt/fsApi/testFs_Api.h>
#include <nn/fs/fs_FileSystemForDebug.h>

using namespace nn::fs;
using namespace nn::fs::fsa;
using namespace nnt::fs::util;

namespace nnt {
namespace fs {
namespace api {
void LoadGetFileTimeStampTests() NN_NOEXCEPT
{
    return;
}

typedef CleanupFileSystemTestFixture GetFileTimeStampCaller;

namespace
{
const int FileSize = 32;

void PrintTimeStamp(const char* mark, const FileTimeStampRaw* pFileTimeStamp)
{
    nn::time::PosixTime posixTime;
    nn::time::CalendarTime calendar;

    posixTime.value = pFileTimeStamp->create;
    calendar = nn::time::ToCalendarTimeInUtc(posixTime);
    NN_LOG("%s:Create %04d/%2d/%2d %02d:%02d:%02d\n", mark, calendar.year, calendar.month, calendar.day, calendar.hour, calendar.minute, calendar.second);

    posixTime.value = pFileTimeStamp->modify;
    calendar = nn::time::ToCalendarTimeInUtc(posixTime);
    NN_LOG("%s:Modify %04d/%2d/%2d %02d:%02d:%02d\n", mark, calendar.year, calendar.month, calendar.day, calendar.hour, calendar.minute, calendar.second);

    posixTime.value = pFileTimeStamp->access;
    calendar = nn::time::ToCalendarTimeInUtc(posixTime);
    NN_LOG("%s:Access %04d/%2d/%2d %02d:%02d:%02d\n", mark, calendar.year, calendar.month, calendar.day, calendar.hour, calendar.minute, calendar.second);
}

bool CreateFileSpecificTime(FileTimeStampRaw* pOutFileTimeStamp, ITestFileSystem& fs, const char* pFileName, bool odd)
{
    int retry = 5;
    while (retry > 0)
    {
        fs.DeleteFile(pFileName);
        NNT_EXPECT_RESULT_SUCCESS(fs.CreateFile(pFileName, FileSize));
        // タイムスタンプ取得
        NNT_EXPECT_RESULT_SUCCESS(fs.GetFileTimeStampRaw(pOutFileTimeStamp, pFileName));
        if (((odd == true)  && (pOutFileTimeStamp->create & 0x1ull) != 0) ||
            ((odd == false) && (pOutFileTimeStamp->create & 0x1ull) == 0))
        {
            return true;
        }
        PrintTimeStamp("Retry", pOutFileTimeStamp);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        retry--;
    }

    fs.DeleteFile(pFileName);
    return false;
}

}

TEST_F(GetFileTimeStampCaller, GetFileTimeStamp_ConfirmFileTimeGap)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedGetFileTimeStamp);

    String fileName1 = GetTestRootPath().append("/testGetFileTimeStamp1.file");
    String fileName2 = GetTestRootPath().append("/testGetFileTimeStamp2.file");

    const int64_t IntervalSeconds = 5;

    // 成否問わず
    GetFs().DeleteFile(fileName1.c_str());
    GetFs().DeleteFile(fileName2.c_str());

    // ファイル 1 作成
    NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateFile(fileName1.c_str(), FileSize));
    // 5 秒待ち
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(IntervalSeconds));
    // ファイル 2 作成
    NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateFile(fileName2.c_str(), FileSize));

    // タイムスタンプ取得
    FileTimeStampRaw fileTimeStamp1;
    FileTimeStampRaw fileTimeStamp2;
    NNT_EXPECT_RESULT_SUCCESS(GetFs().GetFileTimeStampRaw(&fileTimeStamp1, fileName1.c_str()));
    NNT_EXPECT_RESULT_SUCCESS(GetFs().GetFileTimeStampRaw(&fileTimeStamp2, fileName2.c_str()));
    PrintTimeStamp("File1 Time", &fileTimeStamp1);
    PrintTimeStamp("File2 Time", &fileTimeStamp2);

    NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName1.c_str()));
    NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName2.c_str()));

    EXPECT_TRUE((fileTimeStamp2.create - fileTimeStamp1.create) >= IntervalSeconds - 1);
}

TEST_F(GetFileTimeStampCaller, ModifyTimeStamp)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedGetFileTimeStamp);

    String fileName = GetTestRootPath().append("/testGetFileTimeStamp.file");

    /* FAT仕様としてModifyTimeStampの分解能が2秒単位のため、IntervalSecondsは偶数秒である必要がある */
    const int64_t IntervalSeconds = 6;
    FileTimeStampRaw fileTimeStamp1;

    NNT_FS_UTIL_SKIP_TEST_UNLESS(CreateFileSpecificTime(&fileTimeStamp1, GetFs(), fileName.c_str(), false));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(IntervalSeconds));

    // Modify
    std::unique_ptr<ITestFile> file;
    NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Write));
    NNT_EXPECT_RESULT_SUCCESS(file->Write(0, "A", 1, WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(file->Flush());
    file.reset(nullptr);

    // タイムスタンプ取得
    FileTimeStampRaw fileTimeStamp2;
    NNT_EXPECT_RESULT_SUCCESS(GetFs().GetFileTimeStampRaw(&fileTimeStamp2, fileName.c_str()));
    PrintTimeStamp("Before File Time", &fileTimeStamp1);
    PrintTimeStamp("After File Time", &fileTimeStamp2);

    NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));

//    EXPECT_TRUE(fileTimeStamp1.create == fileTimeStamp1.modify);
    EXPECT_TRUE(fileTimeStamp1.create == fileTimeStamp2.create);
    EXPECT_TRUE((fileTimeStamp2.modify - fileTimeStamp1.modify) >= IntervalSeconds - 2);
}

TEST_F(GetFileTimeStampCaller, TimeStampEachSeconds)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedGetFileTimeStamp);

    String fileName = GetTestRootPath().append("/testGetFileTimeStamp.file");

    const int64_t IntervalSeconds = 1;
    FileTimeStampRaw fileTimeStamp1;

    NNT_FS_UTIL_SKIP_TEST_UNLESS(CreateFileSpecificTime(&fileTimeStamp1, GetFs(), fileName.c_str(), false));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(IntervalSeconds));

    // Modify
    std::unique_ptr<ITestFile> file;
    NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Write));
    NNT_EXPECT_RESULT_SUCCESS(file->Write(0, "A", 1, WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(file->Flush());
    file.reset(nullptr);

    // タイムスタンプ取得
    FileTimeStampRaw fileTimeStamp2;
    NNT_EXPECT_RESULT_SUCCESS(GetFs().GetFileTimeStampRaw(&fileTimeStamp2, fileName.c_str()));
    PrintTimeStamp("Before File Time", &fileTimeStamp1);
    PrintTimeStamp("After File Time", &fileTimeStamp2);

    NNT_FS_UTIL_SKIP_TEST_UNLESS(CreateFileSpecificTime(&fileTimeStamp1, GetFs(), fileName.c_str(), true));

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(IntervalSeconds));

    // Modify
    NNT_EXPECT_RESULT_SUCCESS(GetFs().OpenFile(&file, fileName.c_str(), OpenMode_Write));
    NNT_EXPECT_RESULT_SUCCESS(file->Write(0, "A", 1, WriteOption()));
    NNT_EXPECT_RESULT_SUCCESS(file->Flush());
    file.reset(nullptr);

    // タイムスタンプ取得
    NNT_EXPECT_RESULT_SUCCESS(GetFs().GetFileTimeStampRaw(&fileTimeStamp2, fileName.c_str()));
    PrintTimeStamp("Before File Time", &fileTimeStamp1);
    PrintTimeStamp("After File Time", &fileTimeStamp2);

    NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteFile(fileName.c_str()));
}

TEST_F(GetFileTimeStampCaller, DirectoryPath)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedGetFileTimeStamp);

    String dirName = GetTestRootPath().append("/TestDirectory");
    GetFs().DeleteDirectory(dirName.c_str());
    NNT_EXPECT_RESULT_SUCCESS(GetFs().CreateDirectory(dirName.c_str()));

    FileTimeStampRaw fileTimeStamp;
    NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().GetFileTimeStampRaw(&fileTimeStamp, dirName.c_str()));

    NNT_EXPECT_RESULT_SUCCESS(GetFs().DeleteDirectory(dirName.c_str()));
}

TEST_F(GetFileTimeStampCaller, PathNotFound)
{
    NNT_FS_UTIL_SKIP_TEST_UNLESS(GetFsAttribute()->isSupportedGetFileTimeStamp);

    String fileName = GetTestRootPath().append("/PathNotFound");
    GetFs().DeleteFile(fileName.c_str());

    FileTimeStampRaw fileTimeStamp;
    NNT_EXPECT_RESULT_FAILURE(ResultPathNotFound, GetFs().GetFileTimeStampRaw(&fileTimeStamp, fileName.c_str()));
}

}
}
}
