﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

class FriendsFriendRequest : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsFriendRequest, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    for (int i = 0; i < 3; i++)
    {
        nnt::friends::CleanupRelationship(s_Users[i]);
    }
}

TEST_F(FriendsFriendRequest, ReceptionRejection)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::UserSetting setting;
        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[1]));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&context, false));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultFriendRequestReceptionNotAllowed, context.GetResult());
    }
    {
        nn::friends::UserSetting setting;
        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[1]));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsFriendRequest, RequestType1)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::RequestType requestTypes[] =
    {
        nn::friends::RequestType_Faced,
        nn::friends::RequestType_FriendCode,
        nn::friends::RequestType_Facebook,
        nn::friends::RequestType_Twitter,
    };

    for (auto requestType : requestTypes)
    {
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1], requestType));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        nn::friends::FriendRequest requests[2];
        int count = 0;

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[1]);

            EXPECT_FALSE(requests[1].IsValid());
        }

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[0]);

            EXPECT_FALSE(requests[1].IsValid());
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            nn::friends::FriendDetailedInfo info;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[1], s_AccountIds[0]));

            EXPECT_EQ(info.GetRequestType(), requestType);
            EXPECT_EQ(info.GetAccountId(), s_AccountIds[0]);
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFriendRequest, RequestType2)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::RequestType requestTypes[] =
    {
        nn::friends::RequestType_Faced,
        nn::friends::RequestType_InApp,
    };

    for (auto requestType : requestTypes)
    {
        nn::friends::ApplicationInfo appInfo = {{0xFFFFFFFF00000000}, 0xFFFFFFFF00000000};
        nn::friends::InAppScreenName inAppScreenName = {"InAppName1", {"ja"}};
        nn::friends::InAppScreenName myInAppScreenName = {"InAppName0", {"ja"}};

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1], requestType,
                appInfo, inAppScreenName, myInAppScreenName));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        nn::friends::FriendRequest requests[2];
        int count = 0;

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[1]);
            EXPECT_EQ(requests[0].GetRouteInfo().appInfo.appId, appInfo.appId);
            EXPECT_EQ(requests[0].GetRouteInfo().appInfo.presenceGroupId, appInfo.presenceGroupId);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.name, inAppScreenName.name);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.language.string, inAppScreenName.language.string);

            EXPECT_FALSE(requests[1].IsValid());
        }

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[0]);
            EXPECT_EQ(requests[0].GetRouteInfo().appInfo.appId, appInfo.appId);
            EXPECT_EQ(requests[0].GetRouteInfo().appInfo.presenceGroupId, appInfo.presenceGroupId);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.name, myInAppScreenName.name);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.language.string, myInAppScreenName.language.string);

            EXPECT_FALSE(requests[1].IsValid());
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            nn::friends::FriendDetailedInfo info;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[1], s_AccountIds[0]));

            EXPECT_EQ(info.GetRequestType(), requestType);
            EXPECT_EQ(info.GetAccountId(), s_AccountIds[0]);
            EXPECT_EQ(info.GetRequestRouteInfo().appInfo.appId, appInfo.appId);
            EXPECT_EQ(info.GetRequestRouteInfo().appInfo.presenceGroupId, appInfo.presenceGroupId);
            EXPECT_STREQ(info.GetRequestRouteInfo().name.name, myInAppScreenName.name);
            EXPECT_STREQ(info.GetRequestRouteInfo().name.language.string, myInAppScreenName.language.string);
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFriendRequest, RequestType3)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::RequestType requestTypes[] =
    {
        nn::friends::RequestType_FriendCandidate,
    };

    for (auto requestType : requestTypes)
    {
        nn::friends::ExternalApplicationCatalogId catalogId = {{0x77187a302bb9fcd4, 0xf718aa65e0617d6d}};
        nn::friends::InAppScreenName inAppScreenName = {"InAppName1", {"ja"}};
        nn::friends::InAppScreenName myInAppScreenName = {"InAppName0", {"ja"}};

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1], requestType,
                catalogId, inAppScreenName, myInAppScreenName));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        nn::friends::FriendRequest requests[2];
        int count = 0;

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[1]);
            EXPECT_EQ(requests[0].GetExternalApplicationCatalogId().value[0], catalogId.value[0]);
            EXPECT_EQ(requests[0].GetExternalApplicationCatalogId().value[1], catalogId.value[1]);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.name, inAppScreenName.name);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.language.string, inAppScreenName.language.string);

            EXPECT_FALSE(requests[1].IsValid());
        }

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[0]);
            EXPECT_EQ(requests[0].GetExternalApplicationCatalogId().value[0], catalogId.value[0]);
            EXPECT_EQ(requests[0].GetExternalApplicationCatalogId().value[1], catalogId.value[1]);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.name, myInAppScreenName.name);
            EXPECT_STREQ(requests[0].GetRouteInfo().name.language.string, myInAppScreenName.language.string);

            EXPECT_FALSE(requests[1].IsValid());
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            nn::friends::FriendDetailedInfo info;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[1], s_AccountIds[0]));

            EXPECT_EQ(info.GetRequestType(), requestType);
            EXPECT_EQ(info.GetAccountId(), s_AccountIds[0]);
            EXPECT_EQ(info.GetExternalApplicationCatalogId().value[0], catalogId.value[0]);
            EXPECT_EQ(info.GetExternalApplicationCatalogId().value[1], catalogId.value[1]);
            EXPECT_STREQ(info.GetRequestRouteInfo().name.name, myInAppScreenName.name);
            EXPECT_STREQ(info.GetRequestRouteInfo().name.language.string, myInAppScreenName.language.string);
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFriendRequest, RequestType4)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::RequestType requestTypes[] =
    {
        nn::friends::RequestType_3Ds,
        nn::friends::RequestType_Nnid,
    };

    for (auto requestType : requestTypes)
    {
        nn::friends::MiiName miiName = {"MiiName1"};
        nn::friends::MiiName myMiiName = {"MiiName0"};
        nn::friends::MiiImageUrlParam miiImageUrlParam = {"2zbm7pmiozo1s"};
        nn::friends::MiiImageUrlParam myMiiImageUrlParam = {"2zbm7pmiozo1s"};

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1], requestType,
                miiName, miiImageUrlParam, myMiiName, myMiiImageUrlParam));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        nn::friends::FriendRequest requests[2];
        int count = 0;

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[1]);
            EXPECT_STREQ(requests[0].GetMiiName().value, miiName.value);
            EXPECT_STREQ(requests[0].GetMiiImageUrlParam().value, miiImageUrlParam.value);

            EXPECT_FALSE(requests[1].IsValid());
        }

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);

            EXPECT_TRUE(requests[0].IsValid());
            EXPECT_EQ(requests[0].GetRequestType(), requestType);
            EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[0]);
            EXPECT_STREQ(requests[0].GetMiiName().value, myMiiName.value);
            EXPECT_STREQ(requests[0].GetMiiImageUrlParam().value, myMiiImageUrlParam.value);

            EXPECT_FALSE(requests[1].IsValid());
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            nn::friends::FriendDetailedInfo info;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[1], s_AccountIds[0]));

            EXPECT_EQ(info.GetRequestType(), requestType);
            EXPECT_EQ(info.GetAccountId(), s_AccountIds[0]);
            EXPECT_STREQ(info.GetMiiName().value, myMiiName.value);
            EXPECT_STREQ(info.GetMiiImageUrlParam().value, myMiiImageUrlParam.value);
        }
        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFriendRequest, Send1)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 再送
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultAlreadyRequested, context.GetResult());
    }
    // 存在しないユーザー
    {
        nn::account::NetworkServiceAccountId accountId = {};

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], accountId,
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNetworkServiceAccountNotExistsInFriendRequest, context.GetResult());
    }
    // 自分
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultOwnNetworkServiceAccountSpecified, context.GetResult());
    }
}

TEST_F(FriendsFriendRequest, Cancel)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendRequest requests[2];
    int count = 0;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 1);
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFriendRequest(&context, s_Users[0], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 同じフレンド申請をもう一度取り下げてみる。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFriendRequest(&context, s_Users[0], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultRequestStateAlreadyChanged, context.GetResult());
    }
}

TEST_F(FriendsFriendRequest, Send2)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    // 新着順になるので、後者から送信する。
    for (int i = 2; i >= 1; i--)
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[i],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        if (i > 1)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));
        }
    }
}

TEST_F(FriendsFriendRequest, Read)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendRequest sentRequests[2];
    int count = 0;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            sentRequests, s_Users[0], 0, NN_ARRAY_SIZE(sentRequests), nn::friends::RequestListType_Sent));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 2);

        EXPECT_EQ(sentRequests[0].GetAccountId(), s_AccountIds[1]);
        EXPECT_EQ(sentRequests[1].GetAccountId(), s_AccountIds[2]);

        nnt::friends::Dump(sentRequests, count, nn::friends::RequestListType_Sent);
    }

    for (int i = 1; i <= 2; i++)
    {
        nn::friends::FriendRequest receivedRequests[2];

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                receivedRequests, s_Users[i], 0, NN_ARRAY_SIZE(receivedRequests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);
        }

        EXPECT_EQ(receivedRequests[0].GetRequestId().value, sentRequests[i - 1].GetRequestId().value);
        EXPECT_FALSE(receivedRequests[0].IsRead());

        nnt::friends::Dump(receivedRequests, count, nn::friends::RequestListType_Received);

        int unreadCount;
        int readCount;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetReceivedFriendRequestCount(&unreadCount, &readCount, s_Users[i]));

        EXPECT_EQ(unreadCount, 1);
        EXPECT_EQ(readCount, 0);

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::ReadFriendRequest(&context, s_Users[i], receivedRequests[0].GetRequestId()));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

        {
            nn::friends::AsyncContext context;
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
                receivedRequests, s_Users[i], 0, NN_ARRAY_SIZE(receivedRequests), nn::friends::RequestListType_Received));

            nn::os::SystemEvent completionEvent;
            NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

            completionEvent.Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

            ASSERT_EQ(count, 1);
        }

        EXPECT_TRUE(receivedRequests[0].IsRead());

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetReceivedFriendRequestCount(&unreadCount, &readCount, s_Users[i]));

        EXPECT_EQ(unreadCount, 0);
        EXPECT_EQ(readCount, 1);
    }
}

TEST_F(FriendsFriendRequest, Accept)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendRequest requests[2];
    int count = 0;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 1);
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 同じフレンド申請をもう一度承認してみる。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AcceptFriendRequest(&context, s_Users[1], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultRequestStateAlreadyChanged, context.GetResult());
    }

    {
        nn::account::NetworkServiceAccountId accountId;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, &accountId, s_Users[1], 0, 1, filter));

        ASSERT_EQ(count, 1);
        EXPECT_EQ(accountId, requests[0].GetAccountId());
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[1], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 0);
    }
}

TEST_F(FriendsFriendRequest, Reject)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendRequest requests[2];
    int count = 0;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[2], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 1);
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::RejectFriendRequest(&context, s_Users[2], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 同じフレンド申請をもう一度拒否してみる。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::RejectFriendRequest(&context, s_Users[2], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultRequestStateAlreadyChanged, context.GetResult());
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[2], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Received));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 0);
    }
}
