﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;

    nn::Bit8 s_Image1[nn::friends::ProfileImageSizeMax] = {};
    nn::Bit8 s_Image2[nn::friends::ProfileImageSizeMax] = {};

    size_t s_ImageSize1 = 0;
    size_t s_ImageSize2 = 0;
}

namespace
{
    void LoadProfileImages() NN_NOEXCEPT
    {
        static char s_MountRomCacheBuffer[4 * 1024];

        size_t cacheSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));

        NN_ABORT_UNLESS_LESS_EQUAL(cacheSize, sizeof (s_MountRomCacheBuffer));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("rom", s_MountRomCacheBuffer, sizeof (s_MountRomCacheBuffer)));

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount("rom");
        };

        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/user1.jpg", nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&s_ImageSize1, handle, 0, s_Image1, sizeof (s_Image1)));
        }
        {
            nn::fs::FileHandle handle = {};
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, "rom:/user2.jpg", nn::fs::OpenMode_Read));

            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(&s_ImageSize2, handle, 0, s_Image2, sizeof (s_Image2)));
        }
    }
}

class FriendsFacedFriendRequest : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);

        LoadProfileImages();
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsFacedFriendRequest, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nnt::friends::CleanupRelationship(s_Users[0]);

    nn::friends::UserSetting setting;
    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[1]));

    nn::friends::AsyncContext context;
    NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&context, true));

    nn::os::SystemEvent completionEvent;
    NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

    completionEvent.Wait();

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
}

TEST_F(FriendsFacedFriendRequest, OwnError)
{
    nn::friends::FacedFriendRequestRegistrationKey key = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[0]));

    nn::account::Nickname nickname = {"User0"};
    NNT_EXPECT_RESULT_FAILURE(nn::friends::ResultOwnFacedFriendRequestRegistrationKey,
        nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
}

TEST_F(FriendsFacedFriendRequest, BrokenError)
{
    nn::friends::FacedFriendRequestRegistrationKey key = {};
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[0]));

    key.value[63]++;

    nn::account::Nickname nickname = {"User0"};
    NNT_EXPECT_RESULT_FAILURE(nn::friends::ResultFacedFriendRequestRegistrationKeyBroken,
        nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
}

TEST_F(FriendsFacedFriendRequest, AddAndCancel)
{
    nn::friends::DaemonSuspension suspension;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SuspendDaemon(&suspension));

    nn::account::Nickname nickname1 = {"User1"};

    nn::account::Nickname nickname2 = {"User2"};
    nn::friends::InAppScreenName inAppScreenName2 = {"InAppName2", {"ja"}};
    nn::friends::InAppScreenName myInAppScreenName2 = {"InAppName0", {"ja"}};

    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[1]));

        NNT_EXPECT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname1, s_Image1, s_ImageSize1));
    }
    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[2]));

        NNT_EXPECT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname2, s_Image2, s_ImageSize2,
            inAppScreenName2, myInAppScreenName2));
    }

    // 別の UID を指定することで、内部のメモリキャッシュを更新する。
    for (int i = 0; i < s_UserCount; i++)
    {
        nn::friends::FriendRequest requests[3];
        int count = 0;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[i], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Faced));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    nn::friends::Url imageUrls[2] = {};

    {
        nn::friends::FriendRequest requests[3];
        int count = 0;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Faced));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 2);

        nnt::friends::Dump(requests, count, nn::friends::RequestListType_Faced);

        EXPECT_TRUE(requests[0].IsValid());
        EXPECT_EQ(requests[0].GetAccountId(), s_AccountIds[1]);
        EXPECT_STREQ(requests[0].GetNickname().name, nickname1.name);

        EXPECT_TRUE(requests[1].IsValid());
        EXPECT_EQ(requests[1].GetAccountId(), s_AccountIds[2]);
        EXPECT_STREQ(requests[1].GetNickname().name, nickname2.name);
        EXPECT_STREQ(requests[1].GetRouteInfo().name.name, inAppScreenName2.name);
        EXPECT_STREQ(requests[1].GetRouteInfo().name.language.string, inAppScreenName2.language.string);

        EXPECT_FALSE(requests[2].IsValid());

        NNT_ASSERT_RESULT_SUCCESS(requests[0].GetProfileImageUrl(&imageUrls[0], nn::friends::ImageSize_Standard));
        NNT_ASSERT_RESULT_SUCCESS(requests[1].GetProfileImageUrl(&imageUrls[1], nn::friends::ImageSize_Standard));
    }

    static char s_Buffer[nn::friends::ProfileImageSizeMax];
    size_t size = 0;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestProfileImage(&size,
        imageUrls[0].value, s_Buffer, sizeof (s_Buffer)));

    EXPECT_TRUE(std::memcmp(s_Buffer, s_Image1, size) == 0);
    EXPECT_EQ(size, s_ImageSize1);

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestProfileImage(&size,
        s_Users[0], s_AccountIds[1], s_Buffer, sizeof (s_Buffer)));

    EXPECT_TRUE(std::memcmp(s_Buffer, s_Image1, size) == 0);
    EXPECT_EQ(size, s_ImageSize1);

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestProfileImage(&size,
        imageUrls[1].value, s_Buffer, sizeof (s_Buffer)));

    EXPECT_TRUE(std::memcmp(s_Buffer, s_Image2, size) == 0);
    EXPECT_EQ(size, s_ImageSize2);

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestProfileImage(&size,
        s_Users[0], s_AccountIds[2], s_Buffer, sizeof (s_Buffer)));

    EXPECT_TRUE(std::memcmp(s_Buffer, s_Image2, size) == 0);
    EXPECT_EQ(size, s_ImageSize2);

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFacedFriendRequest(s_Users[0], s_AccountIds[1]));
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFacedFriendRequest(s_Users[0], s_AccountIds[2]));

    {
        nn::friends::FriendRequest requests[3];
        int count = 0;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Faced));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 0);
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFacedFriendRequest, BgSend)
{
    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[1]));

        nn::account::Nickname nickname = {"User1"};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
    }

    for (int i = 0; i <= 20; i++)
    {
        ASSERT_LT(i, 20);

        nn::friends::FriendRequest requests[3];
        int count = 0;

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Faced));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        if (count == 0)
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    nn::friends::FriendRequest requests[2];
    int count = 0;

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendRequestList(&context, &count,
            requests, s_Users[0], 0, NN_ARRAY_SIZE(requests), nn::friends::RequestListType_Sent));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        ASSERT_EQ(count, 1);
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::CancelFriendRequest(&context, s_Users[0], requests[0].GetRequestId()));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsFacedFriendRequest, Resend)
{
    nn::friends::DaemonSuspension suspension;
    NNT_ASSERT_RESULT_SUCCESS(nn::friends::SuspendDaemon(&suspension));

    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[1]));

        nn::account::Nickname nickname = {"User1"};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::ResendFacedFriendRequest(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 送信後に再送要求
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::ResendFacedFriendRequest(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNetworkServiceAccountNotExistsGeneral, context.GetResult());
    }

    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[1]));

        nn::account::Nickname nickname = {"User1"};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::ResendFacedFriendRequest(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultAlreadyRequested, context.GetResult());
    }

    {
        nn::friends::FacedFriendRequestRegistrationKey key = {};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFacedFriendRequestRegistrationKey(&key, s_Users[1]));

        nn::account::Nickname nickname = {"User1"};
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::AddFacedFriendRequest(s_Users[0], key, nickname, s_Image1, s_ImageSize1));
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::BlockUser(&context, s_Users[0], s_AccountIds[1],
            nn::friends::BlockReason_BadFriendRequest));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::ResendFacedFriendRequest(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultBlockedByMe, context.GetResult());
    }

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UnblockUser(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::UserSetting setting;
        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[1]));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&context, false));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::ResendFacedFriendRequest(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultFriendRequestReceptionNotAllowed, context.GetResult());
    }
    {
        nn::friends::UserSetting setting;
        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[1]));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFriendRequestReception(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
} // NOLINT(impl/function_size)
