﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testFriends_Common.h"

#include <nn/image/image_JpegDecoder.h>

namespace
{
    nn::account::Uid s_Users[nn::account::UserCountMax] = {};
    nn::account::NetworkServiceAccountId s_AccountIds[nn::account::UserCountMax] = {};
    int s_UserCount = 0;
}

namespace
{
    void ChangePresencePermission(const nn::account::Uid& user,
        nn::friends::PresencePermission permission) NN_NOEXCEPT
    {
        nn::friends::UserSetting setting;

        NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(user));

        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangePresencePermission(&context, permission));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    void UpdatePresence(const nn::account::Uid& user,
        const char* pValue1, const char* pValue2, const char* pDescription, bool isOnlinePlay) NN_NOEXCEPT
    {
        nn::friends::UserPresence presence;

        NNT_ASSERT_RESULT_SUCCESS(presence.Initialize(user));

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

        presence.SetAppValue("key1", pValue1);
        presence.SetAppValue("key2", pValue2);
        presence.SetDescription(pDescription);

NN_PRAGMA_POP_WARNINGS

        if (isOnlinePlay)
        {
            presence.DeclareOpenOnlinePlaySession();
        }
        else
        {
            presence.DeclareCloseOnlinePlaySession();
        }

        NNT_ASSERT_RESULT_SUCCESS(presence.Commit());

        nn::friends::UserPresenceView presenceView;

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetUserPresenceView(&presenceView, user));

        if (isOnlinePlay)
        {
            EXPECT_EQ(presenceView.GetStatus(), nn::friends::PresenceStatus_OnlinePlay);
        }
        else
        {
            EXPECT_EQ(presenceView.GetStatus(), nn::friends::PresenceStatus_Online);
        }

        EXPECT_STREQ(presenceView.GetDescription(), pDescription);
    }

    void WaitForPresenceUpdated(nn::friends::NotificationQueue& queue) NN_NOEXCEPT
    {
        bool isFriendListUpdated = false;

        while (NN_STATIC_CONDITION(true))
        {
            if (!queue.GetSystemEvent()->TimedWait(nn::TimeSpan::FromSeconds(30)))
            {
                // FriendPresenceUpdated 通知は FriendListUpdated 通知にマージされている。
                ASSERT_TRUE(isFriendListUpdated);
                return;
            }

            queue.GetSystemEvent()->Clear();

            while (NN_STATIC_CONDITION(true))
            {
                nn::friends::NotificationInfo notification = {};

                nn::Result result = queue.Pop(&notification);

                if (nn::friends::ResultNotificationNotFound::Includes(result))
                {
                    break;
                }
                NNT_ASSERT_RESULT_SUCCESS(result);

                if (notification.type == nn::friends::NotificationType_FriendPresenceUpdated)
                {
                    return;
                }
                if (notification.type == nn::friends::NotificationType_FriendListUpdated)
                {
                    isFriendListUpdated = true;
                }
            }
        }
    }
}

class FriendsFriend : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

        nnt::friends::LoadAccounts(&s_UserCount, s_Users, s_AccountIds, NN_ARRAY_SIZE(s_Users));
        NN_ABORT_UNLESS_GREATER_EQUAL(s_UserCount, 3);

        nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }
};

TEST_F(FriendsFriend, CleanupRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    for (int i = 0; i < 3; i++)
    {
        nnt::friends::CleanupRelationship(s_Users[i]);
    }
}

TEST_F(FriendsFriend, CreateRelationship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    // 0 - 1
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[1],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[1], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1200));

    // 0 - 2
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[0], s_AccountIds[2],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SendFriendRequest(&context, s_Users[2], s_AccountIds[0],
            nn::friends::RequestType_FriendCode));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(FriendsFriend, Sync)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncFriendList(&context, s_Users[0]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        nn::friends::Friend friends[3];
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, s_Users[0], 0, NN_ARRAY_SIZE(friends), filter));

        ASSERT_EQ(count, 2);

        EXPECT_TRUE(friends[0].IsValid());
        EXPECT_EQ(friends[0].GetAccountId(), s_AccountIds[2]);

        EXPECT_TRUE(friends[1].IsValid());
        EXPECT_EQ(friends[1].GetAccountId(), s_AccountIds[1]);

        EXPECT_FALSE(friends[2].IsValid());

        nnt::friends::Dump(friends, count);

        for (int c = 0; c < count; c++)
        {
            nn::friends::FriendDetailedInfo info;

            EXPECT_FALSE(info.IsValid());
            NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[0], friends[c].GetAccountId()));
            EXPECT_TRUE(info.IsValid());

            NN_LOG("[%d]\n", c);

            nnt::friends::Dump(info);
        }

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendCount(&count, s_Users[0], filter));
        EXPECT_EQ(count, 2);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetNewlyFriendCount(&count, s_Users[0]));
        EXPECT_EQ(count, 2);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DropFriendNewlyFlags(s_Users[0]));

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetNewlyFriendCount(&count, s_Users[0]));
        EXPECT_EQ(count, 0);
    }

    for (int i = 1; i <= 2; i++)
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::SyncFriendList(&context, s_Users[i]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        nn::friends::Friend friends[2];
        int count = 0;
        nn::friends::FriendFilter filter = {};

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, s_Users[i], 0, NN_ARRAY_SIZE(friends), filter));

        ASSERT_EQ(count, 1);

        EXPECT_TRUE(friends[0].IsValid());
        EXPECT_EQ(friends[0].GetAccountId(), s_AccountIds[0]);

        EXPECT_FALSE(friends[1].IsValid());

        nnt::friends::Dump(friends, count);

        nn::friends::FriendDetailedInfo info;

        EXPECT_FALSE(info.IsValid());
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendDetailedInfo(&info, s_Users[i], friends[0].GetAccountId()));
        EXPECT_TRUE(info.IsValid());

        nnt::friends::Dump(info);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendCount(&count, s_Users[i], filter));
        EXPECT_EQ(count, 1);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetNewlyFriendCount(&count, s_Users[i]));
        EXPECT_EQ(count, 1);

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DropFriendNewlyFlags(s_Users[i]));

        NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetNewlyFriendCount(&count, s_Users[i]));
        EXPECT_EQ(count, 0);
    }
} // NOLINT(impl/function_size)

TEST_F(FriendsFriend, FavoriteFlag)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::Friend friends[3];
    int count = 0;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, s_Users[0], 0, NN_ARRAY_SIZE(friends), filter));

    ASSERT_EQ(count, 2);

    filter.isFavoriteOnly = true;

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendCount(&count, s_Users[0], filter));
    EXPECT_EQ(count, 0);

    nn::friends::FriendSetting setting;

    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0], s_AccountIds[1]));

    EXPECT_FALSE(setting.GetFavoriteFlag());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFavoriteFlag(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    EXPECT_TRUE(setting.GetFavoriteFlag());

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendCount(&count, s_Users[0], filter));
    EXPECT_EQ(count, 1);
}

TEST_F(FriendsFriend, OnlineNotificationFlag)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::Friend friends[3];
    int count = 0;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, s_Users[0], 0, NN_ARRAY_SIZE(friends), filter));

    ASSERT_EQ(count, 2);

    nn::friends::FriendSetting setting;

    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0], s_AccountIds[1]));

    EXPECT_FALSE(setting.GetOnlineNotificationFlag());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeOnlineNotificationFlag(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

    EXPECT_TRUE(setting.GetOnlineNotificationFlag());
}

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_DEPRECATED_DECLARATIONS

TEST_F(FriendsFriend, UpdatePresence)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    ASSERT_NO_FATAL_FAILURE(ChangePresencePermission(s_Users[0], nn::friends::PresencePermission_Friends));

    nn::friends::NotificationQueue queue1;
    nn::friends::NotificationQueue queue2;

    NNT_ASSERT_RESULT_SUCCESS(queue1.Initialize(s_Users[1]));
    NNT_ASSERT_RESULT_SUCCESS(queue2.Initialize(s_Users[2]));

    nn::account::UserHandle handles[3] = {};

    for (int i = 0; i < 3; i++)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::OpenUser(&handles[i], s_Users[i]));
    }

    NN_UTIL_SCOPE_EXIT
    {
        for (int i = 0; i < 3; i++)
        {
            nn::account::CloseUser(handles[i]);
        }
    };

    // Open イベント待ち。
    ASSERT_NO_FATAL_FAILURE(WaitForPresenceUpdated(queue1));
    ASSERT_NO_FATAL_FAILURE(WaitForPresenceUpdated(queue2));

    {
        ASSERT_NO_FATAL_FAILURE(UpdatePresence(s_Users[0], "valueA1", "valueB1", "description1", true));
        ASSERT_NO_FATAL_FAILURE(WaitForPresenceUpdated(queue1));

        nn::friends::Friend f1;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UpdateFriendInfo(&f1, s_Users[1], &s_AccountIds[0], 1));

        EXPECT_EQ(f1.GetPresence().GetStatus(), nn::friends::PresenceStatus_OnlinePlay);
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key1"), "valueA1");
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key2"), "valueB1");
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key3"), "");
        EXPECT_STREQ(f1.GetPresence().GetDescription(), "description1");

        ASSERT_NO_FATAL_FAILURE(WaitForPresenceUpdated(queue2));

        nn::friends::Friend f2;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UpdateFriendInfo(&f2, s_Users[2], &s_AccountIds[0], 1));

        EXPECT_EQ(f2.GetPresence().GetStatus(), nn::friends::PresenceStatus_OnlinePlay);
        EXPECT_STREQ(f2.GetPresence().GetAppValue("key1"), "valueA1");
        EXPECT_STREQ(f2.GetPresence().GetAppValue("key2"), "valueB1");
        EXPECT_STREQ(f2.GetPresence().GetAppValue("key3"), "");
        EXPECT_STREQ(f2.GetPresence().GetDescription(), "description1");
    }
    {
        ASSERT_NO_FATAL_FAILURE(ChangePresencePermission(s_Users[0], nn::friends::PresencePermission_Self));
        ASSERT_NO_FATAL_FAILURE(UpdatePresence(s_Users[0], "valueA2", "valueB2", "description2", true));

        // 通知は来ないので待機しない。
    }
    {
        ASSERT_NO_FATAL_FAILURE(ChangePresencePermission(s_Users[0], nn::friends::PresencePermission_FavoriteFriends));
        ASSERT_NO_FATAL_FAILURE(UpdatePresence(s_Users[0], "valueA3", "valueB3", "description3", false));
        ASSERT_NO_FATAL_FAILURE(WaitForPresenceUpdated(queue1));

        nn::friends::Friend f1;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::UpdateFriendInfo(&f1, s_Users[1], &s_AccountIds[0], 1));

        EXPECT_EQ(f1.GetPresence().GetStatus(), nn::friends::PresenceStatus_Online);
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key1"), "valueA3");
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key2"), "valueB3");
        EXPECT_STREQ(f1.GetPresence().GetAppValue("key3"), "");
        EXPECT_STREQ(f1.GetPresence().GetDescription(), "description3");
    }
}

NN_PRAGMA_POP_WARNINGS

TEST_F(FriendsFriend, RequestBgSync)
{
    for (int i = 0; i < 3; i++)
    {
        // 2 回ずつリクエストを出してみる。
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::RequestSyncFriendList(s_Users[i]));
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::RequestSyncFriendList(s_Users[i]));
    }
}

TEST_F(FriendsFriend, NotFriend)
{
    nn::account::NetworkServiceAccountId accountId = {};

    nn::friends::FriendDetailedInfo info = {};
    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, nn::friends::GetFriendDetailedInfo(&info, s_Users[0], accountId));

    nn::friends::FriendSetting setting = {};
    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, setting.Initialize(s_Users[0], accountId));
}

TEST_F(FriendsFriend, ProfileImage)
{
    nn::friends::Friend friends[3];
    int count = 0;
    nn::friends::FriendFilter filter = {};

    NNT_ASSERT_RESULT_SUCCESS(nn::friends::GetFriendList(&count, friends, s_Users[0], 0, NN_ARRAY_SIZE(friends), filter));

    ASSERT_EQ(count, 2);

    static nn::Bit8 s_Image[nn::friends::ProfileImageSizeMax];
    size_t size;

    for (int i = 0; i < count; i++)
    {
        NNT_ASSERT_RESULT_SUCCESS(friends[i].GetProfileImage(&size, s_Image, sizeof (s_Image)));

        nn::image::JpegDecoder decorder;

        decorder.SetImageData(s_Image, size);

        ASSERT_EQ(decorder.Analyze(), nn::image::JpegStatus_Ok);

        nn::image::Dimension dim = decorder.GetAnalyzedDimension();

        ASSERT_EQ(dim.width, 256);
        ASSERT_EQ(dim.height, 256);
    }
}

TEST_F(FriendsFriend, DeleteFriendship)
{
    ASSERT_NO_FATAL_FAILURE(nnt::friends::ConnectNetwork());

    nn::friends::FriendSetting setting = {};
    NNT_ASSERT_RESULT_SUCCESS(setting.Initialize(s_Users[0], s_AccountIds[1]));

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    // 同じ相手をもう一度フレンド削除してみる。
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(nn::friends::DeleteFriend(&context, s_Users[0], s_AccountIds[1]));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, context.GetResult());
    }

    NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, setting.DropNewlyFlag());

    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeFavoriteFlag(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, context.GetResult());
    }
    {
        nn::friends::AsyncContext context;
        NNT_ASSERT_RESULT_SUCCESS(setting.ChangeOnlineNotificationFlag(&context, true));

        nn::os::SystemEvent completionEvent;
        NNT_ASSERT_RESULT_SUCCESS(context.GetSystemEvent(&completionEvent));

        completionEvent.Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::friends::ResultNotFriend, context.GetResult());
    }
}
